(function () {
  "use strict";

  var global = window;
  var AJS = global.AJS;

  function updateProgress(idPrefix, result) {
    AJS.$("#" + idPrefix + "-in-progress-message").text(result.progressMessage);
    AJS.$("#" + idPrefix + "-in-progress-value").text(
      result.progressPercentage,
    );

    var progressValue = result.progressPercentage / 100;
    AJS.progressBars.update("#" + idPrefix + "-progress-bar", progressValue);

    if (hasErrors(result)) {
      showProgressErrorMessage(idPrefix, result);
    }

    if (hasWarnings(result)) {
      showWarningMessage(result);
    }

    return result;
  }

  function showProgressErrorMessage(idPrefix, result) {
    setMessages(result.errors, AJS.$("#" + idPrefix + "-errors"));
  }

  function showWarningMessage() {
    AJS.$("#stp-zip-warning").show();
  }

  function isAvailable(result) {
    return result.hasOwnProperty("available") && result.available;
  }

  function hasErrors(result) {
    return (
      result.hasOwnProperty("errors") &&
      result.errors &&
      result.errors.length > 0
    );
  }

  function hasWarnings(result) {
    return (
      result.hasOwnProperty("warnings") &&
      result.warnings &&
      result.warnings.length > 0
    );
  }

  function whenAvailable(idPrefix, result, onComplete) {
    AJS.$("#stp-zip-warning").hide();
    AJS.$("#" + idPrefix + "-in-progress").hide();
    AJS.$("#" + idPrefix + "-start-again").show();

    if (hasErrors(result)) {
      return whenError(result);
    }

    return whenSuccess(idPrefix, result, onComplete);
  }

  function whenError() {
    // Display a different link text if there's an error occurred when sending the support request
    AJS.$("#start-again-link").text(
      AJS.I18n.getText("stp.create.support.request.error.back.to.form"),
    );
  }

  function whenSuccess(idPrefix, result, onComplete) {
    var withWarning = hasWarnings(result);
    var doneClassName = withWarning
      ? "stp-done-message-warning"
      : "stp-done-message-success";
    var doneWrapper = AJS.$("#" + idPrefix + "-done-message-wrapper");
    doneWrapper.addClass(doneClassName);

    var doneHeaderText = withWarning
      ? AJS.I18n.getText("stp.create.support.zip.success.heading.warning")
      : AJS.I18n.getText("stp.create.support.zip.success.heading.success");
    AJS.$("#" + idPrefix + "-done-message-header").text(doneHeaderText);

    var doneMessage = AJS.$("#" + idPrefix + "-done-message");
    doneMessage.text(result.progressMessage);

    if (idPrefix === "stp-request") {
      // Remove the cached form content from local storage once the create support request has been successfully submitted
      localStorage.removeItem("stp-create-support-request-form-data");
    }

    if (onComplete) {
      onComplete(result);
    }

    AJS.$("#" + idPrefix + "-done").show();
  }

  /**
   * Poll the status of a task
   *
   * @param url base URL of JIRA to locate REST api
   * @param idPrefix the prefix used for element ids in the UI associated with the task
   * @param delay seconds between updates
   * @param onComplete function to call when task completes
   * @returns {void|*}
   */
  function pollForProgress(url, idPrefix, delay, onComplete) {
    var pollProgress = function () {
      setTimeout(function () {
        pollForProgress(url, idPrefix, delay, onComplete);
      }, delay * 1000);
    };

    return AJS.$.ajax({
      async: true,
      cacheBoolean: false,
      dataType: "json",
      timeout: 10 * 1000,
      type: "GET",
      url: url,
    })
      .done(function (result) {
        updateProgress(idPrefix, result);

        if (isAvailable(result)) {
          return whenAvailable(idPrefix, result, onComplete);
        }

        pollProgress();
      })
      .fail(function (xhr, textStatus) {
        if (textStatus === "timeout") {
          pollProgress();
        }
      });
  }

  function setMessages(messages, messageContainer) {
    if (messages && messages.length > 0) {
      var ulElem = messageContainer.find("ul");
      ulElem.empty();
      messages.map(function (message) {
        ulElem.append("<li>" + message.body + "</li>");
      });
      messageContainer.show();
    } else {
      messageContainer.hide();
    }
  }

  function createSupportRequestFormData(url, currentUser) {
    var formData = localStorage.getItem("stp-create-support-request-form-data");

    if (formData === null) {
      // Initialize the form data cache into local storage if existing data is not present
      initFormDataFunction(url);
    } else {
      var jsonData = JSON.parse(formData);

      var url = jsonData.requestUrl;
      var contactEmail = jsonData.contactEmail;

      // If base URL or contact email is undefined, we display the option for user to restore previous content
      // This is handy for testing, where multiple product based doesn't have specific base URL or admin email address
      if (
        (typeof url === "undefined" || url === url) &&
        (typeof contactEmail === "undefined" || contactEmail === currentUser)
      ) {
        var banner = AJS.messages.generic({
          id: "create-request-form-content-message",
          body: AJS.I18n.getText(
            "stp.create.support.request.cache.desc",
            "<a href='#' id='restore-content'>restore</a>",
            "<a href='#' id='discard-content'>discard</a>",
          ),
          closeable: false,
        });

        var $form = AJS.$("#form-create-support-request");
        $form.prepend(banner);

        AJS.$("#restore-content").on("click", function (event) {
          removeBannerFromPage(banner, event);

          var subject = jsonData.subject;
          var description = jsonData.description;

          if (typeof contactEmail !== "undefined") {
            AJS.$("#form-create-support-request input[name=contactEmail]").val(
              contactEmail,
            );
          }
          if (typeof subject !== "undefined") {
            AJS.$("#form-create-support-request input[name=subject]").val(
              subject,
            );
          }
          if (typeof description !== "undefined") {
            AJS.$(
              "#form-create-support-request textarea[name=description]",
            ).val(description);
          }

          AJS.trigger("analyticsEvent", {
            name: "stp.create.support.request.form.restored",
          });

          // Initialize the form data cache after restoring the previous content
          initFormDataFunction(url);
        });

        AJS.$("#discard-content").on("click", function (event) {
          removeBannerFromPage(banner, event);
          localStorage.removeItem("stp-create-support-request-form-data");
          // Initialize the form data cache after the previous content is discarded
          initFormDataFunction(url);
        });
      }
    }
  }

  // The function to remove the restore content banner by sliding up (animation) then remove it from DOM
  function removeBannerFromPage(banner, event) {
    event.preventDefault();
    banner.slideUp("normal", function () {
      AJS.$(this).remove();
    });
  }

  function initFormDataFunction(url) {
    // Store the value of the create request form to browser local storage every 3 seconds
    var delay = 3 * 1000;

    setTimeout(function () {
      storeFormData(delay, url);
    }, delay);
  }

  function storeFormData(delay, url) {
    var $form = AJS.$("#form-create-support-request");
    var formArray = $form.serializeArray();

    var formData = {};
    var shouldStoreContent = false;

    for (var count = 0; count < formArray.length; count++) {
      var name = formArray[count].name;
      // Only store the value of the Contact Email, Summary and Description of the problem
      if (
        name === "contactEmail" ||
        name === "subject" ||
        name === "description"
      ) {
        var value = formArray[count].value;

        if (value !== "") {
          formData[name] = value;

          if (name === "description" || name === "subject") {
            // Set the should store flag to true if subject or description is not empty
            shouldStoreContent = true;
          }
        }
      }
    }

    // Only save content if description or subject has some value in it
    if (shouldStoreContent) {
      formData.requestUrl = url;
      localStorage.setItem(
        "stp-create-support-request-form-data",
        JSON.stringify(formData),
      );
    }

    // From here onwards, we call the function to store the data into local storage in loop, based on the delay set
    setTimeout(function () {
      storeFormData(delay, url);
    }, delay);
  }

  function attachSmtpMailBannerAnalytics() {
    // When any of the element inside the banner is clicked, we check if any of the link was clicked and if yes, trigger the analytics event
    AJS.$("#outgoing-mail-server-needed-reminder").on("click", function (e) {
      var smtpBannerEvents = {
        mail_server_link: "stp.create.support.request.mail-server-link.clicked",
        support_link: "stp.create.support.request.support-link.clicked",
        support_zip_link: "stp.create.support.request.support-zip-link.clicked",
      };

      var eventName = e.target.getAttribute("data-event-name");

      if (smtpBannerEvents.hasOwnProperty(eventName)) {
        AJS.trigger("analyticsEvent", {
          name: smtpBannerEvents[eventName],
        });
      }
    });
  }

  /**
   * Use pollForProgress to update the support zip creation progress, and when it completes, get the zip file name and
   * attach it to the download button
   *
   * @param baseUrl baseUrl of JIRA
   * @param taskId id of the support zip creation task
   */
  function setupSupportZipCreationPolling(baseUrl, taskId) {
    setTimeout(function () {
      pollForProgress(
        baseUrl +
          "/rest/troubleshooting/latest/support-zip/legacy/status/" +
          taskId,
        "stp-zip",
        1,
        function (result) {
          if (!result.zipFileName) {
            AJS.$("#stp-zip-file-name-error-message").text(
              "An error occurred getting the name of the zipfile",
            );
          } else {
            var downloadLinkElem = AJS.$("#support-zip-download");

            var downloadZipPath =
              baseUrl +
              "/plugins/servlet/troubleshooting/support-zip/download?file=" +
              result.zipFileName;

            downloadLinkElem.on("click", function (event) {
              event.preventDefault();
              window.location.replace(downloadZipPath);
            });
          }
        },
      );
    }, 500);
  }

  // Export to global scope for the backward compatibility
  global.pollForProgress = pollForProgress;
  global.createSupportRequestFormData = createSupportRequestFormData;
  global.attachSmtpMailBannerAnalytics = attachSmtpMailBannerAnalytics;
  global.setupSupportZipCreationPolling = setupSupportZipCreationPolling;
})();
