(function (angular) {
  angular
    .module("retrospectiveApp")
    .controller("resultController", ResultController);

  ResultController.$inject = [
    "IMAGES_BASE_PATH",
    "HTTP_PATH",
    "$http",
    "$q",
    "$scope",
    "IdeasService",
    "UserService",
    "DiscussService",
    "TopicService",
    "EventService",
    "SettingsService",
    "WorkflowService",
    "DateService",
    "ConfluenceAccessModeService",
    'UtilService',
  ];

  function ResultController(
    IMAGES_BASE_PATH,
    HTTP_PATH,
    $http,
    $q,
    $scope,
    IdeasService,
    UserService,
    DiscussService,
    TopicService,
    EventService,
    SettingsService,
    WorkflowService,
    DateService,
    ConfluenceAccessModeService,
    UtilService
  ) {
    const vm = this;

    this.readOnlyMode = ConfluenceAccessModeService.isReadOnlyModeEnabled();

    vm.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
    vm.userKey = $scope.$parent.vm.userKey;
    vm.uuid = $scope.$parent.vm.uuid;
    vm.settings = SettingsService.settings;
    vm.groupsMap = {};
    vm.actionItems = [];
    vm.retroParticipants =
      $scope.$parent.$parent.$parent.$parent.vm.retrospectiveParticipants;
    vm.filterActionItemsBy = [];
    vm.activeUsers = $scope.$parent.vm.activeUsers;
    vm.colorsColumn = {};
    vm.countActionItems = [];
    this.showDetailView = false;
    this.firstActionItemId = null;

    vm.confluenceUsers = [];
    vm.userSearch = "";
    vm.confluenceUsersResult = [];

    vm.thinkElements = [];
    this.groupCollection = null;

    let cache = "";
    this.creatingConfluenceTasks = false;

    WorkflowService.subscribe(
      (event) => {
        this.readOnlyMode = true;
      },
      ["readOnlyMode"]
    );

    SettingsService.subscribe(
      (event) => {
        this.settings = event.state.settings;
      },
      ["settings"]
    );

    // Watch for filter changes to reinitialize AUI dropdowns
    $scope.$watchCollection('vm.filterActionItemsBy', function(newVal, oldVal) {
      if (newVal !== oldVal) {
        // Close all open dropdowns immediately before DOM changes
        AJS.$('.aui-dropdown2-trigger.aui-dropdown2-active').each(function() {
          AJS.$(this).trigger('aui-button-invoke');
        });
        
        // Give Angular time to update the DOM, then clean up any orphaned dropdown states
        setTimeout(() => {
          // Clean up any remaining active states
          AJS.$('.aui-dropdown2-trigger').each(function() {
            const $trigger = AJS.$(this);
            const dropdownId = $trigger.attr('aria-owns');
            
            if (dropdownId) {
              const $dropdown = AJS.$('#' + dropdownId);
              
              // Remove active state if present
              if ($trigger.hasClass('aui-dropdown2-active')) {
                $trigger.removeClass('aui-dropdown2-active');
              }
              if ($dropdown.length && $dropdown.hasClass('aui-dropdown2-active')) {
                $dropdown.removeClass('aui-dropdown2-active');
                $dropdown.attr('aria-hidden', 'true');
              }
            }
          });
        }, 150);
      }
    });

    $http
      .get(`${HTTP_PATH}/ideas?uuid=${vm.uuid}`)
      .then((response) => {
        vm.thinkElements = response.data;

        DiscussService.subscribe(
          (event) => {
            const asString = JSON.stringify(event.state.groupList);

            // avoid re-render of the same state
            if (asString === cache) {
              return;
            } else {
              cache = asString;
            }

            this.groupCollection = {
              groupList: event.state.groupList,
            };

            this.settings.columns.forEach((col, position) => {
              let count = 0;
              vm.groupsMap[col.columnId] = [];

              this.groupCollection.groupList.forEach((group) => {
                if (group.ideas[0].columnId == col.columnId) {
                  count += group.actionItems.length;
                  group.description = UtilService.decodeHTML(group.description);
                  group.actionItems = group.actionItems && group.actionItems.map(actionItem => {
                    actionItem.description = UtilService.decodeHTML(actionItem.description);
                    return actionItem;
                  })

                  // Initialize isCollapsed to true for grouped topics with multiple ideas
                  // This ensures groups start in collapsed state (showing stacked view)
                  if (group.isCollapsed === undefined && group.grouped && group.ideas && group.ideas.length > 1) {
                    group.isCollapsed = true;
                  }

                  vm.groupsMap[col.columnId].push(group);
                }
              });
              this.filterActionItemsBy[position] = this.filterActionItemsBy[position] || "all";
              this.countActionItems[col.columnId] = count;
            });

            // this validation is to refresh the info in detail view when a issue is created or assigned
            if (
              this.showDetailView &&
              typeof this.detailViewColumnId !== "undefined"
            ) {
              this.detailGroup = vm.groupsMap[this.detailViewColumnId].find(
                (group) => group.id === this.detailGroup.id
              );
              this.detailGroup.description = UtilService.decodeHTML(this.detailGroup.description);
              this.detailGroup.firstActionItem =
                this.detailGroup.actionItems.find(
                  (actionItem) => actionItem.id === this.firstActionItemId
                );
            }
          },
          ["groupList"]
        );
      })
      .catch((err) => {
        console.error(err);
        const flagError = AJS.flag({
          type: "error",
          body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.',
        });
      });

    vm.confluenceUsersResult = UserService.getUserList();
    this.findColor = (columnId) => {
      const column = this.settings.columns.find(
        (col) => col.columnId == columnId
      );
      if (column) {
        return column.color;
      }
      return "#000";
    };

    this.convertionAvailable = (columnId, $index) => {
      const groups = this.groupsMap[columnId];

      if (!groups) {
        return false;
      }

      const actionItems = groups
        .map((group) =>
          this.filterActionItems(
            group.actionItems,
            this.filterActionItemsBy[$index]
          )
        )
        .reduce((acc, cur) => acc.concat(cur), []);

      return actionItems.length > 0;
    };

    function sanitize(input) {
      var output = input
        .replace(/<script[^>]*?>.*?<\/script>/gi, "")
        .replace(/<[\/\!]*?[^<>]*?>/gi, "")
        .replace(/<style[^>]*?>.*?<\/style>/gi, "")
        .replace(/<![\s\S]*?--[ \t\n\r]*>/gi, "");
      return output;
    }

    this.addCloseEvents = () => {
      //event to listen the esc key
      $(document).keyup(function (e) {
        if (e.which === 27) {
          $(".close-details-btn").click();
        }
      });
      //add event to window.parent (iframe not included)
      $(window.parent).on("click", (e) => {
        this.closeDetailView();
      });
    };

    this.orderActionItem = (actionItem) => {
      return actionItem === this.detailGroup.firstActionItem;
    };

    this.closeDetailView = () => {
      AJS.$("#retrospectives-app-header").show();
      $(window.parent).off("click");
      $(document).off("keyup");
      $scope.$parent.vm.activeDetailView(false);
      initializeUsersForActionItems();
    };

    // this function open the detail view
    this.openDetailView = (group, col, actionItemId) => {
      initializeUsersForActionItems();
      this.addCloseEvents();
      // we set all the variables that we're going to need in this view
      AJS.$("#retrospectives-app-header").hide();
      window.parent.parent.scrollTo(0, 0);
      this.showDetailView = true;
      //we call this function from home.controller.js to switch the view
      $scope.$parent.vm.changeTab(null, this.showDetailView);
      this.detailGroup = group;
      this.detailViewColumnName = col.content;
      this.detailViewColumnId = col.columnId;

      //in topics summary view we can't access to actionItemId, so when is undefined
      // we only show the entire group
      if (typeof actionItemId === "undefined") {
        this.detailGroup.firstActionItem = null;
      } else {
        // we create this variable to know what action item was clicked first
        this.detailGroup.firstActionItem = this.detailGroup.actionItems.find(
          (ai) => ai.id === actionItemId
        );
        this.firstActionItemId = this.detailGroup.firstActionItem.id;
      }
    };

    this.convertToTasks = (columnId, $index) => {
      if (this.creatingConfluenceTasks) {
        return;
      }

      this.creatingConfluenceTasks = true;

      const atltk = top.window.document.getElementById("atlassian-token");
      const atltkContent = atltk ? atltk.getAttribute("content") : null;

      top.window.location.href = `${IMAGES_BASE_PATH}/pages/editpage.action?createTasks=true&pageId=${
        $scope.$parent.vm.pageId
      }&settingsId=${btoa(vm.uuid)}&columnId=${btoa(columnId)}&criteria=${btoa(
        this.filterActionItemsBy[$index]
      )}&atl_token=${atltkContent}`;
    };

    const filterActionItems = (
      actionItems = [],
      criteria,
      usingGroups = false
    ) => {
      if (usingGroups) {
        actionItems = actionItems.reduce((a, c) => a.concat(c.actionItems), []);
      }

      switch (criteria) {
        case "all":
          return actionItems.slice();
        case "onlyResolved":
          return actionItems.filter((item) => item.complete);
        case "onlyUnresolved":
          return actionItems.filter((item) => !item.complete);
        case "onlyUnassigned":
          return actionItems.filter((item) => !item.assigneeUser);
        default: // a specific user
          return actionItems.filter(
            (item) => item.assigneeUser && item.assigneeUser.userKey == criteria
          );
      }
    };

    const filterIdeas = (ideas, criteria) => {
      if (ideas === undefined) {
        return;
      }

      switch (criteria) {
        case "all":
          return ideas.slice();
        case "mostVoted":
          return ideas.slice().sort((a, b) => b.votes - a.votes);
        case "leastVoted":
          return ideas.slice().sort((a, b) => a.votes - b.votes);
        default:
          return ideas.slice();
      }
    };

    const collapseGroup = (group) => {
      // Only allow collapse/expand for groups with 2 or more ideas
      if (group.grouped && group.ideas && group.ideas.length > 1) {
        group.isCollapsed = !group.isCollapsed;
      }
    };

    const getVoteColor = (group) => {
      return typeof group !== "undefined" && group.votes > 0
        ? "vote-blue-number"
        : "vote-gray-number";
    };

    const getVoteTextColor = (group) => {
      return typeof group !== "undefined" && group.votes > 0
        ? "vote-text-voted"
        : "vote-text-unvoted";
    };

    const pluralVote = (group) => {
      return typeof group !== "undefined" && group.votes > 0
        ? "plural-word"
        : "";
    };

    const isCurrentUserMod = () => {
      if (vm.moderator) {
        return vm.moderator.userKey === vm.userKey;
      }
      return UserService.getUserByUserKey(vm.userKey).isModerator;
    };

    const assigneeUser = (actionItem, user, code, group) => {
      if (user) {
        const newUser = Object.assign({}, user);
        delete newUser.isParticipant;
        DiscussService.setAssignee(group.id, actionItem.id, newUser);
      } else {
        DiscussService.clearAssignee(group.id, actionItem.id);
      }
      AJS.$(`#trigger-dropdown${code}${vm.uuid}`).trigger("aui-button-invoke");
    };

    const toggleCompleted = (actionItem, currentValue, group) => {
      // If marking as complete, close any open discuss dropdowns
      if (currentValue) {
        // Close all open AUI dropdowns by triggering close on all dropdown triggers
        AJS.$('.aui-dropdown2-trigger').each(function() {
          const $trigger = AJS.$(this);
          if ($trigger.hasClass('aui-dropdown2-active')) {
            $trigger.trigger('aui-button-invoke');
          }
        });
      }
      
      DiscussService.toggleComplete(group.id, actionItem.id).catch((err) => {
        console.error(err);
        const flagError = AJS.flag({
          type: "error",
          body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
        });
      });
    };

    const isUserAssignee = (actionItem, user) => {
      const assignee = actionItem.assigneeUser;
      return assignee && assignee.userName == user.userName;
    };

    /*
         calculate string width in screen
         */
    const hasTitleEllipsis = (text) => {
      const ellipsisHelper = document.getElementById("ellipsis-calc-helper");
      if (!ellipsisHelper) {
        return false;
      }
      ellipsisHelper.innerHTML = DOMPurify.sanitize(UtilService.encodeHTML(text));
      ellipsisHelper.style.fontSize = "20px";
      const textWidth = ellipsisHelper.clientWidth + 1;
      const widthAvailable = $(".title-think").width();
      return textWidth > widthAvailable;
    };

    const getAssigneesForColumn = (column) => {
      const group = vm.groupsMap[column.columnId];

      if (!group) {
        return [];
      }

      const actionItems = group.reduce((accum, topic) => {
        if (topic.actionItems && topic.actionItems.length) {
          return accum.concat(topic.actionItems);
        }
        return accum;
      }, []);

      if (!actionItems.length) {
        return [];
      }

      const keys = [];
      return actionItems.reduce((accum, actionItem) => {
        if (
          actionItem.assigneeUser &&
          keys.indexOf(actionItem.assigneeUser.userKey) < 0
        ) {
          keys.push(actionItem.assigneeUser.userKey);
          return accum.concat(actionItem.assigneeUser);
        }
        return accum;
      }, []);
    };

    const hasActionItems = (colId) => {
      const groups = vm.groupsMap[colId];
      return (
        groups &&
        groups.find((group) => group.actionItems && group.actionItems.length)
      );
    };

    this.hasMoreThanFourColumns = () => {
      return this.settings.columns.length > 4;
    };

    this.init = () => {
      $scope.$parent.vm.showActionItems = true;
      $scope.$parent.vm.showDetailView = false;
      AJS.$("#btn-action-items1").click();
      if (this.hasMoreThanFourColumns()) {
        $("#retro-container").addClass("more-than-four-columns-container");
        $("#retro-container").removeClass("retro-container");
      }
    };

    const initializeUsersForActionItems = () => {
      vm.confluenceUsersResult = vm.retroParticipants;
    };

    const searchForUser = async () => {
      if (!vm.userSearch.trim().length) {
        initializeUsersForActionItems();
        return;
      }
      vm.confluenceUsersResult = await UserService.getUsersBySearch(
        vm.userSearch.trim()
      );

      if (!vm.confluenceUsersResult.length) {
        initializeUsersForActionItems();
      }
    };

    const cleanUserSearch = () => {
      vm.userSearch = "";
      initializeUsersForActionItems();
    };

    const getFilteredUsers = (actionItem) => {
      if (!vm.confluenceUsersResult || !actionItem.assigneeUser) {
        return vm.confluenceUsersResult;
      }

      return vm.confluenceUsersResult.filter((user) => {
        return user.userKey !== actionItem.assigneeUser.userKey;
      });
    };

    vm.getFilteredUsers = getFilteredUsers;
    vm.hasActionItems = hasActionItems;
    vm.getAssigneesForColumn = getAssigneesForColumn;
    vm.hasTitleEllipsis = hasTitleEllipsis;
    vm.collapseGroup = collapseGroup;
    vm.filterActionItems = filterActionItems;
    vm.filterIdeas = filterIdeas;
    vm.getVoteColor = getVoteColor;
    vm.getVoteTextColor = getVoteTextColor;
    vm.pluralVote = pluralVote;
    vm.isCurrentUserMod = isCurrentUserMod;
    vm.isUserAssignee = isUserAssignee;
    vm.assigneeUser = assigneeUser;
    vm.toggleCompleted = toggleCompleted;
    vm.getColumnClass = SettingsService.getColumnClass;
    vm.isExpired = false;
    vm.searchForUser = searchForUser;
    vm.cleanUserSearch = cleanUserSearch;

    $q.when(
      SettingsService.isRetroExpired(vm.uuid),
      (response) => (vm.isExpired = response)
    ).catch((err) => {
      console.error(err);
      const flagError = AJS.flag({
        type: "error",
        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
      });
    });
  }
})(angular);
