import IBaseStep from "../../declarations/IBaseStep";
import wizardModel from "../../models/WizardModel";
import retrospectiveModel from "../../models/RetrospectiveModel";
import retrospectiveService from "../../services/RetrospectiveService";
import navigation from "../../navigation";
import { normalizeString } from "../../utils/stringUtils";

declare const AJS: any;

export type Column = {
    columnId: string;
    content: string;
    color: string;
};

class ColumnCreate implements IBaseStep {
    private $columnList: JQuery;
    private $columnInput: JQuery;
    private $techniqueSelect: JQuery;
    private $SelectInput: string = ".s2id_technique-select";
    private $ResetButton: JQuery;
    private $happyIcon: JQuery;
    private $colorBand: JQuery;
    private editableIndex: number = -1;
    private colorPickerOpened = false;
    private formats: any;
    private formatId: number;
    private $techniqueSelected: JQuery;
    private $happyIconWizardElement: JQuery;
    private $preferenceIconWizardElement: JQuery;
    private $inputColumnEdition:JQuery;

    constructor(private $element: JQuery) {
        this.$inputColumnEdition = $element.find('#column-name-input')
        this.$columnList = $element.find(".column-list");
        this.$columnInput = $element.find(".column-row-input");
        this.$techniqueSelect = $element.find("#technique-select");
        this.$ResetButton = $element.find("#reset-button");
        this.$ResetButton.focusout();
        this.$happyIcon = $element.find("#preference-icon-happy");
        this.$colorBand = $element.find("#task-create");
        this.$techniqueSelected = $element.find("#format-selected");
        this.$happyIconWizardElement = $element.find("#task-create");
        this.$preferenceIconWizardElement = $element.find("#preference-element");
        this.formats = [];
        this.formatId = retrospectiveModel.getFormatId();
        this.updateColumns(this.formatId);
        this.resetColumns([]);
        if (!wizardModel.getEditSettingsAvailable()) {
            this.$happyIcon.attr("class", "preference-icon-happy-deactivated");
            this.$colorBand.css("box-shadow", "none");
            this.$happyIconWizardElement.removeClass("has-hover");
            this.$preferenceIconWizardElement.removeClass("has-hover");
        }
    }

    bindEvents() {
        this.$preferenceIconWizardElement.on("click", this.onNext.bind(this));
        if (!wizardModel.getEditSettingsAvailable()) {
            this.$preferenceIconWizardElement.unbind("click");
        }
        this.$ResetButton.on("click", this.reset.bind(this));
        this.$techniqueSelect.on("change", this.selectTechnique.bind(this));
        this.$techniqueSelect.on("select2-open", this.selectOpen.bind(this));
        this.$techniqueSelect.on("select2-close", this.selectClose.bind(this));
        this.$columnInput.on(
            "submit",
            ".column-name-form",
            this.addNewColumn.bind(this)
        );
        this.$columnInput.on(
            "submit",
            ".column-name-form",
            this.focusOnColumnAdded.bind(this)
        );
        // Real-time validation for new column input
        this.$columnInput.on(
            "input",
            "#column-name-input",
            this.validateNewColumnName.bind(this)
        );
        this.$columnList.on(
            "blur",
            ".column-name-input",
            this.completeColumnEdition.bind(this)
        );
        // Real-time validation for existing column inputs
        this.$columnList.on(
            "input",
            ".column-name-input",
            this.validateColumnName.bind(this)
        );
        this.$columnList.on(
            "blur",
            ".column-color-input",
            this.completeColumnEdition.bind(this)
        );
        this.$columnList.on(
            "submit",
            ".column-name-form",
            this.completeColumnEditionBySubmit.bind(this)
        );
        this.$columnList.on(
            "click",
            ".column-row-edit",
            this.focusColumnInputForEdition.bind(this)
        );
        this.$columnList.on(
            "click",
            ".column-row-delete",
            this.removeColumn.bind(this)
        );

        this.$columnList.on(
            "click",
            ".column-color-input-container",
            this.openColorPicker.bind(this)
        );
        this.$columnList.on(
            "click",
            ".column-color-input",
            this.openColorPicker.bind(this)
        );

        $(window).click(() => {
            $("#color-picker-container").css({display: "none"});
            this.colorPickerOpened = false;
            this.editableIndex = -1;
        });

        $("#scroll").on("scroll", this.moveColorPicker.bind(this));

        $("#color-picker-container").on("click", this.selectColor.bind(this));

        $(".color-rectangle").on("click", this.selectColor.bind(this));
    }

    reset(e) {
        retrospectiveService.getFormats().then((data) => {
            let columns = data.find((format) => format.id === Number(this.formatId))
                .columns;
            retrospectiveModel.setFormatsReset(this.formatId, columns);
            this.resetColumns(columns);
        });
    }

    selectClose(e) {
        $(this.$SelectInput).removeClass("select-open");
    }

    selectOpen(e) {
        $(this.$SelectInput).addClass("select-open");
    }

    selectTechnique(e) {
        this.$inputColumnEdition.val('')
        let formatText = e.added.text;
        this.$techniqueSelected.text(formatText);
        retrospectiveModel.setTechniqueSelected(formatText);
        this.formatId = this.$techniqueSelect.val();
        retrospectiveModel.setFormatId(this.formatId);
        this.updateColumns(this.formatId);
    }

    unbindEvents() {
        this.$columnInput.off();
        this.$columnList.off();
    }

    onLoad(formatId, isEditing, technique) {
        // We make a request to get the formats from backend
        retrospectiveService
            .getFormats()
            .then((data) => {
                this.formats = data;
                this.formatId = !!formatId ? formatId : this.formats[0].id;
                const formatsData = [];
                this.formats.map((format) =>
                    formatsData.push({id: format.id, text: format.name})
                );
                const auxTechnique = this.formats.find(format => format.name == 'What went well')
                const techniqueName = technique || auxTechnique.name

                if (techniqueName){
                    this.$techniqueSelected.text(techniqueName);
                }else {
                    throw new Error("Error getting formats list")
                }
                
                AJS.$("#technique-select").auiSelect2({
                    multiple: false,
                    minimumResultsForSearch: -1,
                    data: formatsData,
                    initSelection: function (element, callback) {
                        let formatId = retrospectiveModel.getFormatId();
                        if (formatId !== null) {
                            callback(formatsData.find(format => format.id === Number(formatId)));
                        }else if (formatsData.length){
                            callback(formatsData.find(format => format.text === auxTechnique.name))
                        }
                    },}).select2('val',this.formatId);

                //We initialize the select2 element
                retrospectiveModel.setFormatId(this.formatId);
                retrospectiveModel.setTechniqueSelected(techniqueName);
                if (!isEditing) {
                    retrospectiveModel.setFormats(this.formats);
                }
                this.updateColumns(this.formatId);
            })
            .catch((e) => console.log(e));

        if (!wizardModel.getEditSettingsAvailable()) {
            this.$columnInput.prop("disabled", true);
            $(".overlay").css("display", "inherit");
            $("#no-edit-message").css("display", "inherit");
            $(".column-name-input").prop("disabled", true);
            $(".column-color-input").prop("disabled", true);
        }
    }

    onNext() {
        navigation.onNextClick();
    }

    onPrevious() {
    }

    selectColor(evt) {
        evt.stopPropagation();
        const element = $(evt.target);

        if (element.hasClass("color-rectangle")) {
            // is a color item
            $(`#column-color-input${this.editableIndex}`).val(
                evt.target.dataset.color
            );
            $(`#column-color-input-container${this.editableIndex}`).css({
                "background-color": evt.target.dataset.color,
            });
            $("#color-picker-container").css({display: "none"});
            $(".column-color-input").trigger("blur");
        }
        this.closeColorPicker();
    }

    openColorPicker(evt) {
        evt.stopPropagation();
        const input = evt.target;
        this.editableIndex = input.dataset.index;
        const picker = $("#color-picker-container");
        const scroll = document.getElementById("scroll");

        picker.css({
            top: input.offsetTop - scroll.scrollTop,
            left: input.offsetLeft - picker.width(),
        });
        picker.css({display: "flex"});
        this.colorPickerOpened = true;
    }

    closeColorPicker() {
        const picker = $("#color-picker-container");
        picker.css({display: "none"});
        this.colorPickerOpened = false;
    }

    moveColorPicker(evt) {
        if (!this.colorPickerOpened) return;

        const picker = $("#color-picker-container");
        const colorContainer = document.getElementById(
            `column-color-input-container${this.editableIndex}`
        );
        const scroll = evt.target;
        const pickerPosition = colorContainer.offsetTop - scroll.scrollTop;
        const limitTop = scroll.offsetTop;
        const limitBottom = limitTop + scroll.offsetHeight;
        let outOfLimits = pickerPosition < limitTop || pickerPosition > limitBottom;

        if (outOfLimits) {
            $("#color-picker-container").css({display: "none"});
            return;
        }

        picker.css({top: colorContainer.offsetTop - scroll.scrollTop});
        picker.css({display: "flex"});
    }

    generateRandomColor() {
        return (
            "#" + this.generateValue() + this.generateValue() + this.generateValue()
        );
    }

    generateValue() {
        const val = Math.floor(Math.random() * 256).toString(16);
        return val.length < 2 ? "0" + val : val;
    }

    addNewColumn() {
        const inputField = $("#column-name-input");
        const rawColumnName = inputField.val().toString().trim();
        
        // Normalize the column name first (before HTML encoding)
        const normalizedColumnName = normalizeString(rawColumnName);
        
        if (!normalizedColumnName || normalizedColumnName === "") {
            return false;
        }

        // Check for duplicates using normalized name (before encoding)
        if (retrospectiveModel.doesColumnsExist(normalizedColumnName, this.formatId)) {
            $("#column-name-input").addClass("duplicated");
            $("#duplicated").show();
        } else {
            $("#column-name-input").removeClass("duplicated");
            $("#duplicated").hide();

        // Encode HTML entities for safe storage
        const encodedColumnName = normalizedColumnName
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");

        retrospectiveModel.addColumn(
            this.generateNextColumnId(),
            encodedColumnName,
            this.generateRandomColor(),
            Number(this.formatId)
        );

        this.updateColumns(this.formatId);
        inputField.val("");
        setTimeout(function () {
            $("#column-name-input").focus();
        }, 20);
        }
        return false;
    }

    private generateNextColumnId() {
        return `C${retrospectiveModel.getColumns(this.formatId).length + 1}`;
    }

    removeColumn(event) {
        if (retrospectiveModel.getColumns(this.formatId).length == 1) {
            return;
        }
        const columnIndex = event.currentTarget.attributes["data-index"].value;
        retrospectiveModel.removeColumnByIndex(columnIndex, this.formatId);
        this.updateColumns(this.formatId);
    }

    focusColumnInputForEdition(event) {
        const columnIndex = event.currentTarget.attributes["data-index"].value;
        const inputField: any = $(`#column-name-input${columnIndex}`);
        const inputFieldValueLen = inputField.val().length;
        inputField.focus();
        inputField.selectionRange(inputFieldValueLen, inputFieldValueLen);
    }

    completeColumnEditionBySubmit(event) {
        this.completeColumnEdition(event);
        const nextColumnIndex =
            parseInt(event.currentTarget.attributes["data-index"].value) + 1;
        if (nextColumnIndex < retrospectiveModel.getColumns(this.formatId).length) {
            const columnContent = $(`#column-name-input${nextColumnIndex}`).val();
            $(`#column-name-input${nextColumnIndex}`)
                .focus()
                .val("")
                .val(columnContent);
        } else $("#column-name-input").focus();
        return false;
    }

    completeColumnEdition(event) {
        // Helper function to decode HTML entities to readable text
        const decodeHTML = (text: string): string => {
            return text
                .replace(/&amp;/g, "&")
                .replace(/&lt;/g, "<")
                .replace(/&gt;/g, ">")
                .replace(/&quot;/g, '"')
                .replace(/&#039;/g, "'");
        };

        // Helper function to encode characters for safe storage
        const encodeHTML = (text: string): string => {
            return text
                .replace(/&/g, "&amp;")
                .replace(/</g, "&lt;")
                .replace(/>/g, "&gt;")
                .replace(/"/g, "&quot;")
                .replace(/'/g, "&#039;");
        };

        const columnIndex = parseInt(event.currentTarget.attributes["data-index"].value);
        const columnInput = $(`#column-name-input${columnIndex}`);
        const columnColorInput = $(`#column-color-input${columnIndex}`);
        const rawColumnContent = columnInput.val().toString().trim();
        const columnColor = columnColorInput.val();

        // Get stored content and decode it for comparison
        const storedContent = retrospectiveModel.getColumns(this.formatId)[columnIndex].content;
        const decodedStoredContent = decodeHTML(storedContent);
        const normalizedStoredContent = normalizeString(decodedStoredContent);

        if (rawColumnContent == "") {
            // Decode stored content for display
            columnInput.val(decodedStoredContent);
            columnColorInput.val(
                retrospectiveModel.getColumns(this.formatId)[columnIndex].color
            );
         } else {
            // Normalize the input before checking for duplicates
            const normalizedColumnContent = normalizeString(rawColumnContent);
            
            // Check for duplicates, excluding the current column being edited
            if (
                normalizedColumnContent &&
                retrospectiveModel.doesColumnsExist(normalizedColumnContent, this.formatId, columnIndex) &&
                normalizedStoredContent !== normalizedColumnContent
            ) {
                columnInput.addClass("duplicated");

                if (event.type == "submit") {
                    // Decode stored content for display
                    columnInput.val(decodedStoredContent);
                    columnColorInput.val(
                        retrospectiveModel.getColumns(this.formatId)[columnIndex].color
                    );
                }
            } else {
                columnInput.removeClass("duplicated");
                // Encode normalized content before storing
                const encodedColumnName = normalizedColumnContent
                    ? encodeHTML(normalizedColumnContent)
                    : encodeHTML(rawColumnContent);
                retrospectiveModel.updateColumnByIndex(
                    columnIndex,
                    encodedColumnName,
                    columnColor,
                    this.formatId
                );
            }
        }
    }

    resetColumns(columns) {
        // Helper function to decode HTML entities to readable text
        const decodeHTML = (text: string): string => {
            return text
                .replace(/&amp;/g, "&")
                .replace(/&lt;/g, "<")
                .replace(/&gt;/g, ">")
                .replace(/&quot;/g, '"')
                .replace(/&#039;/g, "'");
        };

        const columnsLength = columns.length;
        let trashDisabledClass = columnsLength == 1 ? "disabled" : "";

        this.$columnInput.show();
        this.$columnInput.show();

        if (!columnsLength) {
            $("#required").show();
            $("#duplicated").hide();
        } else {
            $("#required").hide();
        }

        this.$columnList.empty();
        const columnElements = columns.map(
            (column: Column, index) => {
                // Decode the stored encoded content to get readable text for display
                const decodedContent = decodeHTML(column.content.trim());

                // Create main row container
                const row = document.createElement("div");
                row.className = "column-row";

                // Create column number div
                const columnNumber = document.createElement("div");
                columnNumber.className = "column-row-number";
                columnNumber.textContent = `${index + 1}.`;
                row.appendChild(columnNumber);

                // Create form element
                const form = document.createElement("form");
                form.className = "column-name-form";
                form.setAttribute("data-index", index.toString());

                // Create input element - setAttribute automatically escapes for XSS safety
                const input = document.createElement("input");
                input.id = `column-name-input${index}`;
                input.setAttribute("data-index", index.toString());
                input.setAttribute("value", decodedContent);
                input.className = "column-name-input";
                input.setAttribute("type", "text");
                input.setAttribute("placeholder", "Type column name here and hit enter...");
                input.setAttribute("autocomplete", "off");
                form.appendChild(input);
                row.appendChild(form);

                // Create edit icon div
                const editDiv = document.createElement("div");
                editDiv.className = "column-row-edit";
                editDiv.setAttribute("data-index", index.toString());
                const editIcon = document.createElement("span");
                editIcon.className = "aui-icon aui-icon-small aui-iconfont-edit";
                editDiv.appendChild(editIcon);
                row.appendChild(editDiv);

                // Create delete icon div
                const deleteDiv = document.createElement("div");
                deleteDiv.className = `column-row-delete ${trashDisabledClass}`;
                deleteDiv.setAttribute("data-index", index.toString());
                const deleteIcon = document.createElement("span");
                deleteIcon.className = "aui-icon aui-icon-small aui-iconfont-delete";
                deleteDiv.appendChild(deleteIcon);
                row.appendChild(deleteDiv);

                // Create color input container
                const colorContainer = document.createElement("div");
                colorContainer.id = `column-color-input-container${index}`;
                colorContainer.className = "column-color-input-container";
                colorContainer.setAttribute("data-index", index.toString());
                colorContainer.setAttribute("style", `background-color: ${column.color}`);

                // Create hidden color input
                const colorInput = document.createElement("input");
                colorInput.id = `column-color-input${index}`;
                colorInput.setAttribute("data-index", index.toString());
                colorInput.setAttribute("value", column.color);
                colorInput.className = "column-color-input";
                colorInput.setAttribute("type", "hidden");
                colorContainer.appendChild(colorInput);
                row.appendChild(colorContainer);

                return row;
            }
        );

        // Append all DOM nodes to the list
        columnElements.forEach((element) => {
            this.$columnList.append(element);
        });
    }

    updateColumns(formatId: number) {
        // Helper function to decode HTML entities to readable text
        const decodeHTML = (text: string): string => {
            return text
                .replace(/&amp;/g, "&")
                .replace(/&lt;/g, "<")
                .replace(/&gt;/g, ">")
                .replace(/&quot;/g, '"')
                .replace(/&#039;/g, "'");
        };

        const columnsLength = retrospectiveModel.getColumns(formatId).length;
        let trashDisabledClass = columnsLength == 1 ? "disabled" : "";

        this.$columnInput.show();

        if (!columnsLength) {
            $("#required").show();
            $("#duplicated").hide();
        } else {
            $("#required").hide();
        }

        this.$columnList.empty();
        const columnElements = retrospectiveModel.getColumns(formatId).map(
            (column: Column, index) => {
                // Decode the stored encoded content to get readable text for display
                const decodedContent = decodeHTML(column.content.trim());

                // Create main row container
                const row = document.createElement("div");
                row.className = "column-row";

                // Create column number div
                const columnNumber = document.createElement("div");
                columnNumber.className = "column-row-number";
                columnNumber.textContent = `${index + 1}.`;
                row.appendChild(columnNumber);

                // Create form element
                const form = document.createElement("form");
                form.className = "column-name-form";
                form.setAttribute("data-index", index.toString());

                // Create input element - setAttribute automatically escapes for XSS safety
                const input = document.createElement("input");
                input.id = `column-name-input${index}`;
                input.setAttribute("data-index", index.toString());
                input.setAttribute("value", decodedContent);
                input.className = "column-name-input";
                input.setAttribute("type", "text");
                input.setAttribute("placeholder", "Type column name here and hit enter...");
                input.setAttribute("autocomplete", "off");
                form.appendChild(input);
                row.appendChild(form);

                // Create edit icon div
                const editDiv = document.createElement("div");
                editDiv.className = "column-row-edit";
                editDiv.setAttribute("data-index", index.toString());
                const editIcon = document.createElement("span");
                editIcon.className = "aui-icon aui-icon-small aui-iconfont-edit";
                editDiv.appendChild(editIcon);
                row.appendChild(editDiv);

                // Create delete icon div
                const deleteDiv = document.createElement("div");
                deleteDiv.className = `column-row-delete ${trashDisabledClass}`;
                deleteDiv.setAttribute("data-index", index.toString());
                const deleteIcon = document.createElement("span");
                deleteIcon.className = "aui-icon aui-icon-small aui-iconfont-delete";
                deleteDiv.appendChild(deleteIcon);
                row.appendChild(deleteDiv);

                // Create color input container
                const colorContainer = document.createElement("div");
                colorContainer.id = `column-color-input-container${index}`;
                colorContainer.className = "column-color-input-container";
                colorContainer.setAttribute("data-index", index.toString());
                colorContainer.setAttribute("style", `background-color: ${column.color}`);

                // Create hidden color input
                const colorInput = document.createElement("input");
                colorInput.id = `column-color-input${index}`;
                colorInput.setAttribute("data-index", index.toString());
                colorInput.setAttribute("value", column.color);
                colorInput.className = "column-color-input";
                colorInput.setAttribute("type", "hidden");
                colorContainer.appendChild(colorInput);
                row.appendChild(colorContainer);

                return row;
            }
        );

        // Append all DOM nodes to the list
        columnElements.forEach((element) => {
            this.$columnList.append(element);
        });
    }

    focusOnColumnAdded() {
        const lastColumnIndex =
            retrospectiveModel.getColumns(this.formatId).length - 1;
        const inputField: any = $(`#column-name-input${lastColumnIndex}`);

        inputField.focus();
        inputField.blur();
    }

    /**
     * Validates the new column name in real-time as user types
     */
    validateNewColumnName(event) {
        const inputField = $(event.target);
        const rawColumnName = inputField.val().toString().trim();
        const normalizedColumnName = normalizeString(rawColumnName);

        if (!normalizedColumnName || normalizedColumnName === "") {
            inputField.removeClass("duplicated");
            $("#duplicated").hide();
            return;
        }

        if (retrospectiveModel.doesColumnsExist(normalizedColumnName, this.formatId)) {
            inputField.addClass("duplicated");
            $("#duplicated").show();
        } else {
            inputField.removeClass("duplicated");
            $("#duplicated").hide();
        }
    }

    /**
     * Validates an existing column name in real-time as user types
     */
    validateColumnName(event) {
        const inputField = $(event.target);
        const columnIndex = parseInt(inputField.attr("data-index"));
        const rawColumnContent = inputField.val().toString().trim();
        const normalizedColumnContent = normalizeString(rawColumnContent);

        if (!normalizedColumnContent || normalizedColumnContent === "") {
            inputField.removeClass("duplicated");
            return;
        }

        // Check for duplicates, excluding the current column being edited
        if (retrospectiveModel.doesColumnsExist(normalizedColumnContent, this.formatId, columnIndex)) {
            inputField.addClass("duplicated");
        } else {
            inputField.removeClass("duplicated");
        }
    }

}

export default ColumnCreate;
