(function (angular) {
    angular
        .module('retrospectiveApp')
        .controller('voteController', VoteController);

    VoteController.$inject = ['IMAGES_BASE_PATH', 'HTTP_PATH', '$http', '$scope', 'EventService', 'IdeasService', 'VotesService', 'SettingsService', 'WorkflowService', "UtilService"];

    function VoteController(IMAGES_BASE_PATH, HTTP_PATH, $http, $scope, EventService, IdeasService, VotesService, SettingsService, WorkflowService, UtilService) {
        const vm = this;
        vm.readOnlyMode = false;
        vm.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        let creatingVote = false;
        vm.userKey = $scope.$parent.vm.userKey;
        vm.uuid = $scope.$parent.vm.uuid;
        vm.moderator = $scope.$parent.vm.moderator;
        vm.settings = SettingsService.settings;
        vm.groups = {};
        vm.colorsColumn = {};

        vm.thinkElements = [];
        this.groupCollection = null;
        let cache = '';

        this.localVotes = {};
        this.columnMajorities = {};
        this.majoritiesCalculated = false;

        WorkflowService.subscribe((event)=>{
            this.readOnlyMode = true
        },['readOnlyMode']);

        SettingsService.subscribe((event) => {
            this.settings = event.state.settings;
        }, ['settings']);

        $http.get(`${HTTP_PATH}/ideas?uuid=${vm.uuid}`).then((response) => {
            vm.thinkElements = response.data;

            VotesService.subscribe((event) => {
                const asString = JSON.stringify(event.state.groupList);

                // avoid re-render of the same state
                if (asString === cache) {
                    return;
                } else {
                    cache = asString;
                }

                this.groupCollection = {
                    groupList: event.state.groupList
                };

                /**
                 * groups won't change, so let's calculate it only the first time
                 */
                if (!this.majoritiesCalculated) {
                    this.majoritiesCalculated = true;
                    const ids = vm.settings.columns.map(col => col.columnId);

                    for (let j = 0; j < this.groupCollection.groupList.length; j++) {
                        this.findColumnIdMajority(j, this.groupCollection.groupList[j].ideas, ids);
                    }
                }

                for (let i = 0; i < vm.settings.columns.length; i++) {
                    const columnId = vm.settings.columns[i].columnId;
                    const groupsForColumn = [];

                    for (let j = 0; j < this.groupCollection.groupList.length; j++) {
                        if (this.groupCollection.groupList[j].ideas[0] && this.columnMajorities[j] == columnId) {
                            if (vm.groups[columnId] && vm.groups[columnId][j]) {
                                this.groupCollection.groupList[j].isCollapsed = vm.groups[columnId][j].isCollapsed;
                            }

                            groupsForColumn.push(this.groupCollection.groupList[j]);
                        }
                    }

                    vm.groups[columnId] = groupsForColumn;
                }
            }, ['groupList']);
        }).catch(err => {
            console.error(err);
            const flagError = AJS.flag({
                type: 'error',
                body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
            });
        });

        this.findColumnIdMajority = (index, ideas, ids) => {
            const counter = {};

            /**
             * initialize a counter for every id
             */
            ids.forEach(id => counter[id] = 0);

            ideas.forEach(idea => {
                counter[idea.columnId]++;
            });

            const keys = Object.keys(counter);
            let winnerCounter = counter[keys[0]];
            let winnerId = keys[0];

            keys.forEach(key => {
                if (counter[key] > winnerCounter) {
                    winnerCounter = counter[key];
                    winnerId = key;
                }
            });

            this.columnMajorities[index] = winnerId;
        };

        this.findColor = (columnId) => {
            const column = this.settings.columns.find((col) => col.columnId == columnId);
            if (column) {
                return column.color;
            }
            return '#000';
        };

        const countTotalVotes = () => {
            let groups = [];

            const columnIds = Object.keys(vm.groups || {});

            columnIds.forEach((id) => {
                groups = groups.concat(vm.groups[id]);
            });

            const totalVotes = groups.reduce((sum, group) => {
                return sum + (group.votesPerUser && group.votesPerUser[vm.userKey] || 0);
            }, 0);

            SettingsService.setTotalVotes(totalVotes);

            return totalVotes;
        };

        const voteClick = (group, columnId, increased) => {
            if (creatingVote) {
                return;
            }
            creatingVote = true;

            const data = {};
            data.group = JSON.parse(JSON.stringify(group));
            data.columnId = columnId;

            delete data.group['isCollapsed'];

            VotesService.create({
                uuid: vm.uuid,
                data,
                increased
            }).then((response) => {
                if (response.data.applied) {
                    if (increased) {
                        group.votesPerUser[vm.userKey] = (group.votesPerUser[vm.userKey] || 0) + 1;
                    } else {
                        group.votesPerUser[vm.userKey] = group.votesPerUser[vm.userKey] - 1;
                    }
                } else {
                    console.log(`RETROSPECTIVES: Vote request not applied: ${response.data.message}`);
                }
                creatingVote = false;
            }).catch((err) => {
                creatingVote = false;
                console.error(err);
                const flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        const collapseClass = (group) => group.grouped && group.isCollapsed ? '' : 'ideas-grouped';

        const collapseGroup = (group) => {
            if (group.grouped) {
                group.isCollapsed = !group.isCollapsed;
            }
        };

        const getVoteLabel = (numberOfVotes) => {
            if (vm.settings.votesAllowedPerCard === 0) {
                return numberOfVotes;
            }
            return `${numberOfVotes}/${vm.settings.votesAllowedPerCard}`;
        };

        const getVoteText = (collection) => {
            if (vm.settings.votesAllowedPerCard !== 0) {
                return 'Votes';
            }
            return collection.votesPerUser && collection.votesPerUser[vm.userKey] === 1 ? 'Vote' : 'Votes';
        };

        const getVotesForCard = (group) => group.votesPerUser && group.votesPerUser[vm.userKey] || 0;

        vm.getVotingClass = (group, delta) => {
            if (delta > 0) {//increase
                if (vm.settings.votesPerPlayer !== 0 && countTotalVotes() >= vm.settings.votesPerPlayer) {
                    return 'inactive';
                }

                if (vm.settings.votesAllowedPerCard !== 0 && group.votesPerUser && group.votesPerUser[vm.userKey] >= vm.settings.votesAllowedPerCard) {
                    return 'inactive';
                }

                return 'active';
            } else {// decrease
                return getVotesForCard(group) > 0 ? 'active' : 'inactive';
            }
        };

        const ideasCollapsedClass = (group) => group.grouped && group.isCollapsed && group.ideas.length > 1 ? '' : 'collapsed-ideas';

        const isCurrentUserMod = () => vm.moderator && vm.moderator.userKey === vm.userKey;

        vm.getArrowUpTitle = (group, delta) => {
            if (delta > 0) {//increase
                if (vm.settings.votesPerPlayer !== 0 && countTotalVotes() >= vm.settings.votesPerPlayer) {
                    return 'You’ve run out of votes';
                }

                if (vm.settings.votesAllowedPerCard !== 0 && group.votesPerUser && group.votesPerUser[vm.userKey] >= vm.settings.votesAllowedPerCard) {
                    return 'You’ve run out of votes';
                }

                return '';
            } else {// decrease
                return getVotesForCard(group) > 0 ? '' : 'You’ve run out of votes';
            }
        };

        /*
         calculate string width in screen
         */
        const hasTitleEllipsis = (text) => {
            const ellipsisHelper = document.getElementById('ellipsis-calc-helper');
            if(!ellipsisHelper) {
                return false;
            }
            ellipsisHelper.innerHTML = DOMPurify.sanitize(UtilService.encodeHTML(text));
            ellipsisHelper.style.fontSize = '20px';
            const textWidth = ellipsisHelper.clientWidth + 1;
            const widthAvailable = $('.title-think').width();
            return textWidth > widthAvailable;
        };

        this.hasMoreThanFourColumns = () => {
            return this.settings.columns.length > 4;
        }

        this.init = () => {
            if (this.hasMoreThanFourColumns()) {
                $('#retro-container').addClass('more-than-four-columns-container');
                $('#retro-container').removeClass('retro-container');
            }
        }

        vm.getVoteLabel = getVoteLabel;
        vm.hasTitleEllipsis = hasTitleEllipsis;
        vm.voteClick = voteClick;
        vm.getVoteText = getVoteText;
        vm.collapseGroup = collapseGroup;
        vm.collapseClass = collapseClass;
        vm.ideasCollapsedClass = ideasCollapsedClass;
        vm.isCurrentUserMod = isCurrentUserMod;
        vm.getColumnClass = SettingsService.getColumnClass;
    }
}(angular));