(function (angular) {
    angular
        .module('retrospectiveApp')
        .service('WorkflowService', WorkflowService);

    WorkflowService.$inject = [
        'HTTP_PATH',
        'IMAGES_BASE_PATH',
        '$http',
        'PollerService',
        'SubscribableService',
        'StatefulService',
        'IdeasService',
        'GroupsService',
        'VotesService',
        'DiscussService',
        'ReadOnlyModePollerService',
        'HeartbeatPollerService'
    ];

    function WorkflowService(HTTP_PATH, IMAGES_BASE_PATH, $http, Poller, Subscribable, Stateful,
                             IdeasService, GroupsService, VotesService, DiscussService, ReadOnlyModePollerService,
                             HeartbeatPollerService
    ) {
        /*
         turn this into a subscribable service,
         so that controllers can list to its changes
         */
        Subscribable.make(this);

        /*
         add logic to this service for simple state management
         */
        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        const setModerator = (moderator) => {
            if (!moderator) {
                return;
            }

            if (!this.state.exists('moderator')
                || this.state.get('moderator').userKey !== moderator.userKey
                || this.state.get('moderator').isReady !== moderator.isReady
                || this.state.get('moderator').isConnected !== moderator.isConnected
            ) {
                this.state.set('moderator', moderator);

                this.notify({
                    identifier: 'moderator',
                    state: {
                        moderator: moderator
                    }
                });
            }
        };

        this.startPollers = async (presetModerator, userKey, uuid, pageId, globalSettings) => {
            this.state.set('userKey', userKey);
            this.state.set('uuid', uuid);
            this.state.set('pageId', pageId);

            this.state.shared._set('userKey', userKey);
            this.state.shared._set('uuid', uuid);
            this.state.shared._set('pageId', pageId);


            let sessionDataInterval = 1000; 
            if(globalSettings && globalSettings.sessionInterval){
                sessionDataInterval = globalSettings.sessionInterval * 1000; // intervals received in seconds, convert to milliseconds
            }


            /*
             Will take care of:

             - current stage
             - moderator
             - participants
             - who is ready
             - completion date
             */

            const ReadOnlyMode = new ReadOnlyModePollerService((response)=>{
                if(response){
                    this.notify({
                        identifier: 'readOnlyMode',
                        state: {
                            readOnlyMode: true
                        }
                    });
                    IdeasService.stopPoller();
                    GroupsService.stopPoller();
                    VotesService.stopPoller(5000);
                }
            });

            new HeartbeatPollerService(() => {}).start(uuid, userKey);

            // const sessionDataPoller = new Poller(`/workflow/session-data?uuid=${uuid}`, (response) => {
            this.sessionDataPoller = new Poller(`/workflow/session-data?uuid=${uuid}`, (response) => {
                if (!response.data || !response.data.baseUsers) {
                    return;
                }

                // TODO 
                // response.data.globalSettings = response.data.globalSettings || {};
                response.data.participants = (response.data.baseUsers || [])
                    .filter(u => u.isParticipant)
                    .sort((a, b) => a.userKey < b.userKey);

                response.data.baseUsers = (response.data.baseUsers || [])
                    .filter(u => u.isConnected)
                    .sort((a, b) => a.userKey < b.userKey);


                const definedModerator = response.data.moderator;

                if (definedModerator) {
                    setModerator(definedModerator);
                } else {
                    const {presetModeratorUserKey} = response.data;
                    this.getConfluenceUser(presetModeratorUserKey, uuid).then(user => {
                        setModerator(user);
                    });
                }

                /*
                 notify subscribers about newest info
                 */
                if (this.state.get('stage', 'undefined') !== response.data.viewName) {
                    let thinkInterval = 1000;
                    let groupInterval = 1000;
                    let discussInterval = 1000;
                    let summaryInterval = 1000; 
                    let summaryStopAfter = 0; 

                    if(globalSettings && globalSettings.thinkInterval){
                        thinkInterval = globalSettings.thinkInterval * 1000; // intervals received in seconds, convert to milliseconds
                    }
                    if(globalSettings && globalSettings.groupInterval){
                        groupInterval = globalSettings.groupInterval * 1000; // intervals received in seconds, convert to milliseconds
                    }
                    if(globalSettings && globalSettings.discussInterval){
                        discussInterval = globalSettings.discussInterval * 1000; // intervals received in seconds, convert to milliseconds
                    }
                    if(globalSettings && globalSettings.summaryInterval){
                        summaryInterval = globalSettings.summaryInterval * 1000; // intervals received in seconds, convert to milliseconds
                    }
                    if(globalSettings && globalSettings.summaryStopAfter){
                        summaryStopAfter = globalSettings.summaryStopAfter * 60 * 1000; // stopAfter received in minutes, convert to milliseconds
                    }
                    switch (response.data.viewName) {
                        case 'think':
                            this.sessionDataPoller.stop();
                            this.sessionDataPoller.start();
                            IdeasService.startPollers(userKey, uuid, thinkInterval);
                            GroupsService.stopPoller();
                            VotesService.stopPoller();
                            DiscussService.stopPoller();
                            break;
                        case 'group':
                            this.sessionDataPoller.stop();
                            this.sessionDataPoller.start();
                            IdeasService.stopPoller();
                            GroupsService.startPollers(userKey, uuid, groupInterval);
                            VotesService.stopPoller();
                            DiscussService.stopPoller();
                            break;
                        case 'vote':
                            this.sessionDataPoller.stop();
                            this.sessionDataPoller.start();
                            IdeasService.stopPoller();
                            GroupsService.stopPoller();
                            VotesService.startPollers(userKey, uuid);
                            DiscussService.stopPoller();
                            break;
                        case 'discuss':
                            this.sessionDataPoller.stop();
                            this.sessionDataPoller.start();
                            IdeasService.stopPoller();
                            VotesService.stopPoller();
                            GroupsService.stopPoller();
                            DiscussService.stopPoller();
                            DiscussService.startPollers(userKey, uuid, discussInterval);
                            break;
                        case 'result':
                            this.sessionDataPoller.stop();
                            this.sessionDataPoller.start(summaryStopAfter);
                            IdeasService.stopPoller();
                            VotesService.stopPoller();
                            GroupsService.stopPoller();
                            DiscussService.stopPoller();
                            DiscussService.startPollers(userKey, uuid, summaryInterval, summaryStopAfter);
                            break;
                    }

                    this.notify({
                        identifier: 'stage',
                        state: {
                            stage: response.data.viewName
                        }
                    });
                }

                this.state.set('stage', response.data.viewName);

                if (response.data.baseUsers.length) {
                    const user = response.data.baseUsers.find(p => p.userKey == this.state.get('userKey'));

                    this.notify({
                        identifier: 'isUserReady',
                        state: {
                            ready: user ? user.isReady : false,
                        }
                    });
                }

                this.notify({
                    identifier: 'sessionParticipants',
                    state: {
                        participants: response.data.participants
                    }
                });

                this.notify({
                    identifier: 'participants',
                    state: {
                        participants: response.data.baseUsers
                    }
                });

                this.notify({
                    identifier: 'completionDate',
                    state: {
                        completionDate: response.data.completionDate
                    }
                });

                this.notify({
                    identifier: 'discussGroupIndex',
                    state: {
                        index: response.data.discussGroupIndex
                    }
                });
            }, sessionDataInterval);

            this.sessionDataPoller.setIdentifier('Session data poller');
            this.sessionDataPoller.start();
            ReadOnlyMode.start();
        };

        this.initRetro = (uuid) => {
            const pageId = top.AJS.Meta.get('page-id');

            return $http.put(`${HTTP_PATH}/workflow/${uuid}`, pageId);
        }

        /*
         updates the moderator in the backend and in the local service storage
         */
        this.setModerator = (user) => {
            let prevMod = '';

            const moderator = this.state.get('moderator', false);

            if (moderator && moderator.userKey !== user.userKey) {
                prevMod = `&prevMod=${moderator.userKey}`;
            }

            $http.put(`${HTTP_PATH}/workflow/set-moderator?userKey=${user.userKey}&uuid=${this.state.get('uuid')}${prevMod}`)
                .then(() => this.sessionDataPoller.poll())
        };

        this.changeView = (newStage) => {
            if (this.state.get('stage') === newStage) {
                return;
            }

            $http.put(`${HTTP_PATH}/workflow/change-view?userKey=${this.state.get('userKey')}&uuid=${this.state.get('uuid')}&viewName=${encodeURIComponent(newStage)}`)
                .then(() => this.sessionDataPoller.poll())
        };

        this.reopen = () => 
            $http.get(`${HTTP_PATH}/workflow/reopen?uuid=${this.state.get('uuid')}`)
                .then(() => this.sessionDataPoller.poll())

        this.persistData = () =>
            $http.post(`${HTTP_PATH}/workflow/persist/all?uuid=${this.state.get('uuid')}`);

        this.imReady = () =>
            $http.put(`${HTTP_PATH}/workflow/im-ready?userKey=${this.state.get('userKey')}&uuid=${this.state.get('uuid')}`)
                .then(() => this.sessionDataPoller.poll())


        let confluenceUserPromise = null;
        let lastModeratorUserKey = '';
        this.getConfluenceUser = (userKey, uuid) => {
            if (!confluenceUserPromise || lastModeratorUserKey !== userKey) {
                confluenceUserPromise = $http.get(`${IMAGES_BASE_PATH}/rest/api/user?key=${userKey}`)
                    .then(response => {
                        const { data } = response;

                        return {
                            userKey: data.userKey,
                            name: data.username,
                            fullName: data.displayName,
                            pageId: uuid,
                            imageUri: data.profilePicture.path.replace('/confluence', ''),
                            isConnected: false,
                            isModerator: true,
                            isReady: false,
                            isParticipant: false,
                            numericPageId: parseInt(uuid),
                        }
                    })
            }

            return confluenceUserPromise;
        }
    }
}(angular));
