import IBaseStep from "../../declarations/IBaseStep";
import wizardModel from "../../models/WizardModel";
import retrospectiveModel from "../../models/RetrospectiveModel";
import retrospectiveService from "../../services/RetrospectiveService";
import navigation from "../../navigation";
declare const AJS: any;

export type Column = {
    columnId: string;
    content: string;
    color: string;
};

class ColumnCreate implements IBaseStep {
    private $columnList: JQuery;
    private $columnInput: JQuery;
    private $techniqueSelect: JQuery;
    private $SelectInput: string = ".s2id_technique-select";
    private $ResetButton: JQuery;
    private $happyIcon: JQuery;
    private $colorBand: JQuery;
    private editableIndex: number = -1;
    private colorPickerOpened = false;
    private formats: any;
    private formatId: number;
    private $techniqueSelected: JQuery;
    private $happyIconWizardElement: JQuery;
    private $preferenceIconWizardElement: JQuery;
    private $inputColumnEdition:JQuery;

    constructor(private $element: JQuery) {
        this.$inputColumnEdition = $element.find('#column-name-input')
        this.$columnList = $element.find(".column-list");
        this.$columnInput = $element.find(".column-row-input");
        this.$techniqueSelect = $element.find("#technique-select");
        this.$ResetButton = $element.find("#reset-button");
        this.$ResetButton.focusout();
        this.$happyIcon = $element.find("#preference-icon-happy");
        this.$colorBand = $element.find("#task-create");
        this.$techniqueSelected = $element.find("#format-selected");
        this.$happyIconWizardElement = $element.find("#task-create");
        this.$preferenceIconWizardElement = $element.find("#preference-element");
        this.formats = [];
        this.formatId = retrospectiveModel.getFormatId();
        this.updateColumns(this.formatId);
        this.resetColumns([]);
        if (!wizardModel.getEditSettingsAvailable()) {
            this.$happyIcon.attr("class", "preference-icon-happy-deactivated");
            this.$colorBand.css("box-shadow", "none");
            this.$happyIconWizardElement.removeClass("has-hover");
            this.$preferenceIconWizardElement.removeClass("has-hover");
        }
    }

    bindEvents() {
        this.$preferenceIconWizardElement.on("click", this.onNext.bind(this));
        if (!wizardModel.getEditSettingsAvailable()) {
            this.$preferenceIconWizardElement.unbind("click");
        }
        this.$ResetButton.on("click", this.reset.bind(this));
        this.$techniqueSelect.on("change", this.selectTechnique.bind(this));
        this.$techniqueSelect.on("select2-open", this.selectOpen.bind(this));
        this.$techniqueSelect.on("select2-close", this.selectClose.bind(this));
        this.$columnInput.on(
            "submit",
            ".column-name-form",
            this.addNewColumn.bind(this)
        );
        this.$columnInput.on(
            "submit",
            ".column-name-form",
            this.focusOnColumnAdded.bind(this)
        );
        this.$columnList.on(
            "blur",
            ".column-name-input",
            this.completeColumnEdition.bind(this)
        );
        this.$columnList.on(
            "blur",
            ".column-color-input",
            this.completeColumnEdition.bind(this)
        );
        this.$columnList.on(
            "submit",
            ".column-name-form",
            this.completeColumnEditionBySubmit.bind(this)
        );
        this.$columnList.on(
            "click",
            ".column-row-edit",
            this.focusColumnInputForEdition.bind(this)
        );
        this.$columnList.on(
            "click",
            ".column-row-delete",
            this.removeColumn.bind(this)
        );

        this.$columnList.on(
            "click",
            ".column-color-input-container",
            this.openColorPicker.bind(this)
        );
        this.$columnList.on(
            "click",
            ".column-color-input",
            this.openColorPicker.bind(this)
        );

        $(window).click(() => {
            $("#color-picker-container").css({display: "none"});
            this.colorPickerOpened = false;
            this.editableIndex = -1;
        });

        $("#scroll").on("scroll", this.moveColorPicker.bind(this));

        $("#color-picker-container").on("click", this.selectColor.bind(this));

        $(".color-rectangle").on("click", this.selectColor.bind(this));
    }

    reset(e) {
        retrospectiveService.getFormats().then((data) => {
            let columns = data.find((format) => format.id === Number(this.formatId))
                .columns;
            retrospectiveModel.setFormatsReset(this.formatId, columns);
            this.resetColumns(columns);
        });
    }

    selectClose(e) {
        $(this.$SelectInput).removeClass("select-open");
    }

    selectOpen(e) {
        $(this.$SelectInput).addClass("select-open");
    }

    selectTechnique(e) {
        this.$inputColumnEdition.val('')
        let formatText = e.added.text;
        this.$techniqueSelected.text(formatText);
        retrospectiveModel.setTechniqueSelected(formatText);
        this.formatId = this.$techniqueSelect.val();
        retrospectiveModel.setFormatId(this.formatId);
        this.updateColumns(this.formatId);
    }

    unbindEvents() {
        this.$columnInput.off();
        this.$columnList.off();
    }

    onLoad(formatId, isEditing, technique) {
        // We make a request to get the formats from backend
        retrospectiveService
            .getFormats()
            .then((data) => {
                this.formats = data;
                this.formatId = !!formatId ? formatId : this.formats[0].id;
                const formatsData = [];
                this.formats.map((format) =>
                    formatsData.push({id: format.id, text: format.name})
                );
                const auxTechnique = this.formats.find(format => format.name == 'What went well')
                const techniqueName = technique || auxTechnique.name

                if (techniqueName){
                    this.$techniqueSelected.text(techniqueName);
                }else {
                    throw new Error("Error getting formats list")
                }
                
                AJS.$("#technique-select").auiSelect2({
                    multiple: false,
                    minimumResultsForSearch: -1,
                    data: formatsData,
                    initSelection: function (element, callback) {
                        let formatId = retrospectiveModel.getFormatId();
                        if (formatId !== null) {
                            callback(formatsData.find(format => format.id === Number(formatId)));
                        }else if (formatsData.length){
                            callback(formatsData.find(format => format.text === auxTechnique.name))
                        }
                    },}).select2('val',this.formatId);

                //We initialize the select2 element
                retrospectiveModel.setFormatId(this.formatId);
                retrospectiveModel.setTechniqueSelected(techniqueName);
                if (!isEditing) {
                    retrospectiveModel.setFormats(this.formats);
                }
                this.updateColumns(this.formatId);
            })
            .catch((e) => console.log(e));

        if (!wizardModel.getEditSettingsAvailable()) {
            this.$columnInput.prop("disabled", true);
            $(".overlay").css("display", "inherit");
            $("#no-edit-message").css("display", "inherit");
            $(".column-name-input").prop("disabled", true);
            $(".column-color-input").prop("disabled", true);
        }
    }

    onNext() {
        navigation.onNextClick();
    }

    onPrevious() {
    }

    selectColor(evt) {
        evt.stopPropagation();
        const element = $(evt.target);

        if (element.hasClass("color-rectangle")) {
            // is a color item
            $(`#column-color-input${this.editableIndex}`).val(
                evt.target.dataset.color
            );
            $(`#column-color-input-container${this.editableIndex}`).css({
                "background-color": evt.target.dataset.color,
            });
            $("#color-picker-container").css({display: "none"});
            $(".column-color-input").trigger("blur");
        }
        this.closeColorPicker();
    }

    openColorPicker(evt) {
        evt.stopPropagation();
        const input = evt.target;
        this.editableIndex = input.dataset.index;
        const picker = $("#color-picker-container");
        const scroll = document.getElementById("scroll");

        picker.css({
            top: input.offsetTop - scroll.scrollTop,
            left: input.offsetLeft - picker.width(),
        });
        picker.css({display: "flex"});
        this.colorPickerOpened = true;
    }

    closeColorPicker() {
        const picker = $("#color-picker-container");
        picker.css({display: "none"});
        this.colorPickerOpened = false;
    }

    moveColorPicker(evt) {
        if (!this.colorPickerOpened) return;

        const picker = $("#color-picker-container");
        const colorContainer = document.getElementById(
            `column-color-input-container${this.editableIndex}`
        );
        const scroll = evt.target;
        const pickerPosition = colorContainer.offsetTop - scroll.scrollTop;
        const limitTop = scroll.offsetTop;
        const limitBottom = limitTop + scroll.offsetHeight;
        let outOfLimits = pickerPosition < limitTop || pickerPosition > limitBottom;

        if (outOfLimits) {
            $("#color-picker-container").css({display: "none"});
            return;
        }

        picker.css({top: colorContainer.offsetTop - scroll.scrollTop});
        picker.css({display: "flex"});
    }

    generateRandomColor() {
        return (
            "#" + this.generateValue() + this.generateValue() + this.generateValue()
        );
    }

    generateValue() {
        const val = Math.floor(Math.random() * 256).toString(16);
        return val.length < 2 ? "0" + val : val;
    }

    addNewColumn() {
        const inputField = $("#column-name-input");
        const columnName = inputField.val().trim()
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
        if (columnName == "") {
            return false;
        }

        if (retrospectiveModel.doesColumnsExist(columnName, this.formatId)) {
            $("#column-name-input").addClass("duplicated");
            $("#duplicated").show();
        } else {
            $("#column-name-input").removeClass("duplicated");
            $("#duplicated").hide();

            retrospectiveModel.addColumn(
                this.generateNextColumnId(),
                columnName,
                this.generateRandomColor(),
                Number(this.formatId)
            );
            this.updateColumns(this.formatId);
            inputField.val("");
            setTimeout(function () {
                $("#column-name-input").focus();
            }, 20);
        }
        return false;
    }

    private generateNextColumnId() {
        return `C${retrospectiveModel.getColumns(this.formatId).length + 1}`;
    }

    removeColumn(event) {
        if (retrospectiveModel.getColumns(this.formatId).length == 1) {
            return;
        }
        const columnIndex = event.currentTarget.attributes["data-index"].value;
        retrospectiveModel.removeColumnByIndex(columnIndex, this.formatId);
        this.updateColumns(this.formatId);
    }

    focusColumnInputForEdition(event) {
        const columnIndex = event.currentTarget.attributes["data-index"].value;
        const inputField: any = $(`#column-name-input${columnIndex}`);
        const inputFieldValueLen = inputField.val().length;
        inputField.focus();
        inputField.selectionRange(inputFieldValueLen, inputFieldValueLen);
    }

    completeColumnEditionBySubmit(event) {
        this.completeColumnEdition(event);
        const nextColumnIndex =
            parseInt(event.currentTarget.attributes["data-index"].value) + 1;
        if (nextColumnIndex < retrospectiveModel.getColumns(this.formatId).length) {
            const columnContent = $(`#column-name-input${nextColumnIndex}`).val();
            $(`#column-name-input${nextColumnIndex}`)
                .focus()
                .val("")
                .val(columnContent);
        } else $("#column-name-input").focus();
        return false;
    }

    completeColumnEdition(event) {
        const columnIndex = event.currentTarget.attributes["data-index"].value;
        const columnInput = $(`#column-name-input${columnIndex}`);
        const columnColorInput = $(`#column-color-input${columnIndex}`);
        const columnContent = columnInput.val().trim();
        const columnColor = columnColorInput.val();

        if (columnContent == "") {
            columnInput.val(
                retrospectiveModel.getColumns(this.formatId)[columnIndex].content
            );
            columnColorInput.val(
                retrospectiveModel.getColumns(this.formatId)[columnIndex].color
            );
        } else if (
            retrospectiveModel.doesColumnsExist(columnContent, this.formatId) &&
            retrospectiveModel.getColumns(this.formatId)[columnIndex].content !==
            columnContent
        ) {
            columnInput.addClass("duplicated");

            if (event.type == "submit") {
                columnInput.val(
                    retrospectiveModel.getColumns(this.formatId)[columnIndex].content
                );
                columnColorInput.val(
                    retrospectiveModel.getColumns(this.formatId)[columnIndex].color
                );
            }
        } else {
            columnInput.removeClass("duplicated");
            retrospectiveModel.updateColumnByIndex(
                columnIndex,
                columnContent,
                columnColor,
                this.formatId
            );
        }
    }

    resetColumns(columns) {
        const columnsLength = columns.length;
        let trashDisabledClass = columnsLength == 1 ? "disabled" : "";

        this.$columnInput.show();
        this.$columnInput.show();

        if (!columnsLength) {
            $("#required").show();
            $("#duplicated").hide();
        } else {
            $("#required").hide();
        }

        this.$columnList.empty();
        const $columnElement = columns.map(
            (column: Column, index) =>
                {
                    column.content=column.content.trim()
                    .replace(/</g, "&lt;")
                    .replace(/>/g, "&gt;")
                    .replace(/"/g, "&quot;")
                    .replace(/'/g, "&#039;");
                
                return `<div class="column-row">
                <div class="column-row-number">${index + 1}.</div>
                    <form class="column-name-form" data-index="${index}">
                        <input id="column-name-input${index}" data-index="${index}" value="${
                    column.content
                }"
                        class="column-name-input" type="text" placeholder="Type column name here and hit enter..."
                        autocomplete="off" />
                    </form>
                    <div class="column-row-edit" data-index="${index}"><span class="aui-icon aui-icon-small aui-iconfont-edit "></span></div>
                    <div class="column-row-delete ${trashDisabledClass}" data-index="${index}"><span class="aui-icon aui-icon-small aui-iconfont-delete" ></span></div>
                    <div id="column-color-input-container${index}" class="column-color-input-container" data-index="${index}" style="background-color: ${
                    column.color
                }">
                        <input id="column-color-input${index}" data-index="${index}" value="${
                   column.color
                }" class="column-color-input" type="hidden"/>
                    </div>

             </div>`
            }
        );

        this.$columnList.append($columnElement.join(""));
    }

    updateColumns(formatId: number) {
        const columnsLength = retrospectiveModel.getColumns(formatId).length;
        let trashDisabledClass = columnsLength == 1 ? "disabled" : "";

        this.$columnInput.show();

        if (!columnsLength) {
            $("#required").show();
            $("#duplicated").hide();
        } else {
            $("#required").hide();
        }

        this.$columnList.empty();
        const $columnElement = retrospectiveModel.getColumns(formatId).map(
            (column: Column, index) =>
                {
                    column.content=column.content.trim()
                    .replace(/</g, "&lt;")
                    .replace(/>/g, "&gt;")
                    .replace(/"/g, "&quot;")
                    .replace(/'/g, "&#039;");
                    return `<div class="column-row">
                <div class="column-row-number">${index + 1}.</div>
                    <form class="column-name-form" data-index="${index}">
                        <input id="column-name-input${index}" data-index="${index}" value="${
                    column.content
                }"
                        class="column-name-input" type="text" placeholder="Type column name here and hit enter..."
                        autocomplete="off" />
                    </form>
                    <div class="column-row-edit" data-index="${index}"><span class="aui-icon aui-icon-small aui-iconfont-edit"></span></div>
                    <div class="column-row-delete ${trashDisabledClass}" data-index="${index}"><span class="aui-icon aui-icon-small aui-iconfont-delete" ></span></div>
                    <div id="column-color-input-container${index}" class="column-color-input-container" data-index="${index}" style="background-color: ${
                    column.color
                }">
                        <input id="column-color-input${index}" data-index="${index}" value="${
                    column.color
                }" class="column-color-input" type="hidden"/>
                    </div>

             </div>`
            }
        );

        this.$columnList.append($columnElement.join(""));
    }

    focusOnColumnAdded() {
        const lastColumnIndex =
            retrospectiveModel.getColumns(this.formatId).length - 1;
        const inputField: any = $(`#column-name-input${lastColumnIndex}`);

        inputField.focus();
        inputField.blur();
    }
}

export default ColumnCreate;
