import IBaseStep from "../../declarations/IBaseStep";
import wizardModel from "../../models/WizardModel";
import retrospectiveModel from "../../models/RetrospectiveModel";
import userService from '../../services/UserService';
import navigation from "../../navigation";


declare const AJS: any;

class Preferences implements IBaseStep {

    private $unlimitedCardVotes:JQuery;
    private $unlimitedPlayerVotes:JQuery;
    private $playerVotesInput:JQuery;
    private $cardVotesInput:JQuery;
    private $anonymous:JQuery;
    private $presetModeratorRadio:JQuery;
    private $dynamicModeratorRadio:JQuery;
    private $presetModeratorSelect:JQuery;
    private $curElement:JQuery;
    private $preferencesIcon:JQuery
    private $techniqueSelected:JQuery
    private $happyIconWizardElement:JQuery

    constructor(private $element:JQuery) {
        this.$curElement = $element;
        this.$unlimitedCardVotes = $element.find('#card-votes');
        this.$unlimitedPlayerVotes = $element.find('#player-votes');
        this.$preferencesIcon = $element.find('#preferences-icon')
        this.$playerVotesInput = $element.find('#player-votes-input');
        this.$cardVotesInput = $element.find('#card-votes-input');
        this.$anonymous = $element.find('#anonymous');
        this.$presetModeratorRadio = $element.find('#preset-moderator-static-strategy');
        this.$dynamicModeratorRadio = $element.find('#preset-moderator-dynamic-strategy');
        this.$presetModeratorSelect = $element.find('#preset-moderator');
        this.$techniqueSelected = $element.find('#format-selected')
        this.$happyIconWizardElement = $element.find('#task-create')
        if (!wizardModel.getEditSettingsAvailable()) {
            this.$playerVotesInput.prop('disabled', true);
            this.$unlimitedPlayerVotes.prop('disabled', true);
            this.$unlimitedCardVotes.prop('disabled', true);
            this.$cardVotesInput.prop('disabled', true);
            this.$anonymous.prop('disabled', true);
            $('.overlay').css('display', 'inherit');
            $('#no-edit-message').css('display', 'inherit');
        }
    }

    bindEvents() {
        this.$happyIconWizardElement.on('click', this.onPrevious.bind(this));
        this.$unlimitedCardVotes.on('click', this.unlimitedSelected.bind(this));
        this.$unlimitedPlayerVotes.on('click', this.unlimitedSelected.bind(this));
        this.$cardVotesInput.on('blur', this.saveCardVotes.bind(this));
        this.$playerVotesInput.on('blur', this.savePlayerVotes.bind(this));
        this.$anonymous.on('click', this.anonymousSelection.bind(this));
        this.$presetModeratorRadio.on('click', this.presetModeratorStrategySelected.bind(this));
        this.$presetModeratorSelect.on('change', this.presetModeratorStrategySelection.bind(this));
        this.$dynamicModeratorRadio.on('click', this.dynamicModeratorStrategySelected.bind(this));
    }

    unbindEvents() {
        this.$unlimitedCardVotes.off();
        this.$unlimitedPlayerVotes.off();
    }

    onLoad(formatId, isEditing, technique) {
        this.$techniqueSelected.text(technique)
        const votesPerPlayer = retrospectiveModel.getVotesPerPlayer();
        const votesPerCard = retrospectiveModel.getVotesPerCard();

        this.$preferencesIcon.attr('class', 'preference-icon-active')
        if (votesPerCard != 0) {
            this.$cardVotesInput.val(votesPerCard);
            this.$cardVotesInput.prop('disabled', false);
            $('#card-votes').prop('checked', false);
        } else {
            this.$cardVotesInput.val('1');
            this.$cardVotesInput.prop('disabled', true);
            $('#card-votes').prop('checked', true);
        }

        if (votesPerPlayer != 0) {
            this.$playerVotesInput.val(votesPerPlayer);
            this.$playerVotesInput.prop('disabled', false);
            $('#player-votes').prop('checked', false);
        } else {
            this.$playerVotesInput.val('1');
            this.$playerVotesInput.prop('disabled', true);
            $('#player-votes').prop('checked', true);
        }

        this.$anonymous.prop('checked', retrospectiveModel.isAnonymousPlayers());

        const currentPresetModeratorUserKey = retrospectiveModel.getPresetModeratorUserKey();

        if (!currentPresetModeratorUserKey || currentPresetModeratorUserKey === 'RETROS_DYNAMIC_MODERATOR') {
            this.dynamicModeratorStrategySelected(null);
        }

        if (currentPresetModeratorUserKey && currentPresetModeratorUserKey !== 'RETROS_DYNAMIC_MODERATOR') {
            this.$presetModeratorRadio.prop('checked', 'checked');
            this.presetModeratorStrategySelected(null, currentPresetModeratorUserKey);
        }
    }

    onNext() {
    }

    onPrevious() {
        navigation.onBackClick()
    }

    sanitize($control:JQuery) {
        const val = $control.val();
        let sanitized = val.split('').filter(e => !isNaN(e)).join('');
        if( sanitized.length > 0 ) {
            if (parseInt(sanitized) <= 0) {
                sanitized = 1;
            }else{
                sanitized = parseInt(sanitized);
            }
        }else{
            sanitized = 1;
        }

        $control.val(sanitized);
    }

    saveCardVotes() {
        this.sanitize(this.$cardVotesInput);
        this.validValue()
        retrospectiveModel.setVotesPerCard($('#card-votes').prop('checked') ? 0 : parseInt(this.$cardVotesInput.val()));
    }

    savePlayerVotes() {
        this.sanitize(this.$playerVotesInput);
        this.validValue()
        retrospectiveModel.setVotesPerPlayer($('#player-votes').prop('checked') ? 0 : parseInt(this.$playerVotesInput.val()));
    }

    validValue(): boolean {
        if ($('#card-votes').prop('checked') || $('#player-votes').prop('checked')) {
            this.displayValidValuesMessage();
            return true;
        }

        let votesPerCard = parseInt(this.$cardVotesInput.val());
        let votesPerPlayer = parseInt(this.$playerVotesInput.val());
        if (votesPerCard <= votesPerPlayer) {
            this.displayValidValuesMessage();
            return true;
        } else {
            this.displayNoValidValuesMessage();
            return false;
        }
    }

    displayNoValidValuesMessage() {
        $('#dialog-submit-button').prop('disabled', true);
        $('#invalid-values').show();
        $(this.$cardVotesInput).addClass('border-red');
    }

    displayValidValuesMessage() {
        $('#dialog-submit-button').prop('disabled', false);
        $('#invalid-values').hide();
        $(this.$cardVotesInput).removeClass('border-red');
    }

    anonymousSelection(event:Event) {
        const $target = $(event.target);

        if ($target.is(':checked')) {
            retrospectiveModel.setAnonymousPlayers(true);
        } else {
            retrospectiveModel.setAnonymousPlayers(false);
        }
    }

    unlimitedSelected(event:Event) {
        const $target = $(event.target);
        this.validValue();
        if ($target.attr('id') == 'card-votes') {
            if ($target.is(':checked')) {
                retrospectiveModel.setVotesPerCard(0);
                this.$cardVotesInput.prop('disabled', true);
                this.$cardVotesInput.val('1');
            } else {
                let value = parseInt(this.$cardVotesInput.val());
                if( value == 0 ){
                    value = 1;
                    this.$cardVotesInput.val(value);
                }
                retrospectiveModel.setVotesPerCard(value);
                this.$cardVotesInput.prop('disabled', false);
            }
        } else if ($target.attr('id') == 'player-votes') {
            if ($target.is(':checked')) {
                retrospectiveModel.setVotesPerPlayer(0);
                this.$playerVotesInput.prop('disabled', true);
                this.$playerVotesInput.val('1');
            } else {
                let value = parseInt(this.$playerVotesInput.val());
                if( value == 0 ){
                    value = 1;
                    this.$playerVotesInput.val(value);
                }
                retrospectiveModel.setVotesPerPlayer(value);
                this.$playerVotesInput.prop('disabled', false);
            }
        }
    }

    presetModeratorStrategySelected(event:Event, currentPresetModeratorUserKey?: string) {
        this.$presetModeratorSelect.show();
        this.$curElement.find('#s2id_preset-moderator').show();
        if(currentPresetModeratorUserKey){
            this.$presetModeratorSelect.val(currentPresetModeratorUserKey);
        }
        const select2Config = userService.getUsersSelect2Config(this.presetModeratorStrategySelection); // pass the selection function to use after init
        AJS.$("#preset-moderator").auiSelect2(select2Config);
    }

    presetModeratorStrategySelection(event:Event, value?: string) {
        const selectedValue = value || this.$presetModeratorSelect.val();
        retrospectiveModel.setPresetModeratorUserKey(selectedValue);
        console.log('Set preset moderator value', selectedValue);
    }

    dynamicModeratorStrategySelected(event:Event) {
        this.$presetModeratorSelect.hide();
        this.$curElement.find('#s2id_preset-moderator').hide();
        retrospectiveModel.setPresetModeratorUserKey('RETROS_DYNAMIC_MODERATOR');
        console.log('Set preset moderator value RETROS_DYNAMIC_MODERATOR');

    }

}

export default Preferences;
