import Dialog from "../dialog";
import stepsModel from "../models/StepsModel";
import IBaseStep from "../declarations/IBaseStep";
import retrospectiveModel from "../models/RetrospectiveModel";
import wizardModel from "../models/WizardModel";

class Navigation {
    private insertMacro: () => Promise<any>;
    private $contentElement: JQuery;
    private dialog: Dialog;
    private loadedCtrl: IBaseStep;
    // buttons
    private $insertBtn: JQuery;
    private $closeBtn: JQuery;
    private $nextBtn: JQuery;
    private $backBtn: JQuery;

    init(dialog: Dialog, insertMacro: () => Promise<any>, contentId: string, isEditing = false, formatId = 1,) {
        this.insertMacro = insertMacro;
        if (!dialog.$element.length) {
            throw new Error('Dialog not initialized');
        }

        this.$contentElement = dialog.$element.find(`#${contentId}`);
        if (!this.$contentElement.length) {
            throw new Error('Content element not found');
        }
        this.dialog = dialog;
        this.bindEvents();
        this.loadContent(formatId, isEditing, retrospectiveModel.getTechniqueSelected() );
        this.updateButtons();
    }

    bindEvents() {
        const $parent = this.dialog.$element;

        this.$insertBtn = $parent.find('#dialog-submit-button');
        this.$closeBtn = $parent.find('#dialog-close-button');

        this.$insertBtn.on('click', this.onInsert.bind(this));
        this.$closeBtn.on('click', this.onCancel.bind(this));

        this.$nextBtn = $parent.find('.dialog-next-button');
        this.$backBtn = $parent.find('.dialog-back-button');
        this.$nextBtn.on('click', this.onNext.bind(this));
        this.$backBtn.on('click', this.onBack.bind(this));


    }

    onInsert() {
        this.insertMacro().then(this.onCancel.bind(this));
    }

    onCancel() {
        this.dialog.hide();
        this.dialog.$element.remove();
        stepsModel.reset();
        retrospectiveModel.clear();
    }

    updateButtons() {
        const stepConfig = stepsModel.getCurrentConfiguration();
        if (!wizardModel.getEditSettingsAvailable()) {
            this.$insertBtn.prop('disabled', true);
            this.$nextBtn.prop("disabled", true);
            this.$backBtn.prop("disabled", true);
            return
        }

        if (stepConfig.nextStep) {
            this.$nextBtn.prop("disabled", false);
            this.$insertBtn.prop("disabled", true);
            } else{
            this.$nextBtn.prop("disabled", true);
            this.$insertBtn.prop("disabled", false);
        }

        if (stepConfig.previousStep) {
            this.$backBtn.prop("disabled", false);
        } else {
            this.$backBtn.prop("disabled", true);
        }
    }

    bindEventsNavigationButtons($element: JQuery) {
    }

    loadContent(formatId , isEditing , technique) {
        const stepConfig = stepsModel.getCurrentConfiguration();

        if (stepConfig.template) {
            const $element = $(stepConfig.template);
            this.$contentElement.append($element);
            this.loadedCtrl = new stepConfig.entryClass($element);

            this.loadedCtrl.onLoad(formatId, isEditing, technique);
            this.loadedCtrl.bindEvents();

            this.bindEventsNavigationButtons($element);
            this.updateButtons();
        }
    }

    removeStep() {
        if (this.loadedCtrl) {
            this.loadedCtrl.unbindEvents();
            this.$contentElement.empty();
        }
    }

    onNextClick() {
        if(stepsModel.hasNext()) {//we are in the first step
            this.removeStep();
            stepsModel.next();
            let formatId = retrospectiveModel.getFormatId()
            let technique = retrospectiveModel.getTechniqueSelected()
            this.loadContent(formatId,false, technique);
            this.updateButtons();
        }
    }

    onNext() {
        if(stepsModel.hasNext() && retrospectiveModel.getColumns(retrospectiveModel.getFormatId()).length) {//we are in the first step
            this.loadedCtrl.onNext();
            this.removeStep();
            stepsModel.next();

            let formatId = retrospectiveModel.getFormatId()
            let technique = retrospectiveModel.getTechniqueSelected()

            this.loadContent(formatId,false, technique);
            this.updateButtons();
        }
    }

    onBackClick() {
        this.removeStep();
        stepsModel.back();
        let formatId = retrospectiveModel.getFormatId()
        this.loadContent(formatId, true, retrospectiveModel.getTechniqueSelected());
        this.updateButtons();
    }

    onBack() {
        this.loadedCtrl.onPrevious();
        this.removeStep();
        stepsModel.back();
        let formatId = retrospectiveModel.getFormatId()
        this.loadContent(formatId, true, retrospectiveModel.getTechniqueSelected());
        this.updateButtons();
    }
}
const navigation = new Navigation();

export default navigation;