(function (angular) {
    angular
        .module('retrospectiveApp')
        .controller('discussController', DiscussController);

    DiscussController.$inject = ['IMAGES_BASE_PATH', 'HTTP_PATH', '$http', '$scope', 'DiscussService', 'UserService',
        'TopicService', 'SettingsService', 'UtilService', 'WorkflowService'];

    function DiscussController(IMAGES_BASE_PATH, HTTP_PATH, $http, $scope, DiscussService, UserService,
                               TopicService, SettingsService, UtilService, WorkflowService) {
        const vm = this;

        vm.readOnylMode = false;
        vm.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        vm.userKey = $scope.$parent.vm.userKey;
        vm.uuid = $scope.$parent.vm.uuid;
        vm.moderator = $scope.$parent.vm.moderator;
        vm.settings = SettingsService.settings;
        vm.currentTopicIndex = 0;
        vm.tooltipPrevious = 'Previous';
        vm.tooltipNext = 'Next';
        vm.newActionItemDescription = '';
        vm.colorsColumn = {};
        vm.topics = [];
        vm.confluenceUsers=[];
        vm.userSearch = '';
        vm.confluenceUsersResult = [];
        vm.retroParticipants = [];
        vm.participantsHash = '';

        vm.thinkElements = [];
        this.groupCollection = null;

        WorkflowService.subscribe((event)=>{
            this.readOnlyMode = true
        },['readOnlyMode']);

        WorkflowService.subscribe((event) => {
            this.moderator = event.state.moderator;
        }, ['moderator']);

        WorkflowService.subscribe((event) => {
            this.retroParticipants = event.state.participants;
            //we only update retro participants if the participant list has been changed since the last time
            // this is to avoid unnecessary refreshing that will cause conflicts with user search in action items assigning
            let newParticipantsHash = '';
            this.retroParticipants.forEach((participant) => newParticipantsHash = newParticipantsHash.concat(participant.userKey));
            if(vm.participantsHash !== newParticipantsHash){
                initializeUsersForActionItems();
                vm.participantsHash = newParticipantsHash;
            }
        }, ['participants']);

        SettingsService.subscribe((event) => {
            this.settings = event.state.settings;
        }, ['settings']);

        $http.get(`${HTTP_PATH}/ideas?uuid=${vm.uuid}`).then((response) => {
            vm.thinkElements = response.data;

            DiscussService.subscribe((event) => {
                this.groupCollection = {
                    groupList: event.state.groupList
                };

                // update this in another thread
                vm.currentTopic = this.groupCollection.groupList[vm.currentTopicIndex];
                vm.topics = this.groupCollection.groupList;
            }, ['groupList']);
        }).catch(err => {
            console.error(err);
            const flagError = AJS.flag({
                type: 'error',
                body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
            });
        });

        WorkflowService.subscribe((event) => {
            if (this.currentTopicIndex != event.state.index) {
                this.currentTopicIndex = event.state.index;

                // clear users input when we change the action item on discussion
                this.newActionItemDescription = '';
            }
        }, ['discussGroupIndex']);

        this.findColor = (columnId) => {
            const column = this.settings.columns.find((col) => col.columnId == columnId);
            if (column) {
                return column.color;
            }
            return '#000';
        }

        const previousTopic = () => {
            if (vm.currentTopicIndex > 0) {
                DiscussService.previousTopic()
                .then(() => {
                    WorkflowService.sessionDataPoller.poll();
                })
                .catch(err => {
                    console.error(err);
                    const flagError = AJS.flag({
                        type: 'error',
                        body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                    });
                });
            }
        };

        const nextTopic = () => {
            if (vm.currentTopicIndex < vm.topics.length) {
                DiscussService.nextTopic()
                .then(() => {
                    WorkflowService.sessionDataPoller.poll();
                })
                .catch(err => {
                    console.error(err);
                    const flagError = AJS.flag({
                        type: 'error',
                        body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                    });
                });
            }
        };

        const addActionItem = () => {
            if (vm.newActionItemDescription.length) {
                const newActionItem = {};
                newActionItem.description = vm.newActionItemDescription;
                DiscussService.addActionItem(vm.currentTopic.id, newActionItem).catch(err => {
                    console.error(err);
                    const flagError = AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
                    });
                });
                vm.newActionItemDescription = '';
            }
        };

        const deleteActionItem = (actionItem) => {
            vm.currentTopic.actionItems.splice(vm.currentTopic.actionItems.indexOf(actionItem), 1);
            DiscussService.removeActionItem(vm.currentTopic.id, actionItem.id).catch(err => {
                console.error(err);
                const flagError = AJS.flag({
                    type: 'error',
                    body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
                });
            });
        };

        const assigneeUser = (index, actionItem, user) => {
            AJS.$(`#trigger-dropdown${index}${vm.uuid}`).trigger('aui-button-invoke');
            if (user) {
                const newUser = Object.assign({}, user);
                delete newUser.isParticipant;
                DiscussService.setAssignee(vm.currentTopic.id, actionItem.id, newUser).catch(err => {
                    console.error(err);
                    const flagError = AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
                    });
                })
            } else {
                DiscussService.clearAssignee(vm.currentTopic.id, actionItem.id).catch(err => {
                    console.error(err);
                    const flagError = AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
                    });
                });
            }

        };

        const isUserAssignee = (index, user) => {
            const assignee = vm.currentTopic.actionItems[index].assigneeUser;
            return assignee && assignee.userName == user.userName;
        };


        const getVoteText = votes => votes === 1 ? 'Vote' : 'Votes';

        const isCurrentUserMod = () => {
            if (vm.moderator) {
                return vm.moderator.userKey === vm.userKey;
            }

            return UserService.getUserByUserKey(vm.userKey).isModerator;
        };

        const searchForUser = async () => {
            if(!vm.userSearch.trim().length){
                initializeUsersForActionItems();
                return
            }

            vm.confluenceUsersResult = await UserService.getUsersBySearch(vm.userSearch.trim())

            if(!vm.confluenceUsersResult.length){
                initializeUsersForActionItems();
            }
        }

        const cleanUserSearch = () => {
            vm.userSearch = '';
            initializeUsersForActionItems()
        }

        const initializeUsersForActionItems = () => {
            vm.confluenceUsersResult = vm.retroParticipants
        }

        vm.isUserAssignee = isUserAssignee;
        vm.previousTopic = previousTopic;
        vm.nextTopic = nextTopic;
        vm.addActionItem = addActionItem;
        vm.getVoteText = getVoteText;
        vm.assigneeUser = assigneeUser;
        vm.deleteActionItem = deleteActionItem;
        vm.isCurrentUserMod = isCurrentUserMod;
        vm.searchForUser = searchForUser;
        vm.cleanUserSearch = cleanUserSearch;

        //activate();
    }
}(angular));
