(function (angular) {
    angular
        .module('retrospectiveApp')
        .controller('thinkController', thinkController)
        .directive('autofocusIdea', () => {
            return {
                restrict: 'A',
                link: (scope, element, attrs) => {
                    scope.$watch(() => {
                        return scope.$eval(attrs.autofocusIdea);
                    }, (newValue) => {
                        if (newValue === true) {
                            element[0].focus();
                        }
                    })
                }
            }
        });

    thinkController.$inject = ['IMAGES_BASE_PATH', '$timeout', 'StatefulService', 'IdeasService', 'SettingsService', 'WorkflowService', 'UserService'];


    function thinkController(IMAGES_BASE_PATH, $timeout, Stateful, IdeasService, SettingsService, WorkflowService, UserService) {
        Stateful.make(this);
        this.readOnlyMode = false;
        this.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        this.userKey = this.state.shared._get('userKey');
        this.uuid = this.state.shared._get('uuid');
        this.settings = SettingsService.settings;
        this.mapIdeaColumns = {};
        this.mapErrorColumns = {};
        this.mapErrorIdeas = {}
        this.ideasInput = {};
        this.minColumnHeight = 500;
        this.currentUser = {}

        /*
         we'll store an object with keys related to the ids of the items being edited
         this way we can seed the "editing" variables when we get new
         idea from the server
         */
        this.ideasBeingEdited = {};

        /*
         used to store the original idea in text before
         the user changes it in the input
         */
        this.originalIdea = {};

        this.$onInit = () => {
            this.currentUser = UserService.getUserByUserKey(this.userKey);

            if (this.currentUser) {
                this.currentUser.imageURI = this.currentUser.imageUri
            }
        }

        WorkflowService.subscribe((event)=>{
            this.readOnlyMode = true
        },['readOnlyMode']);

        SettingsService.subscribe((event) => {
            this.settings = event.state.settings;
        }, ['settings']);

        IdeasService.subscribe((event) => {
            const columns = SettingsService.settings.columns;
            for (var i = 0; i < columns.length; i++) {
                var columnId = columns[i].columnId;

                if (JSON.stringify(this.mapIdeaColumns[columnId]) != JSON.stringify(event.state.ideas[columnId])) {
                    this.mapIdeaColumns[columnId] = (event.state.ideas[columnId] || []).map((idea) => {
                        idea.editing = this.ideasBeingEdited[idea.id];

                        /*
                         don't update the content if it's being edited
                         */
                        if (idea.editing) {
                            const found = this.mapIdeaColumns[columnId].find(i => i.id === idea.id);

                            if (found) {
                                idea.idea = found.idea;
                            }
                        }

                        return idea;
                    });
                }
            }
        }, ['ideas']);

        const isExisting = (item, collection) => {
            if (!item || item.length < 1) return false;
            const newIdea = item;
            if (collection === undefined) {
                return false;
            }

            return collection.find(element => {
                if (element.idea) {
                    if (element.user.userKey === this.userKey) {
                        if (element.editing) {
                            const originalIdea = JSON.parse(this.originalIdea[element.id]);

                            return originalIdea.toLowerCase() === newIdea.toLowerCase();
                        }

                        return element.idea.toLowerCase() === newIdea.toLowerCase();
                    }
                } else {
                    return element.toLowerCase() !== newIdea;
                }
            });
        };

        const setError = (columnId, show) => {
            this.mapErrorColumns[columnId] = show;
        };

        const postIdea = (idea, columnId) => {
            IdeasService.create({
                userKey: this.userKey,
                uuid: this.uuid,
                idea: idea.idea,
                columnId
            }).catch(err => {
                console.error(err);

                // Remove idea from the array
                const ideaIndex = this.mapIdeaColumns[columnId].findIndex(i => i.id === idea.id);

                if (ideaIndex !== -1) {
                    this.mapIdeaColumns[columnId].splice(ideaIndex, 1)
                }

                AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        const addToCollection = (item, columnId) => {

            if (!item) {
                return;
            }

            if (!this.mapIdeaColumns[columnId]) {
                this.mapIdeaColumns[columnId] = [];
            }

            if (isExisting(item, this.mapIdeaColumns[columnId])) {
                setError(columnId, true);
            } else {
                const newItem = {
                    columnId,
                    editable: true,
                    id: Math.random().toString(36).substr(2),
                    idea: item,
                    user: this.currentUser
                };

                this.mapIdeaColumns[columnId].push(newItem);

                postIdea(newItem, columnId);
                this.ideasInput[columnId] = '';
                setError(columnId, false);
            }
        };

        const insertElement = (keyEvent, columnId) => {
            this.mapErrorColumns[columnId] = false;
            if (keyEvent.which === 13) {
                addToCollection(this.ideasInput[columnId], columnId)
            }
        };

        const removeItem = (idea) => {
            IdeasService.remove(idea.id)
                .catch(err => {
                console.error(err);
                    AJS.flag({
                        type: 'error',
                        body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                    });
                });
        };

        /*
         calculate string width in screen
         */
        const hasTitleEllipsis = (text) => {
            const ellipsisHelper = document.getElementById('ellipsis-calc-helper');
            if(!ellipsisHelper) {
                return false;
            }
            ellipsisHelper.innerHTML = text;
            ellipsisHelper.style.fontSize = '20px';
            const textWidth = ellipsisHelper.clientWidth + 1;
            const widthAvailable = $('.title-think').width();
            return textWidth > widthAvailable;
        };

        const startEdit = (item) => {
            item.editing = true;
            this.ideasBeingEdited[item.id] = true;
            this.originalIdea[item.id] = JSON.stringify(item.idea);
            
            $timeout(() => {
                $(`#input-edit-idea-${item.id}`).focus();
            }, 100);
        };

        const trySave = ($event, item, columnId) => {
            if ($event.keyCode === 13) {
                saveEdit(item, columnId);
            }
        };

        const saveEdit = (item, columnId) => {
            let isDuplicated = false

            if (this.mapIdeaColumns[columnId]) {
                for (const idea of this.mapIdeaColumns[columnId]) {
                    if (idea.id !== item.id && idea.user.userKey === this.userKey) {
                        if(idea.editing) {
                            const originalIdea = JSON.parse(this.originalIdea[idea.id]);

                            if (originalIdea.toLowerCase() === item.idea.toLowerCase()) {
                                isDuplicated = true;
                            }
                        } else if (idea.idea.toLowerCase() === item.idea.toLowerCase()) {
                            isDuplicated = true;
                        }
                    }
                }
            }

            if (isDuplicated) {
                // Set this flag to true to show an error message
                this.mapErrorIdeas[item.id] = true;
            } else {
                // Hide the error message
                this.mapErrorIdeas[item.id] = false;

                // Update idea
                IdeasService.update(item).then(() => {
                    item.editing = false;
                    delete this.originalIdea[item.id];
                    delete this.ideasBeingEdited[item.id];
                }).catch(err => {
                    console.error(err);
                    AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
                    });
                });
            }
        };

        const cancelEdit = (item) => {
            item.editing = false;
            item.idea = JSON.parse(this.originalIdea[item.id]);

            this.mapErrorIdeas[item.id] = false;
            delete this.originalIdea[item.id];
            delete this.ideasBeingEdited[item.id];
        };

        this.getIdeasTextClass = () => {
            return (this.settings.anonymousPlayersAllowed) ? "anonimus-ideas-text-width" : "ideas-text-width";
        }

        this.hasMoreThanFourColumns = () => {
            return this.settings.columns.length > 4;
        }

        this.init = () => {
            if (this.hasMoreThanFourColumns()) {
                $('#retro-container').addClass('more-than-four-columns-container');
                $('#retro-container').removeClass('retro-container');
            }
        }

        this.trySave = trySave;
        this.saveEdit = saveEdit;
        this.cancelEdit = cancelEdit;
        this.startEdit = startEdit;
        this.hasTitleEllipsis = hasTitleEllipsis;
        this.insertElement = insertElement;
        this.removeItem = removeItem;
        this.setError = setError;
        this.isIssueOwner = issue => issue.user && issue.user.userKey === this.userKey;
        this.getColumnClass = SettingsService.getColumnClass;
    }
}(angular));
