'use strict';

(function (angular) {
    console.rlog = function () {
        var _console;

        for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
            args[_key] = arguments[_key];
        }

        (_console = console).log.apply(_console, ['AGILE RETROSPECTIVES'].concat(args));
    };

    angular.module('retrospectiveApp', ['dndLists']).config(function ($httpProvider) {
        /**
         * interceptor to log response time
         */
        //$httpProvider.interceptors.push('performanceTracker');

        //initialize get if not there
        if (!$httpProvider.defaults.headers.get) {
            $httpProvider.defaults.headers.get = {};
        }

        // Answer edited to include suggestions from comments
        // because previous version of code introduced browser-related errors

        //disable IE ajax request caching
        $httpProvider.defaults.headers.get['If-Modified-Since'] = 'Mon, 26 Jul 1997 05:00:00 GMT';
        // extra
        $httpProvider.defaults.headers.get['Cache-Control'] = 'no-cache';
        $httpProvider.defaults.headers.get['Pragma'] = 'no-cache';

        $httpProvider.defaults.headers.common['X-Atlassian-Token'] = 'no-check';
        $httpProvider.defaults.headers.common['Content-Type'] = 'application/json';

        $httpProvider.interceptors.push(['$q', function ($q) {
            return {
                'responseError': function responseError(rejection) {
                    if (rejection.status === 403 && rejection.data.message === 'PERMISSION_REVOKED') {
                        $('#permissions-revoked-modal').modal({ backdrop: 'static', show: true });
                    }
                    return $q.reject(rejection);
                }
            };
        }]);
    });
})(angular);
(function (angular) {
    /*
     The motivation for this logic is that
     when we are viewing a page, after the base url,
     we'll find the /display path, so that we can distinct between the base url and
     the path of the page.
      The approach of /rest is no longer working since users can have
     a base url with a permanent path included
      i.e.
      https://mydomain.com/itservices/
      instead of just
      https://mydomain.com/
     */

    function tryGetBase() {
        var baseUrl = window.location.origin + window.location.pathname;

        // we enter using the confluence UI
        if (baseUrl.indexOf('/display') >= 0) {
            return baseUrl.split('/display')[0];
        }
        // we enter by copy and paste a url
        else if (baseUrl.indexOf('/pages') >= 0) {
                return baseUrl.split('/pages')[0];
            } else if (baseUrl.indexOf('/plugins') >= 0) {
                return baseUrl.split('/plugins')[0];
            }
            // we'll use default value
            else {
                    return null;
                }
    }

    function getBaseUrl() {
        var url = tryGetBase();
        return url ? url : '/rest';
    }

    function getServletBaseUrl() {
        var url = tryGetBase();
        return url ? url : '';
    }

    angular.module('retrospectiveApp').constant('HTTP_PATH', getBaseUrl() + '/rest/retrospective/latest').constant('IMAGES_BASE_PATH', getBaseUrl()).constant('SERVLET_PATH', '' + getServletBaseUrl());
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').service('DiscussService', DiscussService);

    DiscussService.$inject = ['HTTP_PATH', '$http', 'SubscribableService', 'StatefulService', 'PollerService'];

    function DiscussService(HTTP_PATH, $http, Subscribable, Stateful, Poller) {
        var _this = this;

        Subscribable.make(this);

        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        this.groupsPoller = null;

        this.stopPoller = function () {
            if (!_this.groupsPoller) {
                return;
            }
            _this.groupsPoller.stop();
            _this.groupsPoller = null;
        };

        this.startPollers = function (userKey, uuid, interval, stopAfter) {
            if (_this.groupsPoller) {
                return;
            }

            _this.state.set('userKey', userKey);
            _this.state.set('uuid', uuid);

            _this.groupsPoller = new Poller('/group?uuid=' + _this.state.get('uuid'), function (response) {
                if (!response.data) {
                    return;
                }

                //remove group with ungrouped ideas (empty)
                response.data.groupList.splice(0, 1);

                _this.notify({
                    identifier: 'groupList',
                    state: {
                        groupList: response.data.groupList.map(function (item) {
                            if (!item.ideas) {
                                item.ideas = [];
                            }
                            return item;
                        }).sort(function (a, b) {
                            /*
                            if the groups have different votes, we'll order them based on the votes
                            otherwise, we'll order based on the id
                             */
                            if (b.votes != a.votes) {
                                return b.votes - a.votes;
                            } else {
                                return b.id - a.id;
                            }
                        })
                    }
                });
            }, interval);

            _this.groupsPoller.start(stopAfter);
        };

        this.addActionItem = function (groupId, actionItem) {
            return $http.post(HTTP_PATH + '/group/' + groupId + '/actionitems?userKey=' + _this.state.get('userKey') + '&uuid=' + _this.state.get('uuid'), actionItem).then(function () {
                return _this.groupsPoller.poll();
            });
        };

        this.removeActionItem = function (groupId, actionItemId) {
            return $http.delete(HTTP_PATH + '/group/' + groupId + '/actionitems/' + actionItemId + '?userKey=' + _this.state.get('userKey') + '&uuid=' + _this.state.get('uuid')).then(function () {
                return _this.groupsPoller.poll();
            });
        };

        this.setAssignee = function (groupId, actionItemId, user) {
            user.imageURI = user.imageUri;
            delete user.imageUri;
            delete user.isConnected;
            delete user.id;
            return $http.put(HTTP_PATH + '/group/' + groupId + '/actionitems/' + actionItemId + '/assignee?uuid=' + _this.state.get('uuid'), user).then(function () {
                return _this.groupsPoller.poll();
            });
        };

        this.toggleComplete = function (groupId, actionItemId) {
            return $http.put(HTTP_PATH + '/group/' + groupId + '/actionitems/' + actionItemId + '/togglecomplete?userKey=' + _this.state.get('userKey') + '&uuid=' + _this.state.get('uuid')).then(function () {
                return _this.groupsPoller.poll();
            });
        };

        this.clearAssignee = function (groupId, actionItemId) {
            return $http.delete(HTTP_PATH + '/group/' + groupId + '/actionitems/' + actionItemId + '/assignee?uuid=' + _this.state.get('uuid')).then(function () {
                return _this.groupsPoller.poll();
            });
        };

        this.nextTopic = function () {
            return $http.put(HTTP_PATH + '/workflow/nexttopic?userKey=' + _this.state.get('userKey') + '&uuid=' + _this.state.get('uuid'));
        };

        this.previousTopic = function () {
            return $http.put(HTTP_PATH + '/workflow/previoustopic?userKey=' + _this.state.get('userKey') + '&uuid=' + _this.state.get('uuid'));
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').service('EventService', EventService);

    EventService.$inject = ['HTTP_PATH', 'SERVLET_PATH', '$timeout', '$interval', '$http', 'DiscussService', 'VotesService', 'IdeasService', 'UserService', 'NotificationsService', 'WorkflowService'];

    function EventService(HTTP_PATH, SERVLET_PATH, $timeout, $interval, $http, DiscussService, VotesService, IdeasService, UserService, NotificationsService, WorkflowService) {
        var _this2 = this;

        this.init = function (_ref) {
            var userKey = _ref.userKey,
                uuid = _ref.uuid;

            _this2.initEvents(uuid, userKey);
        };

        this.initEvents = function (uuid, userKey) {

            /*
             Will take care of:
              - groups (including action items and votes)
             */
            var groupsPoller = new Poller('/topics?uuid=' + uuid, function (response) {});

            /*
             just start pollers and let's start the party!
             */

            groupsPoller.start();
        };

        this.userKey = '';
        this.uuid = '';
        this.eventSource = {
            value: null
        };

        this.thinkState = {
            value: false
        };
        this.groupState = {
            value: false
        };
        this.voteState = {
            value: false
        };
        this.discussState = {
            value: false
        };
        this.isUserReady = {
            value: false
        };

        this.completionDate = {
            value: ''
        };

        this.activeUsers = [];
        this.moderator = {};
        this.activeView = {
            value: ''
        };

        this.loginStatus = {
            processCompleted: false,
            isValidSession: false
        };

        this.nextAvailable = {
            value: true
        };

        this.heartbeat = null;
        this.heartbeatSender = null;

        this.setModerator = function (user) {
            var prevMod = '';

            if (_this2.moderator && _this2.moderator.userKey !== user.userKey) {
                prevMod = '&prevMod=' + _this2.moderator.userKey;
            }

            $http.put(HTTP_PATH + '/workflow/set-moderator?userKey=' + user.userKey + '&uuid=' + _this2.uuid + prevMod).then(function () {
                _this2.moderator = Object.assign(_this2.moderator, user);
                var index = _this2.activeUsers.indexOf(user);

                if (_this2.activeUsers[index]) {
                    _this2.activeUsers[index].isModerator = true;
                    UserService.setModeratorByIndex(index);
                }
            });
        };

        this.changeHeaderState = function (current) {
            switch (current) {
                case 'think':
                    _this2.thinkState.value = true;
                    _this2.groupState.value = false;
                    _this2.voteState.value = false;
                    _this2.discussState.value = false;
                    break;
                case 'group':
                    _this2.thinkState.value = true;
                    _this2.groupState.value = true;
                    _this2.voteState.value = false;
                    _this2.discussState.value = false;
                    break;
                case 'vote':
                    _this2.thinkState.value = true;
                    _this2.groupState.value = true;
                    _this2.voteState.value = true;
                    _this2.discussState.value = false;
                    break;
                case 'discuss':
                    _this2.thinkState.value = true;
                    _this2.groupState.value = true;
                    _this2.voteState.value = true;
                    _this2.discussState.value = true;
                    break;
                case 'main':
                    _this2.thinkState.value = false;
                    _this2.groupState.value = false;
                    _this2.voteState.value = false;
                    _this2.discussState.value = false;
                    break;
            }
        };

        this.changeView = function (current) {
            if (_this2.activeView.value == current) {
                return;
            }

            _this2.nextAvailable.value = false;
            $http.put(HTTP_PATH + '/workflow/change-view?userKey=' + _this2.userKey + '&uuid=' + _this2.uuid + '&viewName=' + current).then(function (response) {
                if (response.data.completionDate) {
                    _this2.completionDate.value = response.data.completionDate;
                }
                _this2.activeView.value = current;
                _this2.changeHeaderState(current);
                _this2.nextAvailable.value = true;
            });
        };

        var selfLoginEvent = function selfLoginEvent(result) {
            UserService.setUserList([].concat(result));

            _this2.activeUsers.length = 0;

            UserService.getUserList().forEach(function (u) {
                _this2.activeUsers.push(u);
            });
        };

        var changeIdeaEvent = function changeIdeaEvent(result) {

            for (var key in result.ideasMap) {
                if (result.ideasMap.hasOwnProperty(key)) {
                    IdeasService.setList(result.ideasMap[key], key);
                }
            }
            IdeasService.notify();
        };

        var newUserEvent = function newUserEvent(user) {
            var containsUser = _this2.activeUsers.find(function (activeUser) {
                return activeUser.userKey == user.userKey;
            });

            if (!containsUser) {
                _this2.activeUsers.push(user);
                UserService.setUserList(_this2.activeUsers.slice());
            }
        };

        var disconnectedUserEvent = function disconnectedUserEvent(disconnectedUser) {
            _this2.activeUsers.forEach(function (user, index) {
                if (user.userKey === disconnectedUser.userKey) {
                    _this2.activeUsers.splice(index, 1);
                }
            });

            UserService.setUserList(_this2.activeUsers.slice());
            var text = disconnectedUser.fullName + ' has left the session.';
            NotificationsService.sendNotification('', text);
        };

        var newModeratorEvent = function newModeratorEvent(moderator) {
            _this2.moderator = Object.assign(_this2.moderator, moderator);

            _this2.activeUsers.forEach(function (user) {
                user.isModerator = user.userKey === moderator.userKey;
            });

            if (_this2.moderator.userKey === _this2.userKey) {
                var text = 'You are the new Moderator of this Retro. You can transfer this role in the dropdown menu.';
                NotificationsService.sendNotification('Congratulations!', text, 'success');
            } else {
                var _text = 'has been successfully named Moderator.';
                NotificationsService.sendNotification(_this2.moderator.fullName, _text, 'success');
            }
        };

        var updateImReady = function updateImReady(user) {
            _this2.activeUsers.forEach(function (u) {
                if (u.userKey === user.userKey) {
                    u.isReady = user.isReady;
                }
            });

            if (_this2.moderator && _this2.moderator.userKey === user.userKey) {
                _this2.moderator.isReady = user.isReady;
            }
        };

        var resetImReady = function resetImReady() {
            _this2.activeUsers.forEach(function (user) {
                user.isReady = false;
            });

            if (_this2.moderator) {
                _this2.moderator.isReady = false;
            }

            _this2.isUserReady.value = false;
        };

        var groupIdeaEvent = function groupIdeaEvent(result) {
            IdeasService.setGroupsList(result.groupCollection);
            IdeasService.setGroupedIdea(result.groupedIdea);
            IdeasService.setDragIdeasEnabled(result.dragIdeasEnabled);
            IdeasService.notify();
        };

        var newGroupEvent = function newGroupEvent(result) {
            IdeasService.setGroupsList(result.groupCollection);
            IdeasService.setGroupedIdea(undefined);
            IdeasService.setDragIdeasEnabled(result.dragIdeasEnabled);
            IdeasService.notify();
        };

        var changeViewEvent = function changeViewEvent(result) {
            _this2.changeView(result.viewName);
        };

        var newVoteEvent = function newVoteEvent(vote) {
            VotesService.insertVote(vote);
            VotesService.notify();
        };

        var actionItemEvent = function actionItemEvent(groupElement) {
            DiscussService.updateGroupElement(groupElement);
            DiscussService.setIndex(undefined);
            DiscussService.notify();
        };

        var changeTopicEvent = function changeTopicEvent(index) {
            DiscussService.setIndex(index);
            DiscussService.notify();
        };

        var syncView = function syncView(data) {
            if (data.viewName && data.viewName != _this2.activeView.value) {
                _this2.activeView.value = data.viewName;
            }
        };

        var digest = function digest(fn) {
            return function (event) {
                $timeout(function () {
                    try {
                        var data = JSON.parse(event.data);
                        fn(data);
                    } catch (err) {
                        fn({});
                    }
                }, 0);
            };
        };

        var persistData = function persistData() {
            return $http.post(HTTP_PATH + '/workflow/persist/all?uuid=' + _this2.uuid);
        };

        var loginProcessCompletedEvent = function loginProcessCompletedEvent(data) {
            _this2.loginStatus.processCompleted = true;
            _this2.loginStatus.isValidSession = data.validSession;

            if (!data.validSession) {
                _this2.eventSource.value.close();

                if (_this2.heartbeat) {
                    $interval.cancel(_this2.heartbeat);
                    $interval.cancel(_this2.heartbeatSender);
                }
            }
        };

        this.init = function (_ref2) {
            var userKey = _ref2.userKey,
                uuid = _ref2.uuid,
                completionDate = _ref2.completionDate;

            _this2.userKey = userKey;
            _this2.uuid = uuid;
            _this2.completionDate.value = completionDate;

            _this2.eventSource.value = new EventSource(SERVLET_PATH + '/plugins/servlet/sse?userKey=' + userKey + '&uuid=' + uuid);

            _this2.eventSource.onerror = function () {
                console.log('EventSource failed.');
            };

            var secondsConnecting = 0;

            _this2.heartbeat = $interval(function () {
                if (_this2.eventSource.value.readyState == 1) {
                    secondsConnecting = 0;
                    _this2.loginStatus.processCompleted = true;
                    return;
                }

                if (_this2.eventSource.value.readyState != 1) {
                    secondsConnecting++;
                }

                if (secondsConnecting >= 20) {
                    _this2.loginStatus.processCompleted = true;
                    _this2.loginStatus.isValidSession = false;
                    _this2.eventSource.value.close();
                    _this2.eventSource.value = new EventSource(SERVLET_PATH + '/plugins/servlet/sse?userKey=' + userKey + '&uuid=' + uuid);
                    //$interval.cancel(this.heartbeat);
                    //$interval.cancel(this.heartbeatSender);
                }
            }, 1000);

            var time = Math.round(5000 + Math.random() * 5000);
            var time2 = Math.round(2000 + Math.random() * 5000);

            $timeout(function () {
                _this2.heartbeatSender = $interval(function () {
                    $http.get(HTTP_PATH + '/workflow/heartbeat?uuid=' + _this2.uuid + '&ts=' + Date.now());
                }, time2);
            }, time);

            _this2.persistData = persistData;
            _this2.initEvents();
        };

        this.initEvents = function () {
            _this2.eventSource.value.addEventListener('SELF_LOGIN', digest(selfLoginEvent));
            _this2.eventSource.value.addEventListener('NEW_IDEA', digest(changeIdeaEvent));

            _this2.eventSource.value.addEventListener('DELETE_IDEA', digest(changeIdeaEvent));
            _this2.eventSource.value.addEventListener('NEW_USER', digest(newUserEvent));
            _this2.eventSource.value.addEventListener('DISCONNECTED_USER', digest(disconnectedUserEvent));
            _this2.eventSource.value.addEventListener('NEW_MODERATOR', digest(newModeratorEvent));
            _this2.eventSource.value.addEventListener('ERROR', digest(console.error));
            _this2.eventSource.value.addEventListener('IM_READY', digest(updateImReady));
            _this2.eventSource.value.addEventListener('RESET_IM_READY', digest(resetImReady));
            _this2.eventSource.value.addEventListener('GROUP_IDEA', digest(groupIdeaEvent));
            _this2.eventSource.value.addEventListener('NEW_GROUP', digest(newGroupEvent));
            _this2.eventSource.value.addEventListener('CHANGE_VIEW', digest(changeViewEvent));
            _this2.eventSource.value.addEventListener('NEW_VOTE', digest(newVoteEvent));
            _this2.eventSource.value.addEventListener('ACTION_ITEM', digest(actionItemEvent));
            _this2.eventSource.value.addEventListener('CHANGE_TOPIC', digest(changeTopicEvent));
            _this2.eventSource.value.addEventListener('SYNC_VIEW', digest(syncView));
            _this2.eventSource.value.addEventListener('LOGIN_PROCESS', digest(loginProcessCompletedEvent));
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('DateService', DateService);

    function DateService() {

        var MONTHS = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

        return {
            format: format
        };

        /*
            returns a string with the format dd/MMM/YYYY ie. 17/May/2017
         */
        function format(date) {
            /*
                if there is no date, return ''
             */
            if (!date) {
                return '';
            }

            var localDate = new Date(date);

            var month = localDate.getMonth();
            var day = normalize(localDate.getDate());
            var year = localDate.getFullYear();
            return day + "/" + MONTHS[month] + "/" + year;
        }

        function normalize(value) {
            return value < 10 ? '0' + value : value;
        }
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('HeartbeatPollerService', HeartbeatPollerService);

    HeartbeatPollerService.$inject = ['HTTP_PATH', '$http', '$window', '$interval', '$timeout'];

    function HeartbeatPollerService(HTTP_PATH, $http, $window, $interval, $timeout) {
        return Poller;

        function Poller(handler) {
            var _this3 = this;

            var sleep = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 45000;

            this.interval = null;
            this.resolved = true;

            this.start = function (uuid, userKey) {
                var stopAfter = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;

                var counter = 0;
                console.log('RETROSPECTIVES: Start retrospective heartbeat');

                var resolver = function resolver(response, error) {
                    if (error) {
                        console.log('Error while polling', error);
                    } else {
                        handler(response);
                    }

                    $timeout(function () {
                        _this3.resolved = true;
                    }, 100);

                    if (stopAfter) {
                        counter++;
                        if (counter == stopAfter) {
                            _this3.stop();
                        }
                    }
                };

                var poll = function poll() {
                    return $http.post(HTTP_PATH + '/workflow/heartbeat?uuid=' + uuid + '&userKey=' + userKey).then(function (response) {
                        return resolver(response.data);
                    }).catch(function (err) {
                        return resolver(null, err);
                    });
                };

                //call heartbeat so that the user is connected
                poll();
                _this3.interval = $interval(function () {
                    if (_this3.resolved) {
                        _this3.resolved = false;
                        poll();
                    }
                }, sleep);
            };

            this.stop = function () {
                console.log('RETROSPECTIVES: Stop poll READ ONLY MODE');
                $interval.cancel(_this3.interval);
            };
        }
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('PollerService', PollerService);

    PollerService.$inject = ['HTTP_PATH', '$http', '$interval', '$timeout'];

    function PollerService(HTTP_PATH, $http, $interval, $timeout) {
        return Poller;

        /*
         helper for polling. instantiate it to start a new interval that will be
         polling until you stop it
         */
        function Poller(url, handler) {
            var _this4 = this;

            var sleep = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 1000;

            this.interval = null;
            this.identifier = '';
            this.resolved = true;

            this.setIdentifier = function (id) {
                return _this4.identifier = id;
            };

            this.start = function () {
                var stopAfter = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;

                var counter = 0;
                console.log('RETROSPECTIVES: Start poll ' + _this4.identifier + ' for url ' + url);

                var resolver = function resolver(response, error) {
                    if (error) {
                        console.log('Error while polling', _this4.identifier, error);
                    } else {
                        handler(response);
                    }

                    $timeout(function () {
                        _this4.resolved = true;
                    }, 100);

                    if (stopAfter) {
                        // counter++;
                        // if (counter == stopAfter) {
                        counter += sleep;
                        if (counter >= stopAfter) {
                            _this4.stop();
                        }
                    }
                };

                _this4.poll = function () {
                    return $http.get('' + HTTP_PATH + url).then(resolver).catch(function (err) {
                        return resolver(null, err);
                    });
                };;

                _this4.interval = $interval(function () {
                    if (_this4.resolved) {
                        _this4.resolved = false;
                        _this4.poll();
                    }
                }, sleep);

                $timeout(function () {
                    _this4.poll();
                }, 500);
            };

            this.stop = function () {
                console.log('RETROSPECTIVES: Stop poll ' + _this4.identifier + ' for url ' + url);

                $interval.cancel(_this4.interval);
            };
        }
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('ConfluenceAccessModeService', function () {
        return {
            isReadOnlyModeEnabled: isReadOnlyModeEnabled
        };
    });

    function isReadOnlyModeEnabled() {
        var meta = document.querySelector("meta[name='ajs-access-mode']");

        if (meta) {
            var metaContent = meta.getAttribute("content");
            return metaContent === 'READ_ONLY';
        }

        return false;
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('ReadOnlyModePollerService', ReadOnlyModePollerService);

    ReadOnlyModePollerService.$inject = ['IMAGES_BASE_PATH', '$http', '$window', '$interval', '$timeout'];

    function ReadOnlyModePollerService(IMAGES_BASE_PATH, $http, $window, $interval, $timeout) {
        return Poller;

        function Poller(handler) {
            var _this5 = this;

            var sleep = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 180000;

            this.interval = null;
            this.resolved = true;

            this.start = function () {
                var stopAfter = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;

                var counter = 0;
                console.log('RETROSPECTIVES: Start poll READ ONLY MODE');

                var resolver = function resolver(response, error) {
                    if (error) {
                        console.log('Error while polling', error);
                    } else {
                        handler(response);
                    }

                    $timeout(function () {
                        _this5.resolved = true;
                    }, 100);

                    if (stopAfter) {
                        counter++;
                        if (counter == stopAfter) {
                            _this5.stop();
                        }
                    }
                };

                var poll = function poll() {
                    return $http.get(IMAGES_BASE_PATH + '/rest/api/accessmode').then(function (response) {
                        return resolver(response.data === 'READ_ONLY');
                    }).catch(function (err) {
                        return resolver(null, err);
                    });
                };

                poll();
                _this5.interval = $interval(function () {
                    if (_this5.resolved) {
                        _this5.resolved = false;
                        poll();
                    }
                }, sleep);
            };

            this.stop = function () {
                console.log('RETROSPECTIVES: Stop poll READ ONLY MODE');
                $interval.cancel(_this5.interval);
            };
        }
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').service('StatefulService', StatefulService);

    function StatefulService() {
        var shared = new State();

        function State() {
            var _state = {};

            this._set = function (key, value) {
                _state[key] = value;
            };

            this._get = function (key) {
                return _state[key];
            };

            this._exists = function (key) {
                return _state.hasOwnProperty(key);
            };
        }

        this.make = function (scope) {
            var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : { allowNull: true, allowUndefined: true };

            scope.state = new State();

            /*
                use shared object will modify the value for any instance that called Stateful.make(this)
                keep in mind that:
                 - The value you set in a place would be changed in another place
                - you should call _set and _get in order to set and get values
             */
            scope.state.shared = shared;

            scope.state.exists = function (key) {
                return scope.state._exists(key);
            };

            scope.state.set = function (key, value) {
                if (key === null || key === undefined) {
                    throw new Error('Key expected but ' + key + ' provided');
                }

                if (!options.allowNull && value === null) {
                    throw new Error('Null values not allowed. Trying to set ' + key);
                }

                if (!options.allowUndefined && value === undefined) {
                    throw new Error('Undefined values not allowed. Trying to set ' + key);
                }

                scope.state._set(key, value);
            };

            scope.state.get = function (key) {
                var defaultValue = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;

                if (!scope.state._exists(key)) {
                    if (defaultValue === null) {
                        throw new Error(key + ' is not in storage');
                    } else {
                        return defaultValue;
                    }
                }

                return scope.state._get(key);
            };
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('SubscribableService', SubscribableService);

    function SubscribableService() {
        /*
         Association between identifiers and the
         properties that the state object should contain as required
         */
        var propDictionary = {
            'participants': ['participants'],
            'stage': ['stage'],
            'moderator': ['moderator'],
            'completionDate': ['completionDate'],
            'ideas': ['ideas'],
            'isUserReady': ['ready'],
            'groupList': ['groupList'],
            'groupedIdea': ['groupedIdea'],
            'discussGroupIndex': ['index'],
            'settings': ['settings'],
            'totalVotes': ['totalVotes'],
            'sessionParticipants': ['participants'],
            'readOnlyMode': ['readOnlyMode']
        };

        var voicemail = {};

        return {
            make: make
        };

        function make(scope) {
            scope.handlers = [];

            /*
             subscribe controllers to change in the events service
              handler: is jut a function to be called
             identifiers: an array containing ids that the handler should respond to
             */
            scope.subscribe = function (handler, identifiers) {
                if (!handler) {
                    throw new Error('No handler provided');
                }

                scope.handlers.push({
                    handler: handler,
                    identifiers: identifiers
                });

                identifiers.forEach(function (identifier) {
                    (voicemail[identifier] || []).forEach(function (event) {
                        handler(event);
                    });

                    //clear voice mail once the events were sent
                    //voicemail[identifier] = [];
                });
            };

            /*
             call to notify subscribers about a change
              model:
              {
                 identifier: String, // a key for the change so that controllers or services can identify it
                 state: Object // the information itself
             }
             */

            scope.notify = function (event) {
                if (!event.identifier || !event.state) {
                    throw new Error('Identifier or delta not provided');
                }

                /*
                    Since services will be expecting for an schema,
                    we should validate that the event's state contains some properties
                     these properties are defined in the object above this function
                 */
                var requiredProps = propDictionary[event.identifier];

                if (requiredProps.length) {
                    requiredProps.forEach(function (propName) {
                        if (!event.state.hasOwnProperty(propName)) {
                            throw Error('Property ' + propName + ' is expected on state object');
                        }
                    });
                }

                // pollers can start before the controller is subscribed to the tream
                // so let's create a a voice mail for these events
                if (!scope.handlers.length) {
                    voicemail[event.identifier] = voicemail[event.identifier] || [];
                    voicemail[event.identifier].push(event);
                }

                scope.handlers.forEach(function (item) {
                    if (item.identifiers.indexOf(event.identifier) >= 0) {
                        item.handler(event);
                    }
                });
            };
        }
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').service('GroupsService', GroupsService);

    GroupsService.$inject = ['HTTP_PATH', '$http', 'SubscribableService', 'StatefulService', 'PollerService'];

    function GroupsService(HTTP_PATH, $http, Subscribable, Stateful, Poller) {
        var _this6 = this;

        Subscribable.make(this);

        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        this.groupsPoller = null;
        this.groupedIdeaPoller = null;

        this.stopPoller = function () {
            if (!_this6.groupsPoller) {
                return;
            }

            _this6.groupsPoller.stop();

            if (!_this6.groupedIdeaPoller) {
                return;
            }

            _this6.groupedIdeaPoller.stop();

            _this6.groupsPoller = null;
            _this6.groupedIdeaPoller = null;
        };

        this.startPollers = function (userKey, uuid) {
            if (_this6.groupsPoller) {
                return;
            }

            _this6.state.set('groupListHash', '');
            _this6.state.set('userKey', userKey);
            _this6.state.set('uuid', uuid);

            _this6.groupedIdeaPoller = new Poller('/group/current?uuid=' + _this6.state.get('uuid'), function (response) {
                _this6.notify({
                    identifier: 'groupedIdea',
                    state: {
                        groupedIdea: response.data || {
                            name: '',
                            ids: []
                        }
                    }
                });
            });

            _this6.groupsPoller = new Poller('/group?uuid=' + _this6.state.get('uuid'), function (response) {
                if (!response.data) {
                    return;
                }

                // TODO hacer que devuelva collection
                var groupListHash = response.data.groupList.reduce(function (a, c) {
                    return a + ',' + (c.ideas || []).length + ',' + c.description;
                }, '');
                var stored = _this6.state.get('groupListHash', '');

                // TODO compare with state to avoid perpetual refresh
                if (groupListHash !== stored) {
                    _this6.state.set('groupListHash', groupListHash);

                    _this6.notify({
                        identifier: 'groupList',
                        state: { groupList: response.data.groupList.map(function (item) {
                                if (!item.ideas) {
                                    item.ideas = [];
                                }
                                return item;
                            }) }
                    });
                }
            });

            _this6.groupsPoller.start();
            _this6.groupedIdeaPoller.start();
        };

        this.addIdeaToGroupedIdea = function (idea) {
            return $http.put(HTTP_PATH + '/group/idea?uuid=' + _this6.state.get('uuid'), { thinkElement: idea }).then(function () {
                return _this6.groupsPoller.poll();
            });
        };

        this.removeIdeaFromGroupedIdea = function (idea) {
            return $http.delete(HTTP_PATH + '/group/idea/' + idea.id + '?uuid=' + _this6.state.get('uuid')).then(function () {
                return _this6.groupsPoller.poll();
            });
        };

        this.createGroupedIdea = function (idea) {
            return $http.post(HTTP_PATH + '/group/idea?uuid=' + _this6.state.get('uuid'), { thinkElement: idea }).then(function () {
                return _this6.groupedIdeaPoller.poll();
            });
        };

        this.editGroupName = function (name) {
            return $http.put(HTTP_PATH + '/group/idea/name?uuid=' + _this6.state.get('uuid'), { name: name }).then(function () {
                return _this6.groupedIdeaPoller.poll();
            });
        };

        // should this affect the GroupResponse ?
        this.saveGroup = function () {
            return $http.post(HTTP_PATH + '/group?uuid=' + _this6.state.get('uuid')).then(function () {
                return _this6.groupedIdeaPoller.poll();
            });
        };

        this.editGroup = function (group) {
            return $http.put(HTTP_PATH + '/group/' + group.id + '?uuid=' + _this6.state.get('uuid')).then(function () {
                return _this6.groupedIdeaPoller.poll();
            });
        };

        this.breakGroup = function () {
            return $http.delete(HTTP_PATH + '/group?uuid=' + _this6.state.get('uuid')).then(function () {
                return _this6.groupedIdeaPoller.poll();
            });
        };
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').service('IdeasService', IdeasService);

    IdeasService.$inject = ['HTTP_PATH', '$http', 'SubscribableService', 'StatefulService', 'PollerService'];

    function IdeasService(HTTP_PATH, $http, Subscribable, Stateful, Poller) {
        var _this7 = this;

        Subscribable.make(this);
        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        this.ideasPoller = null;

        this.stopPoller = function () {
            if (!_this7.ideasPoller) {
                return;
            }

            _this7.ideasPoller.stop();
            _this7.ideasPoller = null;
        };

        this.startPollers = function (userKey, uuid) {
            if (_this7.ideasPoller) {
                return;
            }

            _this7.state.set('ideasMap', {});
            _this7.state.set('userKey', userKey);
            _this7.state.set('uuid', uuid);

            var countIdeas = function countIdeas(map) {
                return Object.keys(map).reduce(function (a, c) {
                    return a + map[c].length;
                }, 0);
            };

            /*
             Will take care of:
              - ideas in think stage
             */
            _this7.ideasPoller = new Poller('/ideas?uuid=' + uuid, function (response) {
                var ideas = response.data || [];
                var ideasMap = {};

                ideas.forEach(function (idea) {
                    ideasMap[idea.columnId] = ideasMap[idea.columnId] || [];
                    ideasMap[idea.columnId].push(idea);
                });

                var existent = countIdeas(_this7.state.get('ideasMap', {}));
                var fromResponse = ideas.length;

                if (existent !== fromResponse) {
                    _this7.state.set('ideasMap', ideasMap);

                    _this7.notify({
                        identifier: 'ideas',
                        state: {
                            ideas: ideasMap
                        }
                    });
                }
            });

            _this7.ideasPoller.start();
        };

        this.setIdeasMapToState = async function (uuid) {
            try {
                var response = await $http.get(HTTP_PATH + '/ideas?uuid=' + uuid);
                var ideas = response.data || [];
                var ideasMap = {};

                ideas.forEach(function (idea) {
                    ideasMap[idea.columnId] = ideasMap[idea.columnId] || [];
                    ideasMap[idea.columnId].push(idea);
                });
                _this7.state.set('ideasMap', ideasMap);
            } catch (err) {
                console.err(err);
            }
        };

        this.getIdeasListByColumnId = function (columnId) {
            return _this7.state.get('ideasMap', {})[columnId];
        };

        this.create = function (_ref3) {
            var idea = _ref3.idea,
                columnId = _ref3.columnId;
            return $http.post(HTTP_PATH + '/ideas?uuid=' + _this7.state.get('uuid'), { idea: idea, columnId: columnId });
        };

        this.remove = function (ideaId) {
            var uuid = _this7.state.get('uuid');
            return $http.delete(HTTP_PATH + '/ideas?uuid=' + uuid + '&id=' + ideaId);
        };

        this.update = function (idea) {
            var uuid = _this7.state.get('uuid');

            return $http.put(HTTP_PATH + '/ideas/' + idea.id + '?&uuid=' + uuid, { idea: idea.idea });
        };
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').service('NotificationsService', NotificationsService);

    NotificationsService.$inject = ['$rootScope', 'WorkflowService', 'StatefulService'];

    function NotificationsService($rootScope, WorkflowResource, Stateful) {
        var _this8 = this;

        Stateful.make(this);

        WorkflowResource.subscribe(function (event) {
            if (event.state.moderator && (!_this8.state.get('moderator', false) || _this8.state.get('moderator').userKey !== event.state.moderator.userKey)) {
                var isCurrentUserModerator = _this8.state.shared._get('userKey') === event.state.moderator.userKey;

                if (isCurrentUserModerator) {
                    var text = 'You are the new Moderator of this Retro. You can transfer this role in the dropdown menu.';
                    sendNotification('Congratulations!', text, 'success');
                } else {
                    var _text2 = 'has been successfully named Moderator.';
                    sendNotification(event.state.moderator.fullName, _text2, 'success');
                }

                _this8.state.set('moderator', event.state.moderator);
            }
        }, ['moderator']);

        WorkflowResource.subscribe(function (event) {
            var newParticipants = event.state.participants;
            var currentParticipants = _this8.state.get('participants', []);

            // search for new participants

            newParticipants.forEach(function (p) {
                var found = currentParticipants.find(function (cp) {
                    return cp.userKey === p.userKey;
                });

                // is new participant
                if (!found && p.userKey !== _this8.state.shared._get('userKey')) {
                    var text = p.fullName + ' has joined the session.';
                    sendNotification('', text);
                }
            });

            // search for participants that left the session
            currentParticipants.forEach(function (p) {
                var found = newParticipants.find(function (cp) {
                    return cp.userKey === p.userKey;
                });

                // left the session
                if (!found) {
                    var text = p.fullName + ' has left the session.';
                    sendNotification('', text);
                }
            });

            _this8.state.set('participants', newParticipants);
        }, ['participants']);

        function isFunction(functionToCheck) {
            return functionToCheck && {}.toString.call(functionToCheck) === '[object Function]';
        }

        /*
         type can be: error, warning, success, info
         */
        function sendNotification(title, message) {
            var type = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 'info';

            AJS.flag = null;

            if (AJS.flag && isFunction(AJS.flag)) {
                var flag = AJS.flag({
                    type: type,
                    body: title
                });

                setTimeout(function () {
                    flag.close();
                }, 8000);
            } else {
                $rootScope.$broadcast('SHOW_NOTIFICATION', {
                    type: type,
                    message: message,
                    title: title
                });
            }
        }

        return {
            sendNotification: sendNotification
        };
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').service('SettingsService', SettingsService);

    SettingsService.$inject = ['HTTP_PATH', '$http', 'SubscribableService'];

    function SettingsService(HTTP_PATH, $http, Subscribable) {
        var _this9 = this;

        Subscribable.make(this);

        this.settings = {
            anonymousPlayersAllowed: false,
            votesPerPlayer: 0,
            votesAllowedPerCard: 1,
            countTotalVotes: 0,
            columns: [{
                columnId: 'C1',
                content: 'What went well?',
                color: '#7ED321'
            }, {
                columnId: 'C2',
                content: 'What didn\'t go well',
                color: '#D0021B'
            }]

        };

        this.handlers = [];

        this.colClass = ['col-md-12 ', 'col-md-6 ', 'col-md-4 ', 'col-md-3 '];

        this.get = function (uuid) {
            return $http.get(HTTP_PATH + '/settings/' + uuid).then(function (response) {
                _this9.settings = response.data;
                return response.data;
            });
        };

        this.set = function (settings) {
            _this9.settings = Object.assign(_this9.settings, settings);

            _this9.notify({
                identifier: 'settings',
                state: {
                    settings: _this9.settings
                }
            });
        };

        this.getColumnClass = function () {
            return _this9.colClass[_this9.settings.columns.length - 1];
        };

        this.setTotalVotes = function (totalVotes) {
            _this9.settings.countTotalVotes = totalVotes;

            _this9.notify({
                identifier: 'totalVotes',
                state: {
                    totalVotes: totalVotes
                }
            });
        };

        this.isRetroExpired = function (uuid) {
            return $http.get(HTTP_PATH + '/workflow/retros-expired/' + uuid).then(function (response) {
                return _this9.settings.isExpired = response.data;
            });
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').factory('TopicService', TopicService);

    TopicService.$inject = ['HTTP_PATH', '$http', '$rootScope'];

    function TopicService(HTTP_PATH, $http, $rootScope) {

        var get = function get(_ref4) {
            var uuid = _ref4.uuid;
            return $http.get(HTTP_PATH + '/topics?uuid=' + uuid).then(function (response) {
                return response.data;
            }).catch(console.error);
        };

        var notifyChangeTopic = function notifyChangeTopic(_ref5) {
            var userKey = _ref5.userKey,
                uuid = _ref5.uuid,
                index = _ref5.index;
            return $http.put(HTTP_PATH + '/workflow/change-topic?userKey=' + userKey + '&uuid=' + uuid + '&currentTopicIndex=' + index).then(function (response) {
                return response.data;
            }).catch(console.error);
        };

        var create = function create(_ref6) {
            var userKey = _ref6.userKey,
                uuid = _ref6.uuid,
                data = _ref6.data;
            return $http.post(HTTP_PATH + '/workflow/action-item?userKey=' + userKey + '&uuid=' + uuid, data).then(function (response) {
                return response.data;
            }).catch(console.error);
        };

        return {
            get: get,
            notifyChangeTopic: notifyChangeTopic,
            create: create
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').service('UserService', UserService);

    UserService.$inject = ['WorkflowService', 'StatefulService', 'HTTP_PATH', '$http'];

    function UserService(WorkflowService, Stateful, HTTP_PATH, $http) {
        var _this10 = this;

        Stateful.make(this);

        /*
            subscribe to events to always get the newest state
         */
        WorkflowService.subscribe(function (event) {
            _this10.state.set('participants', event.state.participants);
        }, ['participants']);

        WorkflowService.subscribe(function (event) {
            _this10.state.set('moderator', event.state.moderator);
        }, ['moderator']);

        // TODO rename to getParticipants
        this.getUserList = function () {
            return _this10.state.get('participants');
        };

        this.getModerator = function () {
            return _this10.state.get('moderator');
        };

        this.getUserByUserKey = function (userKey) {
            return _this10.state.get('participants', []).find(function (p) {
                return p.userKey === userKey;
            });
        };

        // TODO rename to something related with a number
        this.getReadyUsers = function () {
            return _this10.state.get('participants').filter(function (p) {
                return p.isReady;
            }).length;
        };

        this.getBaseUrl = function () {
            var baseUrl = window.location.origin + window.location.pathname;
            var indexString = baseUrl.indexOf('/plugins/');

            if (indexString >= 0) {
                return baseUrl.substring(0, indexString);
            } else {
                return window.location.origin;
            }
        };

        this.getUsersBySearch = async function (search) {
            try {
                var url = _this10.getBaseUrl();

                var _ref7 = await $http({
                    method: 'GET',
                    url: url + '/rest/api/search',
                    headers: {
                        'X-Atlassian-Token': 'no-check',
                        'Content-Type': 'application/json'
                    },
                    params: {
                        cql: 'type=user AND user~"' + search + '*"',
                        limit: 10,
                        start: 0
                    }
                }),
                    data = _ref7.data;

                var results = data.results;

                var finalUsersPromises = results.map(async function (_ref8) {
                    var user = _ref8.user;

                    return await $http({
                        method: 'GET',
                        url: '' + user._links.self,
                        headers: {
                            'X-Atlassian-Token': 'no-check',
                            'Content-Type': 'application/json'
                        }
                    });
                });
                var confluencesUsers = await Promise.all(finalUsersPromises);
                return confluencesUsers.map(function (_ref9) {
                    var data = _ref9.data;
                    return {
                        // we remove confluence/ prefix for imageUri
                        imageUri: _this10.getImageUrl(data.profilePicture.path),
                        fullName: data.displayName,
                        name: data.username,
                        userKey: data.userKey
                    };
                });
            } catch (error) {
                console.log(error.message);
            }
        };

        this.getImageUrl = function (url) {
            if (url.indexOf('/confluence/') >= 0) {
                return url.replace('/confluence', '');
            } else {
                return url;
            }
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').service('UtilService', UtilService);

    function UtilService() {
        var flatten = function flatten(arr) {
            return arr.reduce(function (acc, val) {
                return val ? acc.concat(Array.isArray(val) ? flatten(val) : val) : acc;
            }, []);
        };

        return {
            flatten: flatten
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').service('VotesService', VotesService);

    VotesService.$inject = ['HTTP_PATH', '$http', 'SubscribableService', 'StatefulService', 'PollerService'];

    function VotesService(HTTP_PATH, $http, Subscribable, Stateful, Poller) {
        var _this11 = this;

        Subscribable.make(this);

        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        this.groupsPoller = null;

        this.stopPoller = function () {
            if (!_this11.groupsPoller) {
                return;
            }
            _this11.groupsPoller.stop();
            _this11.groupsPoller = null;
        };

        this.startPollers = function (userKey, uuid) {
            if (_this11.groupsPoller) {
                return;
            }

            _this11.state.set('userKey', userKey);
            _this11.state.set('uuid', uuid);

            _this11.groupsPoller = new Poller('/group?uuid=' + _this11.state.get('uuid'), function (response) {
                if (!response.data) {
                    return;
                }

                //remove group with ungrouped ideas (empty)
                response.data.groupList.splice(0, 1);

                _this11.notify({
                    identifier: 'groupList',
                    state: {
                        groupList: response.data.groupList.map(function (item) {
                            if (!item.ideas) {
                                item.ideas = [];
                            }
                            return item;
                        })
                    }
                });
            });

            _this11.groupsPoller.start(2);
        };

        this.create = function (_ref10) {
            var uuid = _ref10.uuid,
                data = _ref10.data,
                increased = _ref10.increased;
            return $http.post(HTTP_PATH + '/group/vote?uuid=' + uuid + '&increased=' + increased, data);
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').service('WorkflowService', WorkflowService);

    WorkflowService.$inject = ['HTTP_PATH', 'IMAGES_BASE_PATH', '$http', 'PollerService', 'SubscribableService', 'StatefulService', 'IdeasService', 'GroupsService', 'VotesService', 'DiscussService', 'ReadOnlyModePollerService', 'HeartbeatPollerService'];

    function WorkflowService(HTTP_PATH, IMAGES_BASE_PATH, $http, Poller, Subscribable, Stateful, IdeasService, GroupsService, VotesService, DiscussService, ReadOnlyModePollerService, HeartbeatPollerService) {
        var _this12 = this;

        /*
         turn this into a subscribable service,
         so that controllers can list to its changes
         */
        Subscribable.make(this);

        /*
         add logic to this service for simple state management
         */
        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        var setModerator = function setModerator(moderator) {
            if (!moderator) {
                return;
            }

            if (!_this12.state.exists('moderator') || _this12.state.get('moderator').userKey !== moderator.userKey || _this12.state.get('moderator').isReady !== moderator.isReady || _this12.state.get('moderator').isConnected !== moderator.isConnected) {
                _this12.state.set('moderator', moderator);

                _this12.notify({
                    identifier: 'moderator',
                    state: {
                        moderator: moderator
                    }
                });
            }
        };

        this.startPollers = async function (presetModerator, userKey, uuid, pageId, globalSettings) {
            _this12.state.set('userKey', userKey);
            _this12.state.set('uuid', uuid);
            _this12.state.set('pageId', pageId);

            _this12.state.shared._set('userKey', userKey);
            _this12.state.shared._set('uuid', uuid);
            _this12.state.shared._set('pageId', pageId);

            var sessionDataInterval = 1000;
            var sessionDataStopAfter = 0; // used only in summary step
            if (globalSettings && globalSettings.sessionInterval) {
                sessionDataInterval = globalSettings.sessionInterval * 1000; // intervals received in seconds, convert to milliseconds
            }
            if (globalSettings && globalSettings.summaryStopAfter) {
                sessionDataStopAfter = globalSettings.summaryStopAfter * 60 * 1000; // stopAfter received in minutes, convert to milliseconds
            }

            /*
             Will take care of:
              - current stage
             - moderator
             - participants
             - who is ready
             - completion date
             */

            var ReadOnlyMode = new ReadOnlyModePollerService(function (response) {
                if (response) {
                    _this12.notify({
                        identifier: 'readOnlyMode',
                        state: {
                            readOnlyMode: true
                        }
                    });
                    IdeasService.stopPoller();
                    GroupsService.stopPoller();
                    VotesService.stopPoller();
                }
            });

            new HeartbeatPollerService(function () {}).start(uuid, userKey);

            // const sessionDataPoller = new Poller(`/workflow/session-data?uuid=${uuid}`, (response) => {
            _this12.sessionDataPoller = new Poller('/workflow/session-data?uuid=' + uuid, function (response) {
                if (!response.data || !response.data.baseUsers) {
                    return;
                }

                // TODO 
                // response.data.globalSettings = response.data.globalSettings || {};
                response.data.participants = (response.data.baseUsers || []).filter(function (u) {
                    return u.isParticipant;
                }).sort(function (a, b) {
                    return a.userKey < b.userKey;
                });

                response.data.baseUsers = (response.data.baseUsers || []).filter(function (u) {
                    return u.isConnected;
                }).sort(function (a, b) {
                    return a.userKey < b.userKey;
                });

                var definedModerator = response.data.moderator;

                if (definedModerator) {
                    setModerator(definedModerator);
                } else {
                    var presetModeratorUserKey = response.data.presetModeratorUserKey;

                    _this12.getConfluenceUser(presetModeratorUserKey, uuid).then(function (user) {
                        setModerator(user);
                    });
                }

                /*
                 notify subscribers about newest info
                 */
                if (_this12.state.get('stage', 'undefined') !== response.data.viewName) {
                    var summaryInterval = 1000;
                    var summaryStopAfter = 0;
                    if (globalSettings && globalSettings.summaryInterval) {
                        summaryInterval = globalSettings.summaryInterval * 1000; // intervals received in seconds, convert to milliseconds
                    }
                    if (globalSettings && globalSettings.summaryStopAfter) {
                        summaryStopAfter = globalSettings.summaryStopAfter * 60 * 1000; // stopAfter received in minutes, convert to milliseconds
                    }
                    switch (response.data.viewName) {
                        case 'think':
                            _this12.sessionDataPoller.stop();
                            _this12.sessionDataPoller.start();
                            IdeasService.startPollers(userKey, uuid);
                            GroupsService.stopPoller();
                            VotesService.stopPoller();
                            DiscussService.stopPoller();
                            break;
                        case 'group':
                            _this12.sessionDataPoller.stop();
                            _this12.sessionDataPoller.start();
                            IdeasService.stopPoller();
                            GroupsService.startPollers(userKey, uuid);
                            VotesService.stopPoller();
                            DiscussService.stopPoller();
                            break;
                        case 'vote':
                            _this12.sessionDataPoller.stop();
                            _this12.sessionDataPoller.start();
                            IdeasService.stopPoller();
                            GroupsService.stopPoller();
                            VotesService.startPollers(userKey, uuid);
                            DiscussService.stopPoller();
                            break;
                        case 'discuss':
                            _this12.sessionDataPoller.stop();
                            _this12.sessionDataPoller.start();
                            IdeasService.stopPoller();
                            VotesService.stopPoller();
                            GroupsService.stopPoller();
                            DiscussService.stopPoller();
                            DiscussService.startPollers(userKey, uuid);
                            break;
                        case 'result':
                            _this12.sessionDataPoller.stop();
                            _this12.sessionDataPoller.start(summaryStopAfter);
                            IdeasService.stopPoller();
                            VotesService.stopPoller();
                            GroupsService.stopPoller();
                            DiscussService.stopPoller();
                            DiscussService.startPollers(userKey, uuid, summaryInterval, summaryStopAfter);
                            break;
                    }

                    _this12.notify({
                        identifier: 'stage',
                        state: {
                            stage: response.data.viewName
                        }
                    });
                }

                _this12.state.set('stage', response.data.viewName);

                if (response.data.baseUsers.length) {
                    var user = response.data.baseUsers.find(function (p) {
                        return p.userKey == _this12.state.get('userKey');
                    });

                    _this12.notify({
                        identifier: 'isUserReady',
                        state: {
                            ready: user ? user.isReady : false
                        }
                    });
                }

                _this12.notify({
                    identifier: 'sessionParticipants',
                    state: {
                        participants: response.data.participants
                    }
                });

                _this12.notify({
                    identifier: 'participants',
                    state: {
                        participants: response.data.baseUsers
                    }
                });

                _this12.notify({
                    identifier: 'completionDate',
                    state: {
                        completionDate: response.data.completionDate
                    }
                });

                _this12.notify({
                    identifier: 'discussGroupIndex',
                    state: {
                        index: response.data.discussGroupIndex
                    }
                });
            }, sessionDataInterval);

            _this12.sessionDataPoller.setIdentifier('Session data poller');
            _this12.sessionDataPoller.start();
            ReadOnlyMode.start();
        };

        this.initRetro = function (uuid) {
            var pageId = top.AJS.Meta.get('page-id');

            return $http.put(HTTP_PATH + '/workflow/' + uuid, pageId);
        };

        /*
         updates the moderator in the backend and in the local service storage
         */
        this.setModerator = function (user) {
            var prevMod = '';

            var moderator = _this12.state.get('moderator', false);

            if (moderator && moderator.userKey !== user.userKey) {
                prevMod = '&prevMod=' + moderator.userKey;
            }

            $http.put(HTTP_PATH + '/workflow/set-moderator?userKey=' + user.userKey + '&uuid=' + _this12.state.get('uuid') + prevMod).then(function () {
                return _this12.sessionDataPoller.poll();
            });
        };

        this.changeView = function (newStage) {
            if (_this12.state.get('stage') === newStage) {
                return;
            }

            $http.put(HTTP_PATH + '/workflow/change-view?userKey=' + _this12.state.get('userKey') + '&uuid=' + _this12.state.get('uuid') + '&viewName=' + encodeURIComponent(newStage)).then(function () {
                return _this12.sessionDataPoller.poll();
            });
        };

        this.reopen = function () {
            return $http.get(HTTP_PATH + '/workflow/reopen?uuid=' + _this12.state.get('uuid')).then(function () {
                return _this12.sessionDataPoller.poll();
            });
        };

        this.persistData = function () {
            return $http.post(HTTP_PATH + '/workflow/persist/all?uuid=' + _this12.state.get('uuid'));
        };

        this.imReady = function () {
            return $http.put(HTTP_PATH + '/workflow/im-ready?userKey=' + _this12.state.get('userKey') + '&uuid=' + _this12.state.get('uuid')).then(function () {
                return _this12.sessionDataPoller.poll();
            });
        };

        var confluenceUserPromise = null;
        var lastModeratorUserKey = '';
        this.getConfluenceUser = function (userKey, uuid) {
            if (!confluenceUserPromise || lastModeratorUserKey !== userKey) {
                confluenceUserPromise = $http.get(IMAGES_BASE_PATH + '/rest/api/user?key=' + userKey).then(function (response) {
                    var data = response.data;


                    return {
                        userKey: data.userKey,
                        name: data.username,
                        fullName: data.displayName,
                        pageId: uuid,
                        imageUri: data.profilePicture.path.replace('/confluence', ''),
                        isConnected: false,
                        isModerator: true,
                        isReady: false,
                        isParticipant: false,
                        numericPageId: parseInt(uuid)
                    };
                });
            }

            return confluenceUserPromise;
        };
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').directive('retroNotification', ['$interval', retroNotification]);

    function retroNotification($interval) {
        var link = function link(scope, element) {
            scope.notifications = [];

            var interval$ = void 0;

            var notificationCheck = function notificationCheck() {
                scope.notifications = scope.notifications.map(function (notification) {
                    notification.old++;
                    return notification;
                }).filter(function (notification) {
                    return notification.old < 4;
                });

                if (scope.notifications.length) {
                    try {
                        var notificationsContainer = document.getElementById('notification-id-' + scope.retroId);
                        var retro = document.getElementById(scope.retroId);

                        var position = retro.getBoundingClientRect();

                        notificationsContainer.style.top = position.y - 40 + 'px';
                    } catch (err) {}
                }

                if (!scope.notifications.length) {
                    $interval.cancel($interval);
                    interval$ = null;
                }
            };

            scope.$on('SHOW_NOTIFICATION', function (evt, data) {
                scope.notifications.push({
                    type: data.type,
                    title: data.title,
                    message: data.message,
                    old: 0
                });

                if (scope.notifications.length === 1) {
                    interval$ = $interval(notificationCheck, 2000);
                }
            });

            scope.$on('$destroy', function () {
                if ($interval) {
                    $interval.cancel(interval$);
                }
            });

            scope.removeNotification = function ($index) {
                scope.notifications.splice($index, 1);
            };
        };

        return {
            restrict: 'E',
            scope: {
                retroId: '=',
                contextPath: '@',
                timestamp: '='
            },
            link: link,
            template: '<div id="notification-id-{{retroId}}" class="aui-flags-container retro" ng-if="notifications.length">\n                            <div class="aui-flag retro" aria-hidden="false" ng-repeat="notification in notifications">\n                                <div class="aui-message aui-message-info closeable shadowed">\n                                <div class="retro-notification-icon" ng-switch="notification.type">\n                                    <img ng-switch-when="warning" ng-src="{{contextPath}}/download/resources/retrospectives4confluence:retrospective-resources{{timestamp}}/images/notifications/warning.png"/>\n                                    <img ng-switch-when="error" ng-src="{{contextPath}}/download/resources/retrospectives4confluence:retrospective-resources{{timestamp}}/images/notifications/error.png"/>\n                                    <img ng-switch-when="success" ng-src="{{contextPath}}/download/resources/retrospectives4confluence:retrospective-resources{{timestamp}}/images/notifications/confirm.png"/>\n                                    <img ng-switch-when="info" ng-src="{{contextPath}}/download/resources/retrospectives4confluence:retrospective-resources{{timestamp}}/images/notifications/info.png"/>\n                                </div>\n                                    <span class="title">{{notification.title}}</span> {{notification.message}}\n                                    <span class="aui-icon icon-close" role="button" tabindex="0" ng-click="removeNotification($index)"></span>\n                                </div>\n                           </div>\n                       </div>'
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').directive('tooltipDirective', tooltipDirective);

    function tooltipDirective() {
        var link = function link(scope, element) {
            element.attr("data-placement", "bottom");
            element.attr('data-html', 'true');

            element.on('mouseenter', function () {
                if (scope.$parent.vm.isCurrentUserMod && scope.$parent.vm.isCurrentUserMod()) {
                    element.attr('title', scope.modTitle);
                } else {
                    element.attr('title', scope.userTitle);
                }

                if ((scope.visible == undefined || scope.visible) && element.attr('title') != undefined && element.attr('title').length > 0) $(element).tooltip('show');else element.attr('title', '');
            });

            element.on('mouseleave', function () {
                $(element).tooltip('destroy');
            });
        };

        return {
            restrict: 'A',
            scope: {
                modTitle: '=',
                userTitle: '=',
                visible: '='
            },
            link: link
        };
    }
})(angular);
(function (angular) {
    angular.module('retrospectiveApp').controller('discussController', DiscussController);

    DiscussController.$inject = ['IMAGES_BASE_PATH', 'HTTP_PATH', '$http', '$scope', 'DiscussService', 'UserService', 'TopicService', 'SettingsService', 'UtilService', 'WorkflowService'];

    function DiscussController(IMAGES_BASE_PATH, HTTP_PATH, $http, $scope, DiscussService, UserService, TopicService, SettingsService, UtilService, WorkflowService) {
        var _this13 = this;

        var vm = this;

        vm.readOnylMode = false;
        vm.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        vm.userKey = $scope.$parent.vm.userKey;
        vm.uuid = $scope.$parent.vm.uuid;
        vm.moderator = $scope.$parent.vm.moderator;
        vm.settings = SettingsService.settings;
        vm.currentTopicIndex = 0;
        vm.tooltipPrevious = 'Previous';
        vm.tooltipNext = 'Next';
        vm.newActionItemDescription = '';
        vm.colorsColumn = {};
        vm.topics = [];
        vm.confluenceUsers = [];
        vm.userSearch = '';
        vm.confluenceUsersResult = [];
        vm.retroParticipants = [];
        vm.participantsHash = '';

        vm.thinkElements = [];
        this.groupCollection = null;

        WorkflowService.subscribe(function (event) {
            _this13.readOnlyMode = true;
        }, ['readOnlyMode']);

        WorkflowService.subscribe(function (event) {
            _this13.moderator = event.state.moderator;
        }, ['moderator']);

        WorkflowService.subscribe(function (event) {
            _this13.retroParticipants = event.state.participants;
            //we only update retro participants if the participant list has been changed since the last time
            // this is to avoid unnecessary refreshing that will cause conflicts with user search in action items assigning
            var newParticipantsHash = '';
            _this13.retroParticipants.forEach(function (participant) {
                return newParticipantsHash = newParticipantsHash.concat(participant.userKey);
            });
            if (vm.participantsHash !== newParticipantsHash) {
                initializeUsersForActionItems();
                vm.participantsHash = newParticipantsHash;
            }
        }, ['participants']);

        SettingsService.subscribe(function (event) {
            _this13.settings = event.state.settings;
        }, ['settings']);

        $http.get(HTTP_PATH + '/ideas?uuid=' + vm.uuid).then(function (response) {
            vm.thinkElements = response.data;

            DiscussService.subscribe(function (event) {
                _this13.groupCollection = {
                    groupList: event.state.groupList
                };

                // update this in another thread
                vm.currentTopic = _this13.groupCollection.groupList[vm.currentTopicIndex];
                vm.topics = _this13.groupCollection.groupList;
            }, ['groupList']);
        }).catch(function (err) {
            console.error(err);
            var flagError = AJS.flag({
                type: 'error',
                body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
            });
        });

        WorkflowService.subscribe(function (event) {
            if (_this13.currentTopicIndex != event.state.index) {
                _this13.currentTopicIndex = event.state.index;

                // clear users input when we change the action item on discussion
                _this13.newActionItemDescription = '';
            }
        }, ['discussGroupIndex']);

        this.findColor = function (columnId) {
            var column = _this13.settings.columns.find(function (col) {
                return col.columnId == columnId;
            });
            if (column) {
                return column.color;
            }
            return '#000';
        };

        var previousTopic = function previousTopic() {
            if (vm.currentTopicIndex > 0) {
                DiscussService.previousTopic().then(function () {
                    WorkflowService.sessionDataPoller.poll();
                }).catch(function (err) {
                    console.error(err);
                    var flagError = AJS.flag({
                        type: 'error',
                        body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                    });
                });
            }
        };

        var nextTopic = function nextTopic() {
            if (vm.currentTopicIndex < vm.topics.length) {
                DiscussService.nextTopic().then(function () {
                    WorkflowService.sessionDataPoller.poll();
                }).catch(function (err) {
                    console.error(err);
                    var flagError = AJS.flag({
                        type: 'error',
                        body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                    });
                });
            }
        };

        var addActionItem = function addActionItem() {
            if (vm.newActionItemDescription.length) {
                var newActionItem = {};
                newActionItem.description = vm.newActionItemDescription;
                DiscussService.addActionItem(vm.currentTopic.id, newActionItem).catch(function (err) {
                    console.error(err);
                    var flagError = AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                    });
                });
                vm.newActionItemDescription = '';
            }
        };

        var deleteActionItem = function deleteActionItem(actionItem) {
            vm.currentTopic.actionItems.splice(vm.currentTopic.actionItems.indexOf(actionItem), 1);
            DiscussService.removeActionItem(vm.currentTopic.id, actionItem.id).catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                });
            });
        };

        var assigneeUser = function assigneeUser(index, actionItem, user) {
            AJS.$('#trigger-dropdown' + index + vm.uuid).trigger('aui-button-invoke');
            if (user) {
                var newUser = Object.assign({}, user);
                delete newUser.isParticipant;
                DiscussService.setAssignee(vm.currentTopic.id, actionItem.id, newUser).catch(function (err) {
                    console.error(err);
                    var flagError = AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                    });
                });
            } else {
                DiscussService.clearAssignee(vm.currentTopic.id, actionItem.id).catch(function (err) {
                    console.error(err);
                    var flagError = AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                    });
                });
            }
        };

        var isUserAssignee = function isUserAssignee(index, user) {
            var assignee = vm.currentTopic.actionItems[index].assigneeUser;
            return assignee && assignee.userName == user.userName;
        };

        var getVoteText = function getVoteText(votes) {
            return votes === 1 ? 'Vote' : 'Votes';
        };

        var isCurrentUserMod = function isCurrentUserMod() {
            if (vm.moderator) {
                return vm.moderator.userKey === vm.userKey;
            }

            return UserService.getUserByUserKey(vm.userKey).isModerator;
        };

        var searchForUser = async function searchForUser() {
            if (!vm.userSearch.trim().length) {
                initializeUsersForActionItems();
                return;
            }

            vm.confluenceUsersResult = await UserService.getUsersBySearch(vm.userSearch.trim());

            if (!vm.confluenceUsersResult.length) {
                initializeUsersForActionItems();
            }
        };

        var cleanUserSearch = function cleanUserSearch() {
            vm.userSearch = '';
            initializeUsersForActionItems();
        };

        var initializeUsersForActionItems = function initializeUsersForActionItems() {
            vm.confluenceUsersResult = vm.retroParticipants;
        };

        vm.isUserAssignee = isUserAssignee;
        vm.previousTopic = previousTopic;
        vm.nextTopic = nextTopic;
        vm.addActionItem = addActionItem;
        vm.getVoteText = getVoteText;
        vm.assigneeUser = assigneeUser;
        vm.deleteActionItem = deleteActionItem;
        vm.isCurrentUserMod = isCurrentUserMod;
        vm.searchForUser = searchForUser;
        vm.cleanUserSearch = cleanUserSearch;

        //activate();
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').controller('groupController', GroupController);

    GroupController.$inject = ['IMAGES_BASE_PATH', 'HTTP_PATH', '$http', '$scope', 'IdeasService', 'SettingsService', 'UtilService', 'GroupsService', 'StatefulService', 'WorkflowService'];

    function GroupController(IMAGES_BASE_PATH, HTTP_PATH, $http, $scope, IdeasService, SettingsService, UtilService, GroupsService, Stateful, WorkflowService) {
        var _this14 = this;

        Stateful.make(this);

        var vm = this;

        this.readOnlyMode = false;
        this.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        vm.userKey = $scope.$parent.vm.userKey;
        vm.uuid = $scope.$parent.vm.uuid;
        vm.moderator = $scope.$parent.vm.moderator;
        this.settings = SettingsService.settings;
        this.groupedIdeas = {};
        this.groupedIdeas.list = [];
        this.groupedIdeas.name = '';
        this.lastGroupNameValue = '';
        this.showGroupName = true;
        this.showRepeatedGroupNameError = false;
        this.colorsColumn = SettingsService.settings.columnColors;

        vm.thinkElements = [];
        this.addedIdeas = [];
        this.removedIdeas = [];

        function onlyUnique(value, index, self) {
            return self.indexOf(value) === index;
        }

        $http.get(HTTP_PATH + '/ideas?uuid=' + vm.uuid).then(function (response) {
            vm.thinkElements = response.data;
        }).catch(function (err) {
            console.error(err);
            var flagError = AJS.flag({
                type: 'error',
                body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
            });
        });

        WorkflowService.subscribe(function (event) {
            _this14.readOnlyMode = true;
        }, ['readOnlyMode']);

        SettingsService.subscribe(function (event) {
            _this14.settings = event.state.settings;
        }, ['settings']);

        GroupsService.subscribe(function (event) {
            if (_this14.thinkElements && _this14.thinkElements.length) {
                if (_this14.removedIdeas.length) {
                    _this14.removedIdeas.forEach(function (idea) {
                        event.state.groupedIdea.ids = event.state.groupedIdea.ids.filter(function (id) {
                            return id !== idea.id;
                        });
                    });

                    _this14.removedIdeas = [];
                }

                var lengthBefore = event.state.groupedIdea.ids.length;
                if (event.state.groupedIdea.ids.length && _this14.addedIdeas) {
                    _this14.addedIdeas.map(function (i) {
                        return '' + i.id;
                    }).forEach(function (id) {
                        if (event.state.groupedIdea.ids.indexOf(id) == -1) {
                            event.state.groupedIdea.ids.push(id);
                        }
                    });
                }

                _this14.groupedIdeas.name = event.state.groupedIdea.name;
                _this14.groupedIdeas.list = event.state.groupedIdea.ids.filter(function (id) {
                    return id;
                }).map(function (id) {
                    return vm.thinkElements.find(function (i) {
                        return i.id == id;
                    });
                });

                // if we are sync, delete locally applied ideas
                if (lengthBefore == _this14.groupedIdeas.list.length) {
                    _this14.addedIdeas = [];
                }

                if (!event.state.groupedIdea.ids.length) {
                    _this14.addedIdeas = [];
                    _this14.removedIdeas = [];
                }
            }
        }, ['groupedIdea']);

        // Method to check if a given group id has actually just an idea

        this.isGroup = function (group) {
            if (group.ideas.length === 1 && !group.description) {
                group.grouped = false;
                return false;
            }
            return true;
        };

        GroupsService.subscribe(function (event) {
            if (_this14.addedIdeas.length) {
                _this14.addedIdeas.forEach(function (idea) {
                    event.state.groupList[0].ideas = event.state.groupList[0].ideas.filter(function (i) {
                        return i.id != idea.id;
                    });
                });
            }

            var lengthBefore = event.state.groupList[0].ideas.length;
            if (_this14.removedIdeas.length) {
                _this14.removedIdeas.forEach(function (idea) {
                    event.state.groupList[0].ideas.push(idea);
                });
            }

            _this14.groupCollection = {
                groupList: event.state.groupList.map(function (group) {
                    group.ideas = group.ideas.filter(onlyUnique);
                    return group;
                })
            };

            _this14.groupCollection.groupList.forEach(function (group) {
                if (group.grouped) {
                    _this14.isGroup(group);
                }
            });

            // if we are sync, delete locally applied ideas
            if (lengthBefore === _this14.groupCollection.groupList[0].ideas.length) {
                _this14.removedIdeas = [];
            }
        }, ['groupList']);

        this.compareFirstIdea = function (group) {
            if (_this14.isGroupOpen() && group.ideas.length === 1 && _this14.groupedIdeas.list.length === 1) {
                return group.ideas[0].id === _this14.groupedIdeas.list[0].id;
            }
            return false;
        };

        WorkflowService.subscribe(function (event) {
            _this14.moderator = event.state.moderator;
        }, ['moderator']);

        /*
         apply drag and drop changes locally to avoid
         cards blinking between updates from "server stream"
         */
        this.addToAvailableIdeas = function (idea) {
            _this14.groupCollection.groupList[0].ideas.push(idea);
            var index = _this14.groupedIdeas.list.indexOf(idea);
            _this14.groupedIdeas.list.splice(index, 1);

            _this14.removedIdeas.push(idea);
        };

        this.addToGroupedIdea = function (idea) {
            var index = _this14.groupCollection.groupList[0].ideas.indexOf(idea);
            _this14.groupCollection.groupList[0].ideas.splice(index, 1);
            _this14.groupedIdeas.list.push(idea);

            _this14.addedIdeas.push(idea);
        };

        this.applyToGroupedIdea = function () {};

        this.applyToAvailableIdeas = function () {};

        /*
         other functions
         */
        this.showPlusButton = function () {
            return !_this14.isEmptyGroup();
        };

        this.getTotalIssues = function () {
            if (!_this14.groupCollection || !_this14.groupCollection.groupList) {
                return 0;
            }

            var counter = 0;

            _this14.groupCollection.groupList.forEach(function (groupElement) {
                if (groupElement.grouped) {
                    counter++;
                } else {
                    counter += groupElement.ideas.length;
                }
            });

            return counter;
        };

        this.isCurrentUserMod = function () {
            return _this14.moderator && _this14.moderator.userKey === _this14.userKey;
        };

        this.isGroupOpen = function () {
            return _this14.groupedIdeas.list.length;
        };

        this.isEmptyGroup = function () {
            if (!_this14.groupedIdeas || !_this14.groupedIdeas.list.length) {
                _this14.groupedIdeas.name = '';
                return true;
            }
            return false;
        };

        this.dropGroupListAdd = function (idea) {
            if (!_this14.groupedIdeas.list.length) {
                _this14.createGroupWithButton(idea);
            } else {
                _this14.addIdeaToGroup(idea);
            }

            _this14.startDragginRight = false;
            _this14.startDragginLeft = false;
        };

        this.dropGroupListRemove = function (idea) {
            _this14.removeIdeaFromGroup(idea);
            _this14.startDragginRight = false;
            _this14.startDragginLeft = false;
        };

        this.dndAvailable = function () {
            return true;
            if (_this14.isCurrentUserMod()) {
                return true;
            }
            return !!_this14.groupedIdeas.list.length;
        };

        this.insertGroupName = function () {
            if (vm.groupName === vm.lastGroupNameValue) {
                vm.showGroupName = true;
                vm.showRepeatedGroupNameError = false;
                return;
            }

            if (!vm.groupName) {
                vm.groupName = 'Untitled group';
            }

            GroupsService.editGroupName(vm.groupName).then(function () {
                vm.showGroupName = true;
                vm.showRepeatedGroupNameError = false;
            }).catch(function () {
                vm.showRepeatedGroupNameError = true;
            });
        };

        this.createGroupWithButton = function (idea) {
            GroupsService.createGroupedIdea(idea).catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        this.addIdeaToGroup = function (idea) {
            GroupsService.addIdeaToGroupedIdea(idea).catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        // cuando se da click en el boton de menos, para quitar una idea y mandarla hacia la izquierda
        this.removeIdeaFromGroup = function (idea) {
            _this14.addToAvailableIdeas(idea);
            GroupsService.removeIdeaFromGroupedIdea(idea).catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        this.persistGroup = function () {
            _this14.removedIdeas = [];
            _this14.addedIdeas = [];
            GroupsService.saveGroup().catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        this.deleteGroup = function () {
            _this14.removedIdeas = [];
            _this14.addedIdeas = [];
            GroupsService.breakGroup().catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        // cuando se da click en un grupo para modificar sus tareas
        this.openGroup = function (group) {
            return GroupsService.editGroup(group).catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        this.editGroupName = function () {
            _this14.showGroupName = false;
            vm.groupName = vm.groupedIdeas.name;

            setTimeout(function () {
                $('#group-name-input' + vm.uuid).focus();
                if (vm.groupName == 'Untitled group') {
                    vm.groupName = '';
                }
            }, 300);
        };

        this.shouldShowNewGroupIcon = function (group) {
            return !group.grouped && !_this14.groupedIdeas.list.length;
        };

        this.saveLastValue = function () {
            return _this14.lastGroupNameValue = _this14.groupedIdeas.name;
        };

        this.undoGroupName = function () {
            return _this14.groupedIdeas.name = _this14.lastGroupNameValue;
        };

        this.createGroupWithButtonAndFocus = function (idea) {
            if (!_this14.groupedIdeas.name) {
                _this14.showGroupName = false;
            }

            $.when(_this14.createGroupWithButton(idea)).then(function () {
                if (!_this14.groupedIdeas.name) {
                    $('#group-name-input').focus();
                }
            });
        };

        this.groupDragStartRight = function () {
            return _this14.startDragginRight = true;
        };

        this.groupDragStartLeft = function () {
            return _this14.startDragginLeft = true;
        };

        this.getGroupItemClasses = function (idea, group) {
            if (group.isCollapsed == undefined) {
                group.isCollapsed = true;
            }
            var classes = 'retro-list-item vote-group-transition ';
            classes += group.grouped && group.isCollapsed ? ' ideas-grouped ' : '';
            classes += group.grouped && group.isCollapsed && group.ideas.length > 1 ? ' collapsed-ideas' : '';
            return classes;
        };

        this.collapseGroup = function (group) {
            if (group.grouped) {
                group.isCollapsed = !group.isCollapsed;
            }
        };

        this.dragIdeasEnabled = function () {
            return _this14.isCurrentUserMod() || _this14.isGroupOpen();
        };

        this.getDndTypeNewGroup = function () {
            return 'grouping' + _this14.uuid;
        };
        this.getDndTypeGroups = function () {
            return 'groups' + _this14.uuid;
        };
        $scope.$on('$destroy', this.persistGroup);
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').controller('homeController', homeController);

    homeController.$inject = ['IMAGES_BASE_PATH', 'IdeasService', 'EventService', 'SettingsService', 'DateService', 'WorkflowService', 'ConfluenceAccessModeService'];

    function homeController(IMAGES_BASE_PATH, IdeasService, EventService, SettingsService, DateService, WorkflowService, ConfluenceAccessModeService) {
        var _this15 = this;

        this.readOnlyMode = ConfluenceAccessModeService.isReadOnlyModeEnabled();

        this.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        this.isUserReady = false;
        this.activeView = '';
        this.settings = SettingsService.settings;
        this.completionDate = null;

        this.setUserData = false;
        this.validLicense = false;
        this.sessionCompleted = false;
        this.ideasSize = 0;

        this.thinkState = false;
        this.groupState = false;
        this.voteState = false;
        this.discussState = false;

        this.title = 'Retrospectives App';

        this.activeUsers = EventService.activeUsers;
        this.showDetailView = false;

        WorkflowService.subscribe(function (event) {
            _this15.readOnlyMode = true;
        }, ['readOnlyMode']);

        SettingsService.subscribe(function (event) {
            _this15.settings = event.state.settings;
        }, ['settings']);

        WorkflowService.subscribe(function (event) {
            _this15.moderator = event.state.moderator;
        }, ['moderator']);

        this.transferUser = {};

        this.initRetro = WorkflowService.initRetro;
        this.setModerator = WorkflowService.setModerator;
        this.nextAvailable = true;
        this.showActionItems = true;

        this.retrospectiveParticipants = [];
        var cache = '';

        this.getCompletionDate = function () {
            return DateService.format(_this15.completionDate);
        };

        IdeasService.subscribe(function (event) {
            var asString = JSON.stringify(event.state.ideas);

            // avoid re-render of the same state
            if (asString === cache) {
                return;
            } else {
                cache = asString;
            }

            var size = 0;

            if (_this15.settings.columns.length) {
                var firstId = _this15.settings.columns[0].columnId;

                size = _this15.settings.columns.reduce(function (a, c) {
                    if (event.state.ideas[c.columnId]) {
                        return a + (event.state.ideas[c.columnId] || []).length;
                    }
                    return a;
                }, 0);
                _this15.ideasSize = size;
            }
        }, ['ideas']);

        WorkflowService.subscribe(function (event) {
            _this15.completionDate = event.state.completionDate;
        }, ['completionDate']);

        /*
         get participants from polling stream
         */
        WorkflowService.subscribe(function (event) {
            _this15.activeUsers = event.state.participants;
        }, ['participants']);

        WorkflowService.subscribe(function (event) {
            _this15.retrospectiveParticipants = event.state.participants;
        }, ['sessionParticipants']);

        WorkflowService.subscribe(function (event) {
            _this15.isUserReady = event.state.ready;
        }, ['isUserReady']);

        /*
         subscribe to changes in the view/stage
         */
        WorkflowService.subscribe(function (event) {
            if (_this15.activeView == event.state.stage) {
                return;
            }

            _this15.activeView = event.state.stage;

            switch (event.state.stage) {
                case 'think':
                    _this15.thinkState = true;
                    _this15.groupState = false;
                    _this15.voteState = false;
                    _this15.discussState = false;
                    break;
                case 'group':
                    _this15.thinkState = true;
                    _this15.groupState = true;
                    _this15.voteState = false;
                    _this15.discussState = false;
                    break;
                case 'vote':
                    _this15.thinkState = true;
                    _this15.groupState = true;
                    _this15.voteState = true;
                    _this15.discussState = false;
                    break;
                case 'discuss':
                    _this15.thinkState = true;
                    _this15.groupState = true;
                    _this15.voteState = true;
                    _this15.discussState = true;
                    break;
                case 'main':
                    _this15.thinkState = false;
                    _this15.groupState = false;
                    _this15.voteState = false;
                    _this15.discussState = false;
                    break;
            }
        }, ['stage']);

        this.isAdministratorAlone = function () {
            return false;
        };

        this.onGetStartedClick = function () {
            WorkflowService.changeView('think');
        };

        this.imReady = function () {
            WorkflowService.imReady().then(function () {
                document.getElementById('readyBtn').blur();
            }).catch(function () {
                console.error("error while updating ready status");
            });
        };

        this.isCurrentUserMod = function () {
            return _this15.moderator && _this15.moderator.userKey === _this15.userKey;
        };

        this.getUsersReady = function () {
            return _this15.activeUsers.filter(function (user) {
                return user.isReady;
            }).length;
        };

        this.isValidActiveView = function () {
            return ['think', 'group', 'vote', 'discuss'].indexOf(_this15.activeView) >= 0;
        };

        this.isWaitActive = function () {
            return _this15.activeView == 'wait';
        };

        this.isResultActive = function () {
            return _this15.activeView == 'result';
        };

        this.nextSection = function () {
            if (_this15.nextAvailable) {
                var sections = ['think', 'group', 'vote', 'discuss', 'result'];
                var index = sections.indexOf(_this15.activeView);
                WorkflowService.changeView(sections[index + 1]);
            }
        };

        this.transferModeratorRole = function () {
            _this15.setModerator(_this15.transferUser);
        };

        this.showModalTransferRole = function (user) {
            if (_this15.isCurrentUserMod()) {
                _this15.transferUser = user;
                $('#transferModal' + _this15.uuid).modal();
            }
        };

        this.getTooltipConfig = function (_ref11) {
            var view = _ref11.view;

            var messageMod = '';
            var messageUser = 'Only moderator can move between sections';
            var condition = true;

            if (_this15.isCurrentUserMod()) {
                if (!_this15.activeView) {
                    //waiting screen
                    messageMod = 'You can\'t start a session without your team members';
                    condition = _this15.isAdministratorAlone() || !_this15.isCurrentUserMod();
                } else {
                    //any other step
                    messageMod = 'To move forward, first type an idea';
                    condition = _this15.activeView == 'think' && _this15.ideasSize == 0;
                }
            }

            return {
                messageMod: messageMod,
                messageUser: messageUser,
                condition: condition
            };
        };

        this.isAdministratorAlone = function () {
            return false;
        };

        this.changeStepView = function (view) {
            if (_this15.isAdministratorAlone() || !_this15.isCurrentUserMod() || view != 'think' && !_this15.ideasSize) {
                return;
            }

            WorkflowService.changeView(view);
        };

        this.getTotalVotesText = function () {
            if (_this15.settings.votesPerPlayer) {
                return _this15.settings.countTotalVotes + '/' + _this15.settings.votesPerPlayer + ' Total votes';
            }
            return '';
        };

        this.reopen = function () {
            WorkflowService.reopen().catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        this.activeDetailView = function (show) {
            _this15.showDetailView = show;
        };

        this.changeTab = function (show, showDetailView) {
            if (showDetailView) {
                _this15.activeDetailView(showDetailView);
                return;
            }

            if (show) {
                $('#btn-action-items' + _this15.uuid).addClass('result-header-btn-active');
                $('#btn-action-items' + _this15.uuid).attr('disabled', 'disabled');
                $('#btn-topics' + _this15.uuid).removeClass('result-header-btn-active');
                $('#btn-topics' + _this15.uuid).removeAttr('disabled');
                _this15.activeDetailView(false);
            } else {
                $('#btn-action-items' + _this15.uuid).removeClass('result-header-btn-active');
                $('#btn-action-items' + _this15.uuid).removeAttr('disabled');
                $('#btn-topics' + _this15.uuid).addClass('result-header-btn-active');
                $('#btn-topics' + _this15.uuid).attr('disabled', 'disabled');
                _this15.activeDetailView(false);
            }
            _this15.showActionItems = show;
        };

        var countIdeas = function countIdeas(columns) {
            var size = 0;
            if (columns.length) {
                size = columns.reduce(function (a, c) {
                    var ideasList = IdeasService.getIdeasListByColumnId(c.columnId);
                    if (ideasList) {
                        return a + (ideasList || []).length;
                    }
                    return a;
                }, 0);
            }
            return size;
        };

        this.getModeratorTooltipMessage = function () {
            if (_this15.presetModerator === 'RETROS_DYNAMIC_MODERATOR') {
                return '<strong>Dynamic Moderator</strong>: Role is defined during the session and <br/> is automatically transferred to next in line if moderator disconnects.';
            }

            return '<strong>Preset Moderator:</strong>: This participant controls the session. <br/> Role can be transferred if needed.';
        };

        this.initEvents = function (_ref12) {
            var presetModerator = _ref12.presetModerator,
                userKey = _ref12.userKey,
                validLicense = _ref12.validLicense,
                sessionCompleted = _ref12.sessionCompleted,
                completionDate = _ref12.completionDate,
                uuid = _ref12.uuid,
                pageId = _ref12.pageId,
                appId = _ref12.appId;

            _this15.userKey = userKey;
            _this15.uuid = uuid; // for new macros we are going to use the uuid, for legacy macros, this value will be the pageId
            _this15.pageId = pageId;
            _this15.validLicense = validLicense;
            _this15.sessionCompleted = sessionCompleted;
            _this15.appId = appId;
            _this15.completionDate = completionDate;
            _this15.presetModerator = presetModerator;

            WorkflowService.initRetro(_this15.uuid).then(function (globalSettings) {
                $('#' + _this15.uuid).css('display', 'inherit');

                if (_this15.uuid) {
                    SettingsService.get(_this15.uuid).then(function (settings) {
                        var localSettings = {
                            anonymousPlayersAllowed: settings.anonymousPlayersAllowed,
                            votesPerPlayer: settings.votesPerPlayer,
                            votesAllowedPerCard: settings.votesAllowedPerCard,
                            columns: settings.columns
                        };
                        SettingsService.set(localSettings);
                    }).catch(function (err) {
                        console.error(err);
                        var flagError = AJS.flag({
                            type: 'error',
                            body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                        });
                    });
                }

                if (_this15.validLicense || _this15.sessionCompleted) {
                    var globalSettingsData = {};
                    if (globalSettings.data) {
                        globalSettingsData = globalSettings.data;
                    }
                    WorkflowService.startPollers(_this15.presetModerator, _this15.userKey, _this15.uuid, _this15.pageId, globalSettingsData).catch(function (err) {
                        console.error(err);
                        var flagError = AJS.flag({
                            type: 'error',
                            body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                        });
                    });
                    IdeasService.setIdeasMapToState(_this15.uuid).then(function (value) {
                        _this15.ideasSize = countIdeas(_this15.settings.columns);
                    });
                }
            }).catch(function (err) {
                console.error("Retrospective initialization error:", err);
            });
        };

        this.getFooterStyles = function () {
            var styles = { "margin-left": "-28px" };

            switch (_this15.activeView) {
                case 'wait':
                    {
                        styles.width = "calc(100% + 30px)";
                    }break;
                case 'result':
                    {
                        styles.width = "calc(100% + 43px)";
                        styles['margin-left'] = "-13px";
                    }break;
                default:
                    {
                        styles.width = "calc(100% + 48px)";
                    }
            }

            return styles;
        };
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').controller('resultController', ResultController);

    ResultController.$inject = ['IMAGES_BASE_PATH', 'HTTP_PATH', '$http', '$q', '$scope', 'IdeasService', 'UserService', 'DiscussService', 'TopicService', 'EventService', 'SettingsService', 'WorkflowService', 'DateService', 'ConfluenceAccessModeService'];

    function ResultController(IMAGES_BASE_PATH, HTTP_PATH, $http, $q, $scope, IdeasService, UserService, DiscussService, TopicService, EventService, SettingsService, WorkflowService, DateService, ConfluenceAccessModeService) {
        var _this16 = this;

        var vm = this;

        this.readOnlyMode = ConfluenceAccessModeService.isReadOnlyModeEnabled();

        vm.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        vm.userKey = $scope.$parent.vm.userKey;
        vm.uuid = $scope.$parent.vm.uuid;
        vm.settings = SettingsService.settings;
        vm.groupsMap = {};
        vm.actionItems = [];
        vm.retroParticipants = $scope.$parent.$parent.$parent.$parent.vm.retrospectiveParticipants;
        vm.filterActionItemsBy = [];
        vm.activeUsers = $scope.$parent.vm.activeUsers;
        vm.colorsColumn = {};
        vm.countActionItems = [];
        this.showDetailView = false;
        this.firstActionItemId = null;

        vm.confluenceUsers = [];
        vm.userSearch = '';
        vm.confluenceUsersResult = [];

        vm.thinkElements = [];
        this.groupCollection = null;

        var cache = '';
        this.creatingConfluenceTasks = false;

        WorkflowService.subscribe(function (event) {
            _this16.readOnlyMode = true;
        }, ['readOnlyMode']);

        SettingsService.subscribe(function (event) {
            _this16.settings = event.state.settings;
        }, ['settings']);

        $http.get(HTTP_PATH + '/ideas?uuid=' + vm.uuid).then(function (response) {
            vm.thinkElements = response.data;

            DiscussService.subscribe(function (event) {
                var asString = JSON.stringify(event.state.groupList);

                // avoid re-render of the same state
                if (asString === cache) {
                    return;
                } else {
                    cache = asString;
                }

                _this16.groupCollection = {
                    groupList: event.state.groupList
                };

                _this16.settings.columns.forEach(function (col, position) {
                    var count = 0;
                    vm.groupsMap[col.columnId] = [];

                    _this16.groupCollection.groupList.forEach(function (group) {
                        if (group.ideas[0].columnId == col.columnId) {
                            count += group.actionItems.length;
                            vm.groupsMap[col.columnId].push(group);
                        }
                    });
                    _this16.filterActionItemsBy[position] = 'all';
                    _this16.countActionItems[col.columnId] = count;
                });

                // this validation is to refresh the info in detail view when a issue is created or assigned
                if (_this16.showDetailView && typeof _this16.detailViewColumnId !== 'undefined') {
                    _this16.detailGroup = vm.groupsMap[_this16.detailViewColumnId].find(function (group) {
                        return group.id === _this16.detailGroup.id;
                    });
                    _this16.detailGroup.firstActionItem = _this16.detailGroup.actionItems.find(function (actionItem) {
                        return actionItem.id === _this16.firstActionItemId;
                    });
                }
            }, ['groupList']);
        }).catch(function (err) {
            console.error(err);
            var flagError = AJS.flag({
                type: 'error',
                body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
            });
        });

        vm.confluenceUsersResult = UserService.getUserList();
        this.findColor = function (columnId) {
            var column = _this16.settings.columns.find(function (col) {
                return col.columnId == columnId;
            });
            if (column) {
                return column.color;
            }
            return '#000';
        };

        this.convertionAvailable = function (columnId, $index) {
            var groups = _this16.groupsMap[columnId];

            if (!groups) {
                return false;
            }

            var actionItems = groups.map(function (group) {
                return _this16.filterActionItems(group.actionItems, _this16.filterActionItemsBy[$index]);
            }).reduce(function (acc, cur) {
                return acc.concat(cur);
            }, []);

            return actionItems.length > 0;
        };

        function sanitize(input) {
            var output = input.replace(/<script[^>]*?>.*?<\/script>/gi, '').replace(/<[\/\!]*?[^<>]*?>/gi, '').replace(/<style[^>]*?>.*?<\/style>/gi, '').replace(/<![\s\S]*?--[ \t\n\r]*>/gi, '');
            return output;
        }

        this.addCloseEvents = function () {
            //event to listen the esc key
            $(document).keyup(function (e) {
                if (e.which === 27) {
                    $(".close-details-btn").click();
                }
            });
            //add event to window.parent (iframe not included)
            $(window.parent).on('click', function (e) {
                _this16.closeDetailView();
            });
        };

        this.orderActionItem = function (actionItem) {
            return actionItem === _this16.detailGroup.firstActionItem;
        };

        this.closeDetailView = function () {
            AJS.$('#retrospectives-app-header').show();
            $(window.parent).off('click');
            $(document).off('keyup');
            $scope.$parent.vm.activeDetailView(false);
            initializeUsersForActionItems();
        };

        // this function open the detail view
        this.openDetailView = function (group, col, actionItemId) {
            initializeUsersForActionItems();
            _this16.addCloseEvents();
            // we set all the variables that we're going to need in this view
            AJS.$('#retrospectives-app-header').hide();
            window.parent.parent.scrollTo(0, 0);
            _this16.showDetailView = true;
            //we call this function from home.controller.js to switch the view
            $scope.$parent.vm.changeTab(null, _this16.showDetailView);
            _this16.detailGroup = group;
            _this16.detailViewColumnName = col.content;
            _this16.detailViewColumnId = col.columnId;

            //in topics summary view we can't access to actionItemId, so when is undefined
            // we only show the entire group
            if (typeof actionItemId === 'undefined') {
                _this16.detailGroup.firstActionItem = null;
            } else {
                // we create this variable to know what action item was clicked first
                _this16.detailGroup.firstActionItem = _this16.detailGroup.actionItems.find(function (ai) {
                    return ai.id === actionItemId;
                });
                _this16.firstActionItemId = _this16.detailGroup.firstActionItem.id;
            }
        };

        this.convertToTasks = function (columnId, $index) {
            if (_this16.creatingConfluenceTasks) {
                return;
            }

            _this16.creatingConfluenceTasks = true;

            top.window.location.href = IMAGES_BASE_PATH + '/pages/editpage.action?createTasks=true&pageId=' + $scope.$parent.vm.pageId + '&settingsId=' + btoa(vm.uuid) + '&columnId=' + btoa(columnId) + '&criteria=' + btoa(_this16.filterActionItemsBy[$index]);
        };

        var filterActionItems = function filterActionItems() {
            var actionItems = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
            var criteria = arguments[1];
            var usingGroups = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : false;

            if (usingGroups) {
                actionItems = actionItems.reduce(function (a, c) {
                    return a.concat(c.actionItems);
                }, []);
            }

            switch (criteria) {
                case 'all':
                    return actionItems.slice();
                case 'onlyResolved':
                    return actionItems.filter(function (item) {
                        return item.complete;
                    });
                case 'onlyUnresolved':
                    return actionItems.filter(function (item) {
                        return !item.complete;
                    });
                case 'onlyUnassigned':
                    return actionItems.filter(function (item) {
                        return !item.assigneeUser;
                    });
                default:
                    // a specific user
                    return actionItems.filter(function (item) {
                        return item.assigneeUser && item.assigneeUser.userKey == criteria;
                    });
            }
        };

        var filterIdeas = function filterIdeas(ideas, criteria) {

            if (ideas === undefined) {
                return;
            }

            switch (criteria) {
                case 'all':
                    return ideas.slice();
                case 'mostVoted':
                    return ideas.slice().sort(function (a, b) {
                        return b.votes - a.votes;
                    });
                case 'leastVoted':
                    return ideas.slice().sort(function (a, b) {
                        return a.votes - b.votes;
                    });
                default:
                    return ideas.slice();
            }
        };

        var collapseGroup = function collapseGroup(group) {
            group.isCollapsed = !group.isCollapsed;
        };

        var getVoteColor = function getVoteColor(group) {
            return typeof group !== 'undefined' && group.votes > 0 ? 'vote-blue-number' : 'vote-gray-number';
        };

        var getVoteTextColor = function getVoteTextColor(group) {
            return typeof group !== 'undefined' && group.votes > 0 ? 'vote-text-voted' : 'vote-text-unvoted';
        };

        var pluralVote = function pluralVote(group) {
            return typeof group !== 'undefined' && group.votes > 0 ? 'plural-word' : '';
        };

        var isCurrentUserMod = function isCurrentUserMod() {
            if (vm.moderator) {
                return vm.moderator.userKey === vm.userKey;
            }
            return UserService.getUserByUserKey(vm.userKey).isModerator;
        };

        var assigneeUser = function assigneeUser(actionItem, user, code, group) {
            if (user) {
                var newUser = Object.assign({}, user);
                delete newUser.isParticipant;
                DiscussService.setAssignee(group.id, actionItem.id, newUser);
            } else {
                DiscussService.clearAssignee(group.id, actionItem.id);
            }
            AJS.$('#trigger-dropdown' + code + vm.uuid).trigger('aui-button-invoke');
        };

        var toggleCompleted = function toggleCompleted(actionItem, currentValue, group) {
            DiscussService.toggleComplete(group.id, actionItem.id).catch(function (err) {
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                });
            });
        };

        var isUserAssignee = function isUserAssignee(actionItem, user) {
            var assignee = actionItem.assigneeUser;
            return assignee && assignee.userName == user.userName;
        };

        /*
         calculate string width in screen
         */
        var hasTitleEllipsis = function hasTitleEllipsis(text) {
            var ellipsisHelper = document.getElementById('ellipsis-calc-helper');
            if (!ellipsisHelper) {
                return false;
            }
            ellipsisHelper.innerHTML = text;
            ellipsisHelper.style.fontSize = '20px';
            var textWidth = ellipsisHelper.clientWidth + 1;
            var widthAvailable = $('.title-think').width();
            return textWidth > widthAvailable;
        };

        var getAssigneesForColumn = function getAssigneesForColumn(column) {
            var group = vm.groupsMap[column.columnId];

            if (!group) {
                return [];
            }

            var actionItems = group.reduce(function (accum, topic) {
                if (topic.actionItems && topic.actionItems.length) {
                    return accum.concat(topic.actionItems);
                }
                return accum;
            }, []);

            if (!actionItems.length) {
                return [];
            }

            var keys = [];
            return actionItems.reduce(function (accum, actionItem) {
                if (actionItem.assigneeUser && keys.indexOf(actionItem.assigneeUser.userKey) < 0) {
                    keys.push(actionItem.assigneeUser.userKey);
                    return accum.concat(actionItem.assigneeUser);
                }
                return accum;
            }, []);
        };

        var hasActionItems = function hasActionItems(colId) {
            var groups = vm.groupsMap[colId];
            return groups && groups.find(function (group) {
                return group.actionItems && group.actionItems.length;
            });
        };

        this.hasMoreThanFourColumns = function () {
            return _this16.settings.columns.length > 4;
        };

        this.init = function () {
            $scope.$parent.vm.showActionItems = true;
            $scope.$parent.vm.showDetailView = false;
            AJS.$("#btn-action-items1").click();
            if (_this16.hasMoreThanFourColumns()) {
                $('#retro-container').addClass('more-than-four-columns-container');
                $('#retro-container').removeClass('retro-container');
            }
        };

        var initializeUsersForActionItems = function initializeUsersForActionItems() {
            vm.confluenceUsersResult = vm.retroParticipants;
        };

        var searchForUser = async function searchForUser() {
            if (!vm.userSearch.trim().length) {
                initializeUsersForActionItems();
                return;
            }
            vm.confluenceUsersResult = await UserService.getUsersBySearch(vm.userSearch.trim());

            if (!vm.confluenceUsersResult.length) {
                initializeUsersForActionItems();
            }
        };

        var cleanUserSearch = function cleanUserSearch() {
            vm.userSearch = '';
            initializeUsersForActionItems();
        };

        vm.hasActionItems = hasActionItems;
        vm.getAssigneesForColumn = getAssigneesForColumn;
        vm.hasTitleEllipsis = hasTitleEllipsis;
        vm.collapseGroup = collapseGroup;
        vm.filterActionItems = filterActionItems;
        vm.filterIdeas = filterIdeas;
        vm.getVoteColor = getVoteColor;
        vm.getVoteTextColor = getVoteTextColor;
        vm.pluralVote = pluralVote;
        vm.isCurrentUserMod = isCurrentUserMod;
        vm.isUserAssignee = isUserAssignee;
        vm.assigneeUser = assigneeUser;
        vm.toggleCompleted = toggleCompleted;
        vm.getColumnClass = SettingsService.getColumnClass;
        vm.isExpired = false;
        vm.searchForUser = searchForUser;
        vm.cleanUserSearch = cleanUserSearch;

        $q.when(SettingsService.isRetroExpired(vm.uuid), function (response) {
            return vm.isExpired = response;
        }).catch(function (err) {
            console.error(err);
            var flagError = AJS.flag({
                type: 'error',
                body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
            });
        });
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').controller('thinkController', thinkController).directive('autofocusIdea', function () {
        return {
            restrict: 'A',
            link: function link(scope, element, attrs) {
                scope.$watch(function () {
                    return scope.$eval(attrs.autofocusIdea);
                }, function (newValue) {
                    if (newValue === true) {
                        element[0].focus();
                    }
                });
            }
        };
    });

    thinkController.$inject = ['IMAGES_BASE_PATH', '$timeout', 'StatefulService', 'IdeasService', 'SettingsService', 'WorkflowService', 'UserService'];

    function thinkController(IMAGES_BASE_PATH, $timeout, Stateful, IdeasService, SettingsService, WorkflowService, UserService) {
        var _this17 = this;

        Stateful.make(this);
        this.readOnlyMode = false;
        this.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        this.userKey = this.state.shared._get('userKey');
        this.uuid = this.state.shared._get('uuid');
        this.settings = SettingsService.settings;
        this.mapIdeaColumns = {};
        this.mapErrorColumns = {};
        this.mapErrorIdeas = {};
        this.ideasInput = {};
        this.minColumnHeight = 500;
        this.currentUser = {};

        /*
         we'll store an object with keys related to the ids of the items being edited
         this way we can seed the "editing" variables when we get new
         idea from the server
         */
        this.ideasBeingEdited = {};

        /*
         used to store the original idea in text before
         the user changes it in the input
         */
        this.originalIdea = {};

        this.$onInit = function () {
            _this17.currentUser = UserService.getUserByUserKey(_this17.userKey);

            if (_this17.currentUser) {
                _this17.currentUser.imageURI = _this17.currentUser.imageUri;
            }
        };

        WorkflowService.subscribe(function (event) {
            _this17.readOnlyMode = true;
        }, ['readOnlyMode']);

        SettingsService.subscribe(function (event) {
            _this17.settings = event.state.settings;
        }, ['settings']);

        IdeasService.subscribe(function (event) {
            var columns = SettingsService.settings.columns;
            for (var i = 0; i < columns.length; i++) {
                var columnId = columns[i].columnId;

                if (JSON.stringify(_this17.mapIdeaColumns[columnId]) != JSON.stringify(event.state.ideas[columnId])) {
                    _this17.mapIdeaColumns[columnId] = (event.state.ideas[columnId] || []).map(function (idea) {
                        idea.editing = _this17.ideasBeingEdited[idea.id];

                        /*
                         don't update the content if it's being edited
                         */
                        if (idea.editing) {
                            var found = _this17.mapIdeaColumns[columnId].find(function (i) {
                                return i.id === idea.id;
                            });

                            if (found) {
                                idea.idea = found.idea;
                            }
                        }

                        return idea;
                    });
                }
            }
        }, ['ideas']);

        var isExisting = function isExisting(item, collection) {
            if (!item || item.length < 1) return false;
            var newIdea = item;
            if (collection === undefined) {
                return false;
            }

            return collection.find(function (element) {
                if (element.idea) {
                    if (element.user.userKey === _this17.userKey) {
                        if (element.editing) {
                            var originalIdea = JSON.parse(_this17.originalIdea[element.id]);

                            return originalIdea.toLowerCase() === newIdea.toLowerCase();
                        }

                        return element.idea.toLowerCase() === newIdea.toLowerCase();
                    }
                } else {
                    return element.toLowerCase() !== newIdea;
                }
            });
        };

        var setError = function setError(columnId, show) {
            _this17.mapErrorColumns[columnId] = show;
        };

        var postIdea = function postIdea(idea, columnId) {
            IdeasService.create({
                userKey: _this17.userKey,
                uuid: _this17.uuid,
                idea: idea.idea,
                columnId: columnId
            }).catch(function (err) {
                console.error(err);

                // Remove idea from the array
                var ideaIndex = _this17.mapIdeaColumns[columnId].findIndex(function (i) {
                    return i.id === idea.id;
                });

                if (ideaIndex !== -1) {
                    _this17.mapIdeaColumns[columnId].splice(ideaIndex, 1);
                }

                AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        var addToCollection = function addToCollection(item, columnId) {

            if (!item) {
                return;
            }

            if (!_this17.mapIdeaColumns[columnId]) {
                _this17.mapIdeaColumns[columnId] = [];
            }

            if (isExisting(item, _this17.mapIdeaColumns[columnId])) {
                setError(columnId, true);
            } else {
                var newItem = {
                    columnId: columnId,
                    editable: true,
                    id: Math.random().toString(36).substr(2),
                    idea: item,
                    user: _this17.currentUser
                };

                _this17.mapIdeaColumns[columnId].push(newItem);

                postIdea(newItem, columnId);
                _this17.ideasInput[columnId] = '';
                setError(columnId, false);
            }
        };

        var insertElement = function insertElement(keyEvent, columnId) {
            _this17.mapErrorColumns[columnId] = false;
            if (keyEvent.which === 13) {
                addToCollection(_this17.ideasInput[columnId], columnId);
            }
        };

        var removeItem = function removeItem(idea) {
            IdeasService.remove(idea.id).catch(function (err) {
                console.error(err);
                AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        /*
         calculate string width in screen
         */
        var hasTitleEllipsis = function hasTitleEllipsis(text) {
            var ellipsisHelper = document.getElementById('ellipsis-calc-helper');
            if (!ellipsisHelper) {
                return false;
            }
            ellipsisHelper.innerHTML = text;
            ellipsisHelper.style.fontSize = '20px';
            var textWidth = ellipsisHelper.clientWidth + 1;
            var widthAvailable = $('.title-think').width();
            return textWidth > widthAvailable;
        };

        var startEdit = function startEdit(item) {
            item.editing = true;
            _this17.ideasBeingEdited[item.id] = true;
            _this17.originalIdea[item.id] = JSON.stringify(item.idea);

            $timeout(function () {
                $('#input-edit-idea-' + item.id).focus();
            }, 100);
        };

        var trySave = function trySave($event, item, columnId) {
            if ($event.keyCode === 13) {
                saveEdit(item, columnId);
            }
        };

        var saveEdit = function saveEdit(item, columnId) {
            var isDuplicated = false;

            if (_this17.mapIdeaColumns[columnId]) {
                var _iteratorNormalCompletion = true;
                var _didIteratorError = false;
                var _iteratorError = undefined;

                try {
                    for (var _iterator = _this17.mapIdeaColumns[columnId][Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                        var idea = _step.value;

                        if (idea.id !== item.id && idea.user.userKey === _this17.userKey) {
                            if (idea.editing) {
                                var originalIdea = JSON.parse(_this17.originalIdea[idea.id]);

                                if (originalIdea.toLowerCase() === item.idea.toLowerCase()) {
                                    isDuplicated = true;
                                }
                            } else if (idea.idea.toLowerCase() === item.idea.toLowerCase()) {
                                isDuplicated = true;
                            }
                        }
                    }
                } catch (err) {
                    _didIteratorError = true;
                    _iteratorError = err;
                } finally {
                    try {
                        if (!_iteratorNormalCompletion && _iterator.return) {
                            _iterator.return();
                        }
                    } finally {
                        if (_didIteratorError) {
                            throw _iteratorError;
                        }
                    }
                }
            }

            if (isDuplicated) {
                // Set this flag to true to show an error message
                _this17.mapErrorIdeas[item.id] = true;
            } else {
                // Hide the error message
                _this17.mapErrorIdeas[item.id] = false;

                // Update idea
                IdeasService.update(item).then(function () {
                    item.editing = false;
                    delete _this17.originalIdea[item.id];
                    delete _this17.ideasBeingEdited[item.id];
                }).catch(function (err) {
                    console.error(err);
                    AJS.flag({
                        type: 'error',
                        body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>'
                    });
                });
            }
        };

        var cancelEdit = function cancelEdit(item) {
            item.editing = false;
            item.idea = JSON.parse(_this17.originalIdea[item.id]);

            _this17.mapErrorIdeas[item.id] = false;
            delete _this17.originalIdea[item.id];
            delete _this17.ideasBeingEdited[item.id];
        };

        this.getIdeasTextClass = function () {
            return _this17.settings.anonymousPlayersAllowed ? "anonimus-ideas-text-width" : "ideas-text-width";
        };

        this.hasMoreThanFourColumns = function () {
            return _this17.settings.columns.length > 4;
        };

        this.init = function () {
            if (_this17.hasMoreThanFourColumns()) {
                $('#retro-container').addClass('more-than-four-columns-container');
                $('#retro-container').removeClass('retro-container');
            }
        };

        this.trySave = trySave;
        this.saveEdit = saveEdit;
        this.cancelEdit = cancelEdit;
        this.startEdit = startEdit;
        this.hasTitleEllipsis = hasTitleEllipsis;
        this.insertElement = insertElement;
        this.removeItem = removeItem;
        this.setError = setError;
        this.isIssueOwner = function (issue) {
            return issue.user && issue.user.userKey === _this17.userKey;
        };
        this.getColumnClass = SettingsService.getColumnClass;
    }
})(angular);

(function (angular) {
    angular.module('retrospectiveApp').controller('voteController', VoteController);

    VoteController.$inject = ['IMAGES_BASE_PATH', 'HTTP_PATH', '$http', '$scope', 'EventService', 'IdeasService', 'VotesService', 'SettingsService', 'WorkflowService'];

    function VoteController(IMAGES_BASE_PATH, HTTP_PATH, $http, $scope, EventService, IdeasService, VotesService, SettingsService, WorkflowService) {
        var _this18 = this;

        var vm = this;
        vm.readOnlyMode = false;
        vm.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        var creatingVote = false;
        vm.userKey = $scope.$parent.vm.userKey;
        vm.uuid = $scope.$parent.vm.uuid;
        vm.moderator = $scope.$parent.vm.moderator;
        vm.settings = SettingsService.settings;
        vm.groups = {};
        vm.colorsColumn = {};

        vm.thinkElements = [];
        this.groupCollection = null;
        var cache = '';

        this.localVotes = {};
        this.columnMajorities = {};
        this.majoritiesCalculated = false;

        WorkflowService.subscribe(function (event) {
            _this18.readOnlyMode = true;
        }, ['readOnlyMode']);

        SettingsService.subscribe(function (event) {
            _this18.settings = event.state.settings;
        }, ['settings']);

        $http.get(HTTP_PATH + '/ideas?uuid=' + vm.uuid).then(function (response) {
            vm.thinkElements = response.data;

            VotesService.subscribe(function (event) {
                var asString = JSON.stringify(event.state.groupList);

                // avoid re-render of the same state
                if (asString === cache) {
                    return;
                } else {
                    cache = asString;
                }

                _this18.groupCollection = {
                    groupList: event.state.groupList
                };

                /**
                 * groups won't change, so let's calculate it only the first time
                 */
                if (!_this18.majoritiesCalculated) {
                    _this18.majoritiesCalculated = true;
                    var ids = vm.settings.columns.map(function (col) {
                        return col.columnId;
                    });

                    for (var j = 0; j < _this18.groupCollection.groupList.length; j++) {
                        _this18.findColumnIdMajority(j, _this18.groupCollection.groupList[j].ideas, ids);
                    }
                }

                for (var i = 0; i < vm.settings.columns.length; i++) {
                    var columnId = vm.settings.columns[i].columnId;
                    var groupsForColumn = [];

                    for (var _j = 0; _j < _this18.groupCollection.groupList.length; _j++) {
                        if (_this18.groupCollection.groupList[_j].ideas[0] && _this18.columnMajorities[_j] == columnId) {
                            if (vm.groups[columnId] && vm.groups[columnId][_j]) {
                                _this18.groupCollection.groupList[_j].isCollapsed = vm.groups[columnId][_j].isCollapsed;
                            }

                            groupsForColumn.push(_this18.groupCollection.groupList[_j]);
                        }
                    }

                    vm.groups[columnId] = groupsForColumn;
                }
            }, ['groupList']);
        }).catch(function (err) {
            console.error(err);
            var flagError = AJS.flag({
                type: 'error',
                body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
            });
        });

        this.findColumnIdMajority = function (index, ideas, ids) {
            var counter = {};

            /**
             * initialize a counter for every id
             */
            ids.forEach(function (id) {
                return counter[id] = 0;
            });

            ideas.forEach(function (idea) {
                counter[idea.columnId]++;
            });

            var keys = Object.keys(counter);
            var winnerCounter = counter[keys[0]];
            var winnerId = keys[0];

            keys.forEach(function (key) {
                if (counter[key] > winnerCounter) {
                    winnerCounter = counter[key];
                    winnerId = key;
                }
            });

            _this18.columnMajorities[index] = winnerId;
        };

        this.findColor = function (columnId) {
            var column = _this18.settings.columns.find(function (col) {
                return col.columnId == columnId;
            });
            if (column) {
                return column.color;
            }
            return '#000';
        };

        var countTotalVotes = function countTotalVotes() {
            var groups = [];

            var columnIds = Object.keys(vm.groups || {});

            columnIds.forEach(function (id) {
                groups = groups.concat(vm.groups[id]);
            });

            var totalVotes = groups.reduce(function (sum, group) {
                return sum + (group.votesPerUser && group.votesPerUser[vm.userKey] || 0);
            }, 0);

            SettingsService.setTotalVotes(totalVotes);

            return totalVotes;
        };

        var voteClick = function voteClick(group, columnId, increased) {
            if (creatingVote) {
                return;
            }
            creatingVote = true;

            var data = {};
            data.group = group;
            data.columnId = columnId;

            VotesService.create({
                uuid: vm.uuid,
                data: data,
                increased: increased
            }).then(function (response) {
                if (response.data.applied) {
                    if (increased) {
                        group.votesPerUser[vm.userKey] = (group.votesPerUser[vm.userKey] || 0) + 1;
                    } else {
                        group.votesPerUser[vm.userKey] = group.votesPerUser[vm.userKey] - 1;
                    }
                } else {
                    console.log('RETROSPECTIVES: Vote request not applied: ' + response.data.message);
                }
                creatingVote = false;
            }).catch(function (err) {
                creatingVote = false;
                console.error(err);
                var flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        var collapseClass = function collapseClass(group) {
            return group.grouped && group.isCollapsed ? '' : 'ideas-grouped';
        };

        var collapseGroup = function collapseGroup(group) {
            if (group.grouped) {
                group.isCollapsed = !group.isCollapsed;
            }
        };

        var getVoteLabel = function getVoteLabel(numberOfVotes) {
            if (vm.settings.votesAllowedPerCard === 0) {
                return numberOfVotes;
            }
            return numberOfVotes + '/' + vm.settings.votesAllowedPerCard;
        };

        var getVoteText = function getVoteText(collection) {
            if (vm.settings.votesAllowedPerCard !== 0) {
                return 'Votes';
            }
            return collection.votesPerUser && collection.votesPerUser[vm.userKey] === 1 ? 'Vote' : 'Votes';
        };

        var getVotesForCard = function getVotesForCard(group) {
            return group.votesPerUser && group.votesPerUser[vm.userKey] || 0;
        };

        vm.getVotingClass = function (group, delta) {
            if (delta > 0) {
                //increase
                if (vm.settings.votesPerPlayer !== 0 && countTotalVotes() >= vm.settings.votesPerPlayer) {
                    return 'inactive';
                }

                if (vm.settings.votesAllowedPerCard !== 0 && group.votesPerUser && group.votesPerUser[vm.userKey] >= vm.settings.votesAllowedPerCard) {
                    return 'inactive';
                }

                return 'active';
            } else {
                // decrease
                return getVotesForCard(group) > 0 ? 'active' : 'inactive';
            }
        };

        var ideasCollapsedClass = function ideasCollapsedClass(group) {
            return group.grouped && group.isCollapsed && group.ideas.length > 1 ? '' : 'collapsed-ideas';
        };

        var isCurrentUserMod = function isCurrentUserMod() {
            return vm.moderator && vm.moderator.userKey === vm.userKey;
        };

        vm.getArrowUpTitle = function (group, delta) {
            if (delta > 0) {
                //increase
                if (vm.settings.votesPerPlayer !== 0 && countTotalVotes() >= vm.settings.votesPerPlayer) {
                    return 'You’ve run out of votes';
                }

                if (vm.settings.votesAllowedPerCard !== 0 && group.votesPerUser && group.votesPerUser[vm.userKey] >= vm.settings.votesAllowedPerCard) {
                    return 'You’ve run out of votes';
                }

                return '';
            } else {
                // decrease
                return getVotesForCard(group) > 0 ? '' : 'You’ve run out of votes';
            }
        };

        /*
         calculate string width in screen
         */
        var hasTitleEllipsis = function hasTitleEllipsis(text) {
            var ellipsisHelper = document.getElementById('ellipsis-calc-helper');
            if (!ellipsisHelper) {
                return false;
            }
            ellipsisHelper.innerHTML = text;
            ellipsisHelper.style.fontSize = '20px';
            var textWidth = ellipsisHelper.clientWidth + 1;
            var widthAvailable = $('.title-think').width();
            return textWidth > widthAvailable;
        };

        this.hasMoreThanFourColumns = function () {
            return _this18.settings.columns.length > 4;
        };

        this.init = function () {
            if (_this18.hasMoreThanFourColumns()) {
                $('#retro-container').addClass('more-than-four-columns-container');
                $('#retro-container').removeClass('retro-container');
            }
        };

        vm.getVoteLabel = getVoteLabel;
        vm.hasTitleEllipsis = hasTitleEllipsis;
        vm.voteClick = voteClick;
        vm.getVoteText = getVoteText;
        vm.collapseGroup = collapseGroup;
        vm.collapseClass = collapseClass;
        vm.ideasCollapsedClass = ideasCollapsedClass;
        vm.isCurrentUserMod = isCurrentUserMod;
        vm.getColumnClass = SettingsService.getColumnClass;
    }
})(angular);
(function (angular) {
    $(document).ready(function () {
        angular.element(document).ready(function () {
            var tags = document.getElementsByClassName('retro-container');

            var _loop = function _loop(i) {
                setTimeout(function () {
                    angular.bootstrap(angular.element(tags[i]), ['retrospectiveApp']);
                }, 50);
            };

            for (var i = 0; i < tags.length; i++) {
                _loop(i);
            }
        });
    });
})(angular);
//# sourceMappingURL=app1698948812210.js.map
