import { RetrospectiveData } from './../models/RetrospectiveModel';

declare const Promise:any;

class RetrospectiveService {

    /*
     The motivation for this logic is that
     when we are editing or creating a page, after the base url,
     we'll find the /pages path, so that we can distinct between the base url and
     the path of the page.

     The approach of @@BASE_REST_URL is no longer working since users can have
     a base url with a permanent path included

     i.e.

     https://mydomain.com/itservices/

     instead of just

     https://mydomain.com/
     */
    private getBaseUrl():string {
        const baseUrl = window.location.origin + window.location.pathname;

        // the page is created using a blueprint
        if (baseUrl.indexOf('/plugins/createcontent') >= 0) {
            const split = baseUrl.split('/plugins/createcontent');
            return split[0];
        } else {
            const split = baseUrl.split('/pages');
            return split[0];
        }
    }

    getSettings(uuid:string):Promise<any> {
        return new Promise(resolve => {
            return $.ajax({
                type: "GET",
                dataType: "json",
                headers: {
                    'X-Atlassian-Token': 'no-check',
                    'Content-Type': 'application/json'
                },
                url: `${this.getBaseUrl()}/rest/retrospective/latest/settings/${uuid}`
            }).always(data => {
                this.getEditAvailableBySettingsId(uuid).then(available => {
                    data.editSettingsAvailable = available;
                    resolve(data);
                });
            })
        });
    }

    getEditAvailableByPageId(pageId:string):Promise<any> {
        return new Promise(resolve => {
            return $.ajax({
                type: "GET",
                dataType: "json",
                headers: {
                    'X-Atlassian-Token': 'no-check',
                    'Content-Type': 'application/json'
                },
                url: `${this.getBaseUrl()}/rest/retrospective/latest/settings/page/${pageId}/permissions`
            }).always(data => {
                resolve(data.available);
            })
        });
    }

    getEditAvailableBySettingsId(settingsId:string):Promise<any> {
        return new Promise(resolve => {
            return $.ajax({
                type: "GET",
                dataType: "json",
                headers: {
                    'X-Atlassian-Token': 'no-check',
                    'Content-Type': 'application/json'
                },
                url: `${this.getBaseUrl()}/rest/retrospective/latest/settings/${settingsId}/permissions`
            }).always(data => {
                resolve(data.available);
            })
        });
    }

    saveSettings(settings:RetrospectiveData):Promise<any> {
        return new Promise(resolve => {
            $.ajax({
                type: "POST",
                headers: {
                    'X-Atlassian-Token': 'no-check',
                    'Content-Type': 'application/json'
                },
                dataType: "application/json",
                data: JSON.stringify(settings),
                url: `${this.getBaseUrl()}/rest/retrospective/latest/settings`
            }).always(data => {
                resolve(data.responseText);
            })
        });
    }

    getFormats():Promise<any> {
        return new Promise(resolve => {
            return $.ajax({
                type: "GET",
                dataType: "json",
                headers: {
                    'X-Atlassian-Token': 'no-check',
                    'Content-Type': 'application/json'
                },
                url: `${this.getBaseUrl()}/rest/retrospective/latest/settings/formats/`
            }).always(data => {
                resolve(data);
            })
        });
    }
}

const retrospectiveService = new RetrospectiveService();
export default retrospectiveService;
