(function (angular) {
    angular
        .module('retrospectiveApp')
        .controller('homeController', homeController);

    homeController.$inject = [
        'IMAGES_BASE_PATH',
        'IdeasService',
        'EventService',
        'SettingsService',
        'DateService',
        'WorkflowService',
        'ConfluenceAccessModeService'
    ];

    function homeController(IMAGES_BASE_PATH, IdeasService, EventService,
        SettingsService, DateService, WorkflowService, ConfluenceAccessModeService) {

        this.readOnlyMode = ConfluenceAccessModeService.isReadOnlyModeEnabled();

        this.IMAGES_BASE_PATH = IMAGES_BASE_PATH;
        this.isUserReady = false;
        this.activeView = '';
        this.settings = SettingsService.settings;
        this.completionDate = null;

        this.setUserData = false;
        this.validLicense = false;
        this.sessionCompleted = false;
        this.ideasSize = 0;

        this.thinkState = false;
        this.groupState = false;
        this.voteState = false;
        this.discussState = false;

        this.title = 'Retrospectives App';

        this.activeUsers = EventService.activeUsers;
        this.showDetailView = false;

        WorkflowService.subscribe((event)=>{
            this.readOnlyMode = true
        },['readOnlyMode']);

        SettingsService.subscribe((event) => {
            this.settings = event.state.settings;
        }, ['settings']);

        WorkflowService.subscribe((event) => {
            this.moderator = event.state.moderator;
        }, ['moderator']);

        this.transferUser = {};

        this.initRetro = WorkflowService.initRetro;
        this.setModerator = WorkflowService.setModerator;
        this.nextAvailable = true;
        this.showActionItems = true;

        this.retrospectiveParticipants = [];
        let cache = '';

        this.getCompletionDate = () => {
            return DateService.format(this.completionDate);
        };

        IdeasService.subscribe((event) => {
            const asString = JSON.stringify(event.state.ideas);

            // avoid re-render of the same state
            if (asString === cache) {
                return;
            } else {
                cache = asString;
            }


            var size = 0;

            if (this.settings.columns.length) {
                let firstId = this.settings.columns[0].columnId;

                size = this.settings.columns.reduce((a, c) => {
                    if (event.state.ideas[c.columnId]) {
                        return a + (event.state.ideas[c.columnId] || []).length;
                    }
                    return a;
                }, 0);
                this.ideasSize = size;
            }
        }, ['ideas']);

        WorkflowService.subscribe((event) => {
            this.completionDate = event.state.completionDate;
        }, ['completionDate']);

        /*
         get participants from polling stream
         */
        WorkflowService.subscribe((event) => {
            this.activeUsers = event.state.participants;
        }, ['participants']);

        WorkflowService.subscribe((event) => {
            this.retrospectiveParticipants = event.state.participants;
        }, ['sessionParticipants']);

        WorkflowService.subscribe((event) => {
            this.isUserReady = event.state.ready;
        }, ['isUserReady']);

        /*
         subscribe to changes in the view/stage
         */
        WorkflowService.subscribe((event) => {
            if (this.activeView == event.state.stage) {
                return;
            }

            this.activeView = event.state.stage;

            switch (event.state.stage) {
                case 'think':
                    this.thinkState = true;
                    this.groupState = false;
                    this.voteState = false;
                    this.discussState = false;
                    break;
                case 'group':
                    this.thinkState = true;
                    this.groupState = true;
                    this.voteState = false;
                    this.discussState = false;
                    break;
                case 'vote':
                    this.thinkState = true;
                    this.groupState = true;
                    this.voteState = true;
                    this.discussState = false;
                    break;
                case 'discuss':
                    this.thinkState = true;
                    this.groupState = true;
                    this.voteState = true;
                    this.discussState = true;
                    break;
                case 'main':
                    this.thinkState = false;
                    this.groupState = false;
                    this.voteState = false;
                    this.discussState = false;
                    break;
            }
        }, ['stage']);

        this.isAdministratorAlone = () => false;

        this.onGetStartedClick = () => {
            WorkflowService.changeView('think');
        };

        this.imReady = () => {
            WorkflowService.imReady().then(() => {
                document.getElementById('readyBtn').blur();
            }).catch(() =>{
                console.error("error while updating ready status");
            })
        };

        this.isCurrentUserMod = () => {
            return this.moderator && this.moderator.userKey === this.userKey;
        };

        this.getUsersReady = () => this.activeUsers.filter(user => user.isReady).length;

        this.isValidActiveView = () => ['think', 'group', 'vote', 'discuss'].indexOf(this.activeView) >= 0;

        this.isWaitActive = () => this.activeView == 'wait';

        this.isResultActive = () => this.activeView == 'result';

        this.nextSection = () => {
            if (this.nextAvailable) {
                const sections = ['think', 'group', 'vote', 'discuss', 'result'];
                const index = sections.indexOf(this.activeView);
                WorkflowService.changeView(sections[index + 1]);
            }
        };

        this.transferModeratorRole = () => {
            this.setModerator(this.transferUser);
        };


        this.showModalTransferRole = user => {
            if (this.isCurrentUserMod()) {
                this.transferUser = user;
                $(`#transferModal${this.uuid}`).modal();
            }
        };

        this.getTooltipConfig = ({ view }) => {
            let messageMod = '';
            const messageUser = 'Only moderator can move between sections';
            let condition = true;

            if (this.isCurrentUserMod()) {
                if (!this.activeView) { //waiting screen
                    messageMod = 'You can\'t start a session without your team members';
                    condition = this.isAdministratorAlone() || !this.isCurrentUserMod();
                } else { //any other step
                    messageMod = 'To move forward, first type an idea';
                    condition = this.activeView == 'think' && this.ideasSize == 0;
                }
            }

            return {
                messageMod,
                messageUser,
                condition
            }
        };

        this.isAdministratorAlone = () => false;

        this.changeStepView = (view) => {
            if ((this.isAdministratorAlone() || !this.isCurrentUserMod()) || (view != 'think' && !this.ideasSize)) {
                return;
            }

            WorkflowService.changeView(view);
        };

        this.getTotalVotesText = () => {
            if (this.settings.votesPerPlayer) {
                return `${this.settings.countTotalVotes}/${this.settings.votesPerPlayer} Total votes`;
            }
            return '';
        };

        this.reopen = () => {
            WorkflowService.reopen().catch(err => {
                console.error(err);
                const flagError = AJS.flag({
                    type: 'error',
                    body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                });
            });
        };

        this.activeDetailView = (show) =>{
            this.showDetailView = show;
        }

        this.changeTab = (show,showDetailView) => {
            if(showDetailView){
                this.activeDetailView(showDetailView)
                return
            }

            if (show) {
                $(`#btn-action-items${this.uuid}`).addClass('result-header-btn-active');
                $(`#btn-action-items${this.uuid}`).attr('disabled', 'disabled');
                $(`#btn-topics${this.uuid}`).removeClass('result-header-btn-active');
                $(`#btn-topics${this.uuid}`).removeAttr('disabled');
                this.activeDetailView(false);
            } else {
                $(`#btn-action-items${this.uuid}`).removeClass('result-header-btn-active');
                $(`#btn-action-items${this.uuid}`).removeAttr('disabled');
                $(`#btn-topics${this.uuid}`).addClass('result-header-btn-active');
                $(`#btn-topics${this.uuid}`).attr('disabled', 'disabled');
                this.activeDetailView(false);
            }
            this.showActionItems = show;
        };

        const countIdeas = (columns) => {
            let size = 0;
            if (columns.length) {
                size = columns.reduce((a, c) => {
                    const ideasList = IdeasService.getIdeasListByColumnId(c.columnId);
                    if (ideasList) {
                        return a + (ideasList || []).length;
                    }
                    return a;
                }, 0);
            }
            return size;
        }

        this.getModeratorTooltipMessage = () => {
            if (this.presetModerator === 'RETROS_DYNAMIC_MODERATOR') {
                return '<strong>Dynamic Moderator</strong>: Role is defined during the session and <br/> is automatically transferred to next in line if moderator disconnects.';
            }

            return '<strong>Preset Moderator:</strong>: This participant controls the session. <br/> Role can be transferred if needed.';
        }

        this.initEvents = ({ presetModerator, userKey, validLicense, sessionCompleted, completionDate, uuid, pageId, appId }) => {
            this.userKey = userKey;
            this.uuid = uuid; // for new macros we are going to use the uuid, for legacy macros, this value will be the pageId
            this.pageId = pageId;
            this.validLicense = validLicense;
            this.sessionCompleted = sessionCompleted;
            this.appId = appId;
            this.completionDate = completionDate;
            this.presetModerator = presetModerator;

            WorkflowService.initRetro(this.uuid).then(() => {
                $(`#${this.uuid}`).css('display', 'inherit');

                if (this.uuid) {
                    SettingsService.get(this.uuid).then((settings) => {
                        const localSettings = {
                            anonymousPlayersAllowed: settings.anonymousPlayersAllowed,
                            votesPerPlayer: settings.votesPerPlayer,
                            votesAllowedPerCard: settings.votesAllowedPerCard,
                            columns: settings.columns
                        };
                        SettingsService.set(localSettings);
                    }).catch(err => {
                        console.error(err);
                        const flagError = AJS.flag({
                            type: 'error',
                            body: 'An error has occurred. Please try again and <a href="https://planningpoker.atlassian.net/servicedesk/customer/portal/2/group/2/create/8" target="_blank">contact support</a> if the issue persists.'
                        });
                    })
                }

                if (this.validLicense || this.sessionCompleted) {
                    WorkflowService.startPollers(this.presetModerator, this.userKey, this.uuid, this.pageId).catch(err => {
                        console.error(err);
                        const flagError = AJS.flag({
                            type: 'error',
                            body: 'There has been an error, if this persist please contact to <a href="https://www.google.com/" target="_blank">support</a>',
                        });
                    });
                    IdeasService.setIdeasMapToState(this.uuid).then((value) => {
                        this.ideasSize = countIdeas(this.settings.columns);
                    });
                }
            }).catch(err => {
                console.error("Retrospective initialization error:", err);
            });
        };

        this.getFooterStyles = () => {
            const styles = {"margin-left": "-28px"}

            switch (this.activeView) {
                case 'wait': {
                    styles.width = "calc(100% + 30px)";
                } break;
                case 'result': {
                    styles.width = "calc(100% + 43px)";
                    styles['margin-left'] = "-13px";
                } break;
                default: {
                    styles.width = "calc(100% + 48px)";
                }
            }

            return styles;
        }
    }
}(angular));
