import Dialog from "./dialog";
import navigation from "./navigation";
import "./declarations/polifylls";
import "../assets/css/app-rte.css";
import retrospectiveModel, {RetrospectiveData} from "./models/RetrospectiveModel";
import retrospectiveService from "./services/RetrospectiveService";
import wizardModel from "./models/WizardModel";
import * as JQuery from "jquery";

declare const AJS: any;
declare const tinymce: any;
declare const require: any;
declare const Promise: any;

const MACRO_NAME = 'agile Retrospectives for Confluence';
const CONTENT_ID = 'dialog-content';

function getQueryStringValue(key) {
    return decodeURIComponent(window.location.search.replace(new RegExp("^(?:.*[&\\?]" + encodeURIComponent(key).replace(/[\.\+\*]/g, "\\$&") + "(?:\\=([^&]*))?)?.*$", "i"), "$1"));
}

/**
 * Creates the dialog and initialize the navigation
 * @param macro
 */
function opener(macro: any) {
    const isNewPage = AJS.Confluence.Editor.isNewPage();

    /*
     Confluence provides a content id for new pages.
     Before use this value as the page id, please verify that
     the page already exists
     */
    const pageId = AJS.Confluence.getContentId();

    /*
     For any case, the first thing is to empty models to avoid conflicts
     */
    retrospectiveModel.clear();

    /*
     This is going to be for new pages, don't worry about legacy settings
     */
    if (isNewPage) {
        /*
         ok, this is new, but we can be editing an existing macro...
         */
        if (macro.params && macro.params['agile-retrospectives-for-confluence-uuid']) {
            recoverSettings(macro);
        } else {
            /*
             this is new... there are no permissions to manage
             */
            wizardModel.setEditSettingsAvailable(true);
            openDialog(false, 1);
        }
    }
    /*
     This is going to be for both new and existing pages, can include legacy pages
     */
    else {
        /*
         In this case, we are editing a non-legacy macro
         */
        if (macro.params && macro.params['agile-retrospectives-for-confluence-uuid']) {
            recoverSettings(macro);
        }
        /*
         for legacy macros, we just need to validate if the permission is available
         don't worry about other settings
         */
        else {
            retrospectiveService.getEditAvailableByPageId(pageId).then(available => {
                wizardModel.setEditSettingsAvailable(available);
                openDialog(false, 1);
            });
        }
    }
}

function recoverSettings(macro) {
    const uuid = macro.params['agile-retrospectives-for-confluence-uuid'];

    retrospectiveService.getSettings(uuid).then(settings => {
        retrospectiveModel.removeAllColumns();

        retrospectiveModel.setAnonymousPlayers(settings.anonymousPlayersAllowed);
        retrospectiveModel.setVotesPerCard(settings.votesAllowedPerCard);
        retrospectiveModel.setVotesPerPlayer(settings.votesPerPlayer);
        retrospectiveModel.setPresetModeratorUserKey(settings.presetModeratorUserKey);
        retrospectiveModel.setFormatId(settings.formatId)

        retrospectiveService.getFormats().then(data => {
            retrospectiveModel.setFormats(data)
            let technique = data.find(format => format.id === settings.formatId).name
            retrospectiveModel.setTechniqueSelected(technique)
            retrospectiveModel.setColumnsToFormat(settings.formatId, settings.columns)
            wizardModel.setEditSettingsAvailable(settings.editSettingsAvailable);

            let isEditing = true
            openDialog(isEditing, settings.formatId);
        }).catch(err => console.log('error getting formats'))

    }).catch((err) => {
        console.error('Unable to load settings for this macro, please contact if the error persists');
    });

    retrospectiveService.getFormats().then(formats => {
        retrospectiveModel.setFormats(formats.formats)
    }).catch(err => console.log('error getting formats'))
}

function openDialog(isEditing = false, formatId) {
    const options = {
        titleText: 'Set up the Retrospective',
        content: `<div id="${CONTENT_ID}"></div>`,
        footerActionContent: require('./navigation/navigation.html')
    };

    const dialog = new Dialog(MACRO_NAME, options);
    navigation.init(dialog, includeMacroToDom, CONTENT_ID, isEditing, formatId);
    dialog.show();
    document.getElementById(`${MACRO_NAME}`).classList.add('retro-modal-height');
}

function isReadOnlyMode() {
    return new Promise((resolve, reject) => {
        return $.ajax({
            type: "GET",
            url: AJS.contextPath() + "/rest/api/accessmode",
            contentType: 'application/json',
            dataType: "json",
        }).done(function (xhr) {
            if (xhr.responseText === "READ_ONLY") {
                return resolve(true)
            }
            return resolve(false)
        }).fail(err => {
            return resolve(false)
        });
    })
}

/**
 * Called as a end method to include the macro in the editor
 */

function includeMacroToDom() {
    const settings: RetrospectiveData = retrospectiveModel.getSettings();
    settings.columns = retrospectiveModel.getColumnsFromFormat(Number(settings.formatId))

    return isReadOnlyMode().then(readOnlyMode => {
        if (readOnlyMode) {
            return false
        }

        settings.isTemplate = isTemplate();
        settings.spaceKey = AJS.Meta.get("space-key");

        return retrospectiveService.saveSettings(settings).then((uuid) => {
            tinymce.confluence.macrobrowser.macroBrowserComplete({
                name: MACRO_NAME, bodyHtml: undefined, params: {
                    'agile-retrospectives-for-confluence-uuid': uuid
                }
            });
            return true
        }).catch(err => {
            console.error(err);
            const flagError = AJS.flag({
                type: 'error',
                body: 'There has been an error while creating retro settings',
            });
            return false
        });
    }).catch(err => {
        console.error(err);
        const flagError = AJS.flag({
            type: 'error',
            body: 'There has been an error',
        });
        return false
    });


}

function isTemplate(): boolean {
    const baseUrl = window.location.origin + window.location.pathname;
    // the page is created using a template
    return ((baseUrl.indexOf('/pages/templates') >= 0) || (baseUrl.indexOf('/plugins/createcontent/edit-template') >= 0));
}


function getBaseUrl(): string {
    const baseUrl = window.location.origin + window.location.pathname;

    // the page is created using a blueprint
    if (baseUrl.indexOf('/plugins/createcontent') >= 0) {
        const split = baseUrl.split('/plugins/createcontent');
        return split[0];
    } else {
        const split = baseUrl.split('/pages');
        return split[0];
    }
}

function getConfluenceTasks(settingsId, pageId, columnId, criteria, callback) {
    $.ajax({
        type: "GET",
        dataType: "json",
        headers: {
            'X-Atlassian-Token': 'no-check',
            'Content-Type': 'application/json'
        },
        url: `${getBaseUrl()}/rest/retrospective/latest/settings/${settingsId}/confluence-tasks?pageId=${pageId}&columnId=${columnId}&criteria=${criteria}`
    }).always(data => {
        callback(data);
    });
}

/**
 * Listener for the editor loaded (kinda document.ready from JQuery for Confluence editor)
 */
function onEditorLoaded() {
    //evaluate if we are converting action items to task list
    if (getQueryStringValue("createTasks")) {
        const pageId = getQueryStringValue("pageId");
        const settingsId = atob(getQueryStringValue("settingsId"));
        const columnId = atob(getQueryStringValue("columnId"));
        const criteria = atob(getQueryStringValue("criteria"));

        getConfluenceTasks(settingsId, pageId, columnId, criteria, (data: any) => {
            let errorMessage = hasErrorMessage(data);
            let errorFlagDisplayed = false;

            // wait until the editor is loaded
            const interval = setInterval(() => {
                const iframe: any = document.getElementById('wysiwygTextarea_ifr');
                if (!iframe) {
                    return;
                }

                const doc = iframe.contentDocument || iframe.contentWindow.document;
                if (!doc) {
                    return;
                }

                if (errorMessage && !errorFlagDisplayed) {
                    AJS.flag({type: 'error', body: errorMessage});

                    errorFlagDisplayed = true;

                    setTimeout(() => {
                        setInterval(() => {
                            const buttonById = $('#rte-button-cancel');

                            if (buttonById) {
                                buttonById.click();
                            }
                        }, 1000);
                    }, 4000);

                    return;
                } else if (errorMessage) {
                    return;
                }

                const tiny = doc.getElementById('tinymce');
                if (!tiny) {
                    return;
                }

                tiny.innerHTML = tiny.innerHTML + data.content;

                const isContent = setInterval(() => {
                    const index = tiny.innerHTML.indexOf(data.content);
                    if (index) {
                        clearInterval(interval);
                        const buttonById = $('#rte-button-publish');
                        if (buttonById) {
                            buttonById.click();
                        }
                    }

                    setTimeout(() => {
                        clearInterval(interval);
                        clearInterval(isContent);
                    }, 10000);

                }, 500);

            }, 3000);

            setTimeout(() => {
                clearInterval(interval);
            }, 10000);
        });
    }

    //When the confluence editor is loaded, we look for our macro in the macro browser and we override the default
    //dialog implementation to show the retrospectives preferences view instead of default view when the macro is inserted
    AJS.MacroBrowser.setMacroJsOverride(MACRO_NAME, {
        opener
    });
    console.log('Retrospectives macro loaded');
}

function hasErrorMessage(data: any) {
    if (!(data.status >= 400)) {
        return false;
    }

    let errorMessage = "Could not create tasks, ";

    switch (data.status) {
        case 400:
            errorMessage += "some of the sent data is wrong.";
            break;
        case 403:
            errorMessage += "you do not have permission to perform this action.";
            break;
        case 404:
            errorMessage += "some of the sent data was not found."
            break;
        case 409:
            errorMessage += "the state of the retrospective does not meet the required state to perform this action.";
            break;
        default:
            errorMessage = data.responseJSON.message;
    }

    return errorMessage;
}

onEditorLoaded();
