declare const Promise:any;

class UserService {

    private users;

    constructor() {
        this.users = [];
    }

    searchUserByName(name: string) {
        return this.users.find(user => user.displayName == name);
    }

    /*
     The motivation for this logic is that
     when we are editing or creating a page, after the base url,
     we'll find the /pages path, so that we can distinct between the base url and
     the path of the page.

     The approach of @@BASE_REST_URL is no longer working since users can have
     a base url with a permanent path included

     i.e.

     https://mydomain.com/itservices/

     instead of just

     https://mydomain.com/
     */
    private getBaseUrl():string {
        const baseUrl = window.location.origin + window.location.pathname;

        // the page is created using a blueprint
        if (baseUrl.indexOf('/plugins/createcontent') >= 0) {
            const split = baseUrl.split('/plugins/createcontent');
            return split[0];
        } else {
            const split = baseUrl.split('/pages');
            return split[0];
        }
    }

    public loadUsers() {
        this.users = [];

        return new Promise(resolve => {
            let postFetch = ({results, next}) => {
                this.users = this.users.concat(results);

                if (next !== -1) {
                    return this.loadUsersPage(next, postFetch);
                } else {
                    return resolve(this.users);
                }
            };

            this.loadUsersPage(0, postFetch);
        });
    }

    private loadUsersPage(start, resolve) {
        return $.ajax({
            type: "GET",
            dataType: "json",
            headers: {
                'X-Atlassian-Token': 'no-check',
                'Content-Type': 'application/json'
            },
            url: `${this.getBaseUrl()}/rest/api/group/confluence-users/member?start=${start}&limit=10000`
        }).always(data => {
            if (data.results && data.results.length) {
                return resolve({
                    results: data.results,
                    next: start + parseInt(data.size),
                });
            }

            return resolve({
                results: [],
                next: -1
            });
        })
    }

    public getUsersSelect2Config(presetModeratorStrategySelection){
        const ajaxConfig = {
            type: "GET",
            dataType: "json",
            headers: {
                'X-Atlassian-Token': 'no-check',
                'Content-Type': 'application/json'
            },
            url: `${this.getBaseUrl()}/rest/api/search`,
            data: (value, page) => {
                const query = {
                    cql: `type=user AND user~"${value}*"`,
                    limit: 10,
                    start: (page - 1) * 10
                };
                return query;
            },
            results: (data) => {
                const options = data.results.map(u => ({text: u.user.displayName, id: u.user.userKey}))
                return {
                    results: options,
                    more: (data.start + data.size) < data.totalSize
                };
            }
        };

        const initSelectionRequest = (element, callback) => {
            let url = `${this.getBaseUrl()}/rest/api/user/current`;
            const key = $(element).val();

            if(key){
                url = `${this.getBaseUrl()}/rest/api/user?key=${key}`
            }
            $.ajax(url, {
                dataType: "json",
                headers: {
                    'X-Atlassian-Token': 'no-check',
                    'Content-Type': 'application/json'
                }
            }).done(function(data) {
                callback({id: data.userKey, text: data.displayName});
                presetModeratorStrategySelection(null, data.userKey);
            });
        };

        return {
            ajax: ajaxConfig,
            minimumInputLength: -1,
            initSelection: initSelectionRequest
        }
    }

}

const userService = new UserService();
export default userService;