(function (angular) {
    angular
        .module('retrospectiveApp')
        .service('WorkflowService', WorkflowService);

    WorkflowService.$inject = [
        'HTTP_PATH',
        'IMAGES_BASE_PATH',
        '$http',
        'PollerService',
        'SubscribableService',
        'StatefulService',
        'IdeasService',
        'GroupsService',
        'VotesService',
        'DiscussService',
        'ReadOnlyModePollerService',
        'HeartbeatPollerService'
    ];

    function WorkflowService(HTTP_PATH, IMAGES_BASE_PATH, $http, Poller, Subscribable, Stateful,
                             IdeasService, GroupsService, VotesService, DiscussService, ReadOnlyModePollerService,
                             HeartbeatPollerService
    ) {
        /*
         turn this into a subscribable service,
         so that controllers can list to its changes
         */
        Subscribable.make(this);

        /*
         add logic to this service for simple state management
         */
        Stateful.make(this, {
            allowNull: false,
            allowUndefined: false
        });

        const setModerator = (moderator) => {
            if (!moderator) {
                return;
            }

            if (!this.state.exists('moderator')
                || this.state.get('moderator').userKey !== moderator.userKey
                || this.state.get('moderator').isReady !== moderator.isReady
                || this.state.get('moderator').isConnected !== moderator.isConnected
            ) {
                this.state.set('moderator', moderator);

                this.notify({
                    identifier: 'moderator',
                    state: {
                        moderator: moderator
                    }
                });
            }
        };

        this.startPollers = async (presetModerator, userKey, uuid, pageId) => {
            this.state.set('userKey', userKey);
            this.state.set('uuid', uuid);
            this.state.set('pageId', pageId);

            this.state.shared._set('userKey', userKey);
            this.state.shared._set('uuid', uuid);
            this.state.shared._set('pageId', pageId);

            /*
             Will take care of:

             - current stage
             - moderator
             - participants
             - who is ready
             - completion date
             */

            const ReadOnlyMode = new ReadOnlyModePollerService((response)=>{
                if(response){
                    this.notify({
                        identifier: 'readOnlyMode',
                        state: {
                            readOnlyMode: true
                        }
                    });
                    IdeasService.stopPoller();
                    GroupsService.stopPoller();
                    VotesService.stopPoller();
                }

            });

            new HeartbeatPollerService(() => {}).start(uuid, userKey);

            const sessionDataPoller = new Poller(`/workflow/session-data?uuid=${uuid}`, (response) => {
                if (!response.data || !response.data.baseUsers) {
                    return;
                }

                response.data.participants = (response.data.baseUsers || [])
                    .filter(u => u.isParticipant)
                    .sort((a, b) => a.userKey < b.userKey);

                response.data.baseUsers = (response.data.baseUsers || [])
                    .filter(u => u.isConnected)
                    .sort((a, b) => a.userKey < b.userKey);


                const definedModerator = response.data.moderator;

                if (definedModerator) {
                    setModerator(definedModerator);
                } else {
                    const {presetModeratorUserKey} = response.data;

                    this.getConfluenceUser(presetModeratorUserKey, uuid).then(user => {
                        setModerator(user);
                    });
                }

                /*
                 notify subscribers about newest info
                 */
                if (this.state.get('stage', 'undefined') !== response.data.viewName) {
                    switch (response.data.viewName) {
                        case 'think':
                            IdeasService.startPollers(userKey, uuid);
                            GroupsService.stopPoller();
                            VotesService.stopPoller();
                            DiscussService.stopPoller();
                            break;
                        case 'group':
                            IdeasService.stopPoller();
                            GroupsService.startPollers(userKey, uuid);
                            VotesService.stopPoller();
                            DiscussService.stopPoller();
                            break;
                        case 'vote':
                            IdeasService.stopPoller();
                            GroupsService.stopPoller();
                            VotesService.startPollers(userKey, uuid);
                            DiscussService.stopPoller();
                            break;
                        case 'discuss':
                        case 'result':
                            IdeasService.stopPoller();
                            VotesService.stopPoller();
                            GroupsService.stopPoller();
                            DiscussService.startPollers(userKey, uuid);
                            break;
                    }

                    this.notify({
                        identifier: 'stage',
                        state: {
                            stage: response.data.viewName
                        }
                    });
                }

                this.state.set('stage', response.data.viewName);

                if (response.data.baseUsers.length) {
                    const user = response.data.baseUsers.find(p => p.userKey == this.state.get('userKey'));

                    this.notify({
                        identifier: 'isUserReady',
                        state: {
                            ready: user ? user.isReady : false,
                        }
                    });
                }

                this.notify({
                    identifier: 'sessionParticipants',
                    state: {
                        participants: response.data.participants
                    }
                });

                this.notify({
                    identifier: 'participants',
                    state: {
                        participants: response.data.baseUsers
                    }
                });

                this.notify({
                    identifier: 'completionDate',
                    state: {
                        completionDate: response.data.completionDate
                    }
                });

                this.notify({
                    identifier: 'discussGroupIndex',
                    state: {
                        index: response.data.discussGroupIndex
                    }
                });
            });

            sessionDataPoller.setIdentifier('Session data poller');
            sessionDataPoller.start();
            ReadOnlyMode.start();
        };

        this.initRetro = (uuid) => {
            const pageId = top.AJS.Meta.get('page-id');

            return $http.put(`${HTTP_PATH}/workflow/${uuid}`, pageId);
        }

        /*
         updates the moderator in the backend and in the local service storage
         */
        this.setModerator = (user) => {
            let prevMod = '';

            const moderator = this.state.get('moderator', false);

            if (moderator && moderator.userKey !== user.userKey) {
                prevMod = `&prevMod=${moderator.userKey}`;
            }

            $http.put(`${HTTP_PATH}/workflow/set-moderator?userKey=${user.userKey}&uuid=${this.state.get('uuid')}${prevMod}`);
        };

        this.changeView = (newStage) => {
            if (this.state.get('stage') === newStage) {
                return;
            }

            $http.put(`${HTTP_PATH}/workflow/change-view?userKey=${this.state.get('userKey')}&uuid=${this.state.get('uuid')}&viewName=${encodeURIComponent(newStage)}`);
        };

        this.reopen = () => $http.get(`${HTTP_PATH}/workflow/reopen?uuid=${this.state.get('uuid')}`);

        this.persistData = () =>
            $http.post(`${HTTP_PATH}/workflow/persist/all?uuid=${this.state.get('uuid')}`);

        this.imReady = () =>
            $http.put(`${HTTP_PATH}/workflow/im-ready?userKey=${this.state.get('userKey')}&uuid=${this.state.get('uuid')}`)

        this.confluenceTask = (content) =>
            $http.post(`${HTTP_PATH}/workflow/confluence-tasks-transaction`, {
                content
            })

        let confluenceUserPromise = null;
        let lastModeratorUserKey = '';
        this.getConfluenceUser = (userKey, uuid) => {
            if (!confluenceUserPromise || lastModeratorUserKey !== userKey) {
                confluenceUserPromise = $http.get(`${IMAGES_BASE_PATH}/rest/api/user?key=${userKey}`)
                    .then(response => {
                        const { data } = response;

                        return {
                            userKey: data.userKey,
                            name: data.username,
                            fullName: data.displayName,
                            pageId: uuid,
                            imageUri: data.profilePicture.path.replace('/confluence', ''),
                            isConnected: false,
                            isModerator: true,
                            isReady: false,
                            isParticipant: false,
                            numericPageId: parseInt(uuid),
                        }
                    });
            }

            return confluenceUserPromise;
        }
    }
}(angular));
