export type RetrospectiveData = {
    columns:  RetrospectiveColumn[];
    votes_per_player: number;// (0 for unlimited)
    votes_per_card: number;// (0 for unlimited)
    anonymous_players: boolean;
    preset_moderator_user_key: string;
    isTemplate: boolean
    formatId: number
    techniqueSelected: string;
};

export type formats = [{
    id: number;
    name: string;
    columns: RetrospectiveColumn[];
}]

export type RetrospectiveColumnCopy = {
    columnId: string;
    content: string;
    color: string;
}

export type RetrospectiveColumn = {
    columnId: string;
    content: string;
    color: string;
}

class RetrospectiveModel {
    private retrospective:RetrospectiveData;
    private formats:formats

    getColumnsFromFormat(formatId:number):RetrospectiveColumn[]{
        if (this.formats !==undefined){
            const format = this.formats.find(format => format.id === formatId)
            if (format) return format.columns
            else return []
        }else return []
    }

    constructor() {
        this.retrospective = {
            formatId:null,
            columns: this.getColumnsFromFormat(this.getFormatId()),
            votes_per_player: 0,// (0 for unlimited)
            votes_per_card: 1,// (0 for unlimited)
            anonymous_players: false,
            preset_moderator_user_key: 'RETROS_DYNAMIC_MODERATOR',
            isTemplate: false,
            techniqueSelected: null
        }
    }

    setTechniqueSelected(technique:string){
        this.retrospective.techniqueSelected = technique
    }

    getTechniqueSelected(){
        return this.retrospective.techniqueSelected
    }

    setColumnsToFormat(formatId, columns){
        this.formats.find(format => format.id === formatId).columns = columns
    }

    deleteColumnToFormat(formatId, index){
        this.formats.splice(index,1)
    }

    setPresetModeratorUserKey(userKey:string):void {
        this.retrospective.preset_moderator_user_key = userKey;
    }

    setVotesPerPlayer(votes:number):void {
        this.retrospective.votes_per_player = votes;
    }

    getVotesPerPlayer():number {
        return this.retrospective.votes_per_player;
    }

    getVotesPerCard():number {
        return this.retrospective.votes_per_card;
    }

    setVotesPerCard(votes:number):void {
        this.retrospective.votes_per_card = votes;
    }

    addColumn(columnId: string, content: string, color: string, formatId: number):void {
        if (this.formats !== undefined){
            this.formats.find(format => format.id === formatId).columns.push({
                columnId, content, color
            });
        }
        this.fixColumnsIds();
    }

    //checks if a columns exists based on the content
    doesColumnsExist(content: string, formatId):boolean {
        return !!this.formats.find(format => format.id === Number(formatId)).columns.find(col => col.content == content);
    }

    //removes a column based on the content
    removeColumnByContent(content: string):void {
        this.retrospective.columns.filter(col => col.content != content);
        this.fixColumnsIds();
    }

    removeColumnByIndex(index: number, formatId):void {
        this.formats.find(format=> format.id === Number(formatId)).columns.splice(index, 1);
        this.fixColumnsIds();
    }

    removeAllColumns() {
        this.retrospective.columns.splice(0,this.retrospective.columns.length);
        this.fixColumnsIds();
    }

    /*
        this will prevent duplicate column ids as side effect of delete columns in the UI
     */
    private fixColumnsIds() {
        for(let i = 0; i < this.retrospective.columns.length; i++) {
            this.retrospective.columns[i].columnId = `C${(i + i)}`;
        }
    }

    updateColumnByIndex(index: number, content: string, color: string, formatId:number):void {
        this.formats.find(format => format.id === Number(formatId)).columns[index].content = content;
        this.formats.find(format => format.id === Number(formatId)).columns[index].color = color;
    }

    getColumns(formatId:number) {
        if (this.formats && formatId){
            return this.formats.find(format => format.id === Number(formatId)).columns;
        }
        return this.retrospective.columns;
    }

    setAnonymousPlayers(value: boolean):void {
        this.retrospective.anonymous_players = value;
    }

    isAnonymousPlayers():boolean {
        return this.retrospective.anonymous_players;
    }

    getSettings():RetrospectiveData {
        return this.retrospective;
    }

    getPresetModeratorUserKey():string {
        return this.retrospective.preset_moderator_user_key;
    }

    clear():void {
        this.retrospective = {
            columns:[],
            votes_per_player: 0,// (0 for unlimited)
            votes_per_card: 1,// (0 for unlimited)
            anonymous_players: false,
            preset_moderator_user_key: 'RETROS_DYNAMIC_MODERATOR',
            isTemplate: false,
            formatId: null,
            techniqueSelected:null
        }
    }

    setFormatId(formatId){
        this.retrospective.formatId = formatId
    }

    getFormatId(){
        if (this.retrospective !== undefined){
            return this.retrospective.formatId
        }else{
            return 1
        }
    }

    getFormats() {
        return this.formats
    }

    setFormats(formats):void {
        this.formats = formats;
    }

    setFormatsReset(formatId, columns):void{
        this.formats.find(format => format.id === Number(formatId)).columns = columns
    }
}

const retrospectiveModel = new RetrospectiveModel();
export default retrospectiveModel;
