/*
 * Copyright (c) 2020 Appfire Technologies, Inc.
 * All rights reserved.
 *
 * This software is licensed under the provisions of the "Bob Swift Atlassian Apps EULA"
 * (https://bobswift.atlassian.net/wiki/x/WoDXBQ) as well as under the provisions of
 * the "Standard EULA" from the "Atlassian Marketplace Terms of Use" as a "Marketplace Product"
 * (http://www.atlassian.com/licensing/marketplace/termsofuse).
 *
 * See the LICENSE file for more details.
 */
/**
 * Provides custom behaviour in the macro browser for the sql macro.
 */
(function($) {

    var MacroConfig = function() {};

    MacroConfig.prototype.fields = {
        "string" : {
            "profile" : function(param, options) { // parameter with name: profile

                var paramDiv = $(Confluence.Templates.MacroBrowser.macroParameterSelect());
                var input = $("select", paramDiv);

                // we need to do some trickery when the value is first set, and when the value is subsequently changed, so set this up.
                if (options == null) {
                    options = {};
                }
                options.setValue = function(value) {
                    var targetOption = input.find("option[value='" + value + "']");
                    if (targetOption.length == 0) {
                        var option = $("<option/>");
                        option.val(value);
                        option.text(value);
                        input.append(option);
                    }
                };

                profilesBindAsyncDropDown(input);
                return new AJS.MacroBrowser.Field(paramDiv, input, options);
            }
        }
    };

    AJS.MacroBrowser.Macros["html"] = new MacroConfig(); 
    AJS.MacroBrowser.Macros["html-bobswift"] = new MacroConfig();
    AJS.MacroBrowser.Macros["xslt"] = new MacroConfig(); 

    /**
     * Populates an HTML Select element with the list of configured profiles.
     * @param dropDown JQuery selector for the HTML select element to be populated.
     */
    function profilesBindAsyncDropDown(dropDown) {
        // Load the profiles from Active object
        getProfilesAsync(function(profiles) {
            if (!profiles.length) {
                AJS.log("Configured html profiles were not in the expected format.");
                return;
            }
            var currentValue = dropDown.val();
            dropDown.empty();
            dropDown.append($("<option/>").attr("value", ""));

            $.each(profiles, function(index, profile) {
                var option = $("<option />");
                option.val(profile);
                option.text(profile);
                dropDown.append(option);
            });
            // restore the currently selected value.
            dropDown.val(currentValue);
        }, function(xhr, textStatus, errorThrown) {
            AJS.log("Failed to retrieve profiles from Confluence " + textStatus + " - "
                    + errorThrown);
        });
    }

    /**
     * Asynchronously retrieves the current set of profiles.
     *
     * @param successHandler Callback to invoke if the retrieval is successful
     * @param errorHandler Callback to invoke if the retrieval fails.
     */
    function getProfilesAsync(successHandler, errorHandler) {
        AJS.$.ajax({
            async : true,
            url : AJS.params.contextPath+ "/rest/bobswift/html/1.0/macroprofiles/",
            dataType : "json",
            timeout : 10000, // 10 seconds,
            error : function(xhr, textStatus, errorThrown) {
                if (errorHandler && typeof (errorHandler) == "function") {
                    errorHandler(xhr, textStatus, errorThrown);
                }
            },
            success : function(data) {
                if (successHandler && typeof (successHandler) == "function") {
                    successHandler(data);
                }
            }
        });
    }
})(AJS.$);