AJS.$(function(){
  $("body").append("<div id='git-plugin-global-message'></div>");

  var ValidationMixin = {
    getContext: function(context) {
      if(_.isUndefined(this.defaultContext)) {
        this.defaultContext = '#git-plugin-global-message';
      }

      if(_.isUndefined(context)) {
        return this.defaultContext;
      }
      return context;
    },

    renderValidationMessages: function(errors)  {
      this.$('.error').remove();
      _.each(errors.errors, function(error) {
        if (error.field === 'serverInfo') {
          error.field = 'serverInfo.host';
        }

        this.$('#'+error.field.replace(/\./g,"-")).focus().after('<div class="error">' + error.message + '</div>');
      }.bind(this));
    },

    renderSuccessMessage: function(message, context) {
      context = this.getContext(context);
      AJS.$(context).empty();
      AJS.messages.success(context, {
        body: message,
        fadeout: true
      });
    },

    renderErrorMessage: function(message, context) {
      context = this.getContext(context);
      AJS.$(context).empty();
      AJS.messages.error(context, {
        title : "An error occurred",
        body: message,
        fadeout: false
      });
    }
  };

  var Server = Backbone.Model.extend({

    defaults: {
      "serverAuthentication": {},
      "serverInfo": {}
    },

    getServerAuthentication: function()  {
      return $.extend(true, {}, this.get("serverAuthentication"));
    },

    getServerInfo: function()  {
      return $.extend(true, {}, this.get("serverInfo"));
    }
  });

  var Servers = Backbone.Collection.extend({
    model: Server,
    url: AJS.contextPath() + '/rest/git-plugin/1.0/servers',

    comparator: function(server) {
      return server.get("name");
    }
  });

  var Attachment = Backbone.Model.extend({});

  var Attachments = Backbone.Collection.extend({

    model: Attachment,

    initialize: function(options) {
      this.repositoryId = options.repositoryId;
      this.url = AJS.contextPath() + '/rest/git-plugin/1.0/repositories/' + this.repositoryId + '/attachments';
    }

  });

  var Repository = Backbone.Model.extend({

    defaults: {
      "name" : "",
      "path" : ""
    },

    urlRoot: function() {
      if(this.isNew()) {
        return AJS.contextPath() + '/rest/git-plugin/1.0/repositories/' + this.collection.serverId;
      } else {
        return AJS.contextPath() + '/rest/git-plugin/1.0/repositories';
      }
    },

    archive : function() {
      return AJS.$.ajax(AJS.contextPath() + '/rest/git-plugin/1.0/repositories/' + this.get("id") + '/actions/archive', {
        contentType: 'application/json',
        type: 'POST',
        data: JSON.stringify(""),
        processData: false,
        dataType: 'json',
      })
    },

    unarchive : function() {
      console.log("unarchiving..." + this.get("id"))
      return AJS.$.ajax(AJS.contextPath() + '/rest/git-plugin/1.0/repositories/' + this.get("id") + '/actions/unarchive', {
        contentType: 'application/json',
        type: 'POST',
        data: JSON.stringify(""),
        processData: false,
        dataType: 'json',
      })
    },

  });

  var SecurityGroups = function(repository) {
    var that = this;
    this.repository = repository;

    var allGroups = [];
    var currentGroups = [];

    var all = AJS.$.ajax(AJS.contextPath() + '/rest/git-plugin/1.0/security/groups', {dataType: 'json'});
    var yours = AJS.$.ajax(AJS.contextPath() + '/rest/git-plugin/1.0/security/groups/' + this.repository.get("id"), {dataType: 'json'});

    AJS.$.when(all, yours).then(
      function(allWrapper, yourWrapper) {
        allGroups = allWrapper[0];
        currentGroups = yourWrapper[0];
        that.trigger("reset")
      },
      function(res) {
        handleUnauthorized(res, function(){
          that.trigger("error");
        });
      }
    );

    this.all = function() {
      return allGroups;
    };

    this.save = function(groups) {
      return AJS.$.ajax(AJS.$.ajax(AJS.contextPath() + '/rest/git-plugin/1.0/security/groups/' + this.repository.get("id"), {
        contentType: 'application/json',
        type: 'PUT',
        data: JSON.stringify(groups),
        processData: false,
        dataType: 'json',
        error: handleUnauthorized
      }));
    }.bind(this);

    this.current = function() {
      return currentGroups;
    };

    return this;
  };
  _.extend(SecurityGroups.prototype, Backbone.Events);


  var Repositories = Backbone.Collection.extend({

    model: Repository,

    initialize: function(options) {
      this.serverId = options.serverId;
      this.url = AJS.contextPath() + '/rest/git-plugin/1.0/servers/' + options.serverId + '/repositories';
    }

  });

  var RepositoryStatus = Backbone.Model.extend({

    initialize: function() {
      _.bindAll(this, "startPolling");
      _.bindAll(this, "poll");
      _.bindAll(this, "stopPolling");
    },

    url: function() {
      return AJS.contextPath() + '/rest/git-plugin/1.0/progress/' + this.get("name");
    },

    startPolling: function() {
      this.keepPolling = true;
      this.poll();
    },

    poll: function() {
      if(this.keepPolling === false) {
        return;
      }

      var response = this.fetch();
      response.always(function() {
        setTimeout(this.poll, 1000);
      }.bind(this));
    },

    stopPolling: function() {
      this.keepPolling = false;
    }

  });

  var ServerRepositoryJobsMixin = {
    _prevPollingResponse: {},
    _pollingPid: null,

    startPollingJobs: function(serverId, callback) {
      var self = this;
      this._pollingPid = setInterval(function() {
        AJS.$.ajax({
          dataType: 'json',
          type: 'GET',
          url: AJS.contextPath() + '/rest/git-plugin/1.0/progress/' + serverId,
          contentType: 'application/json',
          processData: false,
          parse: true,
          success: function(resp) {
            if(!_.isEqual(self._prevPollingResponse, resp)) {
              callback(resp);
            }
            self._prevPollingResponse = resp;
          },
          error: handleUnauthorized
        });
      }, 1000);
    },

    stopPollingJobs: function() {
      if(this._pollingPid !== null) {
        clearInterval(this._pollingPid);
      }
    }
  };

  var SecurityGroupsView = Backbone.View.extend({
    tagName: 'div',

    events: {
      "submit form" : "save"
    },

    initialize: function(options) {
      this.serverId = options.serverId;
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.progressIndicator());
      this.securityGroups = new SecurityGroups(options.model);
      this.model = options.model;
      this.securityGroups.on("reset", this.render.bind(this));
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.securityGroupsConfiguration({
        repository: this.model.toJSON(),
        groups: this.securityGroups.all(),
        serverId: this.serverId
      }));
      _.forEach(this.securityGroups.current(), function(group) {
        this.$("input[name='" + group + "']").attr("checked", 1);
      }.bind(this));
    },

    destroy: function(){
      this.securityGroups.off("reset", this.render);
    },

    save: function(e) {
      e.preventDefault();
      var keys = _.reduce(Backbone.Syphon.serialize(this), function(memo, value, key) {
        if(value === true) {
          memo.push(key);
        }
        return memo;
      }, []);
      this.$('.save-progress').show();
      var response = this.securityGroups.save(keys);

      response.done(function() {
        this.renderSuccessMessage(AJS.I18n.getText("gitplugin.configuration.security.saved.message") + " " + this.model.get("name"));
        this.$('.save-progress').hide();
        Window.GitPlugin.adminRouter.navigate("server/" + this.serverId, {trigger: true});
      }.bind(this));

      response.fail(function() {
        handleUnauthorized(response, function(){
          this.renderErrorMessage(AJS.I18n.getText("gitplugin.unknownerror.message"));
        }.bind(this));
      }.bind(this));
    }

  });
  _.extend(SecurityGroupsView.prototype, ValidationMixin);

  var RepositoryView = Backbone.View.extend({

    tagName: 'tr',
    className: 'git-plugin-repository-row',
    defaultContext: '#git-plugin-repository-messages-container',

    events: {
      "click .edit-repository": "renderEditMode",

      "click .save-repository": "save",
      "click .cancel-save-repository": "cancel",
      "submit form": "save"
    },

    initialize: function(options) {
      this.server = options.server;
      this.model.on('progressUpdate', this.handleProgressUpdate.bind(this));
      this.model.on('progressDone', this.render.bind(this));
      this.model.on("change", this.render.bind(this));
      this.model.on("remove", this.remove.bind(this));
    },

    handleProgressUpdate: function(e) {
      this.renderEditMode();
      this.$(".inline-form-element").prop( "disabled", true );
      this.$(".save-progress").show();
      this.$(".aui-iconfont-success").hide();
      this.$('.progressText').html(e.msg + " " + e.percent + "%");
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.repositoryTableRow({repository: this.getRepositoryWithFormattedDateTimes(), server: this.server.toJSON()}));

      this.deleteDialog = AJS.InlineDialog(
          this.$(".permanently-delete-repository"),
          "delete-repository-dialog" + this.model.get("id"),
          function(content, trigger, showPopup) {
            content.html(Confluence.Templates.ConfigureGitPlugin.deleteRepository({repository: this.model.toJSON()}));
            showPopup();
            content.find('button').on("click", function() {
              this.deleteRepository(content, this.deleteDialog);
            }.bind(this));
            return false;
          }.bind(this)
      );

      this.archiveDialog = AJS.InlineDialog(
          this.$(".archive-repository"),
          "archive-repository-dialog" + this.model.get("id"),
          function(content, trigger, showPopup) {
            content.html(Confluence.Templates.ConfigureGitPlugin.archiveRepository({repository: this.model.toJSON()}));
            showPopup();
            content.find('button').on("click", function() {
              this.archiveRepository(content, this.archiveDialog);
            }.bind(this));
            return false;
          }.bind(this)
      );

      this.unarchiveDialog = AJS.InlineDialog(
          this.$(".unarchive-repository"),
          "unarchive-repository-dialog" + this.model.get("id"),
          function(content, trigger, showPopup) {
            content.html(Confluence.Templates.ConfigureGitPlugin.unarchiveRepository({repository: this.model.toJSON()}));
            showPopup();
            content.find('button').on("click", function() {
              this.unarchiveRepository(content, this.unarchiveDialog)
            }.bind(this));
            return false;
          }.bind(this)
      );
    },

    renderEditMode: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.editableRepositoryTableRow({repository: this.getRepositoryWithFormattedDateTimes(), server: this.server.toJSON()}));
    },

    getRepositoryWithFormattedDateTimes: function() {
      var repository = this.model.toJSON();
      if (repository.lastSynchronized) {
        repository.friendlyLastSynchronized = new Intl.DateTimeFormat('default', {dateStyle: 'short', timeStyle: 'long'}).format(new Date(repository.lastSynchronized));
      } else {
        repository.friendlyLastSynchronized = "";
      }

      return repository;
    },

    cancel: function() {
      if(this.model.isNew()) {
        this.remove();
      } else {
        this.render();
      }
    },

    deleteRepository: function(content, dialog) {
      content.find('.delete-progress').show();
      var response = this.model.destroy({wait: true});

      response.done(function() {
        this.renderSuccessMessage("Repository " + this.model.get("name") + " permanently deleted");
      }.bind(this));

      response.fail(function(xhr) {
        if(xhr.status === 422) {
          var json = JSON.parse(xhr.responseText);

          var messageHtml = Confluence.Templates.ConfigureGitPlugin.cannotRemoveRepositoryMessage({
            repositoryId: this.model.get('id'),
            serverId: this.server.get('id'),
            message: json.message
          });

          this.renderErrorMessage(
              messageHtml,
              '#git-plugin-global-message'
          );
        } else {
          handleUnauthorized(xhr, function(){
            this.renderErrorMessage(AJS.I18n.getText("gitplugin.unknownerror.message"), '#git-plugin-global-message');
          }.bind(this));
        }
      }.bind(this));

      response.always(function() {
        content.find('.delete-progress').hide();
        dialog.hide();
      }.bind(this));
    },

    archiveRepository: function(content, dialog) {
      content.find('.archive-progress').show();
      var response = this.model.archive();

      response.done(function() {
        this.model.set({"archived": true});
        this.renderSuccessMessage("Repository " + this.model.get("name") + " archived");
      }.bind(this));

      response.fail(function(xhr) {
        handleUnauthorized(xhr, function() {
          this.renderErrorMessage("Error while archiving repository " + this.model.get("name"));
        }.bind(this))
      }.bind(this));

      response.always(function() {
        content.find('.archive-progress').hide();
        dialog.hide();
      }.bind(this));
    },

    unarchiveRepository: function(content, dialog) {
      content.find('.unarchive-progress').show();
      var response = this.model.unarchive();

      response.done(function() {
        this.model.set({"archived": false});
        this.renderSuccessMessage("Repository " + this.model.get("name") + " restored.");
      }.bind(this));

      response.fail(function(xhr) {
        handleUnauthorized(xhr, function() {
          this.renderErrorMessage("Error while restoring repository " + this.model.get("name") + ".");
        }.bind(this))
      }.bind(this));

      response.always(function() {
        content.find('.unarchive-progress').hide();
        dialog.hide();
      }.bind(this));
    },

    save: function() {
      this.$(".inline-form-element").prop( "disabled", true );
      this.$(".save-progress").show();
      this.$(".aui-iconfont-success").hide();

      var response = this.model.save({
        path: this.$(".edit-path").val(),
        name: this.$(".edit-name").val()
      }, {wait: false});
      this.model.collection.add(this.model, {silent: true});

      response.done(function() {
        this.render();
      }.bind(this));

      response.fail(function(jqXHR, textStatus) {
        try{
          var errors = JSON.parse(jqXHR.responseText);
          this.renderEditMode();
          this.renderValidationMessages(errors);

          if(!_.isUndefined(errors.message)) {
            this.renderErrorMessage(errors.message);
          }
        } catch(e) {
          handleUnauthorized(response, function(){
            this.renderErrorMessage(AJS.I18n.getText("gitplugin.unknownerror.message"));
          }.bind(this));
        }

      }.bind(this));

      return false;
    }

  });

  _.extend(RepositoryView.prototype, ValidationMixin);

  var RepositoriesView = Backbone.View.extend({

    previousProgressKeys: [],

    events: {
      "click .new-repository" : "newRepository"
    },

    initialize: function(options) {
      _.bindAll(this, "renderItem");

      this.dialog = new AJS.Dialog({
        width:800,
        height:100,
        id:"edit-repository-dialog",
        closeOnOutsideClick: false
      });

      this.$el.html(Confluence.Templates.ConfigureGitPlugin.progressIndicator());

      this.collection = new Repositories({serverId: this.model.id});
      this.collection.fetch({
        success: this.render.bind(this),
        error: this.renderNotFound.bind(this)
      });
      this.collection.on('add', this.renderItem);
      this.model.on("change", this.render.bind(this)); // or else the model is sometimes undefined
      this.startPollingJobs(this.model.id, this.handleProgressUpdate.bind(this));
    },

    handleProgressUpdate: function (progressMap) {
      var repositories = this.collection;

      current = _(progressMap).values(progressMap).map(function(repo) {
        return {name: repo.repository.name, path: repo.repository.path};
      });

      var difference = _.difference(this.previousProgressKeys, current);

      difference.forEach(function(repository) {
        if(_.contains(this.previousProgressKeys, repository)) {
          var old = repositories.where({name: repository.name, path: repository.path});
          old[0].trigger('progressDone');
        }
      }, this);

      this.previousProgressKeys = current;


      _(progressMap).forEach(function(progress) {
        var existing = repositories.where({name: progress.repository.name, path: progress.repository.path});
        var repository = existing[0];

        if(!repository) {
          repository = new Repository(progress.repository);
          repositories.add(repository);
        }

        if(repository) {
          var percent = Math.round((100 / progress.totalWork) * progress.completedWork);
          repository.trigger('progressUpdate', {
            percent: percent,
            msg: progress.task
          });
        }
      });
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.repositories({server: this.model.toJSON()}));
      this.$itemContainer = this.$(".git-repositories");
      this.collection.each(this.renderItem);

      AJS.tablessortable.setTableSortable(this.$(".aui-table-sortable"));
    },

    renderItem: function(repository) {
      var itemView = new RepositoryView({model: repository, server: this.model});
      itemView.render();
      this.$itemContainer.append(itemView.el);
    },

    renderNotFound: function(model, xhr) {
      handleUnauthorized(xhr, function(){
        this.renderErrorMessage("Could not find specified server..");
      }.bind(this));
    },

    newRepository: function() {
      var newRepository = new Repository();
      newRepository.collection = this.collection;

      var itemView = new RepositoryView({model: newRepository, server: this.model});
      itemView.renderEditMode();
      this.$itemContainer.append(itemView.el);
      this.$('.git-plugin-repository-row:last-child .edit-name').focus();
    },

    remove: function() {
      this.stopPollingJobs();
    }

  });
  _.extend(RepositoriesView.prototype, ValidationMixin, ServerRepositoryJobsMixin);

  var ServerTableRowView = Backbone.View.extend({
    tagName: 'tr',
    className: 'git-plugin-server-row',

    initialize: function() {
      this.model.on("change", this.render.bind(this));
      this.model.on("remove", this.remove.bind(this));
    },

    events: {
      "click .edit-server": "openEditDialog"
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.serverRow({server: this.model.toJSON()}));

      this.dialog = AJS.InlineDialog(
        this.$(".remove-server"),
        "remove-server-dialog" + this.model.get("id"),
        function(content, trigger, showPopup) {
          content.html(Confluence.Templates.ConfigureGitPlugin.deleteServer({server: this.model.toJSON()}));
          showPopup();
          content.find('button').on("click", function() {
            content.find('.delete-progress').show();
            this.model.destroy({
              wait: true,
              success: function(){
                content.find('.delete-progress').hide();
                this.dialog.hide();
                this.renderSuccessMessage("Server removed");
              }.bind(this),
              error: function(model, xhr){
                handleUnauthorized(xhr, function(){
                  content.find('.delete-progress').hide();
                  this.dialog.hide();
                  this.renderErrorMessage("Could not remove server");
                }.bind(this));
              }.bind(this)
            });
          }.bind(this));
          return false;
        }.bind(this)
      );
    },

    openEditDialog: function(event) {
      var editFormView = new ServerEditFormView({model: this.model});
      editFormView.render();
      return false;
    }
  });
  _.extend(ServerTableRowView.prototype, ValidationMixin);

  var ServerEditFormView = Backbone.View.extend({

    authenticationMethods: {
      "NO_AUTHENTICATION" : [],
      "USERNAME_PASSWORD" : ["#serverAuthentication-username", "#serverAuthentication-password"],
      "SELECT_PRIVATE_KEY": ["#serverAuthentication-privateKey", "#serverAuthentication-username", "#serverAuthentication-password"]
    },

    authenticationFields: ["#serverAuthentication-privateKey", "#serverAuthentication-username", "#serverAuthentication-password"],

    tagName: 'form',
    className: 'aui',

    events: {
      "keyup" : "checkForEnter",
      "change #serverAuthentication-authenticationMethod" : "revealAuthenticationFields",
      "change #serverInfo-protocol" : "changeAuthenticationMethods",
      "keypress #serverInfo-port" : "validatePortKeypress",
      "keydown #serverInfo-port" : "validatePortKeydown"
    },

    checkForEnter: function(event) {
      if(event.which === 13 && event.target.className !== 'select' ) {
        this.save();
      }
    },

    initialize: function() {
      this.dialog = new AJS.Dialog({
        width:500,
        height:500,
        id:"edit-server-dialog",
        closeOnOutsideClick: false
      });
    },


    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.editServerForm({server: this.model.toJSON()}));

      this.dialog.addHeader(AJS.I18n.getText("gitplugin.server.title"));
      this.dialog.addPanel("Edit form", this.el);

      this.dialog.addSubmit(
        AJS.I18n.getText("save.name"),
        function() {
          this.save();
          return false;
        }.bind(this)
      );

      this.dialog.addCancel(AJS.I18n.getText("close.name"), function() {
        this.dialog.hide().remove();
        return false;
      }.bind(this));
      this.dialog.show();

      // kind of ugly
      this.$('#serverInfo-protocol').val(this.model.getdsServerInfo().protocol).trigger("change");
      this.$('#serverAuthentication-authenticationMethod').val(this.model.getServerAuthentication().authenticationMethod).trigger("change");
    },

    save: function() {
      this.dialog.disable();

      var options = {
        success: this.success.bind(this),
        error: this.error.bind(this),
        wait: true
      };

      if(this.model.isNew()) {
        this.collection.create(Backbone.Syphon.serialize(this), options)
      } else {
        this.model.save(Backbone.Syphon.serialize(this), options);
      }

      return false;
    },

    success: function(model, response) {
      this.dialog.hide().remove();
      this.renderSuccessMessage("Server is saved");
      this.dialog.enable();
    },

    error: function(model, xhr) {
      if (xhr.responseText) {
        try {
          var errors = JSON.parse(xhr.responseText);
          // If there is not an errors array it's an unexpected error
          if(!errors.errors) {
            this.renderValidationMessages({errors: []}) // remove all existing errors
            throw errors;
          }
          this.renderValidationMessages(errors)
          this.dialog.updateHeight();
        } catch (e) {
          handleUnauthorized(xhr, function(){
            console.log('Git addon threw an error', e);
            this.renderErrorMessage(AJS.I18n.getText("gitplugin.unknownerror.message"), this.$('.edit-server-errors'));
          }.bind(this));
          this.dialog.updateHeight();
        }
      }
      this.dialog.enable();
    },

    revealAuthenticationFields: function(event) {
      var authenticationMethod = AJS.$(event.currentTarget).val();
      var fields = this.authenticationMethods[authenticationMethod];
      _.each(this.authenticationFields, function(field) {
        var index = _.indexOf(fields, field);
        if(index === -1) {
          AJS.$(field).val("").parent().hide();
        } else {
          AJS.$(field).parent().show();
        }
      });
      this.dialog.updateHeight();
    },

    changeAuthenticationMethods: function(event) {
      var protocol = AJS.$(event.currentTarget).val();
      this.$("#serverAuthentication-authenticationMethod option").removeAttr("disabled");

      if (protocol !== "SSH") {
        this.$("#serverAuthentication-authenticationMethod option[value='SELECT_PRIVATE_KEY']").attr("disabled", "disabled");
        this.$('label[for="serverAuthentication-password"]').text(AJS.I18n.getText("gitplugin.server.password"));
      } else {
        this.$('label[for="serverAuthentication-password"]').text(AJS.I18n.getText("gitplugin.server.passphrase"));
      }
    },

    validatePortKeypress: function(event) {
      AJS.$("#serverInfo-port").next("div.error").remove();
      if (AJS.$(event.currentTarget).val().length > 5) {
        this.$("#serverInfo-port").after('<div class="error">' + AJS.I18n.getText("gitplugin.validation.constraints.PortThreshold.message") + '</div>');
        event.preventDefault();
      } else if (event.keyCode < 48 || event.keyCode > 57) {
        this.$("#serverInfo-port").after('<div class="error">' + AJS.I18n.getText("gitplugin.validation.constraints.DigitsOnly.message") + '</div>');
        event.preventDefault();
      }
    },

    validatePortKeydown: function(event) {
      if (event.keyCode === 8 || event.keyCode === 46) {
        AJS.$("#serverInfo-port").next("div.error").remove();
      }
    }

  });

  _.extend(ServerEditFormView.prototype, ValidationMixin);



  var GitPluginConfigurationView = Backbone.View.extend({

    events: {
      "click .new-server" : "openNewServerDialog"
    },

    initialize: function() {
      _.bindAll(this, "renderItem");

      this.$el.html(Confluence.Templates.ConfigureGitPlugin.progressIndicator());
      this.collection = new Servers();

      // This works for Confluence versions 5 (?) and up
      this.collection.on('reset', this.render.bind(this));

      // added this for older Confluence versions.
      this.collection.on('sync', this.render.bind(this));

      this.listenTo(this.collection, 'reset', this.renderRepositoryMissingMessage);
      this.listenTo(this.collection, 'sync', this.renderRepositoryMissingMessage);

      this.collection.fetch();
    },

    renderItem: function(model) {
      var itemView = new ServerTableRowView({model: model});
      itemView.render();
      this.$itemContainer.append(itemView.el);
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.index());
      this.$itemContainer = this.$(".git-servers");
      this.collection.each(this.renderItem);

      AJS.tablessortable.setTableSortable(this.$(".aui-table-sortable"));
    },

    renderRepositoryMissingMessage: function() {
      new RepositoryMissingMessage({el: this.$(".git-missing-message-container")[0]});
    },

    openNewServerDialog: function(e){
      e.preventDefault();
      var editFormView = new ServerEditFormView({model: new Server(), collection: this.collection});
      editFormView.render();
    }

  });

  var RepositoryMissingMessage = Backbone.View.extend({

      events: {
        "click .fix-git" : "fixGit"
      },

      initialize: function() {
        this.render();
      },

      render: function() {
          var self = this;
          AJS.$.ajax({
             url: AJS.contextPath() + '/rest/git-plugin/1.0/health-check/check',
                 success: function (data) {
                    var repositoryMissingMessage = Confluence.Templates.ConfigureGitPlugin.repositoryMissingMessage({restRepositoryHealth: data});
                    self.$el.append(repositoryMissingMessage);
                 },

                 error: function(data) {
                    self.$(".repositories-missing").remove();
                    var errorMissingRepositoryMessage = Confluence.Templates.ConfigureGitPlugin.errorMissingRepositoryMessage({restRepositoryHealth: JSON.parse(data.responseText), message: AJS.I18n.getText("gitplugin.repository.missing.error.check.call")});
                    self.$el.append(errorMissingRepositoryMessage);
                 }
          });
      },

      fixGit: function(e) {
          e.preventDefault();
          this.$(".duration-warning").append(Confluence.Templates.ConfigureGitPlugin.progressIndicator());
          var self = this;
          AJS.$.ajax({
             url: AJS.contextPath() + '/rest/git-plugin/1.0/health-check/fix',
                 success: function () {
                    self.$(".repositories-missing").remove();
                    var repositoriesFixedMessage = Confluence.Templates.ConfigureGitPlugin.repositoriesFixedMessage();
                    self.$el.append(repositoriesFixedMessage);
                 },

                 error: function(data) {
                    self.$(".repositories-missing").remove();
                    var errorMissingRepositoryMessage = Confluence.Templates.ConfigureGitPlugin.errorMissingRepositoryMessage({restRepositoryHealth: JSON.parse(data.responseText), message: AJS.I18n.getText("gitplugin.repository.missing.error.fix.call")});
                    self.$el.append(errorMissingRepositoryMessage);
                 }
          });
      }
  });

  var AttachmentRow = Backbone.View.extend({
    tagName: 'tr',
    className: 'git-attachment-row',

    initialize: function() {
      this.render();
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.attachmentRow({attachment: this.model.toJSON(), baseUrl: AJS.contextPath()}));
    }
  });


  var AttachmentsView = Backbone.View.extend({

    initialize: function(options) {
      this.serverId = options.serverId;
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.progressIndicator());

      this.listenTo(this.model, 'sync', this.render);
      this.listenTo(this.model, 'sync', this.fetchAttachments);

      this.listenTo(this.collection, 'reset', this.renderTrashMessage);
      this.listenTo(this.collection, 'sync', this.renderTrashMessage);

      this.listenTo(this.collection, 'reset', this.renderAllAttachments);
      this.listenTo(this.collection, 'sync', this.renderAllAttachments);
    },

    fetchAttachments: function() {
      this.collection.fetch();
    },

    renderAttachment: function(attachment) {
      this.$('tbody').append(new AttachmentRow({model: attachment}).$el);
    },

    renderAllAttachments: function() {
      this.$('tbody').empty();
      this.collection.each(this.renderAttachment, this);

      AJS.tablessortable.setTableSortable(this.$(".aui-table-sortable"));
    },

    renderTrashMessage: function() {
        var attachmentsInTrash = this.collection.models.filter(function(element) {
            return element.attributes.deleted;
        });

        var trashMessage = Confluence.Templates.ConfigureGitPlugin.trashMessage({nAttachmentsInTrash: attachmentsInTrash.length, baseUrl: AJS.contextPath()});
        this.$('.trash-message-container').append(trashMessage);
    },

    render: function() {
      this.$el.html(Confluence.Templates.ConfigureGitPlugin.repositoryAttachmentsPage(
        {
          repository: this.model.toJSON(),
          serverId: this.serverId
        }
      ));
    }
  });

  var handleUnauthorized = function(err, callback) {
    if(err.status === 401) {
      window.location.reload()
    } else {
      if(callback) callback();
    }
  };

  var AdminRouter = Backbone.Router.extend({

    $container: AJS.$("#git-plugin-configuration"),
    $current: null,

    routes: {
      '': 'index',
      'server/:serverId': 'openRepositoryView',
      'server/:serverId/repository/:repositoryId/security': 'openSecurityView',
      'server/:serverId/repository/:repositoryId/attachments': 'openAttachmentView'
    },

    index: function() {
      this.renderMainView(new GitPluginConfigurationView());
    },

    openRepositoryView: function(serverId){
      var server = new Server({id: serverId});
      // The 'Server' model depends on a 'Servers' collection to generate the URL, so this (nasty hack) is necessary
      // to add the 'Server' model to a collection.
      new Servers([server]);
      server.fetch();
      var view = new RepositoriesView({model: server});
      this.renderMainView(view);
    },

    openSecurityView: function(serverId, repositoryId){
      var repository = new Repository({id: repositoryId});
      new Repositories([repository]);
      repository.fetch();
      var view = new SecurityGroupsView({model: repository, serverId: serverId});
      this.renderMainView(view);
    },

    openAttachmentView: function(serverId, repositoryId){
      var repository = new Repository({id: repositoryId});
      new Repositories([repository]);
      repository.fetch();
      var view = new AttachmentsView({
        model: repository,
        serverId: serverId,
        collection: new Attachments({repositoryId: repositoryId})
      });

      this.renderMainView(view);
    },

    renderMainView: function(view) {
      if(this.$current) {
        this.$current.remove();
      }
      this.$current = view;
      this.$container.html(this.$current.el);
    }

  });

  Window.GitPlugin = {};
  Window.GitPlugin.adminRouter = new AdminRouter();

  if(!Backbone.history.start()) {
    adminRouter.navigate('');
  }

});
