/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('cmis/cmisproperty-helper', ['jquery', 'misc/Size', 'cmis/cmisconfluence'], function ($, miscSize, cmisConfluence) {
  'use strict';

  /**
   * デフォルト日時フォーマット
   */
  var DEFAULT_DATETIME_FORMAT = "YYYY/MM/DD HH:mm";

  /**
   * デフォルト日時フォーマットUTC
   */
  var DEFAULT_DATETIME_FORMAT_UTC = "YYYY-MM-DDTHH:mm:ss.SSSZZ";

  /**
   * 画面に表示するプロパティ情報を取得する
   * @param properties
   * @param propertyDefinitions
   * @returns {Array}
   */
  function getRenderItems(properties, propertyDefinitions) {
    var renderItems = [];
    // Pull&Push on Issue
    if (properties && propertyDefinitions) {
      $.each(properties, function (propertyKey, propertyValue) {
        // プロパティ定義がない場合は表示対象としない
        var propertyDefinition = getPropertyDefinition(propertyKey, propertyDefinitions);
        if (!propertyDefinition) {
          return true;
        }
        // プレフィックス
        var prefix = propertyKey.split(':')[0];
        // 標準
        if ($.inArray(prefix, getAllPrefixOfType()) != -1) {
          // 標準プロパティのレンダー定義を取得して、visible属性がfalseの場合は表示対象としない
          var propertyRenderDefinition = getPropertyRenderDefinition(propertyKey);
          if (!propertyRenderDefinition || !propertyRenderDefinition.visible) {
            return true;
          }
          // カスタム
        } else {
          // カスタムプロパティのレンダー定義を取得して、標準プロパティのレンダー定義がない、visible属性がfalseの場合は表示対象としない
          var customPropertyDefinition = getCustomPropertyRenderDefinition(propertyDefinition.propertyType)
          if (!customPropertyDefinition || !customPropertyDefinition.visible) {
            return true;
          }
        }
        renderItems.push({
          "propertyKey": propertyKey,
          "propertyValue": propertyValue,
          "propertyDefinition": propertyDefinition,
          "propertyRenderDefinition": propertyRenderDefinition ? propertyRenderDefinition : customPropertyDefinition
        });
      });
    }
    // Pull&Push on Mapping
    if (!properties && propertyDefinitions) {
      $.each(propertyDefinitions, function (propertyKey, propertyDefinition) {
        // プレフィックス
        var prefix = propertyKey.split(':')[0];
        // 標準プロパティ
        if ($.inArray(prefix, getAllPrefixOfType()) != -1) {
          // 標準プロパティのレンダー定義を取得して、visible属性がfalseの場合は表示対象としない
          var propertyRenderDefinition = getPropertyRenderDefinition(propertyKey);
          if (!propertyRenderDefinition || !propertyRenderDefinition.visible) {
            return true;
          }
          // カスタムプロパティ
        } else {
          // カスタムプロパティのレンダー定義を取得して、標準プロパティのレンダー定義がない、visible属性がfalseの場合は表示対象としない
          var customPropertyDefinition = getCustomPropertyRenderDefinition(propertyDefinition.dataType);
          if (!customPropertyDefinition || !customPropertyDefinition.visible) {
            return true;
          }
        }
        renderItems.push({
          "propertyKey": propertyKey,
          "propertyDefinition": propertyDefinition,
          "propertyRenderDefinition": propertyRenderDefinition ? propertyRenderDefinition : customPropertyDefinition
        });
      });
    }

    renderItems.sort(function(a, b) {
      if (!a.propertyRenderDefinition || !b.propertyRenderDefinition) {
        return 0;
      }
      // カスタムプロパティのソート(PropertyKey.toLowerCase 昇順)
      if (!a.propertyRenderDefinition.id && !b.propertyRenderDefinition.id) {
        return compare(getPropertyKey(a), getPropertyKey(b));
      }
      // 標準プロパティのソート(propertyRenderDefinition.order 昇順)
      return compare(getOrder(a.propertyRenderDefinition), getOrder(b.propertyRenderDefinition));

      function compare(a, b) {
        if (!a || a > b) { return 1; }
        if (!b || a < b) { return -1; }
        return 0;
      }
      function getOrder(propertyRenderDefinition) {
        return propertyRenderDefinition ? (propertyRenderDefinition.order ? propertyRenderDefinition.order : undefined) : undefined;
      }
      function getPropertyKey(renderItem) {
        return renderItem.propertyKey ? renderItem.propertyKey.toString().toLowerCase() : undefined;
      }
    });
    return renderItems;
  }

  /**
   * フィールド値を画面に表示するためにフォーマットする
   * @param fieldKey
   * @param fieldValue
   * @param fieldDefinitions
   * @returns {any|*|String}
   */
  function formatFieldValue(fieldKey, fieldValue, fieldDefinitions) {

    var fieldDefinition = fieldDefinitions[fieldKey];

    if (fieldValue && fieldDefinition && (fieldDefinition.schema.type == "user" || fieldDefinition.schema.type == "group")) {
      return fieldValue.name;
    }

    if (fieldValue && fieldDefinition && (fieldDefinition.schema.type == "date" || fieldDefinition.schema.type == "datetime")) {
      return moment(fieldValue).format(DEFAULT_DATETIME_FORMAT);
    }

    return fieldValue;

  }

  /**
   * プロパティ値を画面に表示するためにフォーマットする
   * @param propertyKey
   * @param propertyValue
   * @param propertyDefinition
   * @returns {*}
   */
  function formatPropertyValue(propertyKey, propertyValue, propertyDefinition) {

    if (propertyValue && propertyKey == "cmis:contentStreamLength") {
      propertyValue = miscSize.getReadableSizeString(propertyValue);
    }

    if (propertyValue && propertyKey == "cmis:contentStreamMimeType") {
      var mimetype = getMimeType()[propertyValue];
      propertyValue = mimetype ? mimetype : AJS.I18n.getText("alfresco.for.confluence.property.mimetype.unknown");
    }

    if (propertyValue && propertyDefinition.propertyType == "datetime") {
      // propertyValue = dateformat ? moment(new Date(propertyValue)).format(dateformat) : new Date(propertyValue);
      propertyValue = moment(new Date(propertyValue)).format(DEFAULT_DATETIME_FORMAT);
    }

    if (propertyValue && propertyDefinition.propertyType == "boolean") {
      propertyValue = propertyValue ? AJS.I18n.getText("alfresco.for.confluence.property.boolean.true") : AJS.I18n.getText("alfresco.for.confluence.property.boolean.false") ;
    }

    return propertyValue;

  }

  /**
   * 選択リスト(フィールド)に描画する
   * @param template
   * @param fieldSelectOptions
   * @param propertyKey
   * @param fieldDefinitions
   * @returns {jQuery}
   */
  function renderFieldSelect(template, propertyKey, fieldSelectOptions, fieldDefinitions) {

    return $("input[id='left_key_" + propertyKey + "']", template).auiSelect2({
      data: fieldSelectOptions,
      id: function(data) {
        return data.id;
      },
      formatResult: function(data) {
        var fieldDefinition = fieldDefinitions[data.id];
        var iconRequired = "";
        if (fieldDefinition && fieldDefinition.required) {
          iconRequired = "icon-required";
        }
        return '<span><span class="aui-icon ' + iconRequired + '" style="position: inherit;width: 10px;"></span>' + data.text + '</span>';
      },
      formatSelection: function(data) {
        var fieldDefinition = fieldDefinitions[data.id];
        var iconRequired = "";
        if (fieldDefinition && fieldDefinition.required) {
          iconRequired = "icon-required";
        }
        return '<span><span class="aui-icon ' + iconRequired + '" style="position: inherit;width: 10px;"></span>' + data.text + '</span>';
      }
    });

  }

  /**
   * 選択リスト(フィールド)に表示するオプションを取得する
   * @param fieldDefinitions
   * @param isRenderable {function} - judge renerable option by fieldKey, fieldSchema, fieldOperations, fieldType, fieldRenderDefinition
   * @returns {*}
   */
  function getFieldOptions(fieldDefinitions, isRenderable) {
    var options = $.map(fieldDefinitions, function(fieldDefinition, fieldKey) {
      if (!fieldDefinition.schema) {
        return null;
      }
      var fieldName = fieldDefinition.name;
      var fieldSchema = fieldDefinition.schema;
      var fieldOperations = fieldDefinition.operations;
      var fieldType = fieldSchema ? fieldSchema.system || fieldSchema.custom : fieldDefinition.custom;
      var fieldRenderDefinition = getFieldRenderDefinition(fieldSchema);
      if (!isRenderable(fieldKey, fieldSchema, fieldOperations, fieldType, fieldRenderDefinition)) {
        return null;
      }
      return { id: fieldKey, text: fieldName, fieldRenderDefinition: fieldRenderDefinition };
    });
    options.sort(function(a, b) {

      return compare(getOrder(a.fieldRenderDefinition), getOrder(b.fieldRenderDefinition));

      function compare(a, b) {
        if (!a || a > b) { return 1; }
        if (!b || a < b) { return -1; }
        return 0;
      }
      function getOrder(fieldRenderDefinition) {
        return fieldRenderDefinition ? (fieldRenderDefinition.order ? fieldRenderDefinition.order : undefined) : undefined;
      }

    });
    if (options.length > 0) {
      options.unshift({ id: '', text: '<span style="opacity: 0.5;">' + AJS.I18n.getText('alfresco.for.confluence.property.unmapped') }) + '</span>';
    }
    return options;
  }

  /**
   * 選択済みの選択リスト(フィールド)のオプションを取得する。
   * @param context
   * @param fieldDefinitions
   * @returns {{}}
   */
  function getSelectedFieldOptions(context, fieldDefinitions) {
    var options = {};
    $("input[id^='left_key_'][value!='']", context).filter(function(){
      var fieldKey = $(this).val();
      var fieldDefinition = fieldDefinitions[fieldKey];
      var propertyKey = $(this).attr("id").substring("left_key_".length);
      options[propertyKey] = fieldKey;
    });
    return options;
  }

  /**
   * displayNameのtoLowerCaseに一致するフィールド定義を取得する
   * @param fieldDefinitions
   * @param displayName
   * @returns {String}
   */
  function getFieldKeyByLocalNameToLowerCase(fieldDefinitions, displayName) {
    var keys = $.map(fieldDefinitions, function(fieldDefinition, key) {
      if (!fieldDefinition.schema) {
        console.log(fieldDefinition);
      }
      // システムフィールドの場合
      if (fieldDefinition.schema && fieldDefinition.schema.system) {
        if (fieldDefinition.name.toLowerCase() == displayName.toLowerCase()) {
          return key;
        }
      }
      // カスタムフィールドの場合
      if (fieldDefinition.schema && fieldDefinition.schema.custom) {
        if (fieldDefinition.name.toLowerCase() == displayName.toLowerCase()) {
          return key;
        }
      }
    });
    return keys.length > 0 ? keys[0] : null;
  }

  /**
   * システム・カスタムフィールド表示・非表示の定義を取得する
   */
  function getFieldRenderDefinitions() {
    return [
      { order: 1,  visible: true,  type: "string", system: "key" },
      { order: 2,  visible: true,  type: "string", system: "summary" },
      { order: 3,  visible: true,  type: "string", system: "description" },
      { order: 4,  visible: true,  type: "date", system: "duedate" },
      { order: 5,  visible: true,  type: "user", system: "assignee" },
      { order: 6,  visible: true,  type: "user", system: "reporter" },
      { order: 7,  visible: true,  type: "string", custom: "com.atlassian.jira.plugin.system.customfieldtypes:textfield" },
      { order: 8,  visible: true,  type: "string", custom: "com.atlassian.jira.plugin.system.customfieldtypes:textarea" },
      { order: 9,  visible: true,  type: "string", custom: "com.atlassian.jira.plugin.system.customfieldtypes:url" },
      { order: 10,  visible: true,  type: "number", custom: "com.atlassian.jira.plugin.system.customfieldtypes:float" },
      { order: 11, visible: true,  type: "date",  custom: "com.atlassian.jira.plugin.system.customfieldtypes:datepicker" },
      { order: 12, visible: true,  type: "datetime", custom: "com.atlassian.jira.plugin.system.customfieldtypes:datetime" },
      { order: 13, visible: true,  type: "user",   custom: "com.atlassian.jira.plugin.system.customfieldtypes:userpicker" },
      { order: 99, visible: false, type: "issuetype", system: "issuetype" },
      { order: 99, visible: false, type: "priority", system: "priority" },
      { order: 99, visible: false, type: "array", system: "labels" },
      { order: 99, visible: false, type: "array", system: "comment" },
      { order: 99, visible: false, type: "array", system: "attachment" },
      { order: 99, visible: false, type: "group", custom: "com.atlassian.jira.plugin.system.customfieldtypes:grouppicker" },
      { order: 99, visible: false, type: "project", custom: "com.atlassian.jira.plugin.system.customfieldtypes:project" },
      { order: 99, visible: false, type: "array", custom: "com.atlassian.jira.plugin.system.customfieldtypes:multiuserpicker" },
      { order: 99, visible: false, type: "array", custom: "com.atlassian.jira.plugin.system.customfieldtypes:multigrouppicker" },
      { order: 99, visible: false, type: "string", custom: "com.atlassian.jira.plugin.system.customfieldtypes:select" },
      { order: 99, visible: false, type: "array", custom: "com.atlassian.jira.plugin.system.customfieldtypes:multiselect" },
      { order: 99, visible: false, type: "array", custom: "com.atlassian.jira.plugin.system.customfieldtypes:cascadingselect" },
      { order: 99, visible: false, type: "array", custom: "com.atlassian.jira.plugin.system.customfieldtypes:multicheckboxes" },
      { order: 99, visible: false, type: "string", custom: "com.atlassian.jira.plugin.system.customfieldtypes:radiobuttons" },
      { order: 99, visible: false, type: "version", custom: "com.atlassian.jira.plugin.system.customfieldtypes:version" },
      { order: 99, visible: false, type: "array", custom: "com.atlassian.jira.plugin.system.customfieldtypes:multiversion" },
    ];
  }

  /**
   * システム・カスタムフィールド表示・非表示の定義を取得する
   * @param schema
   * @returns {Object}
   */
  function getFieldRenderDefinition(schema) {
    var renders = getFieldRenderDefinitions().filter(function (render) {
      if (render.type == schema.type) {
        if (render.system && schema.system && render.system == schema.system && render.visible) {
          return render;
        }
      }
      if (render.type == schema.type) {
        if (render.custom && schema.custom && render.custom == schema.custom && render.visible) {
          return render;
        }
      }
    });
    return renders.length > 0 ? renders[0] : null;
  }

  /**
   * 標準プロパティ表示・非表示の定義を取得する
   * # idがなくtypeのみはカスタムプロパティの定義
   * # push, pullは対応可能なフィールドのスキーマ(system|custom)を定義
   * # プロパティの編集が可能かの判断はisPropertyEditableを参照 push, pullの条件にもなっている
   */
  function getPropertyRenderDefinitions() {
    return [
      /**
       * システムプロパティ
       */
      {
        order: 1,  visible: true, id: "cmis:name", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_name'),
      },
      {
        order: 2,  visible: true, id: "cm:title", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cm_title'),
        push: [ "key", "summary", "com.atlassian.jira.plugin.system.customfieldtypes:textfield", ],
        pull: [ "summary", "com.atlassian.jira.plugin.system.customfieldtypes:textfield", ],
      },
      {
        order: 3,  visible: true, id: "cm:description", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cm_description'),
        push: [ "key", "description", "com.atlassian.jira.plugin.system.customfieldtypes:textarea", ],
        pull: [ "description", "com.atlassian.jira.plugin.system.customfieldtypes:textarea", ],
      },
      {
        order: 4,  visible: true, id: "cm:author", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cm_author'),
        push: [ "assignee", "reporter", "com.atlassian.jira.plugin.system.customfieldtypes:userpicker", ],
        pull: [ "assignee", "reporter", "com.atlassian.jira.plugin.system.customfieldtypes:userpicker", ],
      },
      {
        order: 5,  visible: true, id: "cm:owner", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cm_owner'),
        push: [ "assignee", "reporter", "com.atlassian.jira.plugin.system.customfieldtypes:userpicker", ],
        pull: [ "assignee", "reporter", "com.atlassian.jira.plugin.system.customfieldtypes:userpicker", ],
      },
      {
        order: 6,  visible: true, id: "cmis:contentStreamMimeType", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_contentStreamMimeType'),
      },
      {
        order: 7,  visible: true, id: "cmis:contentStreamLength", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_contentStreamLength'),
      },
      {
        order: 8,  visible: true, id: "cmis:createdBy", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_createdBy'),
      },
      {
        order: 9,  visible: true, id: "cmis:creationDate", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_creationDate'),
      },
      {
        order: 10,  visible: true, id: "cmis:lastModifiedBy", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_lastModifiedBy'),
      },
      {
        order: 11, visible: true, id: "cmis:lastModificationDate", displayName: AJS.I18n.getText('alfresco.for.confluence.property.displayname.cmis_lastModificationDate'),
      },
      /**
       * カスタムプロパティー
       */
      {
        order: 99, visible: true, type: [ "string", "d:text", "d:mltext" ], cardinality: [ "single" ], multiValued: false,
        push: [ "key", "summary", "description", "com.atlassian.jira.plugin.system.customfieldtypes:textarea", "com.atlassian.jira.plugin.system.customfieldtypes:textfield", ],
        pull: [ "summary", "description", "com.atlassian.jira.plugin.system.customfieldtypes:textarea", "com.atlassian.jira.plugin.system.customfieldtypes:textfield", ],
      },
      {
        order: 99, visible: true,  type: [ "decimal", "integer", "d:int", "d:long", "d:float", "d:double" ], cardinality: [ "single" ], multiValued: false,
        push: [ "com.atlassian.jira.plugin.system.customfieldtypes:float", ],
        pull: [ "com.atlassian.jira.plugin.system.customfieldtypes:float", ],
      },
      {
        order: 99, visible: true,  type: [ "datetime", "d:date", "d:datetime" ], cardinality: [ "single" ], multiValued: false,
        push: [ "duedate","com.atlassian.jira.plugin.system.customfieldtypes:datepicker", "com.atlassian.jira.plugin.system.customfieldtypes:datetime", ],
        pull: [ "duedate","com.atlassian.jira.plugin.system.customfieldtypes:datepicker", "com.atlassian.jira.plugin.system.customfieldtypes:datetime", ],
      },
      { order: 99, visible: true,  type: "boolean", cardinality: [ "single" ], },
      { order: 99, visible: true,  type: "uri", cardinality: [ "single" ], },
      { order: 99, visible: true,  type: "id", cardinality: [ "single" ], },
      { order: 99, visible: true,  type: "html", cardinality: [ "single" ], },
    ];
  }

  /**
   *
   * @returns
   */
  function getAllPrefixOfType() {
    return [
      'act',
      'alfcmis',
      'aos',
      'app',
      'audio',
      'stcp',
      'blg',
      'bpm',
      'cm',
      'cmis',
      'cmiscustom',
      'cmisext',
      'cmm',
      'dl',
      'download',
      'dp',
      'emailserver',
      'exif',
      'facebook',
      'flickr',
      'fm',
      'gd2',
      'hwf',
      'ia',
      'imap',
      'imwf',
      'inwf',
      'linkedin',
      'lnk',
      'pub',
      'pubwf',
      'qshare',
      'rc',
      'rn',
      'rule',
      'slideshare',
      'smf',
      'srft',
      'st',
      'surf',
      'sync',
      'sys',
      'trx',
      'twitter',
      'usr',
      'ver',
      'ver2',
      'webdav',
      'wf',
      'ws',
      'wswf',
      'youtube',
    ];
  }

  /**
   * 標準プロパティ表示・非表示の定義を取得する
   * @param id
   * @returns {Object}
   */
  function getPropertyRenderDefinition(id) {
    var renders = getPropertyRenderDefinitions().filter(function (render) {
      if (render.id && render.id == id) {
        return render;
      }
    });
    return renders.length > 0 ? renders[0] : null;
  }

  /**
   * カスタムプロパティ表示・非表示の定義を取得する
   * @param type
   * @returns {Object}
   */
  function getCustomPropertyRenderDefinition(type) {
    var renders = getPropertyRenderDefinitions().filter(function (render) {
      if (render.type && $.inArray(type, render.type) != -1) {
        return render;
      }
    });
    return renders.length > 0 ? renders[0] : null;
  }

  /**
   * @see "2.1.3.3.2 Attributes common to ALL Object-Type Property Deﬁnitions"
   *       http://docs.oasis-open.org/cmis/CMIS/v1.1/errata01/os/CMIS-v1.1-errata01-os-complete.html#x1-7400010
   * @param {String} propertyKey
   * @param {Array} propertyDefinitions
   * @returns {Object} { "id": "cmis:objectTypeId", "localName": "cmis:objectTypeId", "localNamespace": "local", "queryName": "cmis:objectTypeId", "displayName": "Type-Id", "description": "This is a Type-Id property.", "propertyType": "string|boolean|decimal|integer|datetime|uri|id|html", "cardinality": "single|multi", "updatability": "readonly|readwrite|whencheckedout|oncreate", "inherited": false, "required": false, "queryable": true, "orderable": false }
   */
  function getPropertyDefinition(propertyKey, propertyDefinitions) {
    var propertyDefinition = null;
    $.each(propertyDefinitions, function (i, definition) {
      propertyDefinition = definition.propertyDefinitions[propertyKey]
      if (propertyDefinition) {
        return false;
      }
    });
    return propertyDefinition;
  }

  /**
   * プロパティの編集可否を確認する
   * @param propertyDefinition
   * @param propertyRenderDefinition
   */
  function isPropertyEditable(propertyDefinition, propertyRenderDefinition) {
    if (propertyDefinition.updatability && propertyDefinition.updatability != 'readwrite') {
      return false;
    }
    if (propertyDefinition.cardinality && propertyRenderDefinition.cardinality && $.inArray(propertyDefinition.cardinality, propertyRenderDefinition.cardinality) == -1) {
      return false;
    }
    if (typeof propertyDefinition.multiValued !== 'undefined' &&  typeof propertyRenderDefinition.multiValued !== 'undefined' && propertyDefinition.multiValued != propertyRenderDefinition.multiValued) {
      return false;
    }
    return true;
  }

  /**
   * MimeTypeを取得する
   * @returns {Object}
   */
  function getMimeType() {
    return {
      "video/3gpp": "3G Video",
      "video/3gpp2": "3G2 Video",
      "application/vnd.adobe.xdp+xml": "Adobe Acrobat XML Data Package",
      "application/vnd.adobe.aftereffects.project": "Adobe AfterEffects Project",
      "application/vnd.adobe.aftereffects.template": "Adobe AfterEffects Template",
      "application/vnd.adobe.air-application-installer-package+zip": "Adobe AIR",
      "image/x-raw-adobe": "Adobe Digital Negative Image",
      "application/x-zip": "Adobe Flex Project File",
      "application/framemaker": "Adobe FrameMaker",
      "application/illustrator": "Adobe Illustrator File",
      "application/x-indesign": "Adobe InDesign Document",
      "application/pagemaker": "Adobe PageMaker",
      "application/pdf": "Adobe PDF Document",
      "image/vnd.adobe.photoshop": "Adobe Photoshop",
      "image/vnd.adobe.premiere": "Adobe Premiere",
      "audio/vnd.adobe.soundbooth": "Adobe SoundBooth",
      "audio/x-aiff": "AIFF Audio",
      "application/acp": "Alfresco Content Package",
      "application/vnd.android.package-archive": "Android Package",
      "image/x-portable-anymap": "Anymap Image",
      "application/vnd.apple.keynote": "Apple iWork Keynote",
      "application/vnd.apple.numbers": "Apple iWork Numbers",
      "application/vnd.apple.pages": "Apple iWork Pages",
      "image/vnd.dwg": "AutoCAD Drawing",
      "image/x-dwt": "AutoCAD Template",
      "audio/basic": "Basic Audio",
      "application/x-dosexec": "Binary File",
      "application/octet-stream": "Binary File (Octet Stream)",
      "image/bmp": "Bitmap Image",
      "image/x-raw-canon": "Canon RAW Image",
      "image/cgm": "CGM Image",
      "text/csv": "Comma Separated Values (CSV)",
      "application/dita+xml": "DITA",
      "message/rfc822": "EMail",
      "application/eps": "EPS Type PostScript",
      "audio/x-flac": "FLAC Audio",
      "application/x-fla": "Flash Source",
      "video/x-flv": "Flash Video",
      "image/x-raw-fuji": "Fuji RAW Image",
      "image/gif": "GIF Image",
      "image/x-portable-graymap": "Greymap Image",
      "application/x-gzip": "GZIP",
      "application/x-gtar": "GZIP Tarball",
      "image/x-raw-hasselblad": "Hasselblad RAW Image",
      "text/html": "HTML",
      "application/vnd.oasis.opendocument.text-web": "HTML Document Template",
      "text/calendar": "iCalendar File",
      "image/ief": "IEF Image",
      "application/java-archive": "Java Archive",
      "application/java": "Java Class",
      "text/x-jsp": "Java Server Page",
      "text/x-java-source": "Java Source File",
      "application/x-javascript": "JavaScript",
      "image/jp2": "JPEG 2000 Image",
      "image/jpeg": "JPEG Image",
      "application/json": "JSON",
      "image/x-raw-kodak": "Kodak RAW Image",
      "application/x-latex": "LaTeX",
      "image/x-raw-leica": "Leica RAW Image",
      "application/x-troff-man": "Man Page",
      "text/x-markdown": "Markdown",
      "text/mediawiki": "MediaWiki Markup",
      "application/vnd.ms-excel": "Microsoft Excel",
      "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet": "Microsoft Excel 2007",
      "application/vnd.ms-excel.addin.macroenabled.12": "Microsoft Excel 2007 add-in",
      "application/vnd.ms-excel.sheet.binary.macroenabled.12": "Microsoft Excel 2007 binary workbook",
      "application/vnd.ms-excel.sheet.macroenabled.12": "Microsoft Excel 2007 macro-enabled workbook",
      "application/vnd.ms-excel.template.macroenabled.12": "Microsoft Excel 2007 macro-enabled workbook template",
      "application/vnd.openxmlformats-officedocument.spreadsheetml.template": "Microsoft Excel template 2007",
      "application/vnd.ms-outlook": "Microsoft Outlook Message",
      "application/vnd.ms-powerpoint": "Microsoft PowerPoint",
      "application/vnd.openxmlformats-officedocument.presentationml.presentation": "Microsoft PowerPoint 2007",
      "application/vnd.ms-powerpoint.addin.macroenabled.12": "Microsoft PowerPoint 2007 add-in",
      "application/vnd.ms-powerpoint.presentation.macroenabled.12": "Microsoft PowerPoint 2007 macro-enabled presentation",
      "application/vnd.ms-powerpoint.template.macroenabled.12": "Microsoft PowerPoint 2007 macro-enabled presentation template",
      "application/vnd.ms-powerpoint.slide.macroenabled.12": "Microsoft PowerPoint 2007 macro-enabled slide",
      "application/vnd.ms-powerpoint.slideshow.macroenabled.12": "Microsoft PowerPoint 2007 macro-enabled slide show",
      "application/vnd.openxmlformats-officedocument.presentationml.slide": "Microsoft PowerPoint 2007 slide",
      "application/vnd.openxmlformats-officedocument.presentationml.slideshow": "Microsoft PowerPoint 2007 slide show",
      "application/vnd.openxmlformats-officedocument.presentationml.template": "Microsoft PowerPoint 2007 template",
      "application/vnd.ms-project": "Microsoft Project",
      "application/vnd.visio": "Microsoft Visio",
      "application/vnd.visio2013": "Microsoft Visio 2013",
      "application/msword": "Microsoft Word",
      "application/vnd.openxmlformats-officedocument.wordprocessingml.document": "Microsoft Word 2007",
      "application/vnd.ms-word.document.macroenabled.12": "Microsoft Word 2007 macro-enabled document",
      "application/vnd.ms-word.template.macroenabled.12": "Microsoft Word 2007 macro-enabled document template",
      "application/vnd.openxmlformats-officedocument.wordprocessingml.template": "Microsoft Word 2007 template",
      "image/x-raw-minolta": "Minolta RAW Image",
      "audio/mpeg": "MPEG Audio",
      "video/mp2t": "MPEG Transport Stream",
      "video/mpeg": "MPEG Video",
      "video/mpeg2": "MPEG2 Video",
      "audio/mp4": "MPEG4 Audio",
      "video/mp4": "MPEG4 Video",
      "video/x-m4v": "MPEG4 Video (m4v)",
      "video/x-ms-asf": "MS ASF Streaming Video",
      "video/x-msvideo": "MS Video",
      "audio/x-ms-wma": "MS WMA Streaming Audio",
      "video/x-ms-wmv": "MS WMV Streaming Video",
      "image/x-raw-nikon": "Nikon RAW Image",
      "audio/ogg": "Ogg Audio",
      "application/ogg": "Ogg Multiplex",
      "video/ogg": "Ogg Video",
      "audio/vorbis": "Ogg Vorbis Audio",
      "image/x-raw-olympus": "Olympus RAW Image",
      "application/vnd.oasis.opendocument.chart": "OpenDocument Chart",
      "application/vnd.oasis.opendocument.database": "OpenDocument Database",
      "application/vnd.oasis.opendocument.graphics": "OpenDocument Drawing",
      "application/vnd.oasis.opendocument.graphics-template": "OpenDocument Drawing Template",
      "application/vnd.oasis.opendocument.formula": "OpenDocument Formula",
      "application/vnd.oasis.opendocument.image": "OpenDocument Image",
      "application/vnd.oasis.opendocument.text-master": "OpenDocument Master Document",
      "application/vnd.oasis.opendocument.presentation": "OpenDocument Presentation",
      "application/vnd.oasis.opendocument.presentation-template": "OpenDocument Presentation Template",
      "application/vnd.oasis.opendocument.spreadsheet": "OpenDocument Spreadsheet",
      "application/vnd.oasis.opendocument.spreadsheet-template": "OpenDocument Spreadsheet Template",
      "application/vnd.oasis.opendocument.text": "OpenDocument Text (OpenOffice 2.0)",
      "application/vnd.oasis.opendocument.text-template": "OpenDocument Text Template",
      "application/vnd.sun.xml.calc": "OpenOffice 1.0/StarOffice6.0 Calc 6.0",
      "application/vnd.sun.xml.calc.template": "OpenOffice 1.0/StarOffice6.0 Calc 6.0 Template",
      "application/vnd.sun.xml.draw": "OpenOffice 1.0/StarOffice6.0 Draw 6.0",
      "application/vnd.sun.xml.impress": "OpenOffice 1.0/StarOffice6.0 Impress 6.0",
      "application/vnd.sun.xml.impress.template": "OpenOffice 1.0/StarOffice6.0 Impress 6.0 Template",
      "application/vnd.sun.xml.writer": "OpenOffice 1.0/StarOffice6.0 Writer 6.0",
      "application/vnd.sun.xml.writer.template": "OpenOffice 1.0/StarOffice6.0 Writer 6.0 Template",
      "image/x-raw-panasonic": "Panasonic RAW Image",
      "image/x-raw-pentax": "Pentax RAW Image",
      "image/x-portable-pixmap": "Pixmap Image",
      "text/plain": "Plain Text",
      "image/png": "PNG Image",
      "image/x-portable-bitmap": "Portable Bitmap",
      "application/postscript": "PostScript",
      "application/remote-printing": "Printer Text File",
      "video/quicktime": "Quicktime Video",
      "video/x-rad-screenplay": "RAD Screen Display",
      "application/x-rar-compressed": "RAR Archive",
      "image/x-cmu-raster": "Raster Image",
      "image/x-raw-red": "RED RAW Image",
      "image/x-rgb": "RGB Image",
      "text/richtext": "Rich Text",
      "application/rtf": "Rich Text Format",
      "application/rss+xml": "RSS",
      "image/svg+xml": "Scalable Vector Graphics Image",
      "video/x-sgi-movie": "SGI Video",
      "text/sgml": "SGML (Human Readable)",
      "application/sgml": "SGML (Machine Readable)",
      "application/x-sh": "Shell Script",
      "application/x-shockwave-flash": "Shockwave Flash",
      "image/x-raw-sigma": "Sigma RAW Image",
      "image/x-raw-sony": "Sony RAW Image",
      "application/vnd.stardivision.chart": "StaChart 5.x",
      "application/vnd.stardivision.calc": "StarCalc 5.x",
      "application/vnd.stardivision.draw": "StarDraw 5.x",
      "application/vnd.stardivision.impress": "StarImpress 5.x",
      "application/vnd.stardivision.impress-packed": "StarImpress Packed 5.x",
      "application/vnd.stardivision.math": "StarMath 5.x",
      "application/vnd.stardivision.writer": "StarWriter 5.x",
      "application/vnd.stardivision.writer-global": "StarWriter 5.x global",
      "text/css": "Style Sheet",
      "text/tab-separated-values": "Tab Separated Values",
      "application/x-tar": "Tarball",
      "application/x-tex": "Tex",
      "application/x-texinfo": "Tex Info",
      "image/tiff": "TIFF Image",
      "x-world/x-vrml": "VRML",
      "audio/x-wav": "WAV Audio",
      "video/webm": "WebM Video",
      "application/wordperfect": "WordPerfect",
      "image/x-xbitmap": "XBitmap Image",
      "application/xhtml+xml": "XHTML",
      "text/xml": "XML",
      "image/x-xpixmap": "XPixmap Image",
      "image/x-xwindowdump": "XWindow Dump",
      "application/x-compress": "Z Compress",
      "application/zip": "ZIP",
    };
  }

  return {

    /**
     * デフォルト日時フォーマット
     */
    DEFAULT_DATETIME_FORMAT: DEFAULT_DATETIME_FORMAT,

    /**
     * デフォルト日時フォーマットUTC
     */
    DEFAULT_DATETIME_FORMAT_UTC: DEFAULT_DATETIME_FORMAT_UTC,

    /**
     * 画面に表示するプロパティ情報を取得する
     * @param properties
     * @param propertyDefinitions
     * @returns {Array}
     */
    getRenderItems: getRenderItems,

    /**
     * フィールド値を画面に表示するためにフォーマットする
     * @param fieldKey
     * @param fieldValue
     * @param fields
     * @param fieldDefinitions
     * @returns {any|*|String}
     */
    formatFieldValue: formatFieldValue,

    /**
     * プロパティ値を画面に表示するためにフォーマットする
     * @param propertyKey
     * @param propertyValue
     * @param propertyDefinition
     * @returns {*}
     */
    formatPropertyValue: formatPropertyValue,

    /**
     * 選択リスト(フィールド)に描画する
     * @param template
     * @param fieldSelectOptions
     * @param propertyKey
     * @param fieldDefinitions
     * @returns {jQuery}
     */
    renderFieldSelect: renderFieldSelect,

    /**
     * 選択リスト(フィールド)に表示するオプションを取得する
     * @param fieldDefinitions
     * @returns {*}
     */
    getFieldOptions: getFieldOptions,

    /**
     * 選択済みの選択リスト(フィールド)のオプションを取得する。
     * @param context
     * @param fieldDefinitions
     * @returns {{}}
     */
    getSelectedFieldOptions: getSelectedFieldOptions,

    /**
     * nameのtoLowerCaseに一致するフィールド定義を取得する
     * @param fieldDefinitions
     * @param name
     * @returns {String}
     */
    getFieldKeyByLocalNameToLowerCase: getFieldKeyByLocalNameToLowerCase,

    /**
     * システム・カスタムフィールド表示・非表示の定義を取得する
     * @param schema
     * @returns {Object}
     */
    getFieldRenderDefinition: getFieldRenderDefinition,

    /**
     * 標準プロパティ表示・非表示の定義を取得する
     * @param id
     * @returns {Object}
     */
    getPropertyRenderDefinition: getPropertyRenderDefinition,

    /**
     * カスタムプロパティ表示・非表示の定義を取得する
     * @param type
     * @returns {Object}
     */
    getCustomPropertyRenderDefinition: getCustomPropertyRenderDefinition,

    /**
     * @see "2.1.3.3.2 Attributes common to ALL Object-Type Property Deﬁnitions"
     *       http://docs.oasis-open.org/cmis/CMIS/v1.1/errata01/os/CMIS-v1.1-errata01-os-complete.html#x1-7400010
     * @param {String} propertyKey
     * @param {Array} propertyDefinitions
     * @returns {Object} { "id": "cmis:objectTypeId", "localName": "cmis:objectTypeId", "localNamespace": "local", "queryName": "cmis:objectTypeId", "displayName": "Type-Id", "description": "This is a Type-Id property.", "propertyType": "string|boolean|decimal|integer|datetime|uri|id|html", "cardinality": "single|multi", "updatability": "readonly|readwrite|whencheckedout|oncreate", "inherited": false, "required": false, "queryable": true, "orderable": false }
     */
    getPropertyDefinition: getPropertyDefinition,

    /**
     * プロパティの編集可否を確認する
     * @param propertyDefinition
     * @param propertyRenderDefinition
     */
    isPropertyEditable: isPropertyEditable,

    /**
     * MimeTypeを取得する
     * @returns {Object}
     */
    getMimeType: getMimeType,

    /**
     * すべてのタイプのPrefixを取得する
     * @return {Array}
     */
    getAllPrefixOfType: getAllPrefixOfType

  };

});
