/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('cmis/cmissearchfilter', ['jquery', 'cmis/confluence-error-handler'],
  function ($, errorHandler) {

    function CmisSearchFilter(searchQueryCallback) {
      this.searchQueryCallback = searchQueryCallback;
      this.regexReserved = new RegExp("[.*+?|()\\[\\]{}\\\\]", "g"); // .*+?|()[]{}\
      this.lastSearch = "";
      this.$container = undefined;
      this.$elem = undefined;
    };

    CmisSearchFilter.prototype.getExpandedSearchFieldWidth = function($container) {
      var $searchInput = $('#cmis-file-search', $container);
      var $quickFilterLabel = $("#cmis-folder-title", $container);
      var quickFilterLabelLeft = $quickFilterLabel.offset().left;
      var searchInputLeft = $searchInput.offset().left;
      var searchInputRight = searchInputLeft + $searchInput.width();
      var activeSearchWidth = searchInputRight - quickFilterLabelLeft;
      return activeSearchWidth;
    };

    CmisSearchFilter.prototype.render = function($container, $parent, searchKeyword) {

      if (!$container.length || !$parent.length) {
        return;
      }

      var self = this;
      var template = JIRA.Templates.cmissearchfilter.renderSearchBox({ searchKeyword: searchKeyword });
      var $elem = $(template).appendTo($parent);
      var $searchInput = $('#cmis-file-search-input', $elem);

      this.$container = $container;
      this.$elem = $elem;

      // when the field is focused do stuff
      $searchInput.focus(function(){
        $elem.addClass('cmis-active');
        $searchInput.css({'width':self.getExpandedSearchFieldWidth($container)});
        $elem.parent().addClass('cmis-search-active');
      });

      // when the field is blurred and empty, remove active state
      $searchInput.blur(function(){
        var rawQuery = $.trim($searchInput.val());
        if (_.isEmpty(rawQuery)) {
          setTimeout(function () {
            $searchInput.val('');
            $elem.removeClass('cmis-active');
            // change the icon to the search icon
            $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
            $searchInput.css({'width':''});
            $elem.parent().removeClass('cmis-search-active');
          }, 400);
        }
      });

      // bind a keyhandler for esc
      $searchInput.bind('keydown', function(event){
        if(event.keyCode == 27) {
          // clear the text
          $searchInput.val('');
          // blur the input
          $searchInput.blur();
          // re-execute search
          self.searchQuery(event);
        }
      });

      // when the icon is clicked do stuff
      $elem.find('.js-search-trigger').click(function(event){
        // if already active close it else open it
        if($elem.hasClass('cmis-active')){
          $elem.removeClass('cmis-active');
          $searchInput.css({'width':''});
          $elem.parent().removeClass('cmis-search-active');
          // change the icon to the search icon
          $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
          // clear the text
          $searchInput.val('');
          // And re-execute the search
          self.searchQuery(event);
        } else {
          $elem.addClass('cmis-active');
          $searchInput.css({'width':self.getExpandedSearchFieldWidth($container)});
          $elem.parent().addClass('cmis-search-active');
          $searchInput.focus();
        }
      });

      // instant search on key type
      $searchInput.keyup(function(event) {
        if ($searchInput.val() !== ''){
          // change the icon to the remove icon
          $elem.find('.aui-iconfont-search-small').removeClass('aui-iconfont-search-small').addClass('aui-iconfont-remove');
        } else {
          // change the icon to the search icon
          $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
        }
        if (event.keyCode == 13) {
          self.searchQuery(event);
        }
      });

      // ensure form is not submitted
      $elem.submit(function(event) {
        event.preventDefault();
        // testing if the input has no text for stylin'
        if ($searchInput.val() !== ''){
          // change the icon to the remove icon
          $elem.find('.aui-iconfont-search-small').removeClass('aui-iconfont-search-small').addClass('aui-iconfont-remove');
        } else {
          // change the icon to the search icon
          $elem.find('.aui-iconfont-remove').removeClass('aui-iconfont-remove').addClass('aui-iconfont-search-small');
        }
        self.searchQuery(event);
      });

      // the field is focused do stuff
      if ($searchInput.val() !== ''){
        // change the icon to the remove icon
        $elem.find('.aui-iconfont-search-small').removeClass('aui-iconfont-search-small').addClass('aui-iconfont-remove');
        $searchInput.focus();
      }

      $(window).resize($.proxy(self.ajustSearchFieldWidth, self));

      return $elem;

    };

    CmisSearchFilter.prototype.clearSearchBox = function() {
      var $formElement = $("#cmis-file-search");
      $formElement.removeClass('cmis-active');
      $formElement.parent().removeClass('cmis-search-active');
      $formElement.find(".cmis-search").val('').css({'width':''});
    };

    CmisSearchFilter.prototype._queryToRegex = function(rawQuery) {
      var finalQuery;
      if (_.isEmpty(rawQuery)) {
        finalQuery = '.*';
      } else {
        finalQuery = _.map(rawQuery.split(" "),
          function (queryItem) {
            return "(?=.*(" + queryItem + "))";
          }).join("");
      }

      console.log("Executing search " + finalQuery);
      return new RegExp(finalQuery, "i");
    };

    CmisSearchFilter.prototype.clearLastSearch = function() {
      this.lastSearch = "";
    };

    CmisSearchFilter.prototype.ajustSearchFieldWidth = function() {
      if(this.$container && this.$elem && this.$elem.hasClass('cmis-active')){
        var $searchInput = this.$container.find('#cmis-file-search-input');
        $searchInput.css({'width':this.getExpandedSearchFieldWidth(this.$container)});
      }
    }

    /**
     * Stops the search (does NOT cancel the already typed text but blurs the input field)
     */
    CmisSearchFilter.prototype.stopEdit = function(container) {
      var $searchInput = this.$container.find('#cmis-file-search-input');
      $searchInput.blur();
    };

    CmisSearchFilter.prototype.searchQuery = function(e) {
      var rawQuery = $.trim($("#cmis-file-search-input").val());
      // Don't rerun the search if it hasn't changed, or if it is empty
      if (rawQuery === this.lastSearch) return;
      this.lastSearch = rawQuery;
      if (rawQuery === '') {
        this.searchQueryCallback(e, rawQuery, null);
      } else {
        var pattern;
        try {
          // Now we munge the query into an appropriate regexp
          pattern = this._queryToRegex(rawQuery);
          this.searchQueryCallback(e, rawQuery, pattern);
        } catch (e) {
          console.log("Regex search threw exception");
          pattern = this._queryToRegex(rawQuery.replace(this.regexReserved, "\\$&"));
          this.searchQueryCallback(e, rawQuery, pattern);
        }
      }
    };

    return {
      /**
       * @param {object} options
       */
      setup: function (searchQueryCallback) {
        return new CmisSearchFilter(searchQueryCallback);
      }
    }

  });
