/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('cmis/cmislink', ['jquery', 'cmis/cmisconfluence', 'cmis/confluence-error-handler', 'confluence/message-controller'], function ($, cmisConfluence, errorHandler, MessageController) {
  var DIALOG_ID = 'cmislink-dialog';
  var DIALOG_SECTION = '<section role="dialog" id="' + DIALOG_ID + '" class="aui-layer aui-dialog2 aui-dialog2-medium aui-help" aria-hidden="true" data-aui-modal="true"></section>';
  var HIDDEN_CLASS = 'hidden';
  var LOADING_CLASS = "loading-documents";

  var linkedDialog_;

  /**
   * リンクを表示する
   * @param {CmisListView} invoker
   * @param {CmisDocument} doc
   * @param {Object} options
   * @returns {*}
   */
  function doShowLinks(invoker, doc, options) {

    var d = new $.Deferred;

    var dateformat = options.dateformat;

    if ($('#' + DIALOG_ID).length <= 0) {
      $('body').append(DIALOG_SECTION);
    }

    var template = JIRA.Templates.cmislink.linkedDialog(links);

    var dialogContainer = $('#' + DIALOG_ID).empty().append(template);

    var links = invoker.getLinkObjectValues(doc.versionSeriesId);

    renderLinkContents(dialogContainer, doc, links, false).done(function() {
      d.resolve();
    });

    $('.closeButton', dialogContainer).click(function () {
      // アップロードダイアログを閉じる
      closeDialog();
    });

    linkedDialog_ = AJS.dialog2('#' + DIALOG_ID);
    linkedDialog_.show();

    return d.promise();

  }

  /**
   * リンク情報を描画する
   * @param context
   * @param doc
   * @param links
   * @param canDelete
   * @returns {*}
   */
  function renderLinkContents(context, doc, links, canDelete) {

    var d = new AJS.$.Deferred;

    (function () {

      var d = new AJS.$.Deferred;

      $("." + LOADING_CLASS, context).removeClass(HIDDEN_CLASS);

      $('.linked-list', context).empty();

      if (!links || links.length == 0) {
        d.resolve({'issues': []});
        return d.promise();
      }

      cmisConfluence.searchPages($.map(links, function (link, index) {
        return link.pageId;
      })).done(function (searchResults) {
        d.resolve(searchResults);
      }).fail(function (error) {
        errorHandler.showError(context, {
          body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.info'),
          error: error
        });
        d.reject();
      });

      return d.promise();

    })().pipe(function (searchResults) {

      if (!links || links.length == 0) {
        $('.linked-list', context).append("There is no linked Pages.");
        d.resolve();
        return;
      }

      $.each(links, function (index, link) {
        var greppedPages = $.grep(searchResults.results, function (page) {
          return page.id == link.pageId;
        });
        if (greppedPages.length === 0) {
          return true;
        }
        if (greppedPages.length > 1) {
          console.log("Duplicated linked issues are detected. [" + link.pageId + "]");
        }
        var page = greppedPages[0];
        $('.linked-list', context).append(JIRA.Templates.cmislink.linkedContent({
          'link': link,
          'page': page,
          'contextPath': contextPath,
          'canDelete': canDelete
        }));
      });

      d.resolve();

    }).fail(function (error) {

      d.reject();

    }).always(function () {

      $("." + LOADING_CLASS, context).addClass(HIDDEN_CLASS);

    });

    return d.promise();

  }

  /**
   * ダイアログを閉じる
   */
  function closeDialog() {
    if (linkedDialog_) {
      linkedDialog_.hide();
    }
  }

  /**
   * リンクする
   * @param {CmisListView} invoker
   * @param {CmisDocument} doc
   * @param {Object} options
   */
  function doLink(invoker, doc, options) {

    var dateformat = options.dateformat;

    // リンクファイルを追加
    cmisConfluence.addLinkByObject(doc, cmisConfluence.getPageId()).done(function (data) {

      // リンク済みリストを更新
      var objectId = doc.versionSeriesId;

      var links = invoker.formatLinks(data.links);

      // リンク済みリストを更新
      invoker.putLinkObject(objectId, links[objectId]);

      // ドキュメントを再描画
      invoker._renderDocument(doc);

      // イベントを飛ばす
      invoker.onLink_.apply(null, [doc, invoker]);

    }).fail(function (error) {
      MessageController.showError(MessageController.parseError(error), MessageController.Location.FLAG);
      errorHandler.showError($(".messages", invoker.el), {
        body: AJS.I18n.getText('alfresco.for.jira.ajax.error.listview.link'),
        error: error
      });
    }).always(function () {
    });

  }

  /**
   * リンク解除する
   * @param {CmisListView} invoker
   * @param {CmisDocument} doc
   * @param {Object} options
   */
  function doUnLink(invoker, doc, options) {

    var dateformat = options.dateformat;

    // リンクファイル削除
    cmisConfluence.removeLinkByObject(doc, cmisConfluence.getPageId()).done(function (data) {

      var objectId = doc.versionSeriesId;

      var links = invoker.formatLinks(data.links);

      // リンク済みリストを更新
      invoker.putLinkObject(objectId, links[objectId]);

      // ドキュメントを再描画
      invoker._renderDocument(doc);

      // イベントを飛ばす
      invoker.onUnlink_.apply(null, [doc, invoker]);

    }).fail(function (error) {
      MessageController.showError(MessageController.parseError(error), MessageController.Location.FLAG);
      errorHandler.showError($(".messages", invoker.el), {
        body: AJS.I18n.getText('alfresco.for.jira.ajax.error.listview.unlink'),
        error: error
      });
    }).always(function () {
    });

  }

  /**
   * リンクする
   * @param {CmisListView} invoker
   * @param {CmisDocument} doc
   * @param {Object} links
   * @param {Object} options
   * @returns {*}
   */
  function doLinks(invoker, doc, links, dateformat) {

    var d = new $.Deferred;

    if ($('#' + DIALOG_ID).length <= 0) {
      $('body').append(DIALOG_SECTION);
    }

    var template = JIRA.Templates.cmislink.linkDialog(links);

    var dialogContainer = $('#' + DIALOG_ID).empty().append(template);

    renderLinkContents(dialogContainer, doc, links, false).done(function() {

      renderPicker(dialogContainer, cmisConfluence.getProjectId());

      d.resolve();

    });

    $('#dialog-cancel-button', dialogContainer).click(function () {
      // アップロードダイアログを閉じる
      closeDialog();
    });

    $('#dialog-submit-button', dialogContainer).click(function () {

      // キー
      var keys = $('#link-picker', dialogContainer).val();

      if (!keys || keys.length == 0) {
        return;
      }

      // リンクファイルを追加
      cmisConfluence.addLinksByObject(doc, keys).done(function (data) {

        var objectId = doc.versionSeriesId;

        var links = invoker.formatLinks(data.links);

        // リンク済みリストを更新
        invoker.putLinkObject(objectId, links[objectId]);

        // ダイアログを再描画
        doLinks(invoker, doc, links[objectId], dateformat);

        // ドキュメントを再描画
        invoker._renderDocument(doc);

        // イベントを飛ばす
        invoker.onLink_.apply(null, [doc, invoker]);

      }).fail(function(error) {
        MessageController.showError(MessageController.parseError(error), MessageController.Location.FLAG);
        errorHandler.showError($(".messages", dialogContainer), { body: AJS.I18n.getText('alfresco.for.jira.ajax.error.listview.link'), error: error });
      }).always(function() {
      });

    });

    linkedDialog_ = AJS.dialog2('#' + DIALOG_ID);
    linkedDialog_.show();

    return d.promise();

  }

  /**
   * リンク解除する
   * @param {CmisListView} invoker
   * @param {CmisDocument} doc
   * @param {Object} links
   * @param {Object} options
   * @returns {*}
   */
  function doUnLinks(invoker, doc, links, dateformat) {

    var d = new $.Deferred;

    if ($('#' + DIALOG_ID).length <= 0) {
      $('body').append(DIALOG_SECTION);
    }

    var template = JIRA.Templates.cmislink.unlinkDialog(links);

    var dialogContainer = $('#' + DIALOG_ID).empty().append(template);

    renderLinkContents(dialogContainer, doc, links, true).done(function() {

      d.resolve();

    });

    $('#dialog-cancel-button', dialogContainer).click(function () {
      // アップロードダイアログを閉じる
      closeDialog();
    });

    $('#dialog-submit-button', dialogContainer).click(function () {

      // キー
      var keys = [];
      $('input[id^=cmis-unlink-][type=checkbox]:checked', dialogContainer).each(function() {
        keys.push($(this).attr("id").replace("cmis-unlink-", ""));
      });

      if (!keys || keys.length == 0) {
        return;
      }

      // リンクファイル削除
      cmisConfluence.removeLinksByObject(doc, keys).done(function (data) {

        var objectId = doc.versionSeriesId;

        var links = invoker.formatLinks(data.links);

        // リンク済みリストを更新
        invoker.putLinkObject(objectId, links[objectId]);

        // ダイアログを再描画
        doUnLinks(invoker, doc, links[objectId], dateformat);

        // ドキュメントを再描画
        invoker._renderDocument(doc);

        // イベントを飛ばす
        invoker.onUnlink_.apply(null, [doc, invoker]);

      }).fail(function(error) {
        MessageController.showError(MessageController.parseError(error), MessageController.Location.FLAG);
        errorHandler.showError($(".messages", dialogContainer), { body: AJS.I18n.getText('alfresco.for.jira.ajax.error.listview.unlink'), error: error });
      }).always(function() {
      });

    });

    linkedDialog_ = AJS.dialog2('#' + DIALOG_ID);
    linkedDialog_.show();

    return d.promise();

  }

  /**
   * ピッカーを描画する
   * @param context
   * @param pid
   * @param options
   */
  function renderPicker(context, pid, options) {

    $(".aui-field-picker-container", context).append(JIRA.Templates.cmislink.picker({
      contextPath: contextPath,
      pid: pid
    }));

    $('.aui-field-picker', context).each(function () {

      new JIRA.IssuePicker({
        element: $(this)
        // ,
        // userEnteredOptionsMsg: formatter.I18n.getText('linkissue.enter.issue.key'),
        // uppercaseUserEnteredOnSelect: true
      });

      $('a.issue-picker-popup').remove();

    }).val("");

  }

  return {
    doShowLinks: doShowLinks,
    doLink: doLink,
    doUnLink: doUnLink,
    doLinks: doLinks,
    doUnLinks: doUnLinks
  }

});
