/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('CMIS', ['jquery', 'jquery/CMIS'], function($, jqueryCmis) {

  var DEFAULT_PREFIXES = {
    'cmis':'cmis:',
    'cm':'cm:',
    'alfcmis':'alfcmis:',
    'qshare':'qshare:'
  }

  function inheritConstructor(parentCtor, childCtor){
    /** @constructor */
    function tempCtor() {};
    tempCtor.prototype = parentCtor.prototype;
    childCtor.superClass_ = parentCtor.prototype;
    childCtor.prototype = new tempCtor();
    /** @override */
    childCtor.prototype.constructor = childCtor;
  }
  
  /**
   * 読み取り専用プロパティを定義
   * @param {Object} obj 定義対象のオブジェクト
   * @param {String} key プロパティ名
   * @param {value} value 値
   */
  function defineReadonly(obj, key, value) {
    Object.defineProperty(obj, key, {
      value: value,
      writable: false,    // 書き込み不可
      enumerable: true,   // 列挙可能
      configurable: true  // 再定義は可能
    });
  }
  
  function args2array(args) {
    return Array.prototype.slice.call(args, 0, args.length); 
  }
  
  /**
   * @param {CmisObject} cmisObject
   * @param {CmisSession} cmisSession
   * @param {object} prefixes
   * @param {object} session
   */
  function getCmisObject(cmisObject, cmisSession, prefixes) {
    var prefixes = $.extend({}, DEFAULT_PREFIXES, prefixes);
    var obj,
        baseTypeId = cmisObject.succinctProperties[prefixes['cmis'] + 'baseTypeId'],
        objectTypeId = cmisObject.succinctProperties[prefixes['cmis'] + 'objectTypeId'];
   
    if (baseTypeId == "cmis:folder") {
      obj = new CmisFolder(cmisObject, cmisSession, prefixes);
    } else if (baseTypeId == "cmis:document") {
      obj = new CmisDocument(cmisObject, cmisSession, prefixes);
    } else if (baseTypeId == "cmis:item") {

      if (objectTypeId == "I:app:folderlink") {
        obj = new CmisFolderLink(cmisObject, cmisSession, prefixes);
      } else if (objectTypeId == "I:app:filelink") {
        obj = new CmisDocumentLink(cmisObject, cmisSession, prefixes);
      }

      // TODO unknown type
    } 
    return obj;
  }
	
  
  function requestReject(defferd) {
    return function(error) {
      defferd.reject(new CmisRequestError(error));
    }
  }
  
  //-------------------------------------------------------------------------
  
  
  /**
   * CMISエラークラス
   */
  function CmisError(message) {
    defineReadonly(this, 'message', message || 'Error');
    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, this.constructor);  
    } else {
      var stack = new Error().stack;
      this.stack = stack;
    }
    
    this.name    = this.constructor.name;
  }
  inheritConstructor(Error, CmisError);
  
  //-------------------------------------------------------------------------
  
  /**
   * CMISリクエストエラークラス
   */
  function CmisRequestError(response) {  
    var message;
    if (response) {
      if (response.body) {
        message = response.body.message;
      } else {
        message = response.statusText;
      }
    } else {
      message = "request error";
    }
    CmisError.call(this, message);
    
    this.response = response;
  }
  inheritConstructor(CmisError, CmisRequestError);
  
  
  
  //-------------------------------------------------------------------------
  // CmisSession
  //-------------------------------------------------------------------------
  
  /**
   * 
   */
  function CmisSession(options) {
    options = options || {};
    
    this.authType = options.authType;
    this.username = options.username;
    this.token = options.token;
    
    if (typeof options.session === 'undefined') {
      if (typeof options.url === 'undefined') {
        throw new CmisError('options.url is undefined');
      }
      // セッションを新たに作成
      this.session = jqueryCmis.createSession(options.url);
    } else {
      this.session = options.session;
    }
    
    if (options.authType === 'Basic') {
      this.session.on('beforesend', function(ajaxParams) {
        ajaxParams.headers = ajaxParams.headers || {};
        ajaxParams.headers['Authorization'] = options.token;
        ajaxParams.timeout = 0;
        ajaxParams.crossDomain = true;
      });
    }
    if (options.authType === 'WSSE') {
      this.session.on('beforesend', function(ajaxParams) {
        ajaxParams.headers = ajaxParams.headers || {};
        ajaxParams.headers['X-WSSE'] = options.token;
        ajaxParams.timeout = 0;
        ajaxParams.crossDomain = true;
      });
    }
  }
  
  CmisSession.prototype.query = function() {
    return this.session.query.apply(this.session, arguments);
  }
  
  /**
   * 使用するリポジトリを選択する
   * @param {String} name
   * @return {Deferred}
   */
  CmisSession.prototype.select = function(name, override) {
    var self = this,
        d = new $.Deferred,
        repository;
     
    function setup(repositories) {
      var repo = repositories[name];
      if (typeof repo === 'undefined') {
        d.reject(new CmisError(name + ' is undefined repository.'));
      } else {
        self.session.defaultRepository = repo;

        repo.repositoryUrl = repo.repositoryUrl.substring(repo.repositoryUrl.indexOf('/cmisbrowser'));
        repo.rootFolderUrl = repo.rootFolderUrl.substring(repo.rootFolderUrl.indexOf('/cmisbrowser'));
        
        if (override) {
          $.extend(repo, override); 
        }
        
        d.resolve(repo, self);
      }
    }
    
    if (typeof this.session.repositories === 'undefined') {
      // まだリポジトリ一覧が読み込まれていない場合は読み込む
      this.session.loadRepositories()
        .done(setup)
        .fail(requestReject(d));
    } else {
      setup(this.session.repositories);
    }
    
    return d.promise();
  }
  
  /**
   * リポジトリ読み込み済みか否か
   * @return {boolean}
   */
  CmisSession.prototype.isLoadedRepository = function() {
    return typeof this.session.defaultRepository !== 'undefined'; 
  }

  /**
   * ベンダ名を取得します
   * @return {String}
   */
  CmisSession.prototype.getVenderName = function() {
    return this.session.defaultRepository.vendorName;
  }

  /**
   * 製品名を取得します
   * @return {String}
   */
  CmisSession.prototype.getProductName = function() {
    return this.session.defaultRepository.productName;
  }

  /**
   * 製品バージョンを取得します
   * @return {String}
   */
  CmisSession.prototype.getProductVersion = function() {
    return this.session.defaultRepository.productVersion;
  }

  /**
   * gets an cmisObject
   * @param {CmisObject} cmisObject
   * @param {CmisSession} cmisSession
   * @param {object} session
   */
  CmisSession.prototype.getCmisObject = function(cmisObject) {
    return getCmisObject(cmisObject, this);
  }
  
  /**
   * gets an object by objectId
   *
   * @param {String} objectId
   * @param {String} returnVersion (if set must be one of 'this', latest' or 'latestmajor')
   * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includeACL, includePolicyIds, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.getObject = function(objectId, returnVersion, options) {
    var self = this,
        d = new $.Deferred;
    
    this.session.getObject.apply(this.session, arguments)
      .done(function(data) {
        var obj = getCmisObject(data, self);
        d.resolve(obj, self);
      })
      .fail(requestReject(d));
    
    return d.promise();
  }
  
  /**
   * ID一覧からオブジェクト一覧を取得します
   * 
   * @param {String} objectId
   * @param {String} returnVersion (if set must be one of 'this', latest' or 'latestmajor')
   * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includeACL, includePolicyIds, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.getObjectByIds = function(objectIds, returnVersion, options) {
    var self = this,
        d = new $.Deferred,
        deffers = [];

    if (objectIds.length <= 0) {
      d.resolve([]);
      return d.promise();
    }
    
    $.each(objectIds, function(i, id) {
      // TODO １つのIDにつき1リクエスト投げてるのでパフォーマンス悪し
      deffers.push(self.getObject(id, returnVersion, options));
    });
    
    $.when.apply(null, deffers).then(function() {
      var documents = [];
      if (deffers.length > 1) {
        for (var i = 0, len = arguments.length; i < len; ++i) {
          documents.push(arguments[i][0]);
        }   
      } else {
        documents.push(arguments[0]);
      }
      // 名前の昇順でソート
      documents.sort(function (a, b) {
        if (a.name > b.name) { return 1; } 
        if (a.name < b.name) { return -1; }
        return 0;
      });
      d.resolve(documents);
    }, function() {
      d.reject();
    });
    
    return d.promise();
  }
  
  /**
   * gets an object by path
   *
   * @param {String} path
   * @param {Object} options
   * @return {Deferred}
   */
  CmisSession.prototype.getObjectByPath = function(path, options) {
	  var self = this,
	      d = new $.Deferred;
	
    this.session.getObjectByPath.apply(this.session, arguments)
      .done(function(data) {
    	  var obj = getCmisObject(data, self);
    	  d.resolve(obj, self);
      })
      .fail(requestReject(d));
    
    return d.promise();
  }
  
  /**
   * Returns children of object specified by path
   * @param {String} path
   * @param {Object} options (possible options: maxItems, skipCount, filter, orderBy, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.getChildrenByPath = function(path, options) {
    var self = this,
        d = new $.Deferred;
    
    this.getObjectByPath(path, options).done(function(folder) {
      folder.getChildren().done(function(children) {
        d.resolve(children, folder, self);
      }).fail(function(error) {
        d.reject(error);
      });
    }).fail(function(error) {
      d.reject(error);
    });
    
    return d.promise();
  }

  /**
   * 指定フォルダIDのフォルダを取得する
   * @param {String} objectId
   * @param {Boolean} searchAllVersions
   * @param {Object} options (possible options: includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.getFolder = function(objectId, searchAllVersions, options) {
    var self = this,
        d = new $.Deferred,
        queryString = "SELECT * FROM cmis:folder WHERE cmis:objectId ='" + objectId + "'";
        
    this.query(queryString, searchAllVersions, options).done(function(data) {
      var result,
          hasItems = data.numItems > 0;
      if (hasItems) {
        result = getCmisObject(data.results[0], self);
      }
      d.resolve(result, self);
    }).fail(requestReject(d));
    
    return d.promise();
  };

  /**
   * 複数の指定フォルダIDのフォルダを取得する
   * @param {Array} objectIds
   * @param {Boolean} searchAllVersions
   * @param {Object} options (possible options: includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.getFolders = function(objectIds, searchAllVersions, options) {
    var self = this,
      d = new $.Deferred,
      queryString = "SELECT * FROM cmis:folder WHERE cmis:objectId IN('" + objectIds.join("','") + "')";

    this.query(queryString, searchAllVersions, options).done(function(data) {
      var result,
        hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
      if (hasItems) {
        var cmisObjects = data.results.map(function(val) {
          return getCmisObject(val, self);
        })
      }
      d.resolve(cmisObjects, self);
    }).fail(requestReject(d));

    return d.promise();
  };

  /**
   * 指定フォルダ名のフォルダを取得する
   * @param {String} name
   * @param {String} parentId
   * @param {Boolean} searchAllVersions
   * @param {Object} options (possible options: includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.getFolderByName = function(name, parentId, searchAllVersions, options) {
    var self = this,
        d = new $.Deferred,
        queryString = "SELECT * FROM cmis:folder WHERE cmis:name IN('" + name + "')";
    
    if (typeof parentId === 'string') {
      queryString += " AND cmis:parentId IN('" + parentId + "')";
    }
    
    this.query(queryString, searchAllVersions, options).done(function(res) {
      var result = [];
      $.each(res.results, function() {
        result.push(getCmisObject(this, self));
      });
      d.resolve(result, self);
    }).fail(requestReject(d));
    
    return d.promise();
  }

  /**
   * 複数のフォルダ内からドキュメントをIN_TREE検索する
   * @param {Array} folders
   * @param {String} keyword
   * @param {Object} options (possible options: maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.queryInTrees = function(folders, keyword, options) {
    var self = this,
      d = new $.Deferred,
      queryString = "SELECT d.* FROM cmis:document AS d ";

    if (!folders || folders.length == 0) {
      d.resolve([], false, 0, self);
      return d.promise();
    }

    var in_trees = [];
    var makeInTreeQuery = function(folders) {
      $.each(folders, function() {
        if ($.isArray(this)) {
          makeInTreeQuery(this);
        } else if (this.id && this.id.length > 0) {
          in_trees.push("IN_TREE(d, '" + this.id + "')");
        }
      });
    };
    makeInTreeQuery(folders);

    queryString += " WHERE (" + in_trees.join(" OR ") + ") ";

    if (keyword) {
      queryString += " AND CONTAINS(d,'\"" + keyword + "\" OR cmis:name:\"" + keyword + "\"')";
    }

    // options's orderBy does not work!
    if (options.orderBy) {
      queryString += " ORDER BY " + options.orderBy;
    }

    this.session.query(queryString, false, options)
      .done(function (data) {
        var result,
          hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
        var result = [];
        if (hasItems) {
          $.each(data.results, function() {
            result.push(getCmisObject(this, self, { 'cmis':'d.cmis:', 'alfcmis':'d.alfcmis:' }));
          });
        }
        d.resolve(result, data.hasMoreItems, data.numItems, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   * 複数のフォルダ内から編集中ドキュメントをIN_TREE検索する
   * @param {Array} folders
   * @param {String} keyword
   * @param {Object} options (possible options: maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.editingInTrees = function(folders, keyword, options) {
    var self = this,
      d = new $.Deferred,
      queryString = "SELECT d.* FROM cmis:document as d join cm:workingcopy as c on d.cmis:objectId = c.cmis:objectId ";

    if (!folders || folders.length == 0) {
      d.resolve([], false, 0, self);
      return d.promise();
    }

    var in_trees = [];
    var makeInTreeQuery = function(folders) {
      $.each(folders, function() {
        if ($.isArray(this)) {
          makeInTreeQuery(this);
        } else if (this.id && this.id.length > 0) {
          in_trees.push("IN_TREE(d, '" + this.id + "')");
        }
      });
    };
    makeInTreeQuery(folders);

    queryString += " WHERE (" + in_trees.join(" OR ") + ")";
    queryString += " AND c.cm:workingCopyOwner = '" + self.username + "'";
    if (keyword) {
      queryString += " AND CONTAINS(d,'\"" + keyword + "\" OR cmis:name:\"" + keyword + "\"')";
    }

    // options's orderBy does not work!
    if (options.orderBy) {
      queryString += " ORDER BY " + options.orderBy;
    }

    this.session.query(queryString, false, options)
      .done(function (data) {
        var result,
          hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
        var result = [];
        if (hasItems) {
          $.each(data.results, function() {
            result.push(getCmisObject(this, self, { 'cmis':'d.cmis:', 'alfcmis':'d.alfcmis:' }));
          });
        }
        d.resolve(result, data.hasMoreItems, data.numItems, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   * 複数のフォルダ内から最近の追加/修正ドキュメントをIN_TREE検索する
   * @param {Array} folders
   * @param {String} keywork
   * @param {String} column
   * @param {Object} options (possible options: maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.recentlyInTrees = function(folders, keyword, column, options) {
    var self = this,
      d = new $.Deferred,
      queryString = "SELECT d.* FROM cmis:document as d ";

    if (!folders || folders.length == 0) {
      d.resolve([], false, 0, self);
      return d.promise();
    }

    var in_trees = [];
    var makeInTreeQuery = function(folders) {
      $.each(folders, function() {
        if ($.isArray(this)) {
          makeInTreeQuery(this);
        } else if (this.id && this.id.length > 0) {
          in_trees.push("IN_TREE(d, '" + this.id + "')");
        }
      });
    };
    makeInTreeQuery(folders);


    queryString += " WHERE (" + in_trees.join(" OR ") + ")";
    queryString += " AND d." + column + " >= TIMESTAMP '" + moment().subtract("days", 7).toISOString() + "'";

    if (keyword) {
      queryString += " AND CONTAINS(d,'\"" + keyword + "\" OR cmis:name:\"" + keyword + "\"')";
    }

    // options's orderBy does not work!
    if (options.orderBy) {
      queryString += " ORDER BY " + options.orderBy;
    }

    this.session.query(queryString, false, options)
      .done(function (data) {
        var result,
          hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
        var result = [];
        if (hasItems) {
          $.each(data.results, function() {
            result.push(getCmisObject(this, self, { 'cmis':'d.cmis:', 'alfcmis':'d.alfcmis:' }));
          });
        }
        d.resolve(result, data.hasMoreItems, data.numItems, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   *
   * @param objectId
   * @param options
   * @returns {*}
   */
  CmisSession.prototype.getShared = function(objectId, options) {

    var self = this,
      d = new $.Deferred,
      queryString = "SELECT * FROM qshare:shared  WHERE cmis:objectId ='" + objectId + "'";

    this.session.query(queryString, false, options)
      .done(function (data) {
        var result,
          hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
        if (hasItems) {
          result = new CmisShared(data.results[0], self);
        }
        d.resolve(result, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   *
   * @param objectId
   * @param options
   * @returns {*}
   */
  CmisSession.prototype.getShareds = function(objectIds, options) {

    var self = this,
      d = new $.Deferred,
      queryString = "SELECT * FROM qshare:shared  WHERE cmis:objectId IN('" + objectIds.join("','") + "')";

    this.session.query(queryString, false, options)
      .done(function (data) {
        var result = [],
          hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
        if (hasItems) {
          $.each(data.results, function() {
            result.push(new CmisShared(this, self));
          });
        }
        d.resolve(result, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   * フォルダを再帰的に作成する
   * @param {String} path
   * @param {Boolean} searchAllVersions
   * @param {Object} options (possible options: includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisSession.prototype.mkdir = function(path, searchAllVersions, options) {
    var self = this,
        d = new $.Deferred,
        elements = $.grep(path.split('/'), function(val) {
                      return val && val.length > 0;
                   });
    
    function create(elements, parentId) {
      
      var el = elements.shift();
      
      // フォルダ情報を取得する
      self.getFolderByName(el, parentId, searchAllVersions, options).done(function(res) {
        if (res.length <= 0) {
          // フォルダ情報が取得できない場合は作成する
          self.session.createFolder(parentId, el).done(function(res) {
            self.session.getObject(getCmisObject(res, self).id, searchAllVersions, options).done(function(res) {
              // サブフォルダを処理する
              next(elements, getCmisObject(res, self));
            }).fail(requestReject(d));
          }).fail(requestReject(d));
        } else {
          // サブフォルダを処理する
          next(elements, res[0]);
        }
      }).fail(requestReject(d));
    }
    
    function next(elements, folder) {
      if (elements.length <= 0) {
        d.resolve(folder);
      } else {
        create(elements, folder.id);
      }
    }
    
    if (elements.length >= 0) {
      create(elements, self.session.defaultRepository.rootFolderId);
    } else {
      d.reject(new CmisError(''));
    }
    
    return d.promise();
  }

  //-------------------------------------------------------------------------
  // CmisObject
  //-------------------------------------------------------------------------
  
  /***
   * 
   */
  function CmisObject(obj, cmisSession, prefixes) {
    this.cmisSession = cmisSession;
    this.session = cmisSession.session;
	  this.succinctProperties = obj.succinctProperties;
	  this.propertiesExtension = obj.propertiesExtension;
	  this.allowableActions = obj.allowableActions;
	  
	  var prop = obj.succinctProperties;
    defineReadonly(this, 'nodeRef',                prop[prefixes['alfcmis'] + 'nodeRef']);
	  defineReadonly(this, 'id',                     prop[prefixes['cmis'] + 'objectId']);
	  defineReadonly(this, 'baseTypeId',             prop[prefixes['cmis'] + 'baseTypeId']);
	  defineReadonly(this, 'objectTypeId',           prop[prefixes['cmis'] + 'objectTypeId']);
	  defineReadonly(this, 'secondaryObjectTypeIds', prop[prefixes['cmis'] + 'secondaryObjectTypeIds']);
	  defineReadonly(this, 'name',                   prop[prefixes['cmis'] + 'name']);
	  defineReadonly(this, 'description',            prop[prefixes['cmis'] + 'description']);
	  defineReadonly(this, 'createdBy',              prop[prefixes['cmis'] + 'createdBy']);
	  defineReadonly(this, 'creationDate',           prop[prefixes['cmis'] + 'creationDate']);
	  defineReadonly(this, 'lastModifiedBy',         prop[prefixes['cmis'] + 'lastModifiedBy']);  
	  defineReadonly(this, 'lastModificationDate',   prop[prefixes['cmis'] + 'lastModificationDate']);
	  defineReadonly(this, 'changeToken',            prop[prefixes['cmis'] + 'changeToken']);
  }
  
  /**
   * deletes an object
   * @param {Boolean} allVersions
   * @param {Object} options (possible options: token)
   * @return {CmisRequest}
   */
  CmisObject.prototype.remove = function(allversions, options) {
    var self = this,
        d = new $.Deferred;

    this.session.deleteObject.apply(this, [this.id].concat(args2array(arguments)))
      .done(function() {
        d.resolve(self);
      })
      .fail(requestReject(d));
    
    return d.promise();
  }

  /**
   * Gets the folders that are the parents of the specified object
   * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
   * @return {CmisRequest}
   */
  CmisObject.prototype.getParents = function (options) {
    var self = this,
      d = new $.Deferred;

    this.session.getParents.apply(this, [this.id].concat(args2array(arguments)))
      .done(function (res) {
        var result = [];
        $.each(res, function () {
          result.push(getCmisObject(this.object, self));
        });
        d.resolve(result, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   * 
   */
  CmisObject.prototype.prop = function(key) {
    return this.succinctProperties[key];
  }

  /**
   *
   */
  CmisObject.prototype.isDocument = function() {
    return this.baseTypeId === "cmis:document";
  }

  /**
   *
   */
  CmisObject.prototype.isFolder = function() {
    return this.baseTypeId === "cmis:folder";
  }

  /**
   *
   */
  CmisObject.prototype.isFolderLink = function() {
    return this.baseTypeId === "cmis:item" && this.objectTypeId === "I:app:folderlink";
  }

  /**
   *
   */
  CmisObject.prototype.isDocumentLink = function() {
    return this.baseTypeId === "cmis:item" &&  this.objectTypeId === "I:app:filelink";
  }

  /**
   *
   */
  CmisObject.prototype.isLink = function() {
    return this.isFolderLink() || this.isDocumentLink();
  }

  //-------------------------------------------------------------------------
  // CmisDocument
  //-------------------------------------------------------------------------
  
  /**
   * 
   */
  function CmisDocument(obj, cmisSession, prefixes) {
    CmisObject.call(this, obj, cmisSession, prefixes);
	  
	  var prop = this.succinctProperties;
	  defineReadonly(this, 'isImmutable',                prop[prefixes['cmis'] + 'isImmutable']);
	  defineReadonly(this, 'isLatestVersion',            prop[prefixes['cmis'] + 'isLatestVersion']);
	  defineReadonly(this, 'isMajorVersion',             prop[prefixes['cmis'] + 'isMajorVersion']);
	  defineReadonly(this, 'isPrivateWorkingCopy',       prop[prefixes['cmis'] + 'isPrivateWorkingCopy']);
	  defineReadonly(this, 'versionLabel',               prop[prefixes['cmis'] + 'versionLabel']);
	  defineReadonly(this, 'versionSeriesId',            prop[prefixes['cmis'] + 'versionSeriesId']);
	  defineReadonly(this, 'isVersionSeriesCheckedOut',  prop[prefixes['cmis'] + 'isVersionSeriesCheckedOut']);
	  defineReadonly(this, 'versionSeriesCheckedOutBy',  prop[prefixes['cmis'] + 'versionSeriesCheckedOutBy']);
	  defineReadonly(this, 'versionSeriesCheckedOutId',  prop[prefixes['cmis'] + 'versionSeriesCheckedOutId']);
	  defineReadonly(this, 'checkinComment',             prop[prefixes['cmis'] + 'checkinComment']);
	  defineReadonly(this, 'contentStreamLength',        prop[prefixes['cmis'] + 'contentStreamLength']);
	  defineReadonly(this, 'contentStreamMimeType',      prop[prefixes['cmis'] + 'contentStreamMimeType']);
	  defineReadonly(this, 'contentStreamFileName',      prop[prefixes['cmis'] + 'contentStreamFileName']);
	  defineReadonly(this, 'contentStreamId',            prop[prefixes['cmis'] + 'contentStreamId']);

	  defineReadonly(this, 'lastThumbnailModification',  prop[prefixes['cm'] + 'lastThumbnailModification']);

    defineReadonly(this, 'workingCopyOwner',           prop[prefixes['cm'] + 'workingCopyOwner']);
    defineReadonly(this, 'workingCopyMode',            prop[prefixes['cm'] + 'workingCopyMode']);
    defineReadonly(this, 'workingCopyLabel',           prop[prefixes['cm'] + 'workingCopyLabel']);
    defineReadonly(this, 'expiryDate',                 prop[prefixes['cm'] + 'expiryDate']);
    defineReadonly(this, 'lockIsDeep',                 prop[prefixes['cm'] + 'lockIsDeep']);
    defineReadonly(this, 'lockType',                   prop[prefixes['cm'] + 'lockType']);
    defineReadonly(this, 'lockOwner',                  prop[prefixes['cm'] + 'lockOwner']);
    defineReadonly(this, 'lockLifeTime',               prop[prefixes['cm'] + 'lockLifeTime']);
    defineReadonly(this, 'lockAdditionalInfo',         prop[prefixes['cm'] + 'lockAdditionalInfo']);

    //TODO Alfresco依存コード
    defineReadonly(this, 'nodeRef',  prop[prefixes['alfcmis'] + 'nodeRef']);
  }
  inheritConstructor(CmisObject, CmisDocument);
  
  /**
   * gets document renditions
   * @param {Object} options (possible options: renditionFilter, maxItems, skipCount, token)
   * @return {Deferred}
   */
  CmisDocument.prototype.getRenditions = function(options) {
    var self = this,
        d = new $.Deferred;

    this.session.getRenditions.apply(this, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        var result = []
        $.each(data, function(i, val) {
          result.push(new CmisRendition(this, self.cmisSession));
        });
        d.resolve(result, self);
      })
      .fail(requestReject(d));
    
    return d.promise();
  }
  
  /**
   * Updates content of document
   * @return {CmisRequest}
   */
  CmisDocument.prototype.cancelCheckOutIfPossible = function() {
    var self = this,
        d = new $.Deferred;
    
    // self.versionSeriesCheckedOutBy
    if (!self.versionSeriesCheckedOutId) {
      return d.resolve().promise();
    }
    
    if (self.versionSeriesCheckedOutBy !== self.cmisSession.username) {
      return d.reject(new CmisRequestError({ 
        status: '409', 
        statusText: 'You can not operate the cancelCheck because it has been checked out by ' + self.versionSeriesCheckedOutBy 
      })).promise();
    }
    
    return self.cancelCheckOut();
  };
  
  /**
   * Updates content of document
   * @param {String/Buffer} content
   * @param {String/Object} input
   * if `input` is a string used as the document name,
   * if `input` is an object it must contain required properties:
   *   {'cmis:name': 'docName', 'cmis:objectTypeId': 'cmis:document'}
   * @param {String} mimeType
   * @param {String} comment
   * @param {Boolean} major
   * @param {Array} policies
   * @param {Object} addACEs
   * @param {Object} removeACEs
   * @param {Object} options (possible options: changeToken, succinct, token)
   * @return {CmisRequest}
   */
  CmisDocument.prototype.update = function(content, input, major, comment, policies, addACEs, removeACEs, options) {
    var self = this,
        d = new $.Deferred,
        pwc;
    
    (function(){
      
      return self.checkOut();
      
    })().pipe(function(p) {
      
      pwc = p;
      return pwc.checkIn(major, input, content, comment, policies, addACEs, removeACEs, options);
      
    }).pipe(function(data) {
      
      d.resolve(getCmisObject(data, self.cmisSession), self);
      
    }, function(error) {
      
      if (pwc) {
        pwc.cancelCheckOut().done(function(){});
      }
      
      d.reject(error);
      
    });
    
    return d.promise();
  };
  
  /**
   * Gets document content
   * @param {Boolean} download
   * @param {Object} options (possible options: streamId, token)
   * @return {Deferred}
   */
  CmisDocument.prototype.getContentStream = function (download, options, ajaxOptions) {
    var self = this,
        d = new $.Deferred;
    
    this.session.getContentStream.apply(this.session, [this.id].concat(args2array(arguments)))
      .done(function(result) {
        d.resolve(result, self);
      })
      .fail(requestReject(d));

    return d.promise();
  };
  
  /**
   * CheckOut document
   * @param {Object} options (possible options: streamId, token)
   * @return {Deferred}
   */
  CmisDocument.prototype.checkOut = function (options, ajaxOptions) {
    var self = this,
        d = new $.Deferred;
    
    this.session.checkOut.apply(this.session, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        d.resolve(getCmisObject(data, self.cmisSession), self);
      })
      .fail(requestReject(d));

    return d.promise();
  };
  
  /**
   * CancelCheckOut document
   * @param {Object} options (possible options: streamId, token)
   * @return {Deferred}
   */
  CmisDocument.prototype.cancelCheckOut = function (options, ajaxOptions) {
    var self = this,
        d = new $.Deferred;
    
    this.session.cancelCheckOut.apply(this.session, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        d.resolve(data, self);
      })
      .fail(requestReject(d));

    return d.promise();
  };
  
  /**
   * CheckIn document
   * @param {String} objectId
   * @param {Boolean} major
   * @param {String/Object} input
   * if `input` is a string used as the document name,
   * if `input` is an object it must contain required properties:
   *   {'cmis:name': 'docName'}
   * @param {String/Buffer} content
   * @param {String} comment
   * @param {Array} policies
   * @param {Object} addACEs
   * @param {Object} removeACEs
   * @param {Object} options (possible options: streamId, token)
   * @return {Deferred}
   */
  CmisDocument.prototype.checkIn = function (major, input, content, comment, policies, addACEs, removeACEs, options, ajaxOptions) {
    var self = this,
        d = new $.Deferred;
    
    this.session.checkIn.apply(this.session, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        d.resolve(getCmisObject(data, self.cmisSession), self);
      })
      .fail(requestReject(d));

    return d.promise();
  };
  
  /**
   * Gets document content URL
   * @param {Boolean} download
   * @param {Object} options (possible options: streamId, token)
   * @return String
   */
  CmisDocument.prototype.getContentStreamURL = function(download, options) {
    return this.session.getContentStreamURL.apply(this, [this.id].concat(args2array(arguments)));
  }
  
  /**
   * サムネイル情報を取得
   */
  CmisDocument.prototype.getThumbnailInfo = function() {
    var self = this,
        d = new AJS.$.Deferred;
    this.getRenditions().done(function(renditions) {
        var ret = $.grep(renditions, function(v) {
        return v.kind == "cmis:thumbnail";
      });
      d.resolve(ret, self);
    }).fail(requestReject(d));
    return d.promise();
  }
  
  /**
   * サムネイルを取得
   */
  CmisDocument.prototype.getThumbnail = function() {
    var self = this,
        d = new AJS.$.Deferred;
    this.getThumbnailInfo().done(function(thumbnailInfo) {
      if (thumbnailInfo.length > 0) {
        thumbnailInfo[0].getContentStream(false).done(function(thubnail) {
          d.resolve(thubnail, thumbnailInfo[0], self);
        }).fail(requestReject(d));
      } else {
        d.resolve(null, null, self);
      }
    }).fail(requestReject(d));
    
    return d.promise();
  }
  
  /**
   * ダウンロード
   */
  CmisDocument.prototype.download = function() {
    var doc = this;
    this.getContentStream(true).done(function(data) {
      var a = document.createElement('a');
      if (window.navigator.msSaveBlob) {
        window.navigator.msSaveBlob(data,doc.name);
      } else {
        a.href = window.URL.createObjectURL(data);
      }
      a.target = '_parent';
      if ('download' in a) {
        a.download = doc.name;
      }
      (document.body || document.documentElement).appendChild(a);
      a.click();
      a.parentNode.removeChild(a);
    }).fail(function(error) {
      // oncole.log(error.message);
    });
  }
  
  
  /**
   * バージョン履歴
   * @param {String} versionSeriesId
   * @return {Deferred}
   */
  CmisDocument.prototype.getVersions = function(versionSeriesId) {
    var self = this,
        d = new $.Deferred;

    self.session.getAllVersions(versionSeriesId)
      .done(function(data) {
        var result = []
        $.each(data, function(i, val) {
          result.push(new CmisVersion(val, self));
        });
        d.resolve(result, self);
      })
      .fail(requestReject(d));
    
    return d.promise();
  }
  
  //-------------------------------------------------------------------------
  // CmisFolder
  //-------------------------------------------------------------------------
  
  function CmisFolder(obj, cmisSession, prefixes) {
    CmisObject.call(this, obj, cmisSession, prefixes);
    
    var prop = this.succinctProperties;
    defineReadonly(this, 'parentId',                   prop[prefixes['cmis'] + 'parentId']);
    defineReadonly(this, 'path',                       prop[prefixes['cmis'] + 'path']);
    defineReadonly(this, 'allowedChildObjectTypeIds',  prop[prefixes['cmis'] + 'allowedChildObjectTypeIds']);
  }
  inheritConstructor(CmisObject, CmisFolder);
  
  /**
   * creates a new document
   *
   * @param {String/Buffer/Blob} content
   * @param {String/Object} input
   * if `input` is a string used as the document name,
   * if `input` is an object it must contain required properties:
   *   {'cmis:name': 'docName', 'cmis:objectTypeId': 'cmis:document'}
   * @param {String} mimeType
   * @param {String} versioningState  (if set must be one of: "none", "major", "minor", "checkedout")
   * @param {Array} policies
   * @param {Object} addACEs
   * @param {Object} removeACEs
   * @param {Object} options (possible options: succinct, token)
   * @return {CmisRequest}
   */
  CmisFolder.prototype.createDocument = function(content, input, mimeType, versioningState, policies, addACEs, removeACEs, options) {
    var self = this,
        d = new $.Deferred;
    this.session.createDocument.apply(this, [this.id].concat(args2array(arguments)))
    .done(function(data) {
      var obj = getCmisObject(data, self.cmisSession);
      d.resolve(obj, self);
    })
    .fail(requestReject(d));

    return d.promise();
  }
    
  /**
   * creates a new folder
   *
   * @param {String/Object} input
   * if `input` is a string used as the folder name,
   * if `input` is an object it must contain required properties:
   *   {'cmis:name': 'aFolder', 'cmis:objectTypeId': 'cmis:folder'}
   * @param {Array} policies
   * @param {Object} addACEs
   * @param {Object} removeACEs
   * @return {Deferred}
   */
  CmisFolder.prototype.createFolder = function(input, policies, addACEs, removeACEs) {
    var self = this,
        d = new $.Deferred;
    
    this.session.createFolder.apply(this, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        var obj = getCmisObject(data, self.cmisSession);
        d.resolve(obj, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }
  
  /**
   * Returns children of object specified by id
   * @param {Object} options (possible options: maxItems, skipCount, filter, orderBy, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
   * @return {Deferred}
   */
  CmisFolder.prototype.getChildren = function(options) {
    var self = this,
        d = new $.Deferred;
    
    this.session.getChildren.apply(this, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        var result = [];
        $.each(data.objects, function() {
          var o = getCmisObject(this.object, self.cmisSession);
          result.push(o);
        });
        d.resolve(result, data.hasMoreItems, data.numItems, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   * フォルダ内に指定名の要素を取得する
   * @param {String} name
   * @param {Boolean} searchAllVersions
   * @param {Object} options (possible options: maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisFolder.prototype.getChildByName = function(name, searchAllVersions, options) {
    var self = this,
        d = new $.Deferred,
        queryString = "SELECT * FROM cmis:document WHERE cmis:name = '" + name + "' AND IN_FOLDER('" + "workspace://SpacesStore/" + this.id + "')";
    
    this.session.query(queryString, searchAllVersions, options)
    .done(function (data) {
      var result,
          hasItems = data.numItems > 0;
      if (hasItems) {
        result = getCmisObject(data.results[0], self.cmisSession);
      }      
      d.resolve(result, self);
    })
    .fail(requestReject(d));
    
    return d.promise();
  }

  /**
   * フォルダ内からドキュメントをIN_TREE検索する
   * @param {String} keyword
   * @param {Object} options (possible options: maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
   * @return {Deferred}
   */
  CmisFolder.prototype.queryInTree = function(keyword, options) {
    var self = this,
      d = new $.Deferred,
      queryString = "SELECT d.* FROM cmis:document AS d WHERE IN_TREE('" + this.id + "')";

    if (keyword) {
      queryString += " AND CONTAINS(d,'\"" + keyword + "\" OR cmis:name:\"" + keyword + "\"')";
    }

    // options's orderBy does not work!
    if (options.orderBy) {
      queryString += " ORDER BY " + options.orderBy;
    }

    this.session.query(queryString, false, options)
      .done(function (data) {
        var result,
          hasItems = data.numItems > 0 || (data.results && data.results.length > 0);
        var result = [];
        if (hasItems) {
          $.each(data.results, function() {
            result.push(getCmisObject(this, self.cmisSession, { 'cmis':'d.cmis:', 'alfcmis':'d.alfcmis:' }));
          });
        }
        d.resolve(result, data.hasMoreItems, data.numItems, self);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  /**
   * フォルダと含まれているものすべてを削除する
   * @param {Boolean} allVersions
   * @param {String} unfileObjects (if set must be one of 'unfile', 'deletesinglefiled', 'delete')
   * @param {Boolean} continueOnFailure
   * @param {Object} options (possible options: token)
   * @return {Deferred}
   */
  CmisFolder.prototype.deleteTree = function(allVersions, unfileObjects, continueOnFailure, options) {
    var self = this,
        d = new $.Deferred;
    
    this.session.deleteTree.apply(this, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        d.resolve(self);
      })
      .fail(requestReject(d));

    return d.promise();
  }
  
  /**
   * フォルダツリーを取得する
   * @param {Integer} depth
   * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
   * @return {Deferred}
   */
  CmisFolder.prototype.getFolderTree = function(depth, options) {
    var self = this,
        d = new $.Deferred;
    
    this.session.getFolderTree.apply(this, [this.id].concat(args2array(arguments)))
      .done(function(data) {
        d.resolve(data);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  //-------------------------------------------------------------------------
  // CmisLink
  //-------------------------------------------------------------------------
  function CmisLink(obj, cmisSession, prefixes) {
    CmisObject.call(this, obj, cmisSession, prefixes);

    var prop = this.succinctProperties;
    defineReadonly(this, 'destination', prop[prefixes['cm'] + 'destination']);
  }
  inheritConstructor(CmisObject, CmisLink);

  /**
   *
   */
  CmisLink.prototype.getDestinationObject = function(returnVersion, options) {
    var self = this,
      d = new $.Deferred;

    this.cmisSession.getObject.apply(this.cmisSession, [this.destination].concat(args2array(arguments)))
      .done(function(data) {
        data.linkObject = self;
        d.resolve(data);
      })
      .fail(requestReject(d));

    return d.promise();
  }

  //-------------------------------------------------------------------------
  // CmisDocumentLink
  //-------------------------------------------------------------------------
  function CmisDocumentLink(obj, cmisSession, prefixes) {
    CmisLink.call(this, obj, cmisSession, prefixes);

    var prop = this.succinctProperties;
  }
  inheritConstructor(CmisLink, CmisDocumentLink);

  //-------------------------------------------------------------------------
  // CmisFolderLink
  //-------------------------------------------------------------------------
  function CmisFolderLink(obj, cmisSession, prefixes) {
    CmisLink.call(this, obj, cmisSession, prefixes);

    var prop = this.succinctProperties;
  }
  inheritConstructor(CmisLink, CmisFolderLink);

  //-------------------------------------------------------------------------
  // CmisRendition
  //-------------------------------------------------------------------------
  
  /**
   * 
   */
  function CmisRendition(data, cmisSession) {
    this.data = data;
    this.cmisSession = cmisSession;
    this.session = cmisSession.session;
    
    defineReadonly(this, 'streamId',             data['streamId']);
    defineReadonly(this, 'mimeType',             data['mimeType']);
    defineReadonly(this, 'length',               data['length']);
    defineReadonly(this, 'title',                data['title']);
    defineReadonly(this, 'kind',                 data['kind']);
    defineReadonly(this, 'height',               data['height']);
    defineReadonly(this, 'width',                data['width']);
    defineReadonly(this, 'renditionDocumentId',  data['renditionDocumentId']);
  }
  
  /**
   * Gets document content
   * @param {Boolean} download
   * @param {Object} options (possible options: streamId, token)
   * @return {CmisRequest}
   */
  CmisRendition.prototype.getContentStream = function (download, options, ajaxOptions) {
    var self = this,
        d = new $.Deferred;
    
    this.session.getContentStream.apply(this.session, [this.renditionDocumentId].concat(args2array(arguments)))
      .done(function(result) {
        d.resolve(result, self);
      })
      .fail(requestReject(d));

    return d.promise();
  };

  /**
   * Gets document content URL
   * @param {Boolean} download
   * @param {Object} options (possible options: streamId, token)
   * @return String
   */
  CmisRendition.prototype.getContentStreamURL = function(download, options) {
    return this.session.getContentStreamURL.apply(this, [this.renditionDocumentId].concat(args2array(arguments)));
  }
  
  //-------------------------------------------------------------------------
  // CmisVersion
  //-------------------------------------------------------------------------  
  /**
   * 
   */
  function CmisVersion(obj, cmisSession) {
    this.cmisSession = cmisSession;
    this.session = cmisSession.session;
    this.succinctProperties = obj.succinctProperties;
    
    var prop = obj.succinctProperties;        
    defineReadonly(this, 'id',                     prop['cmis:objectId']);
    defineReadonly(this, 'baseTypeId',             prop['cmis:baseTypeId']);
    defineReadonly(this, 'objectTypeId',           prop['cmis:objectTypeId']);
    defineReadonly(this, 'name',                   prop['cmis:name']);
    defineReadonly(this, 'versionLabel',           prop['cmis:versionLabel']);
    defineReadonly(this, 'isMajorVersion',         prop['cmis:isMajorVersion']);
    defineReadonly(this, 'isLatestVersion',        prop['cmis:isLatestVersion']);
    defineReadonly(this, 'isLatestMajorVersion',   prop['cmis:isLatestMajorVersion']);
    defineReadonly(this, 'contentStreamFileName',  prop['cmis:contentStreamFileName']);
    defineReadonly(this, 'contentStreamMimeType',  prop['cmis:contentStreamMimeType']);
    defineReadonly(this, 'contentStreamLength',    prop['cmis:contentStreamLength']);
    
  }

  //-------------------------------------------------------------------------
  // CmisShared
  //-------------------------------------------------------------------------
  /**
   *
   * @param obj
   * @param cmisSession
   * @constructor
   */
  function CmisShared(obj, cmisSession, prefixes) {

    var prefixes = $.extend({}, DEFAULT_PREFIXES, prefixes);

    this.cmisSession = cmisSession;
    this.session = cmisSession.session;
    this.succinctProperties = obj.succinctProperties;

    var prop = obj.succinctProperties;

    defineReadonly(this, 'id',                prop[prefixes['cmis'] + 'objectId']);
    defineReadonly(this, 'sharedBy',          prop[prefixes['qshare'] + 'sharedBy']);
    defineReadonly(this, 'sharedId',          prop[prefixes['qshare'] + 'sharedId']);
    defineReadonly(this, 'sharedExpiryDate',  prop[prefixes['qshare'] + 'expiryDate']);
  }

  var session = null;
  
  return {
    createSession: function(options) {
      session = new CmisSession(options);
      return session;
    }
  }
});
