/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('cmis/cmisbrowser', ['jquery', 'cmis/confluence-error-handler'], function($, errorHandler) {

  var DEFAULT_MAX_WIDTH = 650;
  var DEFAULT_MIN_WIDTH = 26;

  function CmisBrowser(options) {
    this.options = options;
    this.cmis_session = options.cmis_session;
    this.el = options.el;
    this.folderOnly = options.folderOnly || false; // フォルダのみ表示するか
    this.rootFolder = options.rootFolder;          // ルートフォルダ
    this.multipleSelect = options.multipleSelect || false; // 複数選択を許可するか
    this.allowFolderSelect = options.allowFolderSelect; // フォルダの選択を許可するか
    this.allowFileSelect = options.allowFileSelect; // ファイルの差選択を許可するか
    this.selectedNodeId = null;
    this.width = options.width || undefined;
    this.maxWidth = options.maxWidth || undefined;
    this.minWidth = options.minWidth || undefined;
    this.resizable = options.resizable || false;
    this.collapsed = options.collapsed || false;
    this.tooltip = options.tooltip || false;
    this.onSidebarResized = options.onSidebarResized || function () {};
    this._init();
  }
  
  CmisBrowser.prototype._init = function() {
    var self = this;
    var onLoad = function(rootFolder) {

      var treedata = self._cmisobjects2nodedata(rootFolder);

      self.el.tree({
        data: treedata,
        //buttonLeft: false,
        onCreateLi: function(node, $li) {
          if (node.dummy) {
            $li.find('.jqtree-title').before('<span class="aui-icon aui-icon-wait"></span>');
          } else if (node.cmisdata.baseTypeId == "cmis:folder") {
            $li.addClass('cmis-folder');
            $li.find('.jqtree-title').before('<span class="' + (node.clazz || "aui-icon aui-icon-small aui-iconfont-devtools-folder-closed") + '"></span>');
          } else {
            // TODO 作成者、サイズ、ダウンロードリンク？を表示
            $li.addClass('cmis-file');
            $li.find('.jqtree-title').before('<span class="aui-icon aui-icon-small aui-iconfont-doc"></span>');
          }
          $('span.jqtree-title', $li).addClass('aui-nav-item-label');
          // ツールチップ
          if (self.tooltip) {
            try { AJS.$('.aui-nav-item', $li).tooltip({gravity: 'w'}); } catch (e) {};
          }
        }
      });
      
      // 
      self.el.bind('tree.open', $.proxy(self._loadChildren, self));
      
      //
      if (self.multipleSelect) {
        self.el.bind('tree.click', $.proxy(self._multipleSelect, self));
      } else {
        self.el.bind('tree.click', $.proxy(self._singleSelect, self));
      }
    }
    
    if (typeof this.rootFolder == 'undefined') {
      this.cmis_session.getObjectByPath('/').done(onLoad);
    } else if (typeof this.rootFolder == 'string') {
      this.cmis_session.getObjectByPath(this.options.rootFolder).done(onLoad);
    } else if (typeof this.rootFolder == 'object') {
      onLoad(this.rootFolder);
    }

    // サイドバー
    self._makeSidebar();

  }

  CmisBrowser.prototype._makeSidebar = function() {
    var self = this;
    var width = self.width || AJS.$(".cmis-sidebar").css("width");
    var maxWidth = self.maxWidth || DEFAULT_MAX_WIDTH;
    var minWidth = self.minWidth || DEFAULT_MIN_WIDTH;

    if (self.collapsed) {
      AJS.$(".cmis-sidebar").attr('aria-expanded', false);
      AJS.$(".cmis-sidebar").css("width", minWidth);
    }

    if (self.resizable) {
      try {
        AJS.$(".cmis-sidebar").resizable("destroy");
      } catch (e) {}
      AJS.$(".cmis-sidebar").resizable({
        handles: {
          e: $(".cmis-sizer")
        },
        grid: [1, 1],
        start: function(event, ui) {
          $('.cmis-sizer').addClass('cmis-sizer-hover');
        },
        stop: function(event, ui) {
          $('.cmis-sizer').removeClass('cmis-sizer-hover');
        },
        resize: function(event, ui) {
          $('.cmis-sizer').addClass('cmis-sizer-hover');
          self.onSidebarResized(event, ui);
        },
        // stop: GH.DetailsView.storeWidth,
        // maxWidth: maxWidth,
        minWidth: minWidth
      });
    }

  }

  /**
   * フォルダの子要素を読み込む
   */
  CmisBrowser.prototype._loadChildren = function(e) {
    var self = this;
    if (e.node.already_loaded) {
      // すでにデータを読み込み済みの場合はスルー
      return ;
    }
    // 子要素を読み込む
    e.node.cmisdata.getChildren({'orderBy':'cmis:name','includeAllowableActions':true}).done(function(children) {
      self.el.tree('updateNode', e.node, {already_loaded:true});
      self.el.tree('removeNode', e.node.children[0]); // ダミーデータを削除
      
      var treedata = self._cmisobjects2nodedata(children);
      self.el.tree('loadData', treedata, e.node);
      var selectedNode = self.el.tree('getNodeById', self.selectedNodeId);
      self.el.tree('selectNode', selectedNode);
      self.selectedNodeId = null;
    }).fail(function(error) {
      self.el.tree('removeNode', e.node.children[0]); // ダミーデータを削除
      errorHandler.showError($(self.el), { body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.browser.render'), error: error });
    });
  }
  
  /**
   * 
   */
  CmisBrowser.prototype._multipleSelect = function(e) {
    // Disable single selection
    e.preventDefault();

    var selected_node = e.node;
    
    if (!this._isSelectableNode(selected_node)) {
      // 選択が許可されていないノード
      return ;
    }

    if (this.el.tree('isNodeSelected', selected_node)) {
        this.el.tree('removeFromSelection', selected_node);
    }
    else {
        this.el.tree('addToSelection', selected_node);
    }
  }
  
  /**
   * 
   */
  CmisBrowser.prototype._singleSelect = function(e) {
    // Disable single selection
    

    var selected_node = e.node;
    
    if (!this._isSelectableNode(selected_node)) {
      // 選択が許可されていないノード
      e.preventDefault();
      return ;
    }
  }
  
  /**
   * 選択できるノードか
   */
  CmisBrowser.prototype._isSelectableNode = function(node) {
    if (node.cmisdata.baseTypeId == "cmis:folder" && !this.allowFolderSelect) {
      // フォルダの選択が許可されていない
      return false;
    }
    if (node.cmisdata.baseTypeId == "cmis:doument" && !this.allowFileSelect) {
      // ファイルの選択が許可されていない
      return false;
    }
    return true;
  }
  
  /**
   * CmisObjectをjqTree用のデータに変換
   */
  CmisBrowser.prototype._cmisobject2nodedata = function(obj) {
    var result = (obj.formatter && typeof obj.formatter === 'function') ? obj.formatter(obj) : {label: obj.name, id: obj.id, cmisdata: obj, clazz: obj.clazz}
    if (result.cmisdata.baseTypeId == 'cmis:folder' && !result.already_loaded) {
      result.already_loaded = false;
      result.children = [{label: 'Loading...', dummy: true}]; // ダミーの子要素を入れておかないとopenイベントが来ない
    }
    return result;
  }
  
  /**
   * CmisObjectの配列をjqTree用のデータ配列に変換
   */
  CmisBrowser.prototype._cmisobjects2nodedata = function(ary) {
    var self = this,
      nodedata = [];
    $.each(ary, function (i, child) {
      if ($.isArray(child)) {
        Array.prototype.push.apply(nodedata, self._cmisobjects2nodedata(child));
      } else if (self.folderOnly && (!child || child.baseTypeId != 'cmis:folder')) {
        return;
      } else {
        nodedata.push(self._cmisobject2nodedata(child));
      }
    });
    return nodedata;
  }
  
  /**
   * 選択されているノードを選択します
   */
  CmisBrowser.prototype.addToSelection = function(id) {
    var node = this.el.tree('getNodeById', id);
    return this.el.tree('addToSelection', node);
  }
  
  /**
   * 選択されているノードを取得します
   */
  CmisBrowser.prototype.getSelectedNode = function() {
    return this.el.tree('getSelectedNode'); 
  }
  
  /**
   * 選択されているノード一覧を取得します
   */
  CmisBrowser.prototype.getSelectedNodes = function() {
    return this.el.tree('getSelectedNodes'); 
  }
  
  /**
   * 選択されているノードを削除します。
   */
  CmisBrowser.prototype.removeSelectedNode = function() {
    var node = this.getSelectedNode();
    var parentNode = node.parent;
    this.el.tree('removeNode', node);
    this.el.tree('selectNode', parentNode);
    return this;
  }
  
  /**
   * 選択されているノードに新たなオブジェクトを追加します。
   */
  CmisBrowser.prototype.appendObjectToSelectedNode = function(obj) {
    var node = this._cmisobject2nodedata(obj);
    var selectedNode = this.getSelectedNode();
    if (!selectedNode) {
      return;
    }
    this.el.tree('appendNode', node, selectedNode);
    return this;
  }

  /**
   * 指定されたノードIDのノードを取得する
   * @param selected
   * @param current
   */
  CmisBrowser.prototype.getNodeById = function(nodeId) {
    return this.el.tree('getNodeById', nodeId);
  }

  /**
   * 指定されたノードを選択してオープンします。
   */
  CmisBrowser.prototype.selectOpenNode = function (selected, current) {
    var self = this;
    var currentNode = this.el.tree('getNodeById', current.id);
    if (currentNode) {
      this.el.tree('openNode', currentNode);
    }
    var selectedNode = this.el.tree('getNodeById', selected.id);
    if (selectedNode) {
      this.el.tree('selectNode', selectedNode);
    } else {
      this.selectedNodeId = selected.id;
    }
    return this;
  }

  /**
   * 指定されたノードを選択してオープンします。
   */
  CmisBrowser.prototype.selectOpenNode = function (selected, current) {
    var self = this;
    var currentNode = this.el.tree('getNodeById', current.id);
    if (currentNode) {
      this.el.tree('openNode', currentNode);
      var selectedNode = this.el.tree('getNodeById', selected.id);
      if (selectedNode) {
        this.el.tree('selectNode', selectedNode);
      } else {
        this.selectedNodeId = selected.id;
      }
    }
    return this;
  }
  
  /**
   * 指定されたノードを削除します。
   */
  CmisBrowser.prototype.removeNode = function(obj) {
    var node = this.el.tree('getNodeById', obj.id);
    var parentNode = node.parent;
    var selectedNode = this.getSelectedNode();
    this.el.tree('removeNode', node);
    if (node.id === selectedNode.id) {
      this.el.tree('selectNode', parentNode);
    }
    return this;
  }
  
  /**
   * 
   */
  CmisBrowser.prototype.bind = function(event, fn) {
    this.el.bind(event, fn);
    return this;
  }

  /**
   *
   */
  CmisBrowser.prototype.setWidth = function (width) {
    AJS.$(".cmis-sidebar").css("width", width);
    return this;
  }
  
  return {
    /**
     * @param {object} options
     */
    setup: function(options) {
      return new CmisBrowser(options);
    }
  }
});
