/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('cmis/cmisuploader', ['jquery', 'cmis/cmisconfluence', 'cmis/confluence-error-handler', 'attachments/attachment-permission'], function($, cmisConfluence, errorHandler, attachmentPermission) {
  var DIALOG_ID = 'cmisuploader-dialog';
  var DIALOG_SECTION = '<section role="dialog" id="'+DIALOG_ID+'" class="aui-layer aui-dialog2 aui-dialog2-xlarge loadingInProgress" aria-hidden="true" data-aui-modal="true"></section>';
  
  var cmis_session_;
  var major_;
  var targetFiles_;
  var uploadFiles_;
  var uploadFolder_;
  var uploaderDialog_;
  var uploaderDialogEl_;
  
  function upload(options) {
    var d = new $.Deferred;
    cmis_session_ = options.cmis_session;
    targetFiles_ = options.targetFiles;
    uploadFolder_ = options.uploadFolder;
    uploadFiles_ = options.uploadFiles;
    major_ = options.major || false;

    var ctx = {
        uploadedDocuments: [],
        onSuccessFile: options.onSuccessFile || function() {
        },
        onErrorFile: options.onErrorFile || function() {
        }
      }
    
    // ダイアログを表示
    showDialog(d, ctx);
    
    // アップロード処理実行
    runUpload(d, ctx);
    
    return d.promise();
  }


  /**
   * ダイアログの表示
   */
  function showDialog(d, ctx) {
    
    if ($('#'+DIALOG_ID).length <= 0) {
      $('body').append(DIALOG_SECTION);
    }
    
    var param = {
      uploadFiles: uploadFiles_
    };
    
    var template = JIRA.Templates.cmisuploader.uploaderDialog(param);
    
    uploaderDialogEl_ = $('#'+DIALOG_ID).empty().append(template);
    uploaderDialogEl_.find('.closeButton').click(function() {
      // アップロードダイアログを閉じる
      closeDialog();
    });
    
    uploaderDialog_ = AJS.dialog2('#'+DIALOG_ID);
    uploaderDialog_.show();
  }
  
  // アップロードダイアログを閉じる
  function closeDialog() {
    if (uploaderDialog_) { 
      uploaderDialog_.hide(); 
    }
  }
  
  
  function runUpload(d, ctx) {

    var currentIndex = 0;
    
    function proc() {

      if (currentIndex >= uploadFiles_.length) {
        // アップロード完了
        uploaderDialogEl_.find('.closeButton').prop('disabled', false);
        d.resolve(ctx.uploadedDocuments);
        return ;
      }
      
      // ステータスuploading
      $(getStatusId()).append('<span class="aui-icon aui-icon-wait">Uploading...</span>');


      var targetFile = targetFiles_ ? targetFiles_[currentIndex] : undefined;

      var uploadFile = uploadFiles_ ? uploadFiles_[currentIndex] : undefined;

      var targetFolder = uploadFolder_;

      // 更新対象のファイルが指定されている場合
      if (targetFile) {

        // 新規バージョン
        update(targetFile, uploadFile);


      } else {

        // 新規 or 更新
        targetFolder.getChildByName(uploadFile.name, false, {'includeAllowableActions': true}).done(function (targetFile) {

          if (targetFile) {

            // 同名のファイルが存在するので更新
            update(targetFile, uploadFile);

          } else {

            // 同名のファイルは存在しないので新規作成
            create(targetFolder, uploadFile);

          }

        }).fail(function (error) {

          setError(error);
          next();

        });

      }

    }
    
    function next() {
      currentIndex = currentIndex + 1;
      proc();
    }
    
    function getStatusId() {
      return '#upload-status-' + currentIndex;
    }
    
    function setSuccess() {
      $(getStatusId()).empty().append('<span class="aui-icon aui-icon-small aui-iconfont-success">Success</span>');
    }

    function setError(error) {
      if ("409" === error.response.status) {
        errorHandler.showError($(getStatusId()).empty(), {
          body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.uploader.locked'),
          error: error
        });
      } else {
        errorHandler.showError($(getStatusId()).empty(), {
          body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.uploader'),
          error: error
        });
      }
    }

    function create(targetFolder, uploadFile) {

      if (!attachmentPermission.isEnableUploadForCreate(targetFolder)) {

        errorHandler.showError($(getStatusId()).empty(), {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.uploader.permission.create')});
        ctx.onErrorFile(uploadFile);
        next();

      }

      var input = $.extend({
          'cmis:name': uploadFile.name, 'cmis:objectTypeId': 'cmis:document'
        },
        cmisConfluence.getAuthorFor('create')
      );

      targetFolder.createDocument(uploadFile, input, "").done(function (doc) {

        setSuccess();
        ctx.onSuccessFile(uploadFile);
        ctx.uploadedDocuments.push(doc);
        next();

      }).fail(function (error) {

        setError(error);
        ctx.onErrorFile(uploadFile);
        next();

      });

    }

    function update(targetFile, uploadFile) {

      (function () {

        if (targetFile.isPrivateWorkingCopy || !targetFile.versionSeriesCheckedOutId) {

          var d = new AJS.$.Deferred;

          d.resolve(targetFile);

          return d.promise();

        }

        return cmisConfluence.getDocumentById(targetFile.versionSeriesCheckedOutId);

      })().pipe(function (targetFile) {

        if (!attachmentPermission.isEnableUploadForUpdate(targetFile)) {

          errorHandler.showError($(getStatusId()).empty(), {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.uploader.permission.update')});
          ctx.onErrorFile(uploadFile);
          next();

        }

        var major = major_;

        var input = cmisConfluence.getAuthorFor('update');

        if (targetFile.isPrivateWorkingCopy) {

          return targetFile.checkIn(major, input, uploadFile);

        }

        return targetFile.update(uploadFile, input, major);

      }).pipe(function (doc) {

        setSuccess();
        ctx.onSuccessFile(uploadFile);
        ctx.uploadedDocuments.push(doc);
        next();

      }, function (error) {

        setError(error);
        ctx.onErrorFile(uploadFile);
        next();

      });

    }

    proc();
  }
  
  return {
    upload: upload
  }
});
