/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('alfresco/main',
    ['jquery', 'cmis/cmisconfluence', 'cmis/cmisbrowser', 'cmis/cmislistview', 'cmis/confluence-listview-helper', 'cmis/confluence-error-handler', 'attachments/attachment-navigation', 'attachments/attachment-permission'],
    function($, cmisConfluence, treeview, listview, listviewHelper, errorHandler, attachmentNavigation, attachmentPermission) {'use strict'
 
  var CONTAINER_ID = 'cmis-container';
  var HIDDEN_CLASS = 'hidden';
  var LOADING_CLASS = "loading-documents";
  
  var cmis_session,
      serverId,
      spaceId,
      pageId,
      publicFolderId,
      privateFolderId,
      permissions,
      expListView,
      expTreeView,
      navigation;
  
  /**
   * 初期化します
   * @param context
   * @return {Deferred}
   */
  function initializeFolder(context) {
    var d = new AJS.$.Deferred;
    
    (function() {
      
      var d = new AJS.$.Deferred;

      if ($('#' + CONTAINER_ID, context).length) {
        d.reject();
        return d.promise();
      }

      $("." + LOADING_CLASS, context).removeClass(HIDDEN_CLASS);

      $(context).append(JIRA.Templates.attachments.container({ resizable: true }));

      cmisConfluence.initPageInfo().done(function(info) {
        d.resolve(info);
      }).fail(function(error) {
        errorHandler.showError(context, {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.info'), error: error});
        d.reject();
      });
      
      return d.promise();
      
    })().pipe(function(data) {
      
      var d = new AJS.$.Deferred;
      
      // フォルダマッピングが設定されていない場合はCMISのセッションを作成しない
      if (!data.isFederatable()) {
        d.reject(data);
        return d.promise();
      }
      
      cmisConfluence.initCmisSession(data, true).done(function() {
        
        // ライセンスが無効な場合は表示しない
        if (cmisConfluence.isInvalidLicense()) {
          errorHandler.showError(context, {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.license')});
          d.reject();
          return d.promise();
        }

        // ユーザーへ認証登録を促す
        if (attachmentPermission.shouldNavigateAccountRegistration(data.permissions)) {
          errorHandler.showError(context, {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.token') + AJS.I18n.getText('alfresco.for.confluence.ajax.error.go-to-profile', cmisConfluence.getUserProfilePanelUrl())});
          d.reject();
          return d.promise();
        }

        // 最大ログイン試行回数を超えた場合はユーザーにログイン確認とリセットを促す
        if (attachmentPermission.isOverMaximumLoginAttemptAllowed(data.permissions)) {
          errorHandler.showError(context, {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.reset-failed-login-count', cmisConfluence.getUserProfilePanelUrl())});
          d.reject();
          return d.promise();
        }

        // 最大ログイン試行回数を超えた場合（admin profile時）
        if (cmisConfluence.isOverMaximumLoginAttemptAllowed()) {
          errorHandler.showError(context, {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.session')});
          d.reject();
          return d.promise();
        }

        cmis_session = cmisConfluence.getCmisSession();
        serverId = data.serverId;
        spaceId = data.spaceId;
        pageId = data.pageId;
        permissions = $.extend(data.permissions, { document_createissue: false });

        d.resolve(data);
      }).fail(function(error) {
        errorHandler.showError(context, {body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.session'), error: error});
        d.reject();
      });
      
      return d.promise();
      
    }).pipe(function(data) {
      
      var d = new AJS.$.Deferred;
      
      if (!getPublicFolderId()) {
        d.resolve(data, null);
        return d.promise();
      }
      
      cmis_session.getFolder(getPublicFolderId(), false, {'includeAllowableActions':true}).done(function(publicFolder) {
        d.resolve(data, publicFolder);
      }).fail(function(error) {
        errorHandler.showError($("#" + CONTAINER_ID, context), { body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.folder.public') + getPublicFolderId(), error: error });
        d.reject();
      });
      
      return d.promise();
      
    }).pipe(function(data) {

      expListView = listview.setup({
        cmis_session: cmis_session,
        el:$('.cmis-listbrowser', context),
        helper: listviewHelper.create(),
        alfrescoUrl:cmisConfluence.getAlfrescoUrl(),
        dateformat:'YYYY/MM/DD',
        autoReloadMode:true,
        permissions: permissions,
        onFolderCreate: doFolderCreate,
        onFolderRemove: doFolderRemove,
        onFolderSelected: onFolderSelected
      });

      navigation = attachmentNavigation.setup({
        cmis_session: cmis_session,
        data: data,
        listView: expListView
      });

      navigation.loadFolders({
        'all': data.advancedConfiguration.enableAllFiles,
        'editing': data.advancedConfiguration.enableImEditing,
        'recent-modified': data.advancedConfiguration.enableRecentlyModified,
        'recent-added': data.advancedConfiguration.enableRecentlyAdded,
        'favorite': attachmentPermission.isEnableFavorite(permissions),
        'shared-links': attachmentPermission.isEnableSharedLinks(permissions),
        'link': {
          getChildren: function () {
            return expListView.getDocumentByPage();
          }
        }
      }).done(function (folders) {

        if ((folders['public'] && folders['public'].length > 0) || folders['private']) {

          expTreeView = treeview.setup({
            cmis_session: cmis_session,
            el: $('.cmis-treebrowser', context),
            rootFolder: folders.asArray(),
            folderOnly: true,
            allowFolderSelect: true,
            tooltip: true,
            collapsed: false,
            resizable: true,
            onSidebarResized: function(e, ui) {
              expListView.searchFilterModule.ajustSearchFieldWidth(e, ui);
            }
          });
          expTreeView.bind('tree.select', function (e) {
            var node = e.node;
            if (!node || !node.cmisdata) {
              return;
            }
            expListView.setPageNum(0);
            navigation.loadByFolder(node);
            expListView.initUpload();
          });
          expTreeView.addToSelection(attachmentNavigation.LINK_FOLDER_ID);

        } else {

          $('.cmis-sidebar', context).remove();

        }

        expListView.setRootFolder(folders['link']);
        expListView.renderFolderInfo(folders['link']);
        expListView.loadByFolder(folders['link']);
        expListView.initUpload();

      });
      
      d.resolve();
      
    }).fail(function(error) {
      
      d.reject();
      
    }).always(function() {

      $("." + LOADING_CLASS, context).addClass(HIDDEN_CLASS);
    
    });
    
    return d.promise();
  }
  
  function doFolderCreate(folder, created, listview) {
    cmisConfluence.getDocumentById(created.id).done(function(doc) {
      expTreeView.appendObjectToSelectedNode(doc);
    });
  }
  
  function doFolderRemove(folder, removed, listview) {
    expTreeView.removeNode(removed);
  }
  
  function onFolderSelected(selected, current, listview) {
    if ($.type(selected) != "string" && expTreeView.getNodeById(selected.id)) {
      // ツリービューに表示対象のフォルダが存在する場合
      expTreeView.selectOpenNode(selected, current);
    } else {
      // ツリービューに表示対象のフォルダが存在しない場合（リンクフォルダを開いた場合、親フォルダをクリックした場合）
      cmisConfluence.getDocumentById($.type(selected) != "string" ? selected.id : selected).done(function (doc) {
        expListView.setPageNum(0);
        navigation.loadByFolder(expTreeView._cmisobject2nodedata(doc));
        expListView.initUpload();
      });
    }
  }
  
  /**
   * 添付ファイルのパブリック・フォルダのIDを取得します
   * @return {String}
   */
  function getPublicFolderId() {
    return publicFolderId;
  }
  
  /**
   * 添付ファイルのプライベート・・フォルダのIDを取得します
   * @return {String}
   */
  function getPrivateFolderId() {
    return privateFolderId;
  }
  
  /**
   * 添付ファイルのプライベート・フォルダのパスを取得します
   * @return {String}
   */
  function getPrivatePath(privateFolder) {
    return privateFolder ? privateFolder.path + "/" + serverId + "/" + spaceId + "/" + pageId : null;
  }
  
  // function getHtmlMeta(name) {
  //   return $("meta[" + "name=" + name + "]").attr("content");
  // }
  
  return {
    initializeFolder: initializeFolder
  };
});

AJS.toInit(function() {
  require(['alfresco/main'], function(main) {
    main.initializeFolder($("#alfresco-attachments-panel"));
  });
});
