/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('attachments/attachment-navigation',
  ['jquery', 'cmis/cmisconfluence', 'cmis/confluence-listview-helper', 'cmis/confluence-error-handler', 'attachments/attachment-permission'],
  function ($, cmisConfluence, listviewHelper, errorHandler, attachmentPermission) {
    'use strict'

    var LINK_FOLDER_ID = "";
    var FAVORITE_FOLDER_ID = "__favorite__";
    var SHARED_LINKS_FOLDER_ID = "__shared-links__";
    var ALL_DOCUMENTS_FOLDER_ID = "__all-documents__";
    var EDITING_DOCUMENTS_FOLDER_ID = "__editing-documents__";
    var RECENT_MODIFIED_FOLDER_ID = "__recent-modified__";
    var RECENT_ADDED_FOLDER_ID = "__recent-added__";

    function CmisNavigation(options) {
      this.cmis_session = options.cmis_session;
      this.listView = options.listView;
      this.data = options.data;
      this.permissions = options.data ? options.data.permissions : {};
      this.publicFolderId = options.data ? options.data.publicFolder : undefined;
      this.privateFolderId = options.data ? options.data.privateFolder : undefined;
      this.publicFolder = undefined;
      this.privateFolder = undefined;
    }

    /**
     * ナビゲーションに表示するメニュー(フォルダー)を読み込んで作成する
     * @param options Object { 'public': true, 'private': true, 'link': true, 'all': true, 'editing': true, 'favorite': true, 'shared-links': true, 'recent-modified': true, 'recent-added': true, }
     * @returns {Deferred}
     */
    CmisNavigation.prototype.loadFolders = function(options) {

      var self = this;

      return (function() {

        var d = new AJS.$.Deferred;

        if (attachmentPermission.isEnableBrowsePanel(self.permissions)) {
          d.reject();
          return d.promise();
        }

        if ((!self.publicFolderId || self.publicFolderId.length <= 0) || attachmentPermission.isEnableBrowsePanelWithoutAccount(self.permissions)) {
          d.resolve(self.data, null);
          return d.promise();
        }

        self.cmis_session.getFolders(self.publicFolderId, false, {'includeAllowableActions':true}).done(function(publicFolders) {
          d.resolve(self.data, publicFolders);
        }).fail(function(error) {
          errorHandler.showError($("#" + CONTAINER_ID, context), { body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.folder.public') + self.publicFolderId, error: error });
          d.reject();
        });

        return d.promise();

      })().pipe(function(data, publicFolders) {

        var d = new AJS.$.Deferred;

        if (!self.privateFolderId || attachmentPermission.isEnableBrowsePanelWithoutAccount(self.permissions)) {
          d.resolve(data, publicFolders, null);
          return d.promise();
        }

        self.cmis_session.getFolder(self.privateFolderId, false, {'includeAllowableActions':true}).done(function(privateFolder) {
          d.resolve(data, publicFolders, privateFolder);
        }).fail(function(error) {
          errorHandler.showError($("#" + CONTAINER_ID, context), { body: AJS.I18n.getText('alfresco.for.confluence.ajax.error.folder.private') + self.privateFolderId, error: error });
          d.reject();
        });

        return d.promise();

      }).pipe(function(data, publicFolders, privateFolder) {

        var d = new AJS.$.Deferred;

        if (!getPrivatePath(privateFolder) || attachmentPermission.isEnableBrowsePanelWithoutAccount(self.permissions)) {
          d.resolve(data, publicFolders, null);
          return d.promise();
        }

        self.cmis_session.mkdir(getPrivatePath(privateFolder), false, {'includeAllowableActions':true}).done(function(privateFolder) {
          d.resolve(data, publicFolders, privateFolder);
        }).fail(function(error) {
          // Ignore HTTP 401(Unauthorized)
          d.resolve(data, publicFolders, null);
        });

        return d.promise();

      }).pipe(function(data, publicFolders, privateFolder) {

        var d = new AJS.$.Deferred;
        self.publicFolders = publicFolders;
        self.privateFolder = privateFolder;
        var folders = self.createFolders(options);
        return d.resolve(folders).promise();

      });

      /**
       * 添付ファイルのプライベート・フォルダのパスを取得します
       * @return {String}
       */
      function getPrivatePath(privateFolder) {
        return privateFolder ? privateFolder.path + "/" + self.data.serverId + "/" + self.data.projectId + "/" + self.data.pageId : null;
      }

    }

    /**
     * ナビゲーションに表示するメニュー(フォルダー)を作成する
     * @param options Object { 'public': true, 'private': true, 'link': true, 'all': true, 'editing': true, 'favorite': true, 'shared-links': true, 'recent-modified': true, 'recent-added': true, }
     * @returns {{}}
     */
    CmisNavigation.prototype.createFolders = function(options) {

      var self = this;

      var folders = {};
      if (attachmentPermission.isEnableBrowsePanelWithoutAccount(self.permissions)) {
        folders['link'] = createLinkFolder(self.listView, options);
        folders.asArray = function() {
          return [folders['link']];
        };
        return folders;
      }
      var enables = $.extend(true, {
        'public': true, 'private': true, 'link': true, 'all': true, 'editing': true, 'favorite': true, 'shared-links': true, 'recent-modified': true, 'recent-added': true,
      }, options);
      if (enables['public'] && self.publicFolders && self.publicFolders.length > 0) {
        folders['public'] = self.publicFolders.map(function(val) {
          return $.proxy(createFolder, self)(val);
        });
      }
      if (enables['private'] && self.privateFolder && self.data.privateFolderLabel) {
        folders['private'] = $.proxy(createFolder, self)(self.privateFolder, {label: self.data.privateFolderLabel});
      }
      if (enables['link']) {
        folders['link'] = createLinkFolder(self.listView, options);
      }
      if (enables['all']) {
        folders['all'] = createAllDocumentsFolder(self.listView, folders);
      }
      if (enables['editing']) {
        folders['editing'] = createEditingDocumentsFolder(self.listView, folders);
      }
      if (enables['favorite']) {
        folders['favorite'] = createFavoriteFolder();
      }
      if (enables['shared-links']) {
        folders['shared-links'] = createSharedLinksFolder();
      }
      if (enables['recent-modified']) {
        folders['recent-modified'] = createRecentFolder(self.listView, folders, { id: RECENT_MODIFIED_FOLDER_ID, label: AJS.I18n.getText('alfresco.for.confluence.listview.recentmodified') }, "cmis:lastModificationDate");
      }
      if (enables['recent-added']) {
        folders['recent-added'] = createRecentFolder(self.listView, folders, { id: RECENT_ADDED_FOLDER_ID, label: AJS.I18n.getText('alfresco.for.confluence.listview.recentadded') }, "cmis:creationDate");
      }
      folders.asArray = function() {
        var foldersAsArray = [folders['link'], folders['all'], folders['recent-modified'], folders['recent-added'], folders['private'], folders['public']];
        if (cmisConfluence.getAccessType() === 'WithOwnProfile') {
          foldersAsArray.splice(2, 0, folders['editing']);
          foldersAsArray.splice(5, 0, folders['favorite']);
          foldersAsArray.splice(6, 0, folders['shared-links']);
        }
        return removeEmpty(foldersAsArray);
      };
      return folders;
    }

    /**
     * ナビゲーションのメニュー(フォルダー)ごとのドキュメントリストの表示処理
     * @param node
     */
    CmisNavigation.prototype.loadByFolder = function(node) {
      var self = this;
      var cmisdata = node.cmisdata;
      // "リンク済み"、"お気に入り"、"共有リンク"、"すべてのファイル"、"編集中のファイル"フォルダの場合は、ソートの初期値を設定してからcmisdataからロードする。
      if ($.inArray(node.id, [LINK_FOLDER_ID, FAVORITE_FOLDER_ID, SHARED_LINKS_FOLDER_ID, ALL_DOCUMENTS_FOLDER_ID, EDITING_DOCUMENTS_FOLDER_ID]) != -1) {
        self.listView.sortColumn_ = "cmis:creationDate";
        self.listView.sortOrder_ = "DESC";
        // "最近の編集"フォルダの場合は、ソートの初期値を設定してからcmisdataからロードする。
      } else if ($.inArray(node.id, [RECENT_MODIFIED_FOLDER_ID]) != -1) {
        self.listView.sortColumn_ = "cmis:lastModificationDate";
        self.listView.sortOrder_ = "DESC";
        // "最近の追加"フォルダの場合は、ソートの初期値を設定してからcmisdataからロードする。
      } else if ($.inArray(node.id, [RECENT_ADDED_FOLDER_ID]) != -1) {
        self.listView.sortColumn_ = "cmis:creationDate";
        self.listView.sortOrder_ = "DESC";
        // パブリック、プライベートフォルダの場合は、アクション制御のためにラップ、ソートの初期値を設定してからcmisdataからロードする。
      } else {
        var publicFolderIds = self.publicFolders ? self.publicFolders.map(function(val) {
          return val.id;
        }) : [];
        self.listView.sortColumn_ = "cmis:lastModificationDate";
        self.listView.sortOrder_ = "DESC";
        // パブリックのルートフォルダの場合は、ドキュメント検索、フォルダ削除を無効化する。
        if ($.inArray(node.id, publicFolderIds) != -1) {
          cmisdata = $.proxy(createFolder, self)(node.cmisdata, {
            actions: {
              folder_create: true,
              folder_remove: false,
              document_create: true,
              document_search: false,
              document_sort: true
            }
          });
          // プライベートのルートフォルダの場合は、ドキュメント検索、フォルダ削除を無効化する。ラベルを指定する。
        } else if (self.privateFolder && node.id == self.privateFolder.id) {
          cmisdata = $.proxy(createFolder, self)(node.cmisdata, {
            actions: {
              folder_create: true,
              folder_remove: false,
              document_create: true,
              document_search: false,
              document_sort: true
            },
            label: self.data.privateFolderLabel
          });
          // サブフォルダの場合は、ドキュメント検索を無効化する、フォルダ削除を有効化する。
        } else {
          cmisdata = $.proxy(createFolder, self)(node.cmisdata, {
            actions: {
              folder_create: true,
              folder_remove: true,
              document_create: true,
              document_search: false,
              document_sort: true
            }
          });
        }
      }
      self.listView.renderFolderInfo(cmisdata);
      self.listView.loadByFolder(cmisdata);
    }

    /**
     * 添付ファイルのフォルダを取得します。
     * @return {Object}
     */
    function createFolder(folder, options) {
      var self = this;
      return folder ? $.extend({}, folder, {
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            id: obj.id,
            label: obj.name,
            clazz: "aui-icon aui-icon-small aui-iconfont-devtools-folder-closed"
          }, options);
        },
        root: (function() {
          // パブリックのルートフォルダの場合
          if ($.inArray(folder.id, self.publicFolders ? self.publicFolders.map(function(val) { return val.id; }) : []) != -1) {
            return true
            // プライベートのルートフォルダの場合
          } else if (self.privateFolder && folder.id == self.privateFolder.id) {
            return true;
          }
          return false;
        })()
      }) : createEmptyFolder();
    }

    /**
     * 添付ファイルのリンクされたフォルダを取得します。
     * @return {Object}
     */
    function createLinkFolder(listView, options) {
      return $.extend({}, {
        baseTypeId: 'cmis:folder',
        allowableActions: {},
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            id: LINK_FOLDER_ID,
            label: AJS.I18n.getText('alfresco.for.confluence.listview.linked'),
            actions: {
              folder_create: false,
              folder_remove: false,
              document_create: false,
              document_search: false,
              document_sort: false
            },
            clazz: "aui-icon aui-icon-small aui-iconfont-devtools-tag",
            already_loaded: true
          }, options);
        },
        getChildren: function () {
          if (options['link'].getChildren) {
            return options['link'].getChildren();
          } else {
            return listviewHelper.create().getDocumentByIds(listView.getLinkObjectKeys());
          }
        }
      });
    }

    /**
     * Alfrescoのお気に入りフォルダを取得します。
     * @return {Object}
     */
    function createFavoriteFolder(options) {
      return $.extend({}, {
        baseTypeId: 'cmis:folder',
        id: FAVORITE_FOLDER_ID,
        allowableActions: {},
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            id: FAVORITE_FOLDER_ID,
            label: AJS.I18n.getText('alfresco.for.confluence.listview.favorite'),
            actions: {
              folder_create: false,
              folder_remove: false,
              document_create: false,
              document_search: false,
              document_sort: false
            },
            clazz: "aui-icon aui-icon-small aui-iconfont-star",
            already_loaded: false
          }, options);
        },
        getChildren: function (options) {
          var d = new AJS.$.Deferred;
          var ids = [];
          var opt = $.extend(options, {"folder": true, "file": true});

          cmisConfluence.getFavorites(opt).done(function (favorites) {
            $.each(favorites.list.entries, function (i, o) {
              var entryObject = o.entry.target.file || o.entry.target.folder;
              ids.push(entryObject.id);
            });

            cmisConfluence.getCmisSession().getObjectByIds(ids, false, {'includeAllowableActions': true}).done(function (objects) {

              // getFavoritesで取得した順に並び替え
              var sortedObjects = [];
              ids.forEach(function (id) {
                objects.forEach(function (obj) {
                  if (id == obj.id || id == obj.versionSeriesId) {
                    sortedObjects.push(obj);
                  }
                });
              });

              var pagination = favorites.list.pagination;
              d.resolve(sortedObjects, pagination.hasMoreItems, pagination.totalItems);
            });
          });
          return d.promise();
        }
      });
    }

    /**
     * Alfrescoの共有リンクフォルダを取得します。
     * @return {Object}
     */
    function createSharedLinksFolder(options) {
      return $.extend({}, {
        baseTypeId: 'cmis:folder',
        id: SHARED_LINKS_FOLDER_ID,
        allowableActions: {},
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            id: SHARED_LINKS_FOLDER_ID,
            label: AJS.I18n.getText('alfresco.for.confluence.listview.sharedlinks'),
            actions: {
              folder_create: false,
              folder_remove: false,
              document_create: false,
              document_search: false,
              document_sort: false
            },
            clazz: "aui-icon aui-icon-small aui-iconfont-share",
            already_loaded: true
          }, options);
        },
        getChildren: function (options) {
          var d = new AJS.$.Deferred;
          var ids = [];

          cmisConfluence.getSharedLinks(options).done(function (sharedLinks) {
            $.each(sharedLinks.list.entries, function (i, o) {
              ids.push(o.entry.nodeId);
            });

            cmisConfluence.getCmisSession().getObjectByIds(ids, false, {'includeAllowableActions': true}).done(function (objects) {

              // getFavoritesで取得した順に並び替え
              var sortedObjects = [];
              ids.forEach(function (id) {
                objects.forEach(function (obj) {
                  if (id == obj.id || id == obj.versionSeriesId) {
                    sortedObjects.push(obj);
                  }
                });
              });

              var pagination = sharedLinks.list.pagination;
              d.resolve(sortedObjects, pagination.hasMoreItems, pagination.totalItems);
            });
          });
          return d.promise();
        }
      });
    }

    /**
     * 全ての文書フォルダを取得します。
     * @return {Object}
     */
    function createAllDocumentsFolder(listView, folders, options) {
      return $.extend({}, {
        baseTypeId: 'cmis:folder',
        allowableActions: {},
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            id: ALL_DOCUMENTS_FOLDER_ID,
            label: AJS.I18n.getText('alfresco.for.confluence.listview.alldocuments'),
            actions: {
              folder_create: false,
              folder_remove: false,
              document_create: false,
              document_search: true,
              document_sort: true
            },
            clazz: "aui-icon aui-icon-small aui-iconfont-devtools-submodule",
            already_loaded: true
          }, options);
        },
        getChildren: function () {
          return listView.quickSearch(
            removeEmpty([ folders['public'], folders['private'] ]),
            listView.getSearchKeyword(),
            listView.getSortColumn(),
            listView.getSortOrder()
          );
        }
      });
    }

    /**
     * 編集中の文書フォルダを取得します。
     * @return {Object}
     */
    function createEditingDocumentsFolder(listView, folders, options) {
      return $.extend({}, {
        baseTypeId: 'cmis:folder',
        allowableActions: {},
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            id: EDITING_DOCUMENTS_FOLDER_ID,
            label: AJS.I18n.getText('alfresco.for.confluence.listview.editingdocuments'),
            actions: {
              folder_create: false,
              folder_remove: false,
              document_create: false,
              document_search: true,
              document_sort: true
            },
            clazz: "aui-icon aui-icon-small aui-iconfont-locked",
            already_loaded: true
          }, options);
        },
        getChildren: function () {
          return listView.editingSearch(
            removeEmpty([ folders['public'], folders['private'] ]),
            listView.getSearchKeyword(),
            listView.getSortColumn(),
            listView.getSortOrder()
          );
        }
      });
    }

    /**
     * 最近の修正/追加した文書フォルダを取得します。
     * @return {Object}
     */
    function createRecentFolder(listView, folders, options, column) {
      return $.extend({}, {
        baseTypeId: 'cmis:folder',
        allowableActions: {},
        formatter: function (obj) {
          return $.extend({
            cmisdata: obj,
            actions: {
              folder_create: false,
              folder_remove: false,
              document_create: false,
              document_search: true,
              document_sort: true
            },
            clazz: "aui-icon aui-icon-small aui-iconfont-edit",
            already_loaded: true
          }, options);
        },
        getChildren: function () {
          return listView.recentlyInTrees(
            removeEmpty([ folders['public'], folders['private'] ]),
            listView.getSearchKeyword(),
            column,
            listView.getSortColumn(),
            listView.getSortOrder()
          );
        }
      });
    }

    /**
     * 空フォルダを取得します。
     * @return {Object}
     */
    function createEmptyFolder() {
      return {
        id: '',
        name: '',
        actions: {},
        allowableActions: {},
        getChildren: function () {
          return d.resolve([]).promise();
        }
      };
    }

    /**
     * Arrayから空文字、null, undefinedなどを取り除きます。
     * @param array
     * @returns {*}
     */
    function removeEmpty(array) {
      return $.grep(array, function(n, i){
        return (n !== "" && n != null);
      });
    }

    return {

      LINK_FOLDER_ID: "",
      FAVORITE_FOLDER_ID: "__favorite__",
      ALL_DOCUMENTS_FOLDER_ID: "__all-documents__",
      EDITING_DOCUMENTS_FOLDER_ID: "__editing-documents__",
      RECENT_MODIFIED_FOLDER_ID: "__recent-modified__",
      RECENT_ADDED_FOLDER_ID: "__recent-added__",

      /**
       * @param {object} options
       */
      setup: function(options) {
        return new CmisNavigation(options);
      }

    };

  });
