define('jquery/CMIS', ['jquery', 'misc/BlobUtil'], function($, blobUtil) {
  'use strict';
  
  /**
   * @class cmis
   * global object
   *
   *      var session = cmis.createSession(url);
   *
   */

  var lib = {};

  //check if we are running on node
  var isCommonJSModules = typeof module !== 'undefined' && module.exports;
  var isNode = isCommonJSModules && typeof window === 'undefined';
  if (isCommonJSModules) {
    module.exports = lib;
  }
  
  /**
   * イベントデータ
   */
  function EventData() {
  }
  EventData.prototype.type = undefined;
  EventData.prototype.data = undefined;
  EventData.prototype.once = false;
  EventData.prototype.handler = undefined;
  
  /**
   * @return {CmisSession}
   *
   */
  lib.createSession = function (url) {

    /**
     * @class CmisSession
     *
     * the session is the enrty point for all cmis requests
     * before making any request, session.loadRepository() must be invoked
     *
     */
    var session = {};    
    
    /**
     * sets token for authentication
     *
     * @param {String} token
     * @return {CmisSession}
     */
    session.setToken = function (token) {
      _defaultOptions.token = token;
      return session;
    };

    /**
     * sets credentials for authentication
     *
     * @param {String} username
     * @param {String} password
     * @return {CmisSession}
     */
    session.setCredentials = function (username, password) {
      _username = username;
      _password = password;
      return session;
    };

    /**
     * sets global handlers for non ok and error responses
     * @param {Function} notOk
     * @param {Function} error
     * @return {CmisSession}
     */
    session.setGlobalHandlers = function (notOk, error) {
      _globalNotOk = notOk || _noop;
      _globalError = error || _noop;
      return session;
    };

    /**
     * Connects to a cmis server and retrieves repositories,
     * token or credentils must already be set
     *
     * @param {Object} options
     * @return {CmisRequest} request
     */
    session.loadRepositories = function (ajaxOptions) {
      var d = new $.Deferred;
      _get(url).send(ajaxOptions).done(function (data) {
        for (var repo in data) {
          session.defaultRepository = data[repo];
          break;
        }
        session.repositories = data;

        d.resolve(data);
      }).fail(function() {
        d.reject.apply(this, arguments);
      });
      return d;
    };


    /**
     * gets an object by objectId
     *
     * @param {String} objectId
     * @param {String} returnVersion (if set must be one of 'this', latest' or 'latestmajor')
     * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includeACL, includePolicyIds, succinct, token)
     * @return {CmisRequest}
     */
    session.getObject = function (objectId, returnVersion, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'object';
      options.objectId = objectId;
      if (returnVersion == 'latestmajor' || returnVersion == 'latest') {
        options.returnVersion = returnVersion;
      }
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * gets an object by path
     *
     * @param {String} path
     * @param {Object} options
     * @return {CmisRequest}
     */
    session.getObjectByPath = function (path, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'object';

      return _get(session.defaultRepository.rootFolderUrl + path).query(options).send(ajaxOptions);
    };

    /**
     * creates a new folder
     *
     * @param {String} parentId
     * @param {String/Object} input
     * if `input` is a string used as the folder name,
     * if `input` is an object it must contain required properties:
     *   {'cmis:name': 'aFolder', 'cmis:objectTypeId': 'cmis:folder'}
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @return {CmisRequest}
     */
    session.createFolder = function (parentId, input, policies, addACEs, removeACEs, options, ajaxOptions) {
      options = _fill(options);
      if ('string' == typeof input) {
        input = {
          'cmis:name': input
        };
      }
      var properties = input || {};
      if (!properties['cmis:objectTypeId']) {
        properties['cmis:objectTypeId'] = 'cmis:folder';
      }
      options.objectId = parentId;
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }
      options.repositoryId = session.defaultRepository.repositoryId;
      options.cmisaction = 'createFolder';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * deletes an object
     * @param {String} objectId
     * @param {Boolean} allVersions
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.deleteObject = function (objectId, allVersions, options, ajaxOptions) {
      options = _fill(options);
      options.repositoryId = session.defaultRepository.repositoryId;
      options.cmisaction = 'delete';
      options.objectId = objectId;
      options.allVersions = !!allVersions;
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };


    /**
     * gets repository informations
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.getRepositoryInfo = function (options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'repositoryInfo';
      return _get(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * gets the types that are immediate children
     * of the specified typeId, or the base types if no typeId is provided
     * @param {String} typeId
     * @param {Boolean} includePropertyDefinitions
     * @param {Object} options (possible options: maxItems, skipCount, token)
     * @return {CmisRequest}
     */
    session.getTypeChildren = function (typeId, includePropertyDefinitions, options, ajaxOptions) {
      options = _fill(options);
      if (typeId) {
        options.typeId = typeId;
      }
      options.includePropertyDefinitions = includePropertyDefinitions;
      options.cmisselector = 'typeChildren'
      return _get(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * gets all types descended from the specified typeId, or all the types
     * in the repository if no typeId is provided
     * @param {String} typeId
     * @param {Integer} depth
     * @param {Boolean} includePropertyDefinitions
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.getTypeDescendants = function (typeId, depth, includePropertyDefinitions, options, ajaxOptions) {
      options = _fill(options);
      if (typeId) {
        options.typeId = typeId;
      }
      options.depth = depth || 1;
      options.includePropertyDefinitions = includePropertyDefinitions;
      options.cmisselector = 'typeDescendants'
      return _get(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * gets definition of the specified type
     * @param {String} typeId
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.getTypeDefinition = function (typeId, options, ajaxOptions) {
      options = _fill(options);
      options.typeId = typeId;
      options.cmisselector = 'typeDefinition'
      return _get(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * gets the documents that have been checked out in the repository
     * @param {String} objectId
     * @param {Object} options (possible options: filter, maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
     * @return {CmisRequest}
     */
    session.getCheckedOutDocs = function (objectId, options, ajaxOptions) {
      options = _fill(options);
      if (objectId) {
        options.objectId = objectId;
      }
      options.cmisselector = 'checkedOut'
      return _get(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * creates a new document
     *
     * @param {String} parentId
     * @param {String/Buffer/Blob} content
     * @param {String/Object} input
     * if `input` is a string used as the document name,
     * if `input` is an object it must contain required properties:
     *   {'cmis:name': 'docName', 'cmis:objectTypeId': 'cmis:document'}
     * @param {String} mimeType
     * @param {String} versioningState  (if set must be one of: "none", "major", "minor", "checkedout")
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.createDocument = function (parentId, content, input, mimeType, versioningState, policies, addACEs, removeACEs, options, ajaxOptions) {
      var options = _fill(options);
      if ('string' == typeof input) {
        input = {
          'cmis:name': input
        };
      }
      var properties = input || {};
      if (!properties['cmis:objectTypeId']) {
        properties['cmis:objectTypeId'] = 'cmis:document';
      }
      if (versioningState) {
        options.versioningState = versioningState;
      }

      options.objectId = parentId;
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }
      options.repositoryId = session.defaultRepository.repositoryId;
      options.cmisaction = 'createDocument';

      return _postMultipart(session.defaultRepository.rootFolderUrl,
        options, content, mimeType, properties['cmis:name'], ajaxOptions);

    };


    /**
     * creates a document object as a copy of the given source document
     *
     * @param {String} parentId
     * @param {String} sourceId
     * @param {String/Buffer/Blob} content
     * @param {String/Object} input
     * if `input` is a string used as the document name,
     * if `input` is an object it must contain required properties:
     *   {'cmis:name': 'docName', 'cmis:objectTypeId': 'cmis:document'}
     * @param {String} mimeType
     * @param {String} versioningState  (if set must be one of: "none", "major", "minor", "checkedout")
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.createDocumentFromSource = function (parentId, sourceId, content, input, mimeType, versioningState, policies, addACEs, removeACEs, options, ajaxOptions) {
      var options = _fill(options);
      if ('string' == typeof input) {
        input = {
          'cmis:name': input
        };
      }
      var properties = input || {};
      if (!properties['cmis:objectTypeId']) {
        properties['cmis:objectTypeId'] = 'cmis:document';
      }
      if (versioningState) {
        options.versioningState = versioningState;
      }
      options.objectId = parentId;
      options.sourceId = sourceId;
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }
      options.repositoryId = session.defaultRepository.repositoryId;
      options.cmisaction = 'createDocumentFromSource';

      return _postMultipart(session.defaultRepository.rootFolderUrl,
        options, content, mimeType, properties['cmis:name'], ajaxOptions);

    };

    /**
     * Creates a relationship
     * @param {Object} properties
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.createRelationship = function (properties, policies, addACEs, removeACEs, options, ajaxOptions) {
      options = _fill(options);
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }
      options.cmisaction = 'createRelationship';
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Creates a policy
     * @param {String} folderId
     * @param {Object} properties
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.createPolicy = function (folderId, properties, policies, addACEs, removeACEs, options, ajaxOptions) {
      options = _fill(options);
      options.objectId = folderId;
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }
      options.cmisaction = 'createPolicy';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Creates an item
     * @param {String} folderId
     * @param {Object} properties
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.createItem = function (folderId, properties, policies, addACEs, removeACEs, options, ajaxOptions) {
      options = _fill(options);
      options.objectId = folderId;
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }
      options.cmisaction = 'createItem';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Updates properties of specified objects
     * @param {Array} objectIds
     * @param {Object} properties
     * @param {Array} addSecondaryTypeIds
     * @param {Array} removeSecondaryTypeIds
     * @param {Options} options (possible options: token)
     * @return {CmisRequest}
     */
    session.bulkUpdateProperties = function (objectIds, properties, addSecondaryTypeIds, removeSecondaryTypeIds, options, ajaxOptions) {
      var options = _fill(options);
      for (var i = objectIds.length - 1; i >= 0; i--) {
        options['objectId[' + i + ']'] = objectIds[i];
      }
      options.objectIds = objectIds;
      _setProps(properties, options);
      if (addSecondaryTypeIds) {
        _setSecondaryTypeIds(addSecondaryTypeIds, 'add', options);
      }
      if (removeSecondaryTypeIds) {
        _setSecondaryTypeIds(removeSecondaryTypeIds, 'remove', options);
      }
      options.cmisaction = 'bulkUpdate';
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * performs a cmis query against the repository
     * @param {String} statement
     * @param {Boolean} searchAllVersions
     * @param {Object} options (possible options: maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
     * @return {CmisRequest}
     */
    session.query = function (statement, searchAllVersions, options, ajaxOptions) {
      options = _fill(options);
      options.cmisaction = 'query';
      options.statement = statement;
      options.searchAllVersions = !!searchAllVersions;
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * gets the changed objects, the list object should contain the next change log token.
     * @param {String} changeLogToken
     * @param {Boolean} includeProperties
     * @param {Boolean} includePolicyIds
     * @param {Boolean} includeACL
     * @param {Object} options (possible options: maxItems, succinct, token)
     * @return {CmisRequest}
     */
    session.getContentChanges = function (changeLogToken, includeProperties, includePolicyIds, includeACL, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'contentChanges';
      if (changeLogToken) {
        options.changeLogToken = changeLogToken;
      }
      options.includeProperties = !!includeProperties;
      options.includePolicyIds = !!includePolicyIds;
      options.includeACL = !!includeACL;
      return _get(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Creates a new type
     * @param {Object} type
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     *
     */
    session.createType = function (type, options, ajaxOptions) {
      options = _fill(options);
      options.cmisaction = 'createType';
      options.type = JSON.stringify(type);
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Updates a type definition
     * @param {Object} type
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.updateType = function (type, options, ajaxOptions) {
      options = _fill(options);
      options.cmisaction = 'updateType';
      options.type = JSON.stringify(type);
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * Deletes specified type
     * @param {String} typeId
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.deleteType = function (typeId, options, ajaxOptions) {
      options = _fill(options);
      options.cmisaction = 'deleteType';
      options.typeId = typeId;
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * gets last result
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.getLastResult = function (options, ajaxOptions) {
      options = _fill(options);
      options.cmisaction = 'lastResult';
      return _post(session.defaultRepository.repositoryUrl)
        .query(options).send(ajaxOptions);
    };


    /**
     * Returns children of object specified by id
     * @param {String} objectId
     * @param {Object} options (possible options: maxItems, skipCount, filter, orderBy, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
     * @return {CmisRequest}
     */
    session.getChildren = function (objectId, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'children';
      options.objectId = objectId;
      return _get(session.defaultRepository.rootFolderUrl).query(options).send(ajaxOptions);
    };

    /**
     * Gets all descendants of specified folder
     * @param {String} folderId
     * @param {Integer} depth
     * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
     * @return {CmisRequest}
     */
    session.getDescendants = function (folderId, depth, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'descendants';
      if (depth) {
        options.depth = depth;
      }
      options.objectId = folderId;
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Gets the folder tree of the specified folder
     * @param {String} folderId
     * @param {Integer} depth
     * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
     * @return {CmisRequest}
     */
    session.getFolderTree = function (folderId, depth, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'folderTree';
      if (depth) {
        options.depth = depth;
      }
      options.objectId = folderId;
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Gets the parent folder of the specified folder
     * @param {String} folderId
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.getFolderParent = function (folderId, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'parent';
      options.objectId = folderId;
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Gets the folders that are the parents of the specified object
     * @param {String} objectId
     * @param {Object} options (possible options: filter, renditionFilter, includeAllowableActions, includeRelationships, includePathSegment, succinct, token)
     * @return {CmisRequest}
     */
    session.getParents = function (objectId, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'parents';
      options.objectId = objectId;
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Gets the allowable actions of the specified object
     * @param {String} objectId
     * @param {Object} options (possible options: filter, maxItems, skipCount, orderBy, renditionFilter, includeAllowableActions, includeRelationships, succinct, token)
     * @return {CmisRequest}
     */
    session.getAllowableActions = function (objectId, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'allowableActions';
      options.objectId = objectId;
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Gets the properties of the specified object
     * @param {String} objectId
     * @param {String} returnVersion (if set must be one of 'this', latest' or 'latestmajor')
     * @param {Object} options (possible options: filter, succinct, token)
     * @return {CmisRequest}
     */
    session.getProperties = function (objectId, returnVersion, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'properties';
      options.objectId = objectId;
      if (returnVersion == 'latestmajor' || returnVersion == 'latest') {
        options.returnVersion = returnVersion;
      }
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Gets document content, WARNING: will not work for binary files (images, documents, ecc..)
     * @param {String} objectId
     * @param {Boolean} download
     * @param {Object} options (possible options: streamId, token)
     * @return {CmisRequest}
     */
    session.getContentStream = function (objectId, download, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'content';
      options.objectId = objectId;
      options.download = (!!download) ? 'attachment' : 'inline';
      return _get(session.defaultRepository.rootFolderUrl).query(options).dataType('binary').send(ajaxOptions);
    };

    /**
     * Gets document content URL
     * @param {String} objectId
     * @param {Boolean} download
     * @param {Object} options (possible options: streamId, token)
     * @return String
     */
    session.getContentStreamURL = function (objectId, download, options) {
      options = _fill(options);
      options.cmisselector = 'content';
      options.objectId = objectId;
      options.download = (!!download) ? 'attachment' : 'inline';

      var pairs = [];
      for (var key in options) {
        pairs.push(encodeURIComponent(key) + '=' + encodeURIComponent(options[key]));
      }
      var query = pairs.join('&');
      return session.defaultRepository.rootFolderUrl + '?' + query;
    };

    /**
     * gets document renditions
     * @param {String} objectId
     * @param {Object} options (possible options: renditionFilter, maxItems, skipCount, token)
     * @return {CmisRequest}
     */
    session.getRenditions = function (objectId, options, ajaxOptions) {
      options = _fill(options);
      options.cmisselector = 'renditions';
      options.objectId = objectId;
      options.renditionFilter = options.renditionFilter || '*';

      return _get(session.defaultRepository.rootFolderUrl).query(options).send(ajaxOptions);

    };

    /**
     * Updates properties of specified object
     * @param {String} objectId
     * @param {Object} properties
     * @param {Object} options (possible options: changeToken, succinct, token)
     * @return {CmisRequest}
     */
    session.updateProperties = function (objectId, properties, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      _setProps(properties, options);
      options.cmisaction = 'update';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Moves an object
     * @param {String} objectId
     * @param {String} sourceFolderId
     * @param {String} targetFolderId
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.moveObject = function (objectId, sourceFolderId, targetFolderId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'move';
      options.targetFolderId = targetFolderId;
      options.sourceFolderId = sourceFolderId;
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Deletes a folfder tree
     * @param {String} objectId
     * @param {Boolean} allVersions
     * @param {String} unfileObjects (if set must be one of 'unfile', 'deletesinglefiled', 'delete')
     * @param {Boolean} continueOnFailure
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.deleteTree = function (objectId, allVersions, unfileObjects, continueOnFailure, options, ajaxOptions) {
      var options = _fill(options);
      options.repositoryId = session.defaultRepository.repositoryId;
      options.cmisaction = 'deleteTree';
      options.objectId = objectId;
      options.allVersions = !!allVersions;
      if (unfileObjects) {
        options.unfileObjects = unfileObjects;
      }
      options.continueOnFailure = !!continueOnFailure;
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * Updates content of document
     * @param {String} objectId
     * @param {String/Buffer} content
     * @param {Boolean} overwriteFlag
     * @param {String} mimeType
     * @param {Object} options (possible options: changeToken, succinct, token)
     * @return {CmisRequest}
     */
    session.setContentStream = function (objectId, content, overwriteFlag, mimeType, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.overwriteFlag = !!overwriteFlag;
      options.cmisaction = 'setContent';

      return _postMultipart(session.defaultRepository.rootFolderUrl,
        options, content, mimeType, undefined, ajaxOptions);

    };

    /**
     * Appends content to document
     * @param {String} objectId
     * @param {String/Buffer} content
     * @param {Boolean} isLastChunk
     * @param {Object} options (possible options: changeToken, succinct, token)
     * @return {CmisRequest}
     */
    session.appendContentStream = function (objectId, content, isLastChunk, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'appendContent';
      options.isLastChunk = !!isLastChunk;
      return _postMultipart(session.defaultRepository.rootFolderUrl,
        options, content, undefined, undefined, ajaxOptions);
    };

    /**
     * deletes object content
     * @param {String} objectId
     * @param {Object} options (possible options: changeToken, succinct, token)
     * @return {CmisRequest}
     */
    session.deleteContentStream = function (objectId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'deleteContent';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Adds specified object to folder
     * @param {String} objectId
     * @param {String} folderId
     * @param {Boolean} allVersions
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.addObjectToFolder = function (objectId, folderId, allVersions, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'addObjectToFolder';
      options.allVersions = !!allVersions;
      options.folderId = folderId;
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * Removes specified object from folder
     * @param {String} objectId
     * @param {String} folderId
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.removeObjectFromFolder = function (objectId, folderId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'removeObjectFromFolder';
      options.folderId = folderId;
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * checks out a document
     * @param {String} objectId
     * @param {Object} options
     * @return {CmisRequest} (possible options: succinct, token)
     */
    session.checkOut = function (objectId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'checkOut';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * cancels a check out
     * @param {String} objectId
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.cancelCheckOut = function (objectId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'cancelCheckOut';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * checks in a document, if needed mimetype may be specified as
     * input['cmis:contentStreamMimeType'] or as option.mimeType
     *
     * @param {String} objectId
     * @param {Boolean} major
     * @param {String/Object} input
     * if `input` is a string used as the document name,
     * if `input` is an object it must contain required properties:
     *   {'cmis:name': 'docName'}
     * @param {String/Buffer} content
     * @param {String} comment
     * @param {Array} policies
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {Object} options
     * @return {CmisRequest}
     */
    session.checkIn = function (objectId, major, input, content, comment, policies, addACEs, removeACEs, options, ajaxOptions) {
      var options = _fill(options);
      if ('string' == typeof input) {
        input = {
          'cmis:name': input
        };
      }
      var properties = input || {};
      if (comment) {
        options.checkinComment = comment;
      }
      options.major = !!major
      options.objectId = objectId;
      _setProps(properties, options);
      if (policies) {
        _setPolicies(policies, options);
      }
      if (addACEs) {
        _setACEs(addACEs, 'add', options);
      }
      if (removeACEs) {
        _setACEs(removeACEs, 'remove', options);
      }

      options.cmisaction = 'checkIn';

      return _postMultipart(session.defaultRepository.rootFolderUrl,
        options, content, options.mimeType || properties['cmis:contentStreamMimeType'],
        properties['cmis:name'], ajaxOptions)

    };

    /**
     * gets versions of object
     * @param {String} objectId
     * @param {String} versionSeriesId
     * @param {Object} options (possible options: filter, includeAllowableActions, succinct, token)
     * @return {CmisRequest}
     */
    session.getAllVersions = function (objectId, versionSeriesId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.versionSeriesId = versionSeriesId;
      options.cmisselector = 'versions';
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * gets object relationships
     * @param {String} objectId
     * @param {Boolean} includeSubRelationshipTypes
     * @param {String} relationshipDirection
     * @param {String} typeId
     * @param {Object} options (possible options: maxItems, skipCount, includeAllowableActions, filter, succinct, token)
     * @return {CmisRequest}
     */
    session.getObjectRelationships = function (objectId, includeSubRelationshipTypes, relationshipDirection, typeId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.includeSubRelationshipTypes = !!includeSubRelationshipTypes;
      options.relationshipDirection = relationshipDirection || 'either';
      if (typeId) {
        options.typeId = typeId;
      }
      options.cmisselector = 'relationships';
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * gets object applied policies
     * @param {String} objectId
     * @param {Object} options (possible options: filter, succinct, token)
     * @return {CmisRequest}
     */
    session.getAppliedPolicies = function (objectId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisselector = 'policies';
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * applies policy to object
     * @param {String} objectId
     * @param {String} policyId
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.applyPolicy = function (objectId, policyId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.policyId = policyId;
      options.cmisaction = 'applyPolicy';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * removes policy from object
     * @param {String} objectId
     * @param {String} policyId
     * @param {Object} options (possible options: succinct, token)
     * @return {CmisRequest}
     */
    session.removePolicy = function (objectId, policyId, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.policyId = policyId;
      options.cmisaction = 'removePolicy';
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);

    };

    /**
     * applies ACL to object
     * @param {String} objectId
     * @param {Object} addACEs
     * @param {Object} removeACEs
     * @param {String} propagation
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.applyACL = function (objectId, addACEs, removeACEs, propagation, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.cmisaction = 'applyACL';
      _setACEs(addACEs, 'add', options);
      _setACEs(removeACEs, 'remove', options);
      return _post(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };

    /**
     * gets object ACL
     * @param {String} objectId
     * @param {Boolean} onlyBasicPermissions
     * @param {Object} options (possible options: token)
     * @return {CmisRequest}
     */
    session.getACL = function (objectId, onlyBasicPermissions, options, ajaxOptions) {
      var options = _fill(options);
      options.objectId = objectId;
      options.onlyBasicPermissions = !!onlyBasicPermissions;
      options.cmisselector = 'acl';
      return _get(session.defaultRepository.rootFolderUrl)
        .query(options).send(ajaxOptions);
    };
    
    var eventHandlers = {};
    
    /**
     * リスナ一覧を取得します
     * @param {String} type
     */
    function getListners(type) {
      var listners = eventHandlers[type];
    if (!listners) {
      eventHandlers[type] = listners = [];
    }
    return listners;
    }
    
    /**
     * イベントを登録します
     * @param {Object} eventData
     */
    function registEventHandler(eventData) {
      var type = eventData.type,
        listners = getListners(type);

      listners.push(eventData);
    }
    
    /**
     * イベントを発火させます
     * @param {String} type イベントタイプ
     * @param {Object} eventObject ハンドラに渡すオブジェクト
     */
    function trigger(type, eventObject) {
      var listners = getListners(type),
          eventData;
      
      for (var i = 0; listners.length > i; i++) {
        eventData = listners[i];
        
        eventData.handler.apply(eventData, [eventObject, eventData.data]);
      }
    }
    
    /**
     * イベントハンドラを登録する
     * @param {String} type
     * @param {Object} [data]
     * @param {function} handler
     */
    session.on = function() {
      var type, data, handler,
          eventData = new EventData();
      
      eventData.type = arguments[0].toLowerCase();
      if (arguments.length == 2) {
        eventData.handler = arguments[1];
      } else if (arguments.length >= 3) {
        eventData.data = arguments[1];
        eventData.handler = arguments[2]; 
      }
      
      registEventHandler(eventData)
    }

    function isObject(obj) {
      return obj === Object(obj);
    }
    
    function serialize(obj) {
      if (!isObject(obj)) return obj;
      var pairs = [];
      for (var key in obj) {
        if (null != obj[key]) {
          pairs.push(encodeURIComponent(key)
            + '=' + encodeURIComponent(obj[key]));
        }
      }
      return pairs.join('&');
    }
    
    function type(str){
      return str.split(/ *; */).shift();
    }
      
    function params(str){
      return reduce(str.split(/ *; */), function(obj, str){
        var parts = str.split(/ *= */)
          , key = parts.shift()
          , val = parts.shift();

        if (key && val) obj[key] = val;
        return obj;
      }, {});
  };
      
    function parseHeader(str) {
      var lines = str.split(/\r?\n/);
      var fields = {};
      var index;
      var line;
      var field;
      var val;

      lines.pop(); // trailing CRLF

      for (var i = 0, len = lines.length; i < len; ++i) {
        line = lines[i];
        index = line.indexOf(':');
        field = line.slice(0, index).toLowerCase();
        val = $.trim(line.slice(index + 1));
        fields[field] = val;
      }

      return fields;
    }
    
    function Response(req, options) {
      options = options || {};
    this.req = req;
    this.xhr = this.req.xhr;
    // responseText is accessible only if responseType is '' or 'text' and on older browsers
    this.text = ((this.req.method !='HEAD' && (this.xhr.responseType === '' || this.xhr.responseType === 'text')) || typeof this.xhr.responseType === 'undefined')
       ? this.xhr.responseText
       : null;
    this.statusText = this.req.xhr.statusText;
    this.setStatusProperties(this.xhr.status);
    this.header = this.headers = parseHeader(this.xhr.getAllResponseHeaders());
    // getAllResponseHeaders sometimes falsely returns "" for CORS requests, but
    // getResponseHeader still works. so we get content-type even if getting
    // other headers fails.
    this.header['content-type'] = this.xhr.getResponseHeader('content-type');
    this.setHeaderProperties(this.header);
    this.body = this.req.method != 'HEAD'
      ? this.parseBody(this.text ? this.text : this.xhr.response)
      : null;
    }
    Response.prototype.setHeaderProperties = function(header){
      // content-type
      var ct = this.header['content-type'] || '';
      this.type = type(ct);

      // params
//      var obj = params(ct);
//      for (var key in obj) this[key] = obj[key];
    }
    
    // TODO
    var parsers = {
      "application/json": $.parseJSON
    }
    
    Response.prototype.parseBody = function(str){
      var parse = parsers[this.type];
      return parse && str && (str.length || str instanceof Object)
          ? parse(str)
          : null;
    }
      
    Response.prototype.setStatusProperties = function(status){
      // handle IE9 bug: http://stackoverflow.com/questions/10046972/msie-returns-status-code-of-1223-for-ajax-request
      if (status === 1223) {
        status = 204;
      }

    var type = status / 100 | 0;
    
    // status / class
    this.status = status;
    this.statusType = type;
    
    // basics
    this.info = 1 == type;
    this.ok = 2 == type;
    this.clientError = 4 == type;
    this.serverError = 5 == type;
    this.error = (4 == type || 5 == type)
        ? this.toError()
        : false;
    
    // sugar
    this.accepted = 202 == status;
    this.noContent = 204 == status;
    this.badRequest = 400 == status;
    this.unauthorized = 401 == status;
    this.notAcceptable = 406 == status;
    this.notFound = 404 == status;
    this.forbidden = 403 == status;
    };
    
    Response.prototype.toError = function() {
      var req = this.req;
      var method = req.method;
      var url = req.url;

      var msg = 'cannot ' + method + ' ' + url + ' (' + this.status + ')';
      var err = new Error(msg);
      err.status = this.status;
      err.method = method;
      err.url = url;

      return err;
    };
    
    var request_types = {
      html: 'text/html',
      json: 'application/json',
      xml: 'application/xml',
      urlencoded: 'application/x-www-form-urlencoded',
      'form': 'application/x-www-form-urlencoded',
      'form-data': 'application/x-www-form-urlencoded'
    };

    // http://visionmedia.github.io/superagent
    var request = function(method, url) {
      var instance_ = {},
        url_ = url,
        query_ = [],
        username_,
        password_,
        ajaxParams = {
          type: method
        };
      
      instance_.auth = function(username, password) {
        ajaxParams.username = username;
        ajaxParams.password = password;
        return instance_;
      };
      
      instance_.async = function(async) {
        ajaxParams.async = async;
        return instance_;
        
      }
      instance_.type = function(type) {
        ajaxParams.contentType = request_types[type] || type;
        return instance_;
      }
      
      instance_.dataType = function(dataType) {
        ajaxParams.dataType = dataType;
        return instance_;
      }
      
      instance_.query = function(val) {
        if ('string' != typeof val) val = serialize(val);
        if (val) query_.push(val);
        return instance_;
      }
      
      instance_.data = function(data) {
        ajaxParams.data = data;
        return instance_;
      }
      
      instance_.processData = function(processData) {
        ajaxParams.processData = processData;
        return instance_;
      }
      
      instance_.send = function(options) {
        var d = new $.Deferred,
          errorObject,
          binaryData;
        
        if (isObject(options)) {
          for (var k in options) {
            ajaxParams[k] = options[k];
          }
        }
        
        ajaxParams.success = function(res) {
          if (ajaxParams.dataType == 'binary') {
            binaryData = res;
          }
        }
        ajaxParams.error = function(xhr, status, errorThrown) {
          errorObject = {
            xhr: xhr,
            status: status,
            errorThrown: errorThrown
          }
        }
        ajaxParams.complete = function(xhr, status) {
          var req = {
            url: url,
            method: method,
          req: instance_,
          xhr: xhr
        };
          var response = new Response(req);
          
          if (response && response.ok) {
            if (ajaxParams.dataType == 'binary') {
              d.resolve(binaryData);
            } else {
              d.resolve(response.body);
            }
          } else {
            d.reject(response);
          }
        }
        
        var query = serialize(query_.join('&'));
        ajaxParams.url = url_ + (~url_.indexOf('?') ? '&' + query : '?' + query);
        
        trigger('beforesend', ajaxParams);
        
        $.ajax(ajaxParams);
        
        return d;
      };
      
      return instance_;
    }


    //Private members and methods
    var _url = url;
    var _token = null;
    var _username = null;
    var _password = null;
    var _afterlogin;

    var _noop = function () {};

    var _globalNotOk = _noop;
    var _globalError = _noop;

    var _http = function (method, url) {
      var r = request(method, url);
      if (_username && _password) {
        return r.auth(_username, _password);
      }
      return r;
    };

    var _get = function (url) {
      return _http('GET', url);
    };

    var _post = function (url, multipart, ajaxOptions) {
      var req = _http('POST', url).type('form');
      if (!multipart) {
//        req.send = req.query;
      }
      return req;
    };
    
    var _postMultipart = function (url, options, content, mimeType, name, ajaxOptions) {

      var data = new FormData();
      if (content) {
        if ('string' == typeof content) {
          content = new Blob([content]); //, {type:mimeType||'text/plain'});
        }
        data.append("content", content);
      }
      
      // マルチバイト文字列が入りうるパラメータはクエリパラメータにする
      var urlparams = [];
      var keyId;
      var keyValue;
      for (var i=0; true; i++) {
        keyId = "propertyId["+i+"]";
        keyValue = "propertyValue["+i+"]";

        if (options[keyId] === "cmis:name") {
          urlparams.push(keyId+"="+options[keyId]);
          urlparams.push(keyValue+"="+encodeURIComponent(options[keyValue]));
          delete options[keyId];
          delete options[keyValue];
        } else if (options[keyId] === undefined) {
          break;
        }
      }
      
      if (url.indexOf('?') === -1) {
        url += '?' + urlparams.join('&');
      } else {
        url += '&' + urlparams.join('&');
      }
      
      var req = _post(url, true);
      for (var k in options) {
        data.append(k, options[k]);
      }
      req.data(data);
      
//      delete req.header['Content-Type'];
      
      req.type(false);
      req.processData(false);
      
      return req.send(ajaxOptions);
    }

    var _defaultOptions = {
      succinct: true
    };

    var _fill = function (options) {
      var o = {};
      for (var k in _defaultOptions) {
        o[k] = _defaultOptions[k];
      }
      if (options === undefined) {
        return o;
      }
      for (k in options) {
        o[k] = options[k];
      }
      return o;
    };

    var _setProps = function (properties, options) {
      var i = 0;
      for (var id in properties) {
        options['propertyId[' + i + ']'] = id;
        var propertyValue = properties[id];
        if (propertyValue !== null && propertyValue !== undefined) {
          if (Object.prototype.toString.apply(propertyValue) == '[object Array]') {
            for (var j = 0; j < propertyValue.length; j++) {
              options['propertyValue[' + i + '][' + j + ']'] = propertyValue[j];
            }
          } else {
            options['propertyValue[' + i + ']'] = propertyValue;
          }
        }
        i++;
      }
    };

    var _setPolicies = function (policies, options) {
      for (var i = 0; i < policies.length; i++) {
        options['policy[' + i + ']'] = policies[i];
      }
    };

    //action must be either 'add' or 'remove'
    var _setACEs = function (ACEs, action, options) {
      var i = 0;
      for (var id in ACEs) {
        options[action + 'ACEPrincipal[' + i + ']'] = id;
        var ace = ACEs[id];
        for (var j = 0; j < ace.length; j++) {
          options[action + 'ACEPermission[' + i + '][' + j + ']'] = ACEs[id][j];
        }
        i++;
      }
    };

    //action must be either 'add' or 'remove'
    var _setSecondaryTypeIds = function (secondaryTypeIds, action, options) {
      for (var i = 0; i < secondaryTypeIds.length; i++) {
        options[action + 'SecondaryTypeId[' + i + ']'] = secondaryTypeIds[i];
      }
    };
    
    return session;
  };

  //use this transport for "binary" data type
  $.ajaxTransport("+binary", function(options, originalOptions, jqXHR){
    // check for conditions and support for blob / arraybuffer response type
    if (window.FormData && ((options.dataType && (options.dataType == 'binary')) || (options.data && ((window.ArrayBuffer && options.data instanceof ArrayBuffer) || (window.Blob && options.data instanceof Blob)))))
    {
      return {
        // create new XMLHttpRequest
        send: function(headers, callback){
        // setup all variables
          var xhr = new XMLHttpRequest(),
            url = options.url,
            type = options.type,
            async = options.async || true,
            // blob or arraybuffer. Default is blob
            dataType = options.responseType || "blob",
            data = options.data || null,
            username = options.username || null,
            password = options.password || null;
            
          xhr.addEventListener('load', function(){
            var data = {};
        data[options.dataType] = xhr.response;
        // make callback and send data
        callback(xhr.status, xhr.statusText, data, xhr.getAllResponseHeaders());
          });
   
          xhr.open(type, url, async, username, password);
          
        // setup custom headers
        for (var i in headers ) {
        xhr.setRequestHeader(i, headers[i] );
        }
          
          xhr.responseType = dataType;
          xhr.send(data);
        },
        abort: function(){
          jqXHR.abort();
        }
      };
    }
  });
  
  return lib;
});
