/*
 * Copyright 2016 Ricksoft Co., Ltd.
 * All rights reserved.
 */
define('cmis/cmisconfluence', ['jquery', 'CMIS'],
    function($, CMIS) {

  var ALFRESCO_FAVORITE_REST_URL = "/api/-default-/public/alfresco/versions/1/people/-me-/favorites";
  var ALFRESCO_SHAREDLINKS_REST_URL = "/api/-default-/public/alfresco/versions/1/shared-links";

  var cmisSession_;
  var cmisSessionInfo_;
  var data_;
  var dataInfo_;
  var browserBindingUrl_;
  var cachePageInfo_;
  
  
  var object = {};
  
  object.initCmisSession = function(info, useCache) {
    var d = new $.Deferred;

    cmisSessionInfo_ = info;

    // フォルダマッピングが設定されていない場合はCMISのセッションを作成しない
    if (info && !info.isFederatable && !info.isFederatable()) {
      d.resolve(info);
      return d.promise();
    }

    // ALFRESCO-529 権限により課題ごとに利用する接続セッションを切り替える必要が出たのでキャッシュをやめる
    // if (useCache) {
    //   // CMISのセッションが作成済みの場合はCMISのセッションを作成しない
    //   if (cmisSession_ && cmisSession_.isLoadedRepository()) {
    //     d.resolve(info);
    //     return d.promise();
    //   }
    // }

    // 認証情報取得
    object.getCredential(info && info.isAdmin && info.isAdmin()).done(function(data) {
      if(!data) {
        d.reject();
        return d.promise();
      }
      
      data_ = $.extend(data, {
        browserBindingUrl: data.alfrescoUrl + "/api/-default-/public/cmis/versions/1.1/browser",
        repositoryUrl: data.alfrescoUrl + "/api/-default-/public/cmis/versions/1.1/browser",
        rootFolderUrl: data.alfrescoUrl + "/api/-default-/public/cmis/versions/1.1/browser/root",
      });

      // ライセンスが無効、ユーザー認証登録がない、または最大ログイン試行回数を超えた場合
      if (object.isInvalidLicense() || !object.hasUserCredential() || object.isOverMaximumLoginAttemptAllowed()) {
        d.resolve(info);
        return d.promise();
      }
      
      // CMISのセッションを作成
      var tmpCmisSession = CMIS.createSession({
        url: data_.browserBindingUrl,
        authType: data_.authType,
        username:data_.username,
        token:data_.token
      });

      // -default- リポジトリを使用する
      tmpCmisSession.select('-default-', {
        repositoryUrl: data_.repositoryUrl,
        rootFolderUrl: data_.rootFolderUrl
      }).done(function(data, cmisSession) {
        cmisSession_ = cmisSession;
        d.resolve(info, cmisSession);
      }).fail(function(error) {
        d.reject(error);
      });
    });
    
    return d.promise();
  }
  
  object.initPageInfo = function() {
    var d = new $.Deferred;
    
    object.getPageInfo().done(function(data) {
      if(!data) {
        d.reject();
        return d.promise();
      }
      data.isFederatable = $.proxy(function() {
        return (this.publicFolder && this.publicFolder.length > 0) || (this.privateFolder && this.privateFolder.length > 0);
      }, data);
      dataInfo_ = data;
      d.resolve(data);
    });
    
    return d.promise();
  }
  
  
  object.getCmisSession = function() {
    return cmisSession_;  
  }

  object.isAdmin = function() {
    return cmisSessionInfo_ && cmisSessionInfo_.isAdmin && cmisSessionInfo_.isAdmin();
  }
  
  object.getAlfrescoUrl = function() {
    return data_.alfrescoUrl;
  }

  object.getShareUrl = function () {
    return data_.shareUrl;
  }

  object.getBrowserBindingUrl = function() {
    return data_.browserBindingUrl;
  }
  
  object.getAccessType = function() {
    return data_.accessType; // WithAdminProfile or WithOwnProfile
  }

  object.getUserName = function () {
        return data_.username;
      }

  object.isInvalidLicense = function() {
    return data_.invalidLicense;
  }

  object.isLimitMaximumLoginAttemptAllowed = function () {
    return data_.limitMaximumLoginAttemptAllowed;
  }

  object.isOverMaximumLoginAttemptAllowed = function () {
    return data_.overMaximumLoginAttemptAllowed;
  }

  object.hasUserCredential = function () {
    return data_.username || data_.token ? true : false;
  }

  /**
   * 認証情報を取得します
   * @param {Boolean}
   * @return {Deferred}
   */
  object.getCredential = function(admin) {
    return $.ajax({
      type: 'GET',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/credential",
      data: (function () {
        if (object.getSpaceKey()) {
          return {'admin': admin, spaceKey: object.getSpaceKey(),};
        }
        return {'admin': admin,};
      })(),
      cache: false
    });
  }

  /**
   * ページ情報を取得します
   * @return {Deferred}
   */
  object.getPageInfo = function() {
    return $.ajax({
      type : 'GET',
      dataType : 'json',
      contentType : 'application/json',
      url : contextPath + "/rest/alfresco/latest/pageinfo",
      data: {
        spaceKey: object.getSpaceKey()
      },
      cache : false
    });
  }
  
  /**
   * 
   */
  object.sanitizeIds = function(ids) {
    var d = new $.Deferred;
    
    if (ids.length <= 0) {
      d.resolve([]);
      return d.promise();
    }
    
    var query = "select cmis:objectId from cmis:document where cmis:objectId in (";
    query += (function() {
      return $.map(ids, function(n) {
        return "'" + n + "'";
      }).join(',');
    })();
    query += ")";
    
    cmisSession_.query(query).done(function(data) {
      var results = [];
      $.each(data.results, function(i, ret) {
        results.push(ret.succinctProperties['cmis:objectId']);
      });
      d.resolve(results);
    }).fail(function(error) {
      d.reject(error);
    });
    
    return d.promise();
  }
  
  /**
   * 
   */
  object.getDocumentById = function(id) {
    return cmisSession_.getObject(id, false, {'includeAllowableActions':true});
  }
  
  /**
   * 
   */
  object.getDocumentByIds = function(ids) {
    var d = new $.Deferred;
    object.sanitizeIds(ids).pipe(function(sanitizeIds){
      return cmisSession_.getObjectByIds(sanitizeIds, false, {'includeAllowableActions':true});
    }).pipe(function(documents) {
      d.resolve(documents);
    }, function(error) {
      d.reject(error);
    });
    return d.promise();
  }

  object.isReadOnlyModeEnabled = function() {
    return AJS.Meta.get('access-mode') === 'READ_ONLY';
  }

  object.isGreaterThanOrEqualTo = function(ver1, ver2) {
    return object.compareTo(ver1, ver2) !== -1;
  }

  object.compareTo = function(str1, str2) {
    var parts1 = str1.split(".");
    var parts2 = str2.split(".");
    var ret = -1;
    var len = parts1.length;
    if (parts1.length > parts2.length) {
      len = parts2.length;
      ret = 1;
    } else if (parts1.length === parts2.length) {
      ret = 0;
    }
    // Make sure 1.09 < 1.3, 1.30 > 1.9, 1.030 < 1.09
    for (var i = 0; i < len; i++) {
      var i1;
      var i2;
      var p1 = parts1[i];
      var p2 = parts2[i];
      var pat = /^(0+)(.*)$/;
      var m1 = p1.match(pat);
      var m2 = p2.match(pat);
      if (m1 != null) {
        p1 = "0." + p1;
        i1 = parseFloat(p1);
      } else {
        i1 = parseInt(p1, 10);
      }
      if (m2 != null) {
        p2 = "0." + p2;
        i2 = parseFloat(p2);
      } else {
        i2 = parseInt(p2, 10);
      }
      if (i1 < i2) {
        return -1;
      }
      if (i1 > i2) {
        return 1;
      }
    }
    return ret;
  }
  
  /**
   * スペースキーを取得します
   * @return {String}
   */
  object.getSpaceKey = function() {
    var meta = $("meta[name=ajs-space-key]");
    if (meta.length <= 0 && window.parent) {
      meta = $(window.parent.document).find('meta[name=ajs-space-key]');
    }
    return meta.attr("content");
  }

  object.getSpaceId = function() {
      var $picker = $("#s2id_cmis-field-space");

      if ($picker.length > 0) {
        var space = $picker.select2("data");
        if (space) {
          return space.entity.id;
        }
      }
  }
  
  /**
   * ページIDを取得します
   * @return {String}
   */
  object.getPageId = function() {
    var meta = $("meta[name=ajs-page-id]");
    if (meta.length <= 0 && window.parent) {
      meta = $(window.parent.document).find('meta[name=ajs-page-id]');
    }
    return meta.attr("content");
  }
  
  /**
   * リモートユーザーを取得します。
   * @return {String}
   */
  object.getRemoteUser = function() {
    return $("meta[name=ajs-remote-user]").attr("content");
  }
  
  /**
   * Authorを取得します。
   * @param {String} operation (create or update)
   * @return {String}
   */
  object.getAuthorFor = function(operation) {
    var author = {
      'WithAdminProfile': {
        'create': {'cmis:secondaryObjectTypeIds': ['P:cm:author'],'cm:author':object.getRemoteUser()},
        'update': {'cm:author':object.getRemoteUser()}
      },
      'WithOwnProfile': {
        'create': {},
        'update': {}
      }
    };
    return author[object.getAccessType()][operation];
  }
  
  /**
   * CreatedByを取得する
   * @return {String}
   */
  object.getCreatedBy = function(document) {
    var createdBy = {
      'WithAdminProfile': document.prop('cm:author') || document.createdBy,
      'WithOwnProfile': document.lastModifiedBy
    };
    return createdBy[object.getAccessType()];
  }

  /**
   * 課題に紐づくオブジェクトIDを取得します
   * @param issueKey
   * @param options
   * @returns {Deferred}
   */
  object.getObjectIdByPage = function (pageId, options) {
    return $.ajax({
      type: 'GET',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/page/" + pageId + "/object",
      data: { 'start': options.skipCount, 'limit': options.maxItems }
    });
  }

  // /**
  //  * ドキュメントをページにリンクします
  //  */
  // object.linkDocuments = function(documents) {
  //   return $.ajax({
  //     type : 'POST',
  //     dataType : 'json',
  //     contentType : 'application/json',
  //     url : contextPath + "/rest/alfresco/latest/pageinfo",
  //     data: JSON.stringify({
  //       pageId: object.getPageId(),
  //       objectId: $.map(documents, function(doc) {
  //         return doc['id'];
  //       })
  //     })
  //   });
  // }
  
  // /**
  //  * ドキュメントをページにリンクします
  //  */
  // object.linkDocument = function(doc) {
  //   return object.linkDocuments([doc]);
  // }
  //
  // /**
  //  * ドキュメントをページからアンリンクします
  //  */
  // object.unlinkDocuments = function(documents) {
  //   return $.ajax({
  //     type : 'DELETE',
  //     dataType : 'json',
  //     contentType : 'application/json',
  //     url : contextPath + "/rest/alfresco/latest/pageinfo",
  //     data: JSON.stringify({
  //       pageId: object.getPageId(),
  //       objectId: $.map(documents, function(doc) {
  //         return doc['id'];
  //       })
  //     })
  //   });
  // }

  object.getAdminCredentialUrl = function () {
    return contextPath + "/admin/plugins/alfresco_for_confluence/credential.action";
  }

  object.getUserProfilePanelUrl = function() {
    if (data_.accessType !== "WithOwnProfile") {
      return undefined;
    }
    return contextPath + "/users/plugins/alfresco_for_confluence/profileView.action";
  }

  object.getShareDocumentDetailUrl = function (doc) {
    if (data_.accessType !== "WithOwnProfile") {
      return undefined;
    }

    if (object.getShareUrl()) {
      return object.getShareUrl() + "/page/document-details?nodeRef=" + doc['nodeRef'];
    } else {
      // Share URLが定義されていない
      return undefined;
    }
  }

  object.getItemActions = function(item, override_actions) {
    var default_actions = {
      document_create:true, 
      document_update:true, 
      document_remove:true,
      document_revert:true,
      document_download:true, 
      document_preview:true, 
      document_version:true,  
      document_link:true, 
      folder_create:true, 
      folder_remove:true
    };
    var default_mapping = {
      'document_create':{'canCreateDocument':true},
      'document_update':{'canCheckOut':true},
      'document_remove':{'canDeleteObject':true},
      'document_revert':{'canCheckOut':true},
      'document_download':{},
      'document_preview':{},
      'document_version':{},
      'document_link':{},
      'folder_create':{'canCreateFolder':true},
      'folder_remove':{'canDeleteObject':true,'canDeleteTree':true}
    };
    var self = this;
    
    if (!item) {
      throw new Error('Item is missing...');
    }
    if (!item.allowableActions) {
      throw new Error('AllowableActions is missing on [' + item.name + ']...');
    }
    
    function resolve(mapping_operations, allowableActions) {
      var result = {};
      for(key_operation in mapping_operations){
        var mapping_actions = mapping_operations[key_operation];
        var allowable = true;
        for(key_action in mapping_actions){
          if (mapping_actions[key_action] !== allowableActions[key_action]) {
            allowable = false;
            break;
          }
        }
        result[key_operation] = allowable;
      }
      return result;
    }
    
    return $.extend({}, default_actions, resolve(default_mapping, item.allowableActions), override_actions);
  }

  /**
   * 共有リンクのURLを取得します
   * @param doc
   * @returns {*}
   */
  object.getSharedLinkUrl = function (shared) {
    if (data_.accessType !== "WithOwnProfile") {
      return undefined;
    }

    if (object.getShareUrl() && shared.sharedId) {
      return object.getShareUrl() + "/s/" + shared.sharedId;
    } else {
      // Share URLが定義されていない
      return undefined;
    }
  }

  /**
   * ドキュメントをページからアンリンクします
   */
  object.unlinkDocument = function(doc) {
    return object.unlinkDocuments([doc]);
  }
  
  /**
   * 認証情報を取得します
   * @return {Deferred}
   */
  object.search = function(query) {
    return $.ajax({
      type : 'GET',
      dataType : 'json',
      contentType : 'application/json',
      url : contextPath + "/rest/alfresco/latest/search",
      data: {
        q: query
      },
      cache : false
    });
  }
  
  object.getPreviewLink = function(doc) {
    var d = new $.Deferred;
    var cmisSession = doc.cmisSession;
    var property = doc.succinctProperties;
    var nodeRef = property["alfcmis:nodeRef"]; // TODO Alfresco依存コード
    var mimeType = property["cmis:contentStreamMimeType"];

    if (!mimeType) {
      d.resolve();
    } else if (mimeType.indexOf('video/') === 0 || mimeType.indexOf('audio/') === 0) {        
      d.resolve(object.createMovieURL(nodeRef, property), "movie");
    } else if ("application/pdf" === mimeType) {
      d.resolve(doc.getContentStreamURL(), "pdf");
    } else {
      // mimeTypeがpdf,image以外の場合は、サムネイル定義を調べる
      $.ajax({
        url: object.createThumbnailDefinitionsURL(nodeRef),
        type: "GET",
          beforeSend: function(xhr) {
            if (cmisSession.authType == 'Basic') {
              xhr.setRequestHeader('Authorization', cmisSession.token);
            }
            if (cmisSession.authType == 'WSSE') {
              xhr.setRequestHeader('X-WSSE', cmisSession.token);
            }
          },
        timeout: 0,
        success: function(definitions){
          if (!!definitions && definitions.indexOf("pdf") != -1) {
            d.resolve(object.createThumbnailURL(nodeRef), "pdf");
          } else if (!!definitions && definitions.indexOf("imgpreview") != -1) {
            d.resolve(object.createThumbnailURL(nodeRef), "image");
          } else {
            d.resolve();
          }
        },
        error: function() {
          d.resolve();
        }
      });
    }

    return d.promise();
  }
  
  /**
   * 添付ファイルのフォルダを取得します。
   * @return {Object}
   */
  object.createFolder = function(folder, options) {
    return folder ? $.extend({}, folder, { 
      formatter: function(obj) {
        return $.extend({
          cmisdata: obj,
          id: obj.id, 
          label: obj.name,
          actions: {folder_remove: false}
        }, options);
      }
    }) : null;
  }
  
  /**
   * 添付ファイルのリンクされたフォルダを取得します。
   * @return {Object}
   */
  object.createLinkFolder = function(folder, options, fn_getChildren) {
    return folder ? $.extend({}, folder, {
      formatter: function(obj) {
        return $.extend({
          cmisdata: obj, 
          label: AJS.I18n.getText('alfresco.for.confluence.listview.linked'), 
          actions: {folder_create: false, folder_remove: false, document_create: false},
          clazz: "aui-icon aui-icon-small aui-iconfont-devtools-tag",
          already_loaded: true
        }, options);
      },
      getChildren: fn_getChildren
    }) : null;
  }
  
  // TODO Alfresco依存コード
  object.createThumbnailDefinitionsURL = function(node) {
    var api = data_.alfrescoUrl + "/service/api";
    var nodeRefAsLink = node.replace(":/", "");

    return api + "/node/" + nodeRefAsLink + "/content/thumbnaildefinitions";
  }
  
  //TODO Alfresco依存コード
  object.createThumbnailURL = function(node) {
    var api = data_.alfrescoUrl + "/service/api";
    var nodeRefAsLink = node.replace(":/", "");

    return api + "/node/" + nodeRefAsLink + "/content/thumbnails/pdf?c=force";
  }
  
  //TODO Alfresco依存コード
  object.createMovieURL = function(node, contentStreamFileName) {
    var api = data_.alfrescoUrl + "/service/api";
    var nodeRefAsLink = node.replace(":/", "");
    return api + "/node/" + nodeRefAsLink + "/content/" + contentStreamFileName;
  }

  object.getImgPreview = function(doc) {
    var d = new AJS.$.Deferred;
    var url = "/s/api/node/workspace/SpacesStore/" + doc.versionSeriesId + "/content/thumbnails/imgpreview";

    object.requestAlfresco(url, d, {
      dataType: 'binary',
      data: {
        c: 'queue',
        ph: 'true',
        lastModified: 'imgpreview'
      }
    });

    return d.promise();
  }

  /**
   * ドキュメントに紐づくリンクを取得します
   */
  object.getLinkByDocuments = function (docs) {
    return $.ajax({
      type: 'POST',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/objects",
      data: JSON.stringify({
        objectId: $.map(docs, function (doc) {
          return doc.isDocumentLink() ? doc['destination'] : doc['id'];
        })
      })
    });
  }

  /**
   * ドキュメントに紐づくリンクを取得します
   */
  object.getLinkByDocument = function (doc) {
    return object.getlinks([doc]);
  }

  /**
   * 課題にドキュメントをリンクします
   */
  object.addLinksByPage = function (docs, pageId) {
    return $.ajax({
      type: 'POST',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/page/" + pageId,
      data: JSON.stringify({
        objectId: $.map(docs, function (doc) {
          return doc['id'];
        })
      })
    });
  }

  /**
   * 課題にドキュメントをリンクします
   */
  object.addLinkByPage = function (doc, pageId) {
    return object.addLinksByPage([doc], pageId);
  }

  /**
   * 課題にドキュメントをリンクします
   */
  object.addLinksByObject = function (doc, pageIds) {
    return $.ajax({
      type: 'POST',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/object/" + doc['id'],
      data: JSON.stringify({
        pageId: pageIds
      })
    });
  }

  /**
   * 課題にドキュメントをリンクします
   */
  object.addLinkByObject = function (doc, pageId) {
    return object.addLinksByObject(doc, [pageId]);
  }

  /**
   * 課題にリンクしているドキュメントを解除します
   */
  object.removeLinksByPage = function (docs, pageId) {
    return $.ajax({
      type: 'DELETE',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/page/" + pageId,
      data: JSON.stringify({
        objectId: $.map(docs, function (doc) {
          return doc['id'];
        })
      })
    });
  }

  /**
   * 課題にリンクしているドキュメントを解除します
   */
  object.removeLinkByPage = function (doc, pageId) {
    return object.removeLinksByPage([doc], pageId);
  }

  /**
   * 課題にリンクしているドキュメントを解除します
   */
  object.removeLinksByObject = function (doc, pageIds) {
    return $.ajax({
      type: 'DELETE',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/object/" + doc['id'],
      data: JSON.stringify({
        pageId: pageIds
      })
    });
  }

  /**
   * 課題にリンクしているドキュメントを解除します
   */
  object.removeLinkByObject = function (doc, pageId) {
    return object.removeLinksByObject(doc, [pageId]);
  }

  /**
   * 現在のログイン失敗回数"を更新します
   */
  object.updateCurrentLoginFailed = function (admin) {
    var d = new AJS.$.Deferred;
    $.ajax({
      type: 'POST',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/updateCurrentLoginFailed",
      data: JSON.stringify({'admin': admin })
    }).done(function (data) {
      data_.overMaximumLoginAttemptAllowed = data.overMaximumLoginAttemptAllowed;
      d.resolve(data);
    }).fail(function (error) {
      d.reject(error);
    });
    return d.promise();

  }

  /**
   * https://docs.atlassian.com/jira/REST/server/#api/2/search
   */
  object.searchPages = function (pageIds) {
    return $.ajax({
      type: 'GET',
      dataType: 'json',
      url: contextPath + "/rest/api/content/search",
      data: {
        "cql": pageIds && pageIds.length > 0 ? "id in (" + pageIds.join(",") + ")" : "",
        "start": 0,
        "limit": 50,
        "expand": "space"
      }
    });
  }

  /**
   * Alfrescoのお気に入り一覧を取得します
   */
  object.getFavorites = function (options) {
    options = options || {};

    var exists = [];
    var data = {};

    ["file", "folder", "site"].forEach(function (key) {
      if (options[key]) {
        exists.push("EXISTS(target/" + key + ")")
      }
    });
    if (exists.length > 0) {
      data['where'] = "(" + exists.join(" OR ") + ")";
    }
    ['maxItems', 'skipCount'].forEach(function (key) {
      if (options[key]) {
        data[key] = options[key];
      }
    });

    var d = new AJS.$.Deferred;
    object.requestAlfresco(ALFRESCO_FAVORITE_REST_URL, d, {
      type: "GET",
      data: data
    });
    return d.promise();
  }

  /**
   * ドキュメントに紐づくリンクを取得します
   */
  object.getLinkByDocuments = function (docs) {
    return $.ajax({
      type: 'POST',
      dataType: 'json',
      contentType: 'application/json',
      url: contextPath + "/rest/alfresco/latest/link/objects",
      data: JSON.stringify({
        objectId: $.map(docs, function (doc) {
          return doc['id'];
        })
      })
    });
  }

  /**
   * AlfrescoからClasses(aspect,type)を取得します
   */
  object.getClasses = function (excludeNamePrefix) {
    var d = new AJS.$.Deferred;
    (function () {
      var d = new AJS.$.Deferred;
      object.requestAlfresco("/service/api/classes", d, {
        type: "GET"
      });
      return d.promise();
    })().done(function (classes) {
      if (!excludeNamePrefix) {
        d.resolve(classes);
      } else {
        var result = [];
        $.each(classes, function (i, clazz) {
          var name = clazz.name;
          var prefix = name.split(':')[0];
          if ($.inArray(prefix, excludeNamePrefix) != -1) {
            return true;
          }
          result.push(clazz);
        });
        d.resolve(result);
      }
    });
    return d.promise();
  }

  /**
   * AlfrescoからPropertiesを取得します
   */
  object.getProperties = function (excludeNamePrefix) {
    var d = new AJS.$.Deferred;
    (function () {
      var d = new AJS.$.Deferred;
      object.requestAlfresco("/service/api/classes", d, {
        type: "GET"
      });
      return d.promise();
    })().done(function (classes) {
      if (!excludeNamePrefix) {
        d.resolve(classes);
      } else {
        var result = {};
        $.each(classes, function (i, clazz) {
          var name = clazz.name;
          var properties = clazz.properties;
          var prefix = name.split(':')[0];
          if ($.inArray(prefix, excludeNamePrefix) == -1) {
            $.extend(result, properties);
          }
        });
        // d.resolve($.map(result, function(value, key) {
        //     return value;
        // }));
        d.resolve(result);
      }
    });
    return d.promise();
  }

  /**
   * Alfrescoのお気に入りを取得します
   * @param {String} id
   */
  object.getFavorite = function (id) {
    var d = new AJS.$.Deferred;

    var url = ALFRESCO_FAVORITE_REST_URL + "/" + id;

    object.requestAlfresco(url, null, {
      type: "GET",
      success: function (favorite) {
        d.resolve(favorite);
      },
      error: function (xhr) {
        if (xhr.status == 404) {
          // ファイルがお気に入り登録されていない場合は404で返る
          d.resolve();
        } else {
          d.reject.apply(this, arguments);
        }
      }
    });
    return d.promise();
  }

  /**
   * Alfrescoのお気に入りを追加します
   * @param {CmisObject} cmisObject
   */
  object.addFavorite = function (cmisObject) {
    var d = new AJS.$.Deferred;

    var data = {'target': {}};

    var objectName = '';
    if (cmisObject.baseTypeId == 'cmis:document') {
      objectName = 'file';
    }
    if (cmisObject.baseTypeId == 'cmis:folder') {
      objectName = 'folder';
    }

    data.target[objectName] = {
      'guid': cmisObject['versionSeriesId'] || cmisObject['id']
    };

    object.requestAlfresco(ALFRESCO_FAVORITE_REST_URL, d, {
      type: "POST",
      data: JSON.stringify(data),
      contentType: 'application/json',
      dataType: "json"
    });
    return d.promise();
  }

  /**
   * Alfrescoのお気に入りを削除します
   * @param {CmisObject} cmisObject
   */
  object.delFavorite = function (cmisObject) {
    var d = new AJS.$.Deferred;

    var deleteId = cmisObject['versionSeriesId'] || cmisObject['id'];

    var url = ALFRESCO_FAVORITE_REST_URL + "/" + deleteId;

    object.requestAlfresco(url, d, {
      type: "DELETE"
    });
    return d.promise();
  }

  /**
   * Alfrescoの共有リンクを取得します
   */
  object.getSharedLinks = function (options) {
    options = options || {};

    var data = {};

    ['maxItems', 'skipCount'].forEach(function (key) {
      if (options[key]) {
        data[key] = options[key];
      }
    });

    var d = new AJS.$.Deferred;
    object.requestAlfresco(ALFRESCO_SHAREDLINKS_REST_URL, d, {
      type: "GET",
      data: data
    });
    return d.promise();
  }

  /**
   * Alfrescoの共有リンクを設定します
   */
  object.shareLink = function (doc, expire) {

    var data = {
      "nodeId": doc.versionSeriesId
    };

    if (expire) {
      data["expiresAt"] = expire;
    }

    var d = new AJS.$.Deferred;
    object.requestAlfresco(ALFRESCO_SHAREDLINKS_REST_URL, d, {
      type: "POST",
      data: JSON.stringify(data),
      contentType: 'application/json',
      dataType: "json"
    });
    return d.promise();
  }

  /**
   * Alfrescoの共有リンクを解除します
   */
  object.unshareLink = function (sharedId) {

    var d = new AJS.$.Deferred;
    object.requestAlfresco(ALFRESCO_SHAREDLINKS_REST_URL + "/" + sharedId, d, {
      type: "DELETE",
    });

    return d.promise();
  }

  /**
   * Alfrescoへリクエストを投げます
   */
  object.requestAlfresco = function (url, deferred, ajaxOptions) {
    var requestOptions = $.extend({
      url: data_.alfrescoUrl + url,
      beforeSend: function (xhr) {
        if (cmisSession_.authType == 'Basic') {
          xhr.setRequestHeader('Authorization', cmisSession_.token);
        }
        if (cmisSession_.authType == 'WSSE') {
          xhr.setRequestHeader('X-WSSE', cmisSession_.token);
        }
      },
      success: function () {
        deferred.resolve.apply(this, arguments);
      },
      error: function () {
        deferred.reject.apply(this, arguments);
      },
      timeout: 0
    }, ajaxOptions || {});
    return $.ajax(requestOptions);
  }
  
  object.moment = function(date) {
    return moment(date);
  }

  object.reloadPage = function() {
    window.location.reload();
  }
  
  return object;
});