import {createDropdownMenu, createDropdownMenuNonPredefined} from "./EasyDropdownMenu";
import {createStatusDropdown} from "./Status-Menu";

AJS.bind("init.rte", function () {
    if ($('body').is('.contenteditor')) {

        AJS.$(window).on("created.property-panel", () => {
            const propertyPanel = Confluence.PropertyPanel.current;
            const macroPlaceholder = propertyPanel.anchor;
            const macroName = macroPlaceholder.dataset.macroName;
            if (macroName.startsWith("easy-dropdown-menu") && macroName !== "easy-dropdown-menu-edit-mode") {
                createDropdownInPropertyPanel(propertyPanel, macroPlaceholder, macroName);
            }
        });

    }
});

const createDropdownInPropertyPanel = (propertyPanel, macroPlaceholder, macroName) => {
    const specificMacroObject = getSpecificMacroObject(macroName);
    const parameters = Confluence.MacroParameterSerializer.deserialize(macroPlaceholder.dataset.macroParameters);
    const spanElement = addSpanToPropertyPanel(propertyPanel, getSpanClassName(parameters, specificMacroObject.placeholderCssClass));
    specificMacroObject.createSelect2Function(macroPlaceholder, specificMacroObject, spanElement, parameters);
}

const getSpecificMacroObject = (macroName) => {
    switch (macroName) {
        case "easy-dropdown-menu":
            return {
                createSelect2Function: createSelect2ForPropertyPanelNonPredefined,
                placeholderCssClass: "edm-dropdown-menu-non-predefined",
                adjustParametersFunction: adjustParametersNonPredefined,
                createDropdownFunction: createDropdownMenuNonPredefined
            };
        case "easy-dropdown-menu-predefined":
            return {
                createSelect2Function: createSelect2ForPropertyPanelPredefined,
                placeholderCssClass: "edm-dropdown-menu",
                adjustParametersFunction: adjustParametersPredefined,
                createDropdownFunction: createDropdownMenu
            };
        case "easy-dropdown-menu-status":
            return {
                createSelect2Function: createSelect2ForPropertyPanelStatus,
                placeholderCssClass: "",
                adjustParametersFunction: adjustParametersStatus,
                createDropdownFunction: createStatusDropdown
            };
    }
}

const addSpanToPropertyPanel = (propertyPanel, spanClass) => {
    const span = document.createElement("span");
    $(span).addClass(spanClass);
    propertyPanel.panel.children("div.aui-property-panel").children("div.panel-buttons").append(span);
    return span;
}

const getSpanClassName = (parameters, placeholderCssClass) => {
    if (parameters["multi-dropdown"] === "true") {
        placeholderCssClass = placeholderCssClass.concat("-multiple");
    }
    let spanClassName = `edmSelect2Editor ${placeholderCssClass}`;
    if (parameters["css-style-class"]) {
        spanClassName = spanClassName.concat(` ${parameters["css-style-class"]}`);
    }
    return spanClassName;
}

const createSelect2ForPropertyPanelNonPredefined = (macroPlaceholder, specificMacroObject, spanElement, parameters) => {
    createSelect2(macroPlaceholder, specificMacroObject, spanElement, parameters);
}

const createSelect2ForPropertyPanelPredefined = (macroPlaceholder, specificMacroObject, spanElement, parameters) => {
    Edm.Rest.getTextSet(parameters["set-id"]).done((dropdownSet) => {
        createSelect2(macroPlaceholder, specificMacroObject, spanElement, parameters, dropdownSet);
    });
}

const createSelect2ForPropertyPanelStatus = (macroPlaceholder, specificMacroObject, spanElement, parameters) => {
    Edm.Rest.getStatusSet(parameters["set-id"]).done((dropdownSet) => {
        createSelect2(macroPlaceholder, specificMacroObject, spanElement, parameters, dropdownSet);
    });
}

export const createSelect2 = (macroPlaceholder, specificMacroObject, spanElement, parameters, dropdownSet) => {
    const adjustedParameters = getAdjustedParametersForSelect2(parameters, specificMacroObject.adjustParametersFunction, dropdownSet);
    specificMacroObject.createDropdownFunction(spanElement, adjustedParameters.options, adjustedParameters.currentOptions, adjustedParameters.isMultiSelect);
    addClickHandlerForSingleSelect(spanElement, adjustedParameters.isMultiSelect);
    addChangeListenerToDropdown(macroPlaceholder, spanElement, parameters);
}

const getAdjustedParametersForSelect2 = (parameters, adjustParametersFunction, dropdownSet) => {
    const adjustedParameters = {};
    adjustedParameters.isMultiSelect = parameters["multi-dropdown"] === "true";
    adjustedParameters.isIncludeBlank = parameters["include-blank"] !== "false" && !parameters["multi-dropdown"];
    adjustParametersFunction(adjustedParameters, parameters, dropdownSet);
    return adjustedParameters;
}

const adjustParametersNonPredefined = (adjustedParameters, parameters) => {
    parameters["current-option"] ?
        adjustedParameters.currentOptions = parameters["current-option"] :
        adjustedParameters.currentOptions = "";
    parameters["options"] ?
        adjustedParameters.options = parameters["options"].split(",") :
        adjustedParameters.options = ["Yes", "No"];
    if (adjustedParameters.isIncludeBlank) {
        adjustedParameters.options = [" "].concat(adjustedParameters.options);
    }
}

const adjustParametersPredefined = (adjustedParameters, parameters, dropdownSet) => {
    parameters["option-id"] ?
        adjustedParameters.currentOptions = getCurrentOptionsArray(parameters["option-id"], dropdownSet) :
        adjustedParameters.currentOptions = [];
    adjustedParameters.options = dropdownSet.optionsList;
    if (adjustedParameters.isIncludeBlank) {
        adjustedParameters.options = [{id: "", name: ""}].concat(adjustedParameters.options);
    }
}

const adjustParametersStatus = (adjustedParameters, parameters, dropdownSet) => {
    adjustedParameters.options = dropdownSet.optionsList;
    parameters["option-id"] ?
        adjustedParameters.currentOptions = parameters["option-id"] :
        adjustedParameters.currentOptions = adjustedParameters.options[0].id;
}

export const getCurrentOptionsArray = (optionIds, dropdownSet) => {
    return  optionIds.split(",").map(optionId => dropdownSet.optionsList.find(option => option.id === optionId) || {id: optionId, name: "OptionNotFound"});
}

const addClickHandlerForSingleSelect = (spanElement, isMultiSelect) => {
    if (!isMultiSelect) {
        $(spanElement).select2("disable");
        $(spanElement).select2("container").on("click", () => {
            $(spanElement).select2("enable");
            $(spanElement).select2("open");
        });
        $(spanElement).on("select2-close", () => {
            $(spanElement).select2("disable");
        });
    }
}

export const addChangeListenerToDropdown = (macroPlaceholder, spanElement, parameters) => {
    $(spanElement).on("change", () => {
        updateParameters(macroPlaceholder.dataset.macroName, spanElement, parameters);
        updateMacro(macroPlaceholder, parameters);
    })
}

const updateParameters = (macroName, spanElement, parameters) => {
    let selectedOptions = $(spanElement).select2("data");
    if (parameters["multi-dropdown"] !== "true") {
        selectedOptions = [selectedOptions];
    }
    if (macroName === "easy-dropdown-menu-status") {
        parameters["current-option-value"] = selectedOptions[0].text.slice(selectedOptions[0].text.indexOf(">") + 1, -7);
        parameters["option-id"] = selectedOptions[0].id;
    } else {
        const currentOptionValues = selectedOptions.map(option => option.text).join(", ");
        const optionIds = selectedOptions.map(option => option.id).join(",");
        if (macroName === "easy-dropdown-menu") {
            updateParameter(parameters, "current-option", optionIds);
        } else if (macroName === "easy-dropdown-menu-predefined") {
            updateParameter(parameters, "current-option-value", currentOptionValues);
            updateParameter(parameters, "option-id", optionIds);
        }
    }
}

const updateParameter = (parameters, parameterName, parameterValue) => {
    parameterValue === "" ? delete parameters[parameterName] :
        parameters[parameterName] = parameterValue;
}

const updateMacro = (macroPlaceholder, parameters) => {
    Confluence.PropertyPanel.destroy();
    const macroRenderRequest = {
        contentId: Confluence.Editor.getContentId(),
        macro: {
            name: macroPlaceholder.dataset.macroName,
            params: parameters,
            body: ""
        }
    }
    tinymce.confluence.MacroUtils.insertMacro(macroRenderRequest, macroPlaceholder).then((newNode) => {
        const options = {
            originalHeight: false,
            anchorIframe: AJS.Rte.getEditorFrame()
        }
        Confluence.PropertyPanel.createFromButtonModel("macro", newNode, getDefaultButtonsArray(newNode), options);
    });
}

const getDefaultButtonsArray = (newNode) => {
    const buttons = [];
    buttons.push({
        className: "macro-placeholder-property-panel-edit-button",
        text: "Edit",
        click: function () {
            AJS.Confluence.PropertyPanel.destroy();
            tinymce.confluence.macrobrowser.editMacro(newNode);
        }
    });
    buttons.push({
        className: "macro-placeholder-property-panel-remove-button",
        text: "Remove",
        click: function () {
            AJS.Confluence.PropertyPanel.destroy();
            AJS.Rte.getEditor().execCommand('mceConfRemoveMacro', newNode);
        }
    });
    return buttons;
}

