import {renderUserInterface} from './ui/renderer';
import {bindAnchorLinks} from '../util/anchorlinks';
import {getUser} from '../rest/getUser';
import {BUBBLE_USER_IMAGE_TEMPLATE} from './templates/bubbleUserImage.template';
import {WELCOME_MESSAGE_TEMPLATE} from './templates/welcomeEntry.template';
import {parseEntries} from './parse-entries';
import {$} from '../util/environment';
import {getEntryUiTemplate} from './dom-diary-util';

export function renderDiary(diary) {
	return parseEntries(diary).then(entries => {
		return renderEntries(diary, entries);
	}).then(function() {
		const $body = $('body');
		const $diary = $(diary);
		const isPreviewMode = $body.hasClass('content-preview') || $body.hasClass('contenteditor');

		if (!isPreviewMode) {
			renderUserInterface(diary);
			bindAnchorLinks(diary);
		}

		renderUserImages(diary);

		$diary.find('.diary-params .macro-body').remove();
		initPagination($diary);
	}).catch(error => {
		AJS.flag({
			type: 'error',
			title: 'Diary experienced an error',
			body: error
		});
	});
}

function renderEntries(diary, entries) {
	return new Promise(function(resolve) {
		const length = entries.length;
		const diaryContent = diary.querySelector('.content');
		while (diaryContent.lastChild) {
			diaryContent.removeChild(diaryContent.lastChild);
		} // faster than innerHTML = '';
		const domEntryList = document.createElement('ul');

		domEntryList.className = 'entries';
		diaryContent.appendChild(domEntryList);

		let entryListHtml = '';

		if (length > 0) {
			const entryTemplate = getEntryUiTemplate(diary);

			for (let i = 0; i < length; i++) {
				entryListHtml += entryTemplate(entries[i], diary);
			}
		} else if (diary.dataset.hideWelcomeMessage !== 'true') {
			entryListHtml = WELCOME_MESSAGE_TEMPLATE();
		}

		// Setting a 0 second timeout allows other functions waiting in AJS.toInit() to execute and makes it seem like
		// the page is loading quicker. Without this it seems like the website is frozen for a short amount of time when
		// rendering a very big diary.
		setTimeout(() => {
			domEntryList.innerHTML = entryListHtml;

			const $diary = $(diary);
			for (let i = 0; i < length; i++) {
				const entry = entries[i];
				const $entryBody = $diary.find('.entries .entry[data-index="' + entry.index + '"] .body');
				if ($entryBody[0]) {
					$entryBody[0].innerHTML = '';
				} else {
					AJS.warn('Could not find entry with id ' + i + ' in diary ' + diary.id);
				}
				$(entry.contentNode).children().detach().appendTo($entryBody);

			}

			resolve();
		}, 0);
	});
}

function renderUserImages(diary) {
	if (diary.dataset.style === 'bubble') {
		const entries = diary.querySelector('.entries').children;
		const length = entries.length;

		for (let i = 0; i < length; i++) {
			const entry = entries[i];
			const userLogo = entry.querySelector('.user-logo');

			if (userLogo && userLogo.dataset.rendered === 'false') {
				getUser(userLogo.parentNode.dataset.username, userInfo => {
					userLogo.parentNode.innerHTML = BUBBLE_USER_IMAGE_TEMPLATE(userInfo.avatarURL);
				});
			}
		}
	}
}

function initPagination($diary) {
	const entriesPerPage = $diary.data('entries-per-page');
	const updateHasMore = $diary => $diary.attr('data-hasmore', !!$diary.find('.entry.hidden').length);

	if ($.isNumeric(entriesPerPage)) {
		// Hide all elements beyond the configured number of entriesPerPage
		$diary.find(`.entry:gt(${entriesPerPage - 1})`).addClass('hidden');
		updateHasMore($diary);
		$diary.find('.diary-more-button').click(() => {
			// Show the next entriesPerPage items
			$diary.find('.entry:not(.hidden)').last().nextAll(`*:lt(${entriesPerPage})`).removeClass('hidden');
			updateHasMore($diary);
		});
	}
}
