import moment from 'moment/moment';

export function parseEntries(diary) {
	return new Promise(function(resolve, reject) {
		try {
			const macroBody = diary.querySelector('.diary-params .macro-body');
			const tableHead = macroBody.querySelector('table > thead');
			const tableBody = macroBody.querySelector('table > tbody');
			const entryNodes = [];
			if (tableHead) {
				entryNodes.push(...tableHead.children);
			}
			if (tableBody) {
				entryNodes.push(...tableBody.children);
			}
			const entries = [];
			for (let i = 0; i < entryNodes.length; i++) {
				const entryNode = entryNodes[i];
				const entry = tableRowToEntry(entryNode, i);
				entry && entries.push(entry);
			}

			sortEntries(entries, diary.dataset.order === 'descending');
			resolve(entries);
		} catch (e) {
			const message = 'Error getting entries of diary with id: ' + diary.id;
			AJS.warn(message, e);
			reject(message, e);
		}
	});
}

function sortEntries(entries, reverse) {
	const comparator = reverse
		? (a, b) => b.unix - a.unix
		: (a, b) => a.unix - b.unix;

	entries.sort(comparator);
	return entries;
}

function tableRowToEntry(tr, index) {
	function isValidEntry(entry) {
		return entry.userName
			&& entry.userHref
			&& entry.displayName
			&& moment(entry.date, 'YYYY-MM-DD', true).isValid()
			&& moment(entry.time, 'HH:mm', true).isValid()
			&& entry.contentNode;
	}

	try {
		if (tr.childElementCount === 5) {
			const dataCells = tr.children;

			const maybeTimeTag = dataCells[0].getElementsByTagName('time')[0];
			const date = maybeTimeTag
				&& maybeTimeTag.attributes
				&& maybeTimeTag.attributes.dateTime
				&& maybeTimeTag.attributes.dateTime.value;

			const time = dataCells[1].innerHTML;

			const userLink = dataCells[2].querySelectorAll('.confluence-userlink')[0];
			const userName = userLink && userLink.dataset && userLink.dataset.username;
			const userHref = userLink && userLink.attributes && userLink.attributes.href && userLink.attributes.href.value;
			const displayName = userLink && userLink.innerHTML;

			const contentNode = dataCells[3];

			const unix = moment(date + ' ' + time, 'YYYY-MM-DD HH:mm').unix();

			const entry = {
				date,
				time,
				userName,
				userHref,
				displayName,
				contentNode,
				index,
				unix
			};

			if (isValidEntry(entry)) {
				return entry;
			} else {
				AJS.warn('Encountered an invalid diary entry:', entry, tr);
			}
		}
	} catch (e) {
		AJS.warn('Encountered an error while trying to create a diary entry:', e, tr);
	}
}
