
import {getEditorFormat} from '../../rest/getEditorFormat';
import {SIMPLE_EDITOR_TEMPLATE} from '../templates/simpleEditor.template';
import {scrollToEntryAndFocus} from '../../util/anchorlinks';
import {createEditor} from './editor/editor';
import {showUnableToGetEditorFormat} from '../../util/flags/editButtonFlags';
import {$} from '../../util/environment';
import {createEditEntrySaveHandler} from './editor/editEntrySaveHandler';
import {deleteEntry} from '../../rest/deleteEntry';
import {showDeleteEntryError, showDeleteEntryInvalidResponse} from '../../util/flags/deleteEntryFlags';

function copyToClipboard(clipboardValue) {
	const temporaryInput = document.createElement('input');
	temporaryInput.value = clipboardValue;
	document.body.appendChild(temporaryInput);
	temporaryInput.select();
	document.execCommand('copy');
	document.body.removeChild(temporaryInput);
}

export function renderEntryMenu($entry, diary) {
	const $container = $entry.find('.menu-container').first();
	const $shareButton = $container.find('.share-button');
	const $editButton = $container.find('.edit-button');
	const $deleteButton = $container.find('.delete-button');

	$entry.on('mouseenter', () => {
		if (!$entry.hasClass('being-edited')) {
			$container.show();
		}
	});

	$entry.on('mouseleave', () => {
		$container.hide();
	});

	$shareButton.on('click', e => {
		e.preventDefault();
		const currentHref = window.location.origin + window.location.pathname;
		const linkToEntry = currentHref + $entry.find('.entry-anchor').attr('href');

		copyToClipboard(linkToEntry);
		AJS.flag({
			type: 'info',
			title: AJS.I18n.getText('de.scandio.diary.macro.entry.menu.share.flag.title'),
			body: AJS.I18n.getText('de.scandio.diary.macro.entry.menu.share.flag.body'),
			close: 'auto'
		});
	});

	$deleteButton.on('click', e => {
		$deleteButton.attr('disabled', true);
		e.preventDefault();

		if (confirm('Are you sure you want to delete this entry?')) {
			const deleteEntryOptions = {
				version: $entry.attr('data-page-version') || diary.dataset.pageVersion,
				contentid: diary.dataset.pageId,
				macroid: diary.id,
				entryid: $entry.find('.entry-anchor').attr('data-index')
			};

			deleteEntry(deleteEntryOptions, response => {
				if (response.status !== 'success') {
					showDeleteEntryError(response.message);
					$deleteButton.removeAttr('disabled');
				} else {
					location.reload();
				}
			}, error => {
				$deleteButton.removeAttr('disabled');
				showDeleteEntryInvalidResponse(error);
			});
		} else {
			$deleteButton.removeAttr('disabled');
		}
	});

	$editButton.on('click', e => {
		e.preventDefault();
		$editButton.attr('disabled', true);

		const openEditor = AJS.Rte && AJS.Rte.getEditor();
		if (openEditor && openEditor.isDirty()) {
			if (confirm('Your comment will be lost.')) { // TODO i18n
				showEntryEditor($entry, diary, $editButton);
			}
		} else {
			showEntryEditor($entry, diary, $editButton);
		}
	});
}

function showEntryEditor($entry, diary, $editButton) {
	const $container = $entry.find('.menu-container').first();
	$container.hide();

	const editorFormatOptions = {
		pageId: diary.dataset.pageId,
		pageVersion: $entry.attr('data-page-version') || diary.dataset.pageVersion,
		macroId: diary.id,
		entryId: $entry.find('.entry-anchor').attr('data-index')
	};

	getEditorFormat(editorFormatOptions, response => {
		if (response.status === 'success') {
			const $body = $entry.find('.body').first().hide();
			$body.after(SIMPLE_EDITOR_TEMPLATE()); // TODO this is bad. We need an "edit entry editor" template.
			$entry.find('.quick-editor-prompt').first().remove();

			if (diary.dataset.style === 'simple') {
				$entry.find('.quick-comment-body').first().css('margin-top', '4px');
			}

			const editorOptions = {
				quickCommentPrompt: $entry.find('.quick-editor-prompt').first(),
				quickCommentForm: $entry.find('.quick-comment-form').first(),
				quickCommentContainer: $entry.find('.quick-comment-container').first(),
				postInitialise: () => {
					const openEditor = AJS.Rte && AJS.Rte.getEditor();
					openEditor.setContent(response.content);
					openEditor.setDirty(false);
					$entry.find('#rte-button-tasklist').remove();
					scrollToEntryAndFocus($entry);
				},
				preActivate: () => {
					$entry.addClass('being-edited');
					$editButton.removeAttr('disabled');
				},
				saveHandler: createEditEntrySaveHandler($entry, diary, editorFormatOptions.entryId),
				cancelHandler: () => {
					$('.being-edited').each((_, currentEntry) => {
						const $currentEntry = $(currentEntry);
						$currentEntry.removeClass('being-edited').find('entry-editor').first().remove();
						$currentEntry.find('.body').first().show();
						$currentEntry.find('.edit-container').first().show();
						$currentEntry.find('.entry-editor').first().remove();
					});
				}
			};

			createEditor(diary, editorOptions).activate();
		} else {
			$editButton.removeAttr('disabled');
			showUnableToGetEditorFormat(response.message);
		}
	}, error => {

		showUnableToGetEditorFormat(error);
	});
}
