/*
 * Decompiled with CFR 0.152.
 */
package de.resolution.commons.cert.impl;

import de.resolution.commons.cert.api.CertUtil;
import de.resolution.commons.cert.api.CertUtilException;
import de.resolution.commons.cert.api.CertificateAndKey;
import de.resolution.commons.cert.api.CertificateCheckResult;
import de.resolution.commons.cert.impl.CertificateCheckResultBuilder;
import de.resolution.commons.cert.impl.ImmutableCertificateAndKey;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.Cipher;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMEncryptedKeyPair;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.bc.BcPEMDecryptorProvider;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertUtilImpl
implements CertUtil {
    private static final String SIGNATURE_ALGORITHM = "SHA256withRSA";
    private static final int YEARS_VALID = 10;
    private static final int KEY_SIZE = 2048;
    private static final JcaX509CertificateConverter certConverter = new JcaX509CertificateConverter().setProvider(new BouncyCastleProvider());
    private final JcaPEMKeyConverter keyConverter = new JcaPEMKeyConverter();
    private static final Logger logger = LoggerFactory.getLogger(CertUtilImpl.class);
    private static final String BEGIN_CERTIFICATE = "-----BEGIN CERTIFICATE-----\n";
    private static final String END_CERTIFICATE = "\n-----END CERTIFICATE-----";

    @Override
    public CertificateAndKey createCertificate(String subject) throws CertUtilException {
        KeyPair keyPair = this.createKeyPair();
        X509Certificate certificate = this.createCertificate(keyPair, subject);
        return new ImmutableCertificateAndKey(certificate, this.getPEM(certificate), keyPair.getPrivate(), this.encodeBase64(keyPair.getPrivate()));
    }

    @Override
    public X509Certificate createCertificate(KeyPair keyPair, String subject) throws CertUtilException {
        return this.createCertificate(keyPair.getPublic(), keyPair.getPrivate(), subject);
    }

    @Override
    public X509Certificate createCertificate(PublicKey publicKey, PrivateKey privateKey, String subject) throws CertUtilException {
        Date notBefore = new Date();
        Calendar cal = Calendar.getInstance();
        cal.setTime(notBefore);
        cal.add(1, 10);
        Date notAfter = cal.getTime();
        return this.createCertificate(publicKey, privateKey, subject, notBefore, notAfter);
    }

    @Override
    public X509Certificate createCertificate(PublicKey publicKey, PrivateKey privateKey, String subject, Date notBefore, Date notAfter) throws CertUtilException {
        ContentSigner signer;
        X500Name issuer = new X500Name(subject);
        BigInteger serial = BigInteger.valueOf(System.currentTimeMillis());
        SubjectPublicKeyInfo publicKeyInfo = SubjectPublicKeyInfo.getInstance(publicKey.getEncoded());
        X509v3CertificateBuilder certificateBuilder = new X509v3CertificateBuilder(issuer, serial, notBefore, notAfter, issuer, publicKeyInfo);
        try {
            signer = new JcaContentSignerBuilder(SIGNATURE_ALGORITHM).build(privateKey);
        }
        catch (OperatorCreationException e) {
            throw new CertUtilException(e);
        }
        X509CertificateHolder holder = certificateBuilder.build(signer);
        try {
            return certConverter.getCertificate(holder);
        }
        catch (CertificateException e) {
            throw new CertUtilException(e);
        }
    }

    @Override
    public KeyPair createKeyPair() throws CertUtilException {
        try {
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
            keyGen.initialize(2048, new SecureRandom());
            return keyGen.generateKeyPair();
        }
        catch (NoSuchAlgorithmException e) {
            throw new CertUtilException(e);
        }
    }

    @Override
    public String encodeBase64(PrivateKey privateKey) throws CertUtilException {
        StringWriter stringWriter = new StringWriter();
        JcaPEMWriter pemWriter = new JcaPEMWriter(stringWriter);
        try {
            pemWriter.writeObject(privateKey);
            pemWriter.close();
        }
        catch (IOException e) {
            throw new CertUtilException(e);
        }
        return stringWriter.toString();
    }

    @Override
    public String getPEM(X509Certificate cert) throws CertUtilException {
        StringWriter writer = new StringWriter();
        JcaPEMWriter pemWriter = new JcaPEMWriter(writer);
        try {
            pemWriter.writeObject(cert);
            pemWriter.close();
            return writer.toString();
        }
        catch (IOException e) {
            throw new CertUtilException(e);
        }
    }

    @Override
    public X509Certificate readCertificate(String pemString) throws CertUtilException {
        String pemStringWithLines = pemString;
        Pattern patternBegin = Pattern.compile("^-+BEGIN CERTIFICATE-+");
        Matcher matcherBegin = patternBegin.matcher(pemStringWithLines);
        pemStringWithLines = !matcherBegin.find() ? BEGIN_CERTIFICATE + pemStringWithLines : matcherBegin.replaceAll(BEGIN_CERTIFICATE);
        Pattern patternEnd = Pattern.compile("-+END CERTIFICATE-+$");
        Matcher matcherEnd = patternEnd.matcher(pemStringWithLines);
        pemStringWithLines = !matcherEnd.find() ? pemStringWithLines + END_CERTIFICATE : matcherEnd.replaceAll(END_CERTIFICATE);
        StringReader stringReader = new StringReader(pemStringWithLines);
        return this.readCertificate(stringReader);
    }

    private X509Certificate readCertificate(Reader reader) throws CertUtilException {
        try {
            PEMParser pr = new PEMParser(reader);
            Object readObject = pr.readObject();
            pr.close();
            if (readObject == null) {
                throw new CertUtilException("PEMParser returned null");
            }
            logger.debug("Object read from PEM is a {}", (Object)readObject.getClass().getCanonicalName());
            if (readObject instanceof X509CertificateHolder) {
                X509CertificateHolder holder = (X509CertificateHolder)readObject;
                return certConverter.getCertificate(holder);
            }
            throw new CertUtilException("Read Object is no X509CertificateHolder but a " + readObject.getClass().getCanonicalName());
        }
        catch (IOException | CertificateException e) {
            throw new CertUtilException(e);
        }
    }

    @Override
    public PrivateKey readPrivateKey(String base64) throws CertUtilException {
        return this.readPrivateKey(base64, null);
    }

    @Override
    public PrivateKey readPrivateKey(String pemString, String password) throws CertUtilException {
        return this.readPrivateKey(new StringReader(pemString), password);
    }

    private PrivateKey readPrivateKey(Reader reader, String password) throws CertUtilException {
        try {
            PEMParser pr = new PEMParser(reader);
            Object readObject = pr.readObject();
            pr.close();
            if (readObject == null) {
                throw new CertUtilException("PEMParser returned null");
            }
            logger.debug("Object read from PEM is a {}", (Object)readObject.getClass().getCanonicalName());
            if (readObject instanceof PrivateKeyInfo) {
                PrivateKeyInfo privateKeyInfo = (PrivateKeyInfo)readObject;
                return this.keyConverter.getPrivateKey(privateKeyInfo);
            }
            if (readObject instanceof PEMEncryptedKeyPair) {
                PEMEncryptedKeyPair encryptedKeyPair = (PEMEncryptedKeyPair)readObject;
                BcPEMDecryptorProvider decProvider = new BcPEMDecryptorProvider(password.toCharArray());
                PEMKeyPair keyPair = encryptedKeyPair.decryptKeyPair(decProvider);
                PrivateKeyInfo privateKeyInfo = keyPair.getPrivateKeyInfo();
                return this.keyConverter.getPrivateKey(privateKeyInfo);
            }
            if (readObject instanceof PEMKeyPair) {
                PrivateKeyInfo privateKeyInfo = ((PEMKeyPair)readObject).getPrivateKeyInfo();
                return this.keyConverter.getPrivateKey(privateKeyInfo);
            }
            throw new CertUtilException("Read Object is no PrivateKeyInfo or PEMEncryptedKeyPair but a " + readObject.getClass().getCanonicalName());
        }
        catch (IOException e) {
            throw new CertUtilException(e);
        }
    }

    @Override
    public CertificateCheckResult check(String certificatePEM, String privateKeyPEM) {
        CertificateCheckResultBuilder checkBuilder = new CertificateCheckResultBuilder();
        try {
            if (certificatePEM == null || certificatePEM.trim().isEmpty()) {
                checkBuilder.inValid();
                return checkBuilder.build();
            }
            X509Certificate cert = this.readCertificate(certificatePEM);
            checkBuilder.certificate(cert);
            checkBuilder.valid();
            this.checkExpired(cert, checkBuilder);
            if (privateKeyPEM == null || privateKeyPEM.trim().isEmpty()) {
                checkBuilder.noPrivateKey();
            } else {
                checkBuilder.privateKey();
                PrivateKey privateKey = this.readPrivateKey(privateKeyPEM);
                PublicKey publicKey = cert.getPublicKey();
                Cipher encCipher = Cipher.getInstance("RSA");
                encCipher.init(1, publicKey);
                byte[] source2 = "thisIsAtestString".getBytes();
                byte[] encrypted = encCipher.doFinal(source2);
                Cipher decCipher = Cipher.getInstance("RSA");
                decCipher.init(2, privateKey);
                byte[] decrypted = decCipher.doFinal(encrypted);
                checkBuilder.privateKey();
                if (Arrays.equals(source2, decrypted)) {
                    checkBuilder.valid();
                } else {
                    checkBuilder.inValid().message("Certificate and key do not match");
                }
            }
        }
        catch (Exception e) {
            logger.info("Certificate check failed", (Throwable)e);
            checkBuilder.inValid().message(e.getMessage());
        }
        return checkBuilder.build();
    }

    private void checkExpired(X509Certificate cert, CertificateCheckResultBuilder checkBuilder) {
        try {
            cert.checkValidity();
            checkBuilder.expired(false);
        }
        catch (CertificateExpiredException | CertificateNotYetValidException e) {
            checkBuilder.message("Certificate is expired or not yet valid: " + e.getMessage());
            checkBuilder.expired(true);
        }
    }
}

