/*global
    jQuery, NAME, window
*/
/**
 * Accessibility Helpers
 *
 MIT License

 Copyright (c) 2018 Adina Halter

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all
 copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 SOFTWARE.
 */
define('communardo/confluence/accessibility/autocomplete',
    [
        'jquery',
        'communardo/confluence/accessibility/name'
    ],
    function (
        $,
        NAME
    ) {
        'use strict';

        const I18N = {
            removeButtonLabels: {
                'user': AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.removeUser'),
                'group': AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.removeGroup'),
                'email': AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.removeEmail')
            },
            messages: {
                ADD_ANOTHER_RECIPIENT: AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.addAnotherRecipient'),
                ADD_ANOTHER_RECIPIENT_USER: AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.addAnotherRecipientUser')
            }
        };

        var recipientCount = 1,
            $widget = null,
            $hiddenInput = null,
            isUsersOnly = false,
            $input = null,
            $username = null,
            $userKey = null,
            $groupname = null,
            $clearText = null,
            $addUser = null,
            $recipientList = null,
            inputVal = "",
            $results = null,
            results = [],
            $live = null,
            key = NAME.keyboard,
            directions = AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.autocomplete.directions'),
            liMarkup = '<li id="" class="autocomplete-item" role="option" aria-selected="false" tabindex="-1" ';


        var Autocomplete = {

            // get results from Confluence's REST API
            getUserResults: function () {
                // this is for too short username requests
                if ($input.val().length < 2) {
                    return [];
                } else {
                    var resultList = [];
                    var restUrl;
                    //check if we have to search for users only or users + groups
                    if ($widget.attr('data-widget') === "accessible-autocomplete-groups") {
                        restUrl = "/rest/prototype/1/search/user-or-group.json?max-results=10&query=";
                    } else if ($widget.attr('data-widget') === "accessible-autocomplete-users") {
                        restUrl = "/rest/prototype/1/search/user.json?max-results=10&query=";
                    }
                    if (restUrl) {
                        $.ajax({
                            url: contextPath + restUrl + $input.val(),
                            async: false
                        }).done(function (data) {
                            if (data.result) {
                                resultList = data.result;
                            }
                        });
                        return resultList;
                    } else {
                        console.warn("Could not determine the autocomplete widget. Please check the 'data-widget' attribute.");
                        return [];
                    }
                }
            },

            positionResults: function () {
                // stop if this has already been set
                if ($results.is('[style*="width"]')) {
                    return;
                }
                $results.css({
                    left: $input.position().left + "px",
                    top: $input.position().top + $input.outerHeight() + "px",
                    "min-width": $input.outerWidth() + "px"
                });

            },

            rePositionResults: function (resultList, input) {
                var $resultList = $(resultList),
                    $inputField = $(input);
                $resultList.css({
                    left: $inputField.position().left + "px",
                    top: $inputField.position().top + $inputField.outerHeight() + "px",
                    "min-width": $inputField.outerWidth() + "px"
                });
            },

            buildListHtml: function (results) {
                var resultsMarkup = "", i;
                for (i = 0; i < results.length; i += 1) {
                    if (results[i].type === "user")
                        resultsMarkup += liMarkup + 'data-type="' + results[i].type + '"' + ' data-username="' + results[i].username + '"' + ' data-userKey="' + results[i].userKey + '">' + results[i].title + "</li>";
                    else if (results[i].type === "group")
                        resultsMarkup += liMarkup + 'data-type="' + results[i].type + '">' + results[i].name + "</li>";
                }
                $results.html(resultsMarkup);
                $results.show();
                $input.attr('aria-expanded', 'true');
            },

            announceResults: function () {
                var number = results.length,
                    textToRead = number + " " + AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.autocomplete.resultsAvailable') + " " + directions;
                // if results length === 0 then say "no search results"
                if (results.length === 0) {
                    textToRead = AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.autocomplete.noResults');
                }
                NAME.access.announcements($live, textToRead);
            },

            markSelected: function ($selectionToMark) {
                // don't mark anything on the results list if we're back at the input field
                if ($selectionToMark.length === 0) {
                    return;
                }
                var activeItemId = 'selectedOption';
                $selectionToMark.attr('aria-selected', 'true').attr('id', activeItemId);
                $input.attr('aria-activedescendant', activeItemId);
            },

            clearSelected: function () {
                $input.attr('aria-activedescendant', '');
                $results.find('[aria-selected="true"]').attr('aria-selected', 'false').attr('id', '');
            },

            closeResults: function () {
                this.clearSelected();
                $results.hide();
                $input.attr('aria-expanded', 'false');
            },

            isEmail: function (value) {
                return value.indexOf("@") > -1;
            },

            isUser: function () {
                // this method should be reworked too but since the check currently relies on global variables and
                // changing that would need to much effort we currently just use it as it is
                return $username.val() !== "" && $userKey.val() !== "";
            },

            isElementAlreadyInList: function (selectedElementText) {
                var isElementAlreadyInList = false;
                $recipientList.find('li').each(function () {
                    if (!isElementAlreadyInList) {
                        if (jQuery(this).attr("data-type") === "user") {
                            isElementAlreadyInList = $(this).find('a').text() === selectedElementText;
                        } else {
                            // overkill check for the text of the html element
                            isElementAlreadyInList = $(this).clone()    //clone the element
                                .children() //select all the children
                                .remove()   //remove all the children
                                .end()  //again go back to selected element
                                .text() === selectedElementText;
                        }
                    }
                });
                return isElementAlreadyInList;
            },

            buildRemoveButton: function ($parentListElement, type) {
                var $removeButton = AJS.$('<button>');
                $removeButton.text(I18N.removeButtonLabels[type]);
                $removeButton.click(function () {
                    $parentListElement.remove();
                    // remove username from hidden input
                    if ($hiddenInput && $hiddenInput.val().indexOf($username.val()) > -1) {
                        $hiddenInput.val($hiddenInput.val().replace("," + $username.val()), "");
                    }
                });
                return $removeButton;
            },

            buildBaseListElement: function (type) {
                var $listElement = AJS.$('<li>');
                $listElement.data('type', type);
                return $listElement;
            },

            buildProfileLink: function (username) {
                var $profileLink = AJS.$('<a>');
                $profileLink.attr('href', AJS.contextPath() + AJS.contextPath() + '/display/~' + username);
                $profileLink.text(username);
                return $profileLink;
            },

            buildProfileListElement: function (userKey, username) {
                var $profileLink, $removeButton, $elementLi;
                $elementLi = this.buildBaseListElement('user');
                $elementLi.data('userKey', userKey);
                $elementLi.data('username', username);
                $profileLink = this.buildProfileLink(username);
                $removeButton = this.buildRemoveButton($elementLi, 'user');
                $elementLi.append($profileLink);
                $elementLi.append($removeButton);
                return $elementLi;
            },

            buildGenericListElement: function (elementText, type) {
                var $elementLi;
                $elementLi = this.buildBaseListElement(type);
                $elementLi.data('value', elementText);
                $elementLi.text(elementText);
                $elementLi.append(this.buildRemoveButton($elementLi, type));
                return $elementLi;
            },

            clearHiddenInputFields: function () {
                // clear username and groupname
                $username.val("");
                $userKey.val("");
                $groupname.val("");
            },

            addReadFeedbackMessage: function (textToRead) {
                var messageToAdd = $widget.data('widget') === "accessible-autocomplete-groups" ? I18N.messages.ADD_ANOTHER_RECIPIENT : I18N.messages.ADD_ANOTHER_RECIPIENT_USER;
                NAME.access.announcements($live, textToRead + "! " + messageToAdd);
            },

            addRecipientToList: function () {
                var inputValue, textToRead, $elementLi;

                inputValue = $input.val();
                if (inputValue === "") return; // doing nothing since no value provided..

                if (this.isElementAlreadyInList(inputValue)) {
                    textToRead = inputValue + " " + AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.alreadyInList');
                } else {
                    if (this.isUser(inputValue)) {
                        $elementLi = this.buildProfileListElement($userKey.val(), $username.val());
                        // add username to hidden input field if it exists - not sure why, but its probably because of
                        // something in the create space from teams template dialog. Should be changed someday...
                        if ($hiddenInput) {
                            var valueToSet = $hiddenInput.val() === "" ? $username.val() : $hiddenInput.val() + "," + $username.val();
                            $hiddenInput.val(valueToSet);
                        }
                    } else if (this.isEmail(inputValue) && !isUsersOnly) {
                        $elementLi = this.buildGenericListElement(inputValue, 'email');
                    } else if ($groupname.val() !== "" && !isUsersOnly) {
                        $elementLi = this.buildGenericListElement(inputValue, 'group');
                    }

                    if ($elementLi) {
                        $recipientList.append($elementLi);
                        this.clearHiddenInputFields();
                        textToRead = inputValue + " " + AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.addedToList');
                    } else {
                        textToRead = AJS.I18n.getText('de.communardo.confluence.plugins.apc.messages.sharepage.provideData');
                    }
                }
                this.addReadFeedbackMessage(textToRead);
            },

            autocomplete: function () {
                // if input value didn't change, return
                if ($input.val() === inputVal) {
                    return;
                }
                // save new input value
                inputVal = $input.val();
                if (!this.isEmail(inputVal)) {
                    // get and save autocomplete results
                    results = this.getUserResults();
                    // build list HTML
                    if (results.length === 0) {
                        this.closeResults();
                    } else {
                        this.buildListHtml(results);
                    }
                    // aria-live results
                    this.announceResults();
                }
            },

            arrowing: function (kc) {
                var $thisActiveItem = $results.find('[aria-selected="true"]'),
                    $nextMenuItem;
                // don't do anything if no results
                if (!results.length) {
                    return;
                }
                if (kc === key.down) {
                    // find the next list item to be arrowed to
                    $nextMenuItem = ($thisActiveItem.length !== 0)
                        ? $thisActiveItem.next('li')
                        : $results.children().eq(0); //first item in list
                }
                if (kc === key.up) {
                    // find the previous list to be arrowed to
                    $nextMenuItem = ($thisActiveItem.length !== 0)
                        ? $thisActiveItem.prev('li')
                        : $results.children().eq(-1); //last item in list
                }
                this.clearSelected();
                this.markSelected($nextMenuItem);
            },

            populating: function () {
                var selectedElement = $results.find('[aria-selected="true"]');
                var selectedText = selectedElement.text();
                if (selectedText === "") {
                    selectedText = inputVal;
                }
                $input.val(selectedText);
                if (selectedElement.length) {
                    // clear hidden fields
                    $userKey.val("");
                    $username.val("");
                    $groupname.val("");
                    if (selectedElement.attr("data-type") === "user") {
                        $username.val(selectedElement.attr("data-username"));
                        $userKey.val(selectedElement.attr("data-userKey"));
                    } else if (selectedElement.attr("data-type") === "group")
                        $groupname.val(selectedElement.text())
                }
            },

            eventListeners: function () {
                var self = this;
                // close results if click outside $input and $results
                $(document).on('click', function (e) {
                    var $container = $input.add($results);
                    if (NAME.general.senseClickOutside($(e.target), $container)) {
                        Autocomplete.closeResults();
                        return;
                    }
                });
                $input.on('keyup', function (e) {
                    var kc = e.keyCode;
                    if (kc === key.up || kc === key.down || kc === key.tab || kc === key.enter || kc === key.esc) {
                        return;
                    }
                    Autocomplete.autocomplete();
                });
                $input.on('keydown', function (e) {
                    var kc = e.keyCode;
                    if (kc === key.tab) {
                        Autocomplete.closeResults();
                        return;
                    }
                    if (kc === key.enter) {
                        e.preventDefault();
                        Autocomplete.closeResults();
                        return;
                    }
                    if (kc === key.up || kc === key.down) {
                        e.preventDefault();
                        Autocomplete.arrowing(kc);
                        Autocomplete.populating();
                        return;
                    }
                    if (kc === key.esc) {
                        $input.val(inputVal);
                        Autocomplete.closeResults();
                    }
                });
                $results.on('click', function (e) {
                    $input.val(e.target.textContent);
                    Autocomplete.clearSelected();
                    Autocomplete.markSelected($(e.target));
                    Autocomplete.populating();
                    Autocomplete.closeResults();
                    $input.focus();
                });
                $results.hover(function () {
                    Autocomplete.clearSelected();
                });
                $clearText.on('click', function () {
                    inputVal = '';
                    $input.val(inputVal);
                    self.clearHiddenInputFields();
                });
                $addUser.on('click', function () {
                    Autocomplete.addRecipientToList();
                    recipientCount++;
                    inputVal = '';
                    $input.val(inputVal);
                })
            },

            /*
                How to use the autocomplete user/group widget:
                    - create an autocomplete html widget (see e.g.: sharepage.vm)
                    - load this js module via: var Autocomplete = require('communardo/confluence/accessibility/autocomplete')
                    - initialize widget via: Autocomplete.init(widget, hiddenInput)
                    - widget: html widget with following data attribute: data-widget="accessible-autocomplete-groups"
                    - hiddenInput: optional hidden input of an existing AJS autocomplete field. e.g.: when you replace an
                    Atlassian autocomplete field in a form with the accessibility autocomplete. the input data of the accessibility
                    widget will be transferred to the hidden field
             */
            init: function (widget, hiddenInput) {
                // initialize the html autocomplete widget
                $widget = $(widget);
                // optional: register hidden input of autocomplete fields. e.g.: for creating a team space
                $hiddenInput = hiddenInput ? $(hiddenInput) : null;
                // if we have a hidden input -> only users can be added to the results list (e.g. create team space)
                isUsersOnly = !!hiddenInput;
                $input = $widget.find('#search');
                $username = $widget.find('#apcusername');
                $userKey = $widget.find('#apcuserKey');
                $groupname = $widget.find('#apcgroupname');
                $clearText = $widget.find('#clearText');
                $addUser = $widget.find('#addUser');
                $recipientList = $widget.find('#recipientList');
                $results = $widget.find('#results');
                $live = $widget.find('[aria-live]');
                this.eventListeners();
                this.positionResults();
            }
        };

        return Autocomplete;
    });
