define("ite/page-restrictions",
    [
        'jquery', 'ajs', 'backbone', 'underscore',
        'ite/restrictions/model/restriction', 'ite/core/flags', 'ite/restrictions/model/restriction-collection'
    ],
    function ($, AJS, Backbone, _, Restriction, flag, RestrictionCollection) {
        "use strict";

        var restrictionChanged = false;
        var ignoreChangeEvents = false;
        var _this;
        var entityType = -1;

        var PageRestriction = Backbone.View.extend({
            el: "body",
            tagName: "div",
            className: "ite-page-restrictions-view",
            template: ACTONIC.INLINETABLE.RESTRICTIONS.PAGE,
            events: {
                "click #ite-restrictions": "render",
                "click #submit": "destroy",

                "click #add-restriction": "addRestriction",
                "click #remove-restrictions": "removeAllRestrictions"
            },

            initialize: function (options) {
                _this = this;
                _.bindAll(this, "render", "destroy");

                if (AJS.Meta.get('page-id') && (AJS.Meta.get('page-id') === AJS.Meta.get('latest-page-id')))
                    entityType = 1;
                else {
                    entityType = 0;
                    this.render();
                }
            },

            render: function (event) {
                if (event) event.preventDefault();


                var restrictions = new RestrictionCollection();
                restrictions.fetch({ cache: false }).done(function () {
                    if (AJS.Meta.get('page-id') && (AJS.Meta.get('page-id') === AJS.Meta.get('latest-page-id'))) {
                        $("#content").append("<div id='ite-restriction-container'></div>")
                    }

                    _this.collection = restrictions;

                    if (entityType == 1) {
                        _this.$el.append(_this.template.dialog());
                        AJS.dialog2("#ite-restriction-page-dialog").show();
                    }
                    _this._onRestrictionLoadSuccess();
                });
            },

            destroy: function (event) {
                if (restrictionChanged)
                    location.reload();
                else {
                    event.preventDefault();
                    AJS.dialog2("#ite-restriction-page-dialog").hide();
                    $("#ite-restriction-page-dialog").remove();
                }
            },

            editRestriction: function () {
                //console.log($(this).closest("tr"));
                var restrictionId = $(this).closest("tr").attr("data-id");
                $(this).closest("tr").prop("id", "change-restriction");
                $(this).closest("tr").html($(_this.template.newRestriction(restrictionId)).children())
                changeRestrictionButtonsState(false);

                $("#type").auiSelect2({
                    minimumResultsForSearch: -1
                });

                initializeUserPicker();
                initializeGroupPicker();

                _this.restrictionGlobalButtonEvents();
            },

            addRestriction: function () {
                checkRestrictionLengthNotification();
                $("#restriction-body").append(_this.template.newRestriction(0));
                changeRestrictionButtonsState(false);

                initializeUserPicker();
                initializeGroupPicker();

                _this.restrictionGlobalButtonEvents();
            },

            restrictionGlobalButtonEvents: function () {
                $("#save").off("click");
                $("#save").on("click", _this.saveRestriction);

                $("#cancel").off("click");
                $("#cancel").on("click", _this.cancelRestriction);
            },

            restrictionRowButtonsEvents: function () {
                $(".restrictions.edit").off("click");
                $(".restrictions.edit").on("click", _this.editRestriction);

                $(".restrictions.remove").off("click");
                $(".restrictions.remove").on("click", _this.removeRestriction);
            },

            saveRestriction: function () {
                var isUpdate = $("#change-restriction").attr("data-id") && _this.collection.get($("#change-restriction").attr("data-id")) === 0;
                var restriction;
                if (isUpdate) {
                    restriction = _this.collection.get($("#change-restriction").attr("data-id"));

                    restriction.set({
                        users: $("#change-restriction #users").val(),
                        groups: $("#change-restriction #groups").val(),
                        restrictionType: $("#change-restriction #type").val()
                    });
                    restrictionChanged = true;
                    _this._onRestrictionLoadSuccess();
                } else {
                    restriction = new Restriction({
                        // id: $("#change-restriction").attr("data-id"),
                        entityType: entityType,
                        spaceKey: AJS.Meta.get("space-key"),
                        pageId: AJS.Meta.get("page-id"),
                        users: $("#change-restriction #users").val(),
                        groups: $("#change-restriction #groups").val(),
                    });
                    restriction.save(null, {
                        success: function (response) {
                            //console.log(response);
                            if (response.attributes.id) {
                                _this.collection.push(new Restriction(response.attributes));
                            }
                            restrictionChanged = true;
                            _this._onRestrictionLoadSuccess();

                            // function sendToAmplitude put to window object when we initialize amplitude on front
                            // (in cases when amplitude tracking is checked in the ITE configuration)
                            if (window.sendToAmplitude) {
                                sendToAmplitude('inlineTableRestrictions-save');
                            }
                        },
                        error: function (model, xhr, options) {
                            flag.simpleWarning(AJS.I18n.getText(xhr.responseText));
                        }
                    });
                }
            },

            removeRestriction: function () {
                var restrictionId = $(this).closest("tr").attr("data-id");
                var restriction = _this.collection.get(restrictionId);
                restriction.destroy();
                _this.collection.remove(restriction);
                $("#restriction-" + restrictionId).remove();
                checkRestrictionLengthNotification();
                restrictionChanged = true;

                // function sendToAmplitude put to window object when we initialize amplitude on front
                // (in cases when amplitude tracking is checked in the ITE configuration)
                if (window.sendToAmplitude) {
                    sendToAmplitude('inlineTableRestrictions-remove');
                }
            },

            _onRestrictionLoadSuccess: function () {
                //console.log(_this.collection.toJSON());
                $("#restriction-body").html(_this.template.renderTableData({
                    data: _.extend(_this.collection.toJSON()),
                    entityType: entityType
                }));
                if (_this.collection.getEntitiesSize({ entityType: 0 }) > 0) $(".space-permission-notification").show();
                _this.restrictionRowButtonsEvents();
                changeRestrictionButtonsState(true);
                checkRestrictionLengthNotification();
            },

            cancelRestriction: function () {
                _this._onRestrictionLoadSuccess();
                checkRestrictionLengthNotification();
            },

            removeAllRestrictions: function () {
                var options = {
                    path: AJS.Meta.get("page-id") || AJS.Meta.get("space-key"),
                    type: AJS.Meta.get("page-id") ? "page" : "space",
                    successCallback: function () {
                        _this.collection.reset(null);
                        _this._onRestrictionLoadSuccess();
                    }
                }
                _this.collection.removeAll(options);

                // function sendToAmplitude put to window object when we initialize amplitude on front
                // (in cases when amplitude tracking is checked in the ITE configuration)
                if (window.sendToAmplitude) {
                    sendToAmplitude('inlineTableRestrictions-removeAll');
                }
            }
        })

        function checkRestrictionLengthNotification() {
            if ($(".restriction-row").length && $(".restriction-row").length > 0) {
                $(".no-restrictions").hide();
                $(".restriction-everyone-notification").show();
            } else {
                $(".no-restrictions").show();
                $(".restriction-everyone-notification").hide();
            }
        }

        function changeRestrictionButtonsState(active) {
            $("#add-restriction").prop("disabled", !active);
        };

        function initializeGroupPicker() {
            var parent = $("#groups").parent();
            $("#groups").remove();
            $("#s2id_groups").remove();
            $(parent).append("<input type=\"text\" class=\"select\" id=\"groups\">");

            AJS.$("#groups").auiSelect2({
                placeholder: AJS.I18n.getText("ite.restrictions.dialog.table.selects.groupPicker.placeholder"),
                formatNoMatches: () => AJS.I18n.getText("ite.restrictions.dialog.table.selects.noMatchesFound"),
                formatSearching: () => AJS.I18n.getText("ite.restrictions.dialog.table.selects.searching"),
                allowClear: true,
                ajax: {
                    type: "GET",
                    cache: true,
                    dataType: "json",
                    url: AJS.contextPath() + "/rest/ctiperestresource/1.0/picker/groups",
                    data: function (term) {
                        return {
                            query: term,
                            maxResults: 100
                        };
                    },
                    results: function (data) {
                        //log.debug("initializeTargetGroupPicker = " + JSON.stringify(data));
                        var results = [];
                        $.each(data, function (index, item) {
                            results.push({ id: item.text, text: item.text });
                        });
                        return {
                            results: results
                        };
                    }
                }
            });

            $("#groups").off("change");
            $("#groups").on("change", function () {
                if (!ignoreChangeEvents) {
                    $("#users").select2("disable");
                    $("#s2id_users a.select2-choice").addClass("disabled-picker");
                }
            });
        }

        function initializeUserPicker() {
            var parent = $("#users").parent();
            $("#users").remove();
            $("#s2id_users").remove();
            $(parent).append("<input type=\"text\" class=\"select\" id=\"users\">");

            AJS.$("#users").auiSelect2({
                placeholder: AJS.I18n.getText("ite.restrictions.dialog.table.selects.userPicker.placeholder"),
                formatNoMatches: () => AJS.I18n.getText("ite.restrictions.dialog.table.selects.noMatchesFound"),
                formatSearching: () => AJS.I18n.getText("ite.restrictions.dialog.table.selects.searching"),
                dropdownAutoWidth: true,
                ajax: {
                    url: AJS.contextPath() + "/rest/ctiperestresource/1.0/picker/users",
                    dataType: "json",
                    data: function (term) {
                        return {
                            q: term
                        };
                    },
                    results: function (data) {
                        return {
                            results: $.map(data, function (el) {
                                return {
                                    id: el.id,
                                    text: el.text + ' (' + el.id + ')'
                                }
                            })
                        };
                    }
                },
            });

            $("#users").off("change");
            $("#users").on("change", function () {
                if (!ignoreChangeEvents) {
                    $("#groups").select2("disable");
                    $("#s2id_groups a.select2-choice").addClass("disabled-picker");
                }
            });
        }

        return PageRestriction;
    }
)