define("ite/inline-editing/table-grid", ["jquery", "ite/inline-editing/utils"], function ($, InlineEditingUtils) {
    function TableGrid() {
        var _this = this;
        var grid;
        var startPos;
        var endPos;
        var selectedCell;

        var table, dom;

        this.init = function (_table, _dom) {
            table = _table;
            dom = _dom;

            buildGrid();

            selectedCell = dom.select('td.mceSelected,th.mceSelected')[0];
            if (selectedCell) {
                startPos = getPos(selectedCell);
                endPos = findEndPos();
                selectedCell = getCell(startPos.x, startPos.y);
            }
        }

        this.insertCol = function (before) {
            var posX;
            var lastCell;
            var cells = [];

            each(grid, function (row, y) {
                each(row, function (cell, x) {
                    if (isCellSelected(cell)) {
                        posX = x;

                        if (before) {
                            return false;
                        }
                    }
                });

                if (before) {
                    return !posX;
                }
            });

            if (posX == 0 && before && hasNumCol()) {
                return false;
            }

            each(grid, function (row, y) {
                var cell;
                var rowSpan;
                var colSpan;

                if (!row[posX]) {
                    return;
                }

                cell = row[posX].elm;

                if (AJS.DarkFeatures.isEnabled('confluence.table.resizable')) {
                    if (y === 0) {
                        var $table = $(cell).closest('table');

                        var $targetCol = $table.find('>colgroup>col').eq(posX);
                        var $newCol = $('<col>');

                        if (before) {
                            $newCol.insertBefore($targetCol);
                        } else {
                            $newCol.insertAfter($targetCol);
                        }

                        if ($table.hasClass('fixed-table')) {
                            if (!copiedColumnWidth || parseFloat(copiedColumnWidth) === 0) {
                                $newCol.css({ width: DEFAULT_COLUMN_WIDTH + CELL_PADDING_LEFT_RIGHT * 2 + CELL_BORDER_WIDTH });
                            } else {
                                $newCol.css({ width: copiedColumnWidth });
                            }
                        }
                    }
                }

                if (cell != lastCell) {
                    colSpan = getSpanVal(cell, 'colspan');
                    rowSpan = getSpanVal(cell, 'rowspan');

                    if (colSpan == 1) {
                        var newCell = cloneCell(cell);
                        if (!before) {
                            dom.insertAfter(newCell, cell);
                            fillLeftDown(posX, y, rowSpan - 1, colSpan);
                        } else {
                            cell.parentNode.insertBefore(newCell, cell);
                            fillLeftDown(posX, y, rowSpan - 1, colSpan);
                        }
                        cells.push(newCell);
                    } else {
                        setSpanVal(cell, 'colSpan', cell.colSpan + 1);
                    }

                    lastCell = cell;
                }
            });

            return cells;
        }

        this.split = function () {
            each(grid, function (row, y) {
                each(row, function (cell, x) {
                    var colSpan;
                    var rowSpan;
                    var i;

                    if (isCellSelected(cell)) {
                        cell = cell.elm;
                        colSpan = getSpanVal(cell, 'colspan');
                        rowSpan = getSpanVal(cell, 'rowspan');

                        if (colSpan > 1 || rowSpan > 1) {
                            setSpanVal(cell, 'rowSpan', 1);
                            setSpanVal(cell, 'colSpan', 1);

                            for (i = 0; i < colSpan - 1; i++) {
                                dom.insertAfter(cloneCell(cell), cell);
                            }

                            fillLeftDown(x, y, rowSpan - 1, colSpan);
                        }
                    }
                });
            });
        }

        this.merge = function (cell) {
            var startX;
            var startY;
            var endX;
            var endY;
            var x;
            var y;
            var startCell;
            var endCell;
            var children;
            var count;

            startX = startPos.x;
            startY = startPos.y;
            endX = endPos.x;
            endY = endPos.y;

            if (startX == 0 && hasNumCol()) {
                return;
            }

            startCell = getCell(startX, startY);
            endCell = getCell(endX, endY);

            if (startCell && endCell && startCell.part == endCell.part) {
                _this.split();
                buildGrid();

                startCell = getCell(startX, startY).elm;
                setSpanVal(startCell, 'colSpan', (endX - startX) + 1);
                setSpanVal(startCell, 'rowSpan', (endY - startY) + 1);

                for (y = startY; y <= endY; y++) {
                    for (x = startX; x <= endX; x++) {
                        if (!grid[y] || !grid[y][x]) {
                            continue;
                        }

                        cell = grid[y][x].elm;

                        if (cell != startCell) {
                            children = tinymce.grep(cell.childNodes);
                            // TODO: avoid usage of some custom "each" method, change implementation to default javascript
                            each(children, function (node) {
                                startCell.appendChild(node);
                            });

                            if (children.length) {
                                children = tinymce.grep(startCell.childNodes);
                                count = 0;
                                each(children, function (node) {
                                    if (node.nodeName === 'BR' && dom.getAttrib(node, 'data-mce-bogus') && count++ < children.length - 1) {
                                        startCell.removeChild(node);
                                    }
                                });
                            }

                            dom.remove(cell);
                        }
                    }
                }

                cleanup();
            }
        }

        this.deleteCols = function () {
            var cols = [];

            each(grid, function (row, y) {
                each(row, function (cell, x) {
                    if (isCellSelected(cell) && tinymce.inArray(cols, x) === -1) {
                        if (AJS.DarkFeatures.isEnabled('confluence.table.resizable')) {
                            $(cell.elm).closest('table').find('>colgroup>col').eq(x).remove();
                        }
                        each(grid, function (row) {
                            if (row[x]) {
                                var cell = row[x].elm;
                                var colSpan;

                                colSpan = getSpanVal(cell, 'colSpan');

                                if (colSpan > 1) {
                                    setSpanVal(cell, 'colSpan', colSpan - 1);
                                } else {
                                    var $cell = $(cell);
                                    var $tr = $cell.closest('tr');
                                    var trNumber = $tr.data('trNumber');
                                    if (trNumber !== undefined) {
                                        window.updates.push({
                                            tableNumber: InlineEditingUtils.getTableNumber($tr),
                                            trNumber: $tr.data('trNumber'),
                                            tdNumber: $cell.parent().children().index($cell),
                                            remove: true
                                        });
                                    }
                                    dom.remove(cell);
                                }
                            }
                        });
                        cols.push(x);
                    }
                });
            });

            cleanup();
        }

        function getPos(target) {
            var pos;

            each(grid, function (row, y) {
                each(row, function (cell, x) {
                    if (cell.elm == target) {
                        pos = { x: x, y: y };
                        return false;
                    }
                });

                return !pos;
            });

            return pos;
        }

        function buildGrid() {
            var startY = 0;

            grid = [];

            each(['thead', 'tbody', 'tfoot'], function (part) {
                var rows = dom.select(part + ' tr', table);

                if (part == 'tbody') {
                    each(rows, function (row, index) {
                        if (index > 0 && row && row.className == 'tablesorter-headerRow') {
                            rows.splice(index, 1);
                        }
                    });
                }

                each(rows, function (tr, y) {
                    y += startY;

                    each(dom.select('td, th', tr), function (td, x) {
                        var x2;
                        var y2;
                        var rowspan;
                        var colspan;

                        if (grid[y]) {
                            while (grid[y][x]) {
                                x++;
                            }
                        }

                        rowspan = getSpanVal(td, 'rowspan');
                        colspan = getSpanVal(td, 'colspan');

                        for (y2 = y; y2 < y + rowspan; y2++) {
                            if (!grid[y2]) {
                                grid[y2] = [];
                            }

                            for (x2 = x; x2 < x + colspan; x2++) {
                                grid[y2][x2] = {
                                    part: part,
                                    real: y2 == y && x2 == x,
                                    elm: td,
                                    rowspan: rowspan,
                                    colspan: colspan
                                };
                            }
                        }
                    });
                });

                startY += rows.length;
            });
        }

        function getCell(x, y) {
            var row;

            row = grid[y];
            if (row) {
                return row[x];
            }
        }

        function hasNumCol() {
            return $(table.rows[0]).children(':first').hasClass('numberingColumn');
        }

        function setSpanVal(td, name, val) {
            if (td) {
                val = parseInt(val);

                if (val === 1) {
                    td.removeAttribute(name, 1);
                } else {
                    td.setAttribute(name, val, 1);
                }
                var $td = $(td);
                var $tr = $td.closest('tr');
                window.updates.push({
                    tableNumber: InlineEditingUtils.getTableNumber($tr),
                    trNumber: $tr.data('trNumber'),
                    tdNumber: $td.parent().children().index($td),
                    colspan: val
                });
            }
        }

        function isCellSelected(cell) {
            return cell && (dom.hasClass(cell.elm, 'mceSelected') || cell == selectedCell);
        }

        function fillLeftDown(x, y, rows, cols) {
            var tr;
            var x2;
            var r;
            var c;
            var cell;

            tr = grid[y][x].elm.parentNode;
            for (r = 1; r <= rows; r++) {
                tr = dom.getNext(tr, 'tr');

                if (tr) {
                    for (x2 = x; x2 >= 0; x2--) {
                        cell = grid[y + r][x2].elm;

                        if (cell.parentNode == tr) {
                            for (c = 1; c <= cols; c++) {
                                dom.insertAfter(cloneCell(cell), cell);
                            }

                            break;
                        }
                    }

                    if (x2 == -1) {
                        for (c = 1; c <= cols; c++) {
                            tr.insertBefore(cloneCell(tr.cells[0]), tr.cells[0]);
                        }
                    }
                }
            }
        }

        function cleanup() {
            var rng = dom.createRng();

            each(dom.select('tr', table), function (tr) {
                if (tr.cells.length === 0) {
                    dom.remove(tr);
                }
            });

            if (dom.select('tr', table).length === 0) {
                rng.setStartAfter(table);
                rng.setEndAfter(table);
                dom.remove(table);
                return;
            }

            each(dom.select('thead,tbody,tfoot', table), function (part) {
                if (part.rows.length === 0) {
                    dom.remove(part);
                }
            });

        }

        function cloneCell(cell) {
            var highlightColour;
            var newCell = dom.create(cell.tagName);

            newCell.colSpan = cell.colSpan;
            newCell.width = cell.width;
            newCell.className = cell.className;
            newCell.innerHTML = tinymce.isIE ? '&nbsp;' : '<br data-mce-bogus="1"/>';

            highlightColour = dom.getAttrib(cell, 'data-highlight-colour');
            if (highlightColour) {
                dom.setAttrib(newCell, 'data-highlight-colour', highlightColour);
            }

            $(newCell).removeClass('numberingColumn');

            return newCell;
        }

        function findEndPos() {
            var maxX;
            var maxY;

            maxX = maxY = 0;

            each(grid, function (row, y) {
                each(row, function (cell, x) {
                    var colSpan;
                    var rowSpan;

                    if (isCellSelected(cell)) {
                        cell = grid[y][x];

                        if (x > maxX) {
                            maxX = x;
                        }

                        if (y > maxY) {
                            maxY = y;
                        }

                        if (cell.real) {
                            colSpan = cell.colspan - 1;
                            rowSpan = cell.rowspan - 1;

                            if (colSpan) {
                                if (x + colSpan > maxX) {
                                    maxX = x + colSpan;
                                }
                            }

                            if (rowSpan) {
                                if (y + rowSpan > maxY) {
                                    maxY = y + rowSpan;
                                }
                            }
                        }
                    }
                });
            });

            return { x: maxX, y: maxY };
        }

        function getSpanVal(td, name) {
            return parseInt(td.getAttribute(name) || 1);
        }
    }

    return new TableGrid();
});

