define("actonic-gdpr/permission-monitoring/filter", ["jquery", "actonic-gdpr/permission-monitoring/intro", "actonic-gdpr/permission-monitoring/rest", "actonic-gdpr/core/flags"], function ($, Intro, Rest, flags) {

    function Filter() {
        var events = [];
        var ignoreChangeEvents = false;
        var currentRequestId;

        function isCurrentPermission() {
            var checked = $(".permission-type:checked");
            return checked.val() === "true";
        }

        this.init = function () {
            $("#help-btn").click(function () {
                AJS.dialog2("#help-dialog").show();
            });

            $("#show-intro").click(function () {
                AJS.dialog2("#help-dialog").hide();
                Intro.showIntro();
            });

            $("#filter-button").click(function () {
                filterAuditLogs(1);
            });

            $("#from-date-picker, #to-date-picker").on("keydown keyup", function () {
                var tempDate = $(this).val().replace(/[^0-9-]/g, "");
                $(this).val(tempDate);
            });

            $("#clear-button").click(function () {
                //log.debug("#clear-button.click");
                clearFilters();
            });

            $("#export-button").click(function () {
                var requestLink;
                if (isCurrentPermission())
                    requestLink = AJS.contextPath() + "/rest/actonic-gdpr/1.0/permission-monitoring/txt/current?" + getExportLink();
                else
                    requestLink = AJS.contextPath() + "/rest/actonic-gdpr/1.0/permission-monitoring/txt/historical?" + getExportLink();

                Rest.getExport(requestLink,
                    function (data) {
                        window.location = requestLink;
                    }, function (error) {
                        if (error.status == 406)
                            flags.simpleWarning(AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.export-limit"));
                        else flags.simpleWarning(error.responseText);
                    }
                );
            });

            $(".permission-type").on("change", function (e) {
                if (isCurrentPermission()) {
                    $("#change-type-picker").disable();
                    $("#from-date-picker").disable();
                    $("#to-date-picker").disable();
                    $("#sort-by-picker").disable();
                    $("#result-table-historical").hide();
                    $("#result-table-current").show();
                } else {
                    $("#change-type-picker").enable();
                    $("#from-date-picker").enable();
                    $("#to-date-picker").enable();
                    $("#sort-by-picker").enable();
                    $("#result-table-historical").show();
                    $("#result-table-current").hide();
                }
                $("#filter-button").click();
            });

            initializeGroupPicker();
            initializeUserPicker();
            initializeSpacePicker();
            $("#from-date-picker").val(moment().subtract(1, "months").format("YYYY-MM-DD"));
        };

        function clearFilters() {
            ignoreChangeEvents = true;
            initializeGroupPicker();
            initializeUserPicker();
            initializeSpacePicker();
            ignoreChangeEvents = false;

            $("#change-type-picker").val("ALL");
            $("#from-date-picker").val(null);
            $("#to-date-picker").val(null);

            $("#result-table-historical tbody tr").remove();
            $("#result-table-current tbody tr").remove();
            $("#pagination-container").empty();
            $("#export-button").hide();

            currentRequestId = 0;
        }

        function getExportLink() {
            var selectedUser = $("#user-picker").val();
            var selectedGroup = $("#group-picker").val();
            var selectedSpace = $("#space-picker").val();
            var eventType = $("#change-type-picker").val();
            var orderBy = $("#sort-by-picker").val();
            var fromDate = moment($("#from-date-picker").val()).startOf("day").valueOf();
            var toDate = moment($("#to-date-picker").val()).endOf("day").valueOf();

            var request = {
                userName: selectedUser,
                groupName: selectedGroup,
                spaceKey: selectedSpace,
                eventType: eventType,
                orderBy: orderBy
            };
            if (fromDate) request.fromDate = fromDate;
            if (toDate) request.toDate = toDate;

            return jQuery.param(request);
        }

        function filterAuditLogs(currentPage) {
            $("#result-table-historical tbody tr").remove();
            $("#result-table-current tbody tr").remove();
            $("#pagination-container").empty();
            $("#export-button").hide();

            var selectedUser = $("#user-picker").val();
            var selectedGroup = $("#group-picker").val();
            var selectedSpace = $("#space-picker").val();
            var eventType = $("#change-type-picker").val();
            var orderBy = $("#sort-by-picker").val();

            if (!selectedUser && !selectedGroup && !selectedSpace) {
                flags.show({
                    type: "warning",
                    close: "auto",
                    body: AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.select-user-group-space")
                });
                return;
            }

            var fromDate = moment($("#from-date-picker").val()).startOf("day").valueOf();
            var toDate = moment($("#to-date-picker").val()).endOf("day").valueOf();
            var valueFromDate = $("#from-date-picker").val();
            var valueToDate = $("#to-date-picker").val();

            if (valueFromDate && !valueFromDate.match(/([1|2][9|0][0-9]{2})\-([1-9]|0[1-9]|1[0,1,2])\-([1-9](?!\w)|0[1-9](?!\w)|[12][0-9](?!\w)|3[01](?!\w))/g)) {
                flags.show({
                    type: "warning",
                    close: "auto",
                    body: AJS.I18n.getText("js-announcing-statistics.filter.flag.warning.date-example")
                });
                return;
            }

            if (valueToDate && !valueToDate.match(/([1|2][9|0][0-9]{2})\-([1-9]|0[1-9]|1[0,1,2])\-([1-9](?!\w)|0[1-9](?!\w)|[12][0-9](?!\w)|3[01](?!\w))/g)) {
                flags.show({
                    type: "warning",
                    close: "auto",
                    body: AJS.I18n.getText("js-announcing-statistics.filter.flag.warning.date-example")
                });
                return;
            }

            if (fromDate > toDate) {
                flags.show({
                    type: "warning",
                    close: "auto",
                    body: AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.date-before")
                });
                return;
            }

            if (fromDate > moment().endOf("day").valueOf()) {
                flags.show({
                    type: "warning",
                    close: "auto",
                    body: AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.date-after")
                });
                return;
            }

            $("#progress-indicator").show();
            var page = currentPage ? currentPage : 1;
            var request = {
                page: page,
                userName: selectedUser,
                groupName: selectedGroup,
                spaceKey: selectedSpace,
                eventType: eventType,
                orderBy: orderBy
            };

            if (fromDate) request.fromDate = fromDate;
            if (toDate) request.toDate = toDate;

            // log.debug("filterEvents request = " + JSON.stringify(request));
            items = [];
            if (isCurrentPermission()) {
                Rest.getCurrentPermissions(request,
                    function (data) {
                        if (typeof data === "string") data = JSON.parse(data);
                        initPagination(data.page, data.totalPages);

                        var receivedEvents = typeof data.items === "string" ? JSON.parse(data.items) : data.items;
                        receivedEvents.forEach(function (event) {
                            items.push(event);
                        });

                        onSearchEnd(true);
                    },
                    function (error) { // error
                        //log.error(error);
                        if (error.status == 400) {
                            flags.showFlag(AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.title.error-code") + " 400", error.responseText, "warning");
                        }
                        if (error.status == 403) {
                            flags.showFlag(AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.title.error-code") + " 403", error.responseText, "warning");
                        }

                        onSearchEnd(true);
                    });
            } else {
                var uniqueRequestId = Math.random();
                currentRequestId = uniqueRequestId;

                var tooSlowQueryTimer = setTimeout(function () {
                    if (uniqueRequestId == currentRequestId) flags.showFlag("Too slow", "The request works too slow. Please try to change search parameters", "warning");
                }, 30000);

                Rest.getHistoricalPermissions(request,
                    function (data) { // success
                        clearTimeout(tooSlowQueryTimer);
                        //log.debug("getHistoricalPermissions data = " + JSON.stringify(data));

                        if (uniqueRequestId == currentRequestId) {
                            if (typeof data === "string") data = JSON.parse(data);
                            initPagination(data.page, data.totalPages);

                            var receivedEvents = typeof data.items === "string" ? JSON.parse(data.items) : data.items;
                            receivedEvents.forEach(function (event) {
                                items.push(event);
                            });

                            if (data.statusCode == 1) {
                                flags.showFlag("Search process aborted", "The search process aborted because of TimeOutException. Some results are not shown. Please try to change search parameters", "warning");
                            }

                            onSearchEnd(false);
                        }
                    },
                    function (error) { // error
                        //log.error(error);
                        if (error.status == 400) {
                            flags.showFlag(AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.title.error-code") + " 400", error.responseText, "warning");
                        }
                        if (error.status == 403) {
                            flags.showFlag(AJS.I18n.getText("js-permission-monitoring.filter.flags.warning.title.error-code") + " 403", error.responseText, "warning");
                        }

                        onSearchEnd(false);
                    });
            }
        }

        function onSearchEnd(isCurrentPermissions) {
            $("#progress-indicator").hide();
            $("#export-button").show();

            items.forEach(function (event) {
                if (isCurrentPermission()) {
                    var tr = ACTONIC.GDPR.PERMISSIONMONITORING.createCurrentTableRow({
                        author: event.author,
                        user: event.userName,
                        eventType: event.summary,
                        details: event.details
                    });
                    $("#result-table-current tbody").append(tr);
                } else {
                    var newDate = moment(event.date).format("MMMM Do YYYY, h:mm:ss a");

                    var tr = ACTONIC.GDPR.PERMISSIONMONITORING.createHistoricalTableRow({
                        date: newDate,
                        author: event.author,
                        user: event.userName,
                        eventType: event.summary,
                        details: event.details
                    });
                    $("#result-table-historical tbody").append(tr);
                }
            });
        }

        function initPagination(currentPage, totalPages) {
            $("#pagination-container").empty();

            for (i = 1; i <= totalPages; i++) {
                var badge = $("<aui-badge/>").attr("data-page", i).text(i);
                if (currentPage == i) $(badge).addClass("aui-badge-primary");

                $("#pagination-container").append(badge);
            }

            $("#pagination-container aui-badge").off("click");
            $("#pagination-container aui-badge").on("click", function () {
                var pageNumber = $(this).attr("data-page");
                filterAuditLogs(pageNumber);
            });
        }

        function initializeGroupPicker() {
            var parent = $("#group-picker").parent();
            $("#group-picker").remove();
            $("#s2id_group-picker").remove();
            $(parent).append("<input type=\"text\" class=\"select\" id=\"group-picker\">");

            AJS.$("#group-picker").auiSelect2({
                placeholder: AJS.I18n.getText("js-permission-monitoring.filter.placeholder.select-group"),
                allowClear: true,
                ajax: {
                    type: "GET",
                    cache: true,
                    dataType: "json",
                    url: AJS.contextPath() + "/rest/actonic-gdpr/1.0/picker/groups",
                    data: function (term) {
                        return {
                            query: term,
                            maxResults: 100
                        };
                    },
                    results: function (data) {
                        //log.debug("initializeTargetGroupPicker = " + JSON.stringify(data));
                        var results = [];
                        $.each(data, function (index, item) {
                            results.push({id: item.text, text: item.text});
                        });
                        return {
                            results: results
                        };
                    }
                }
            });

            $("#group-picker").off("change");
            $("#group-picker").on("change", function () {
                if (!ignoreChangeEvents) {
                    $("#user-picker").select2("disable");
                    $("#s2id_user-picker a.select2-choice").addClass("disabled-picker");

                    $("#space-picker").select2("disable");
                    $("#s2id_space-picker a.select2-choice").addClass("disabled-picker");

                    $("#filter-button").click();
                }
            });
        }

        function initializeUserPicker() {
            var parent = $("#user-picker").parent();
            $("#user-picker").remove();
            $("#s2id_user-picker").remove();
            $(parent).append("<input type=\"text\" class=\"select\" id=\"user-picker\">");

            AJS.$("#user-picker").auiSelect2({
                placeholder: AJS.I18n.getText("js-permission-monitoring.filter.placeholder.select-user"),
                dropdownAutoWidth: true,
                ajax: {
                    url: AJS.contextPath() + "/rest/api/search?start=0&limit=100",
                    dataType: "json",
                    data: function (term) {
                        if (term) {
                            return {
                                cql: "user ~ '" + term + "'"
                            };
                        } else {
                            return {
                                cql: "user ~ '*'"
                            };
                        }
                    },
                    results: function (data) {
                        return {
                            results: $.map(data.results, function (el) {
                                //log.debug(el);
                                return {
                                    id: el.user.username,
                                    text: el.user.displayName + ' (' + el.user.username + ')'
                                }
                            })
                        };
                    }
                },
            });

            $("#user-picker").off("change");
            $("#user-picker").on("change", function () {
                if (!ignoreChangeEvents) {
                    $("#group-picker").select2("disable");
                    $("#s2id_group-picker a.select2-choice").addClass("disabled-picker");

                    $("#space-picker").select2("disable");
                    $("#s2id_space-picker a.select2-choice").addClass("disabled-picker");

                    $("#filter-button").click();
                }
            });
        }

        function initializeSpacePicker() {
            var parent = $("#space-picker").parent();
            $("#space-picker").remove();
            $("#s2id_space-picker").remove();
            $(parent).append("<input class=\"select\" type=\"text\" id=\"space-picker\">");

            AJS.$("#space-picker").auiSelect2({
                placeholder: AJS.I18n.getText("js-permission-monitoring.filter.placeholder.select-space"),
                allowClear: true,
                //minimumInputLength: 2,
                ajax: {
                    url: AJS.contextPath() + "/rest/actonic-gdpr/1.0/picker/spaces",
                    dataType: "json",
                    data: function (term) {
                        return {
                            q: term
                        };
                    },
                    results: function (data) {
                        return {
                            results: $.map(data, function (el) {
                                return {
                                    id: el.id,
                                    text: el.text + ' (' + el.id + ')'
                                }
                            })
                        };
                    }
                }
            });

            $("#space-picker").off("change");
            $("#space-picker").on("change", function () {
                if (!ignoreChangeEvents) {
                    $("#group-picker").select2("disable");
                    $("#s2id_group-picker a.select2-choice").addClass("disabled-picker");

                    $("#user-picker").select2("disable");
                    $("#s2id_user-picker a.select2-choice").addClass("disabled-picker");

                    $("#filter-button").click();
                }
            });
        }
    }

    return new Filter();
});