/**
 * @param animation {function(number):void}
 * @param duration {number}
 * @returns {Promise}
 */
function animate(animation, duration) {
    return new Promise((resolve) => {
        let start
        let first = true

        const animateImpl = (timestamp) => {
            if (first) {
                first = false
                start = timestamp

                animation(0)
                requestAnimationFrame(animateImpl)
            } else {
                const elapsed = timestamp - start
                if (duration > elapsed) {
                    animation(elapsed / duration)
                    requestAnimationFrame(animateImpl)
                } else {
                    animation(1)
                    resolve()
                }
            }
        }
        requestAnimationFrame(animateImpl)
    })
}

/**
 * @param animation {function(number):void}
 */
function ease(animation) {
    const tacc = 0.5
    const tdec = 0.5
    const max = 2 / (1 - tacc + tdec)
    const easeImpl = (t) => {
        if (0.0 >= t) {
            return 0;
        }
        if (1.0 <= t) {
            return 1.0;
        }

        if (tacc > t) {
            return max / (2.0 * tacc) * t * t;
        }

        if (tdec < t) {
            const tmp = 1 - t;
            return 1 - max / (2.0 - 2.0 * tdec) * tmp * tmp;
        }

        return max * (t - tacc / 2.0);
    }
    return (time) => animation(easeImpl(time))
}

/**
 * @param document {Document}
 * @param rule {string}
 */
function addStyleSheetRule(document, rule) {
    let style = document.getElementById('graphity-transition-rules')
    if (!style) {
        style = document.createElement('style');
        style.setAttribute('id', 'graphity-transition-rules')
        document.head.appendChild(style)
    }
    style.textContent += `${rule}\n`
    return true
}

/**
 * @param buttons {NodeList}
 * @param enabled {boolean}
 */
function setButtonsEnabled(buttons, enabled) {
    const n = buttons ? buttons.length : 0
    for (let i = 0; i< n; ++i) {
        const item = buttons.item(i)
        if (enabled) {
            item.removeAttribute('disabled')
            item.style.opacity = 1.0
        } else {
            item.setAttribute('disabled', 'true')
            item.style.opacity = 0.5
        }
    }
}

/**
 * @param div {HTMLDivElement}
 * @param enabled {boolean}
 */
function setDivEnabled(div, enabled) {
    if (enabled) {
        div.style.opacity = 1.0
        div.style.pointerEvents = 'initial'
    } else {
        div.style.pointerEvents = 'none'
        div.style.opacity = 0.5
    }
}