(function ($) {
    "use strict";

    var TOKENS_API = AJS.contextPath() + "/rest/api-tokens/latest/";
    var CONFIG_API = AJS.contextPath() + "/rest/api-tokens/latest/config";
    var SERVER_TIME = TOKENS_API + "server-time"

    var foo = getServerTime();
    console.log(foo);

    AJS.toInit(function () {
        let recommendedExpiration;
        let unlimitedExpiration;
        $.get(CONFIG_API, function (config) {
            unlimitedExpiration = config.unlimitedExpiration;
            if (config.recommendedExpiration) {
                recommendedExpiration = config.recommendedExpiration !== null ? config.recommendedExpiration : ""
            }
            if (!config.unlimitedExpiration) {
                $('label[for="expiration-date-input"]').append('<span class="aui-icon icon-required">required</span>');
            }
        })
        checkLicense();
        updateTable();

        // Create token dialog
        $("#create-token").click(function () {
            $("#expiration-date-input").val(recommendedExpiration);
            AJS.dialog2("#create-token-dialog").show();
        });
        $("#create-token-dialog .create").click(() => createTokenHandler(unlimitedExpiration));
        $("#create-token-dialog .close").click(function () {
            AJS.dialog2("#create-token-dialog").hide();
        });

        // Show token
        $("#show-token-dialog .copy-to-clipboard").click(function () {
            var $temp = $("<input>");
            $("body").append($temp);
            $temp.val($("#plain-token").val()).select();
            document.execCommand("copy");
            $temp.remove();
            $("#plain-token").val("");
            $('#plain-token').prop("type", "password");
            $('#view-password').text("Show");
            AJS.dialog2("#show-token-dialog").hide();
            showFlag("Copied to clipboard");
        });
        $("#view-password").click(function () {
            if ($("#plain-token").prop("type") === "text") {
                $("#plain-token").prop("type", "password");
                $("#view-password").text("Show");
            } else {
                $("#plain-token")
                    .prop("type", "text")
                    .focus()
                    .select();
                $("#view-password").text("Hide");
            }
        });
        $("#show-token-dialog .close").click(function () {
            $("#plain-token").val("");
            $('#plain-token').prop("type", "password");
            $('#view-password').text("Show");
            AJS.dialog2("#show-token-dialog").hide();
        });

        // Revoke dialog
        $("#revoke-token-dialog .revoke").click(function () {
            revoke($("#revoking-token-id").val());
            AJS.dialog2("#revoke-token-dialog").hide();
        });
        $("#revoke-token-dialog .close").click(function () {
            AJS.dialog2("#revoke-token-dialog").hide();
        });

        // Revoke All dialog
        $("#app-specific-passwords-revoke-all").click(function () {
            AJS.dialog2("#revoke-all-tokens-dialog").show();
        });
        $("#revoke-all-tokens-dialog .revoke-all").click(function () {
            revokeAll();
            AJS.dialog2("#revoke-all-tokens-dialog").hide();
        });
        $("#revoke-all-tokens-dialog .close").click(function () {
            AJS.dialog2("#revoke-all-tokens-dialog").hide();
        });
        $("#tokens_table").on("click", ".revoke-token", function () {
            $("#revoking-token-label").text($(this).attr("data-asp-label"));
            $("#revoking-token-id").val($(this).attr("data-asp-id"));
            AJS.dialog2("#revoke-token-dialog").show();
        });

        $('#create-api-token-form').on('submit', function (event) {
            event.preventDefault();
            createTokenHandler();
        });
    });

    function updateTable() {
        Promise.all([$.get(TOKENS_API), $.get(CONFIG_API)])
            .then(([tokens, config]) => {
                handleCreateTokenButton(tokens, config);
                fillTable(tokens);
            })
            .catch(error => {
                console.error("Failed to fetch tokens:", error);
                $("#create-token").prop("disabled", true).prop("title", "Error loading tokens. Please try again later.");
            });
    }

    function handleCreateTokenButton(tokens, config) {
        const $createTokenButton = $("#create-token");

        if (!config.creationAllowed) {
            $createTokenButton
                .prop("disabled", true)
                .prop("title", "Creation of new tokens was disabled by administrator");
            return;
        }

        if (tokens.length >= config.maxTokensPerUser) {
            $createTokenButton
                .prop("disabled", true)
                .prop("title", "Maximum amount of tokens: " + config.maxTokensPerUser + ". Please contact your administrator");
        } else {
            $createTokenButton.prop("disabled", false).prop("title", "");
        }
    }

    function fillTable(tokens) {
        $("#tokens_table").empty();

        if (!$.isEmptyObject(tokens)) {
            $.each(tokens, function (i, token) {
                var tr = $("<tr>").append(
                    $("<td>").text(token.label),
                    $("<td>").append(expirationBuilder(token.expirationDate)),
                    $("<td>")
                        .attr("title", new Date(token.createdAt))
                        .text(formatDate(token.createdAt)),
                    $("<td>")
                        .attr("title", token.lastAccessed === null ? null : new Date(token.lastAccessed))
                        .text(formatDate(token.lastAccessed)),
                    $("<td>").text(token.usageCount),
                    $("<td>").append(
                        $("<a>")
                            .attr("data-asp-id", token.id)
                            .attr("data-asp-label", token.label)
                            .addClass("revoke-token")
                            .text("Revoke")
                    )
                );
                $("#tokens_table").append(tr);
                $(".date-tooltip").tooltip();
            });
            $(".no-password").hide();
            $(".app-specific-passwords-list.has-password").show();
            $("#app-specific-passwords-revoke-all").show();
        } else {
            $(".app-specific-passwords-list.has-password").hide();
            $("#app-specific-passwords-revoke-all").hide();
            $(".no-password").show();
        }
    }

    function timeBetween(firstDate, secondDate) {
        var date = new Date(firstDate);
        if (secondDate > date) {
            var seconds = Math.floor((secondDate - date) / 1000);
        } else {
            var seconds = Math.floor((date - secondDate) / 1000);
        }
        var intervalType;
        var interval = Math.floor(seconds / 31536000);

        if (interval >= 1) {
            intervalType = "year";
        } else {
            interval = Math.floor(seconds / 2592000);

            if (interval >= 1) {
                intervalType = "month";
            } else {
                interval = Math.floor(seconds / 86400);

                if (interval >= 1) {
                    intervalType = "day";
                } else {
                    interval = Math.floor(seconds / 3600);

                    if (interval >= 1) {
                        intervalType = "hour";
                    } else {
                        interval = Math.floor(seconds / 60);

                        if (interval >= 1) {
                            intervalType = "minute";
                        } else {
                            interval = seconds;
                            intervalType = "second";
                        }
                    }
                }
            }
        }

        if (interval > 1 || interval === 0) {
            intervalType += "s";
        }

        return "".concat(interval, " ").concat(intervalType);
    }

    function formatDate(timestamp) {
        return timestamp === null
            ? "Never accessed"
            : "".concat(timeBetween(new Date(timestamp), new Date()), " ago");
    }

    function revoke(id) {
        $.ajax({
            url: TOKENS_API + id,
            type: "DELETE",
            success: function success() {
                showFlag("Your API token has been revoked");
                updateTable();
            }
        });
    }

    function showFlag(body) {
        var type =
            arguments.length > 1 && arguments[1] !== undefined
                ? arguments[1]
                : "success";
        var close =
            arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : "auto";
        AJS.flag({
            type: type,
            body: body,
            close: close
        });
    }

    function revokeAll() {
        var _$;

        var requests = [];
        $("#tokens_table a[data-asp-id]").each(function (i, element) {
            var id = $(element).attr("data-asp-id");

            if (id) {
                requests.push(
                    $.ajax({
                        url: TOKENS_API + id,
                        type: "DELETE"
                    })
                );
            }
        });

        (_$ = $).when.apply(_$, requests).done(function () {
            showFlag("Your API token(s) have been revoked");
            updateTable();
        });
    }

    function isEmpty(value) {
        return !value || !(value.trim && value.trim());
    }

    function createToken(label, expiration) {
        $.ajax({
            type: "post",
            url: TOKENS_API,
            dataType: "json",
            contentType: "application/json; charset=utf-8",
            data: JSON.stringify({
                label: label,
                expiration: expiration
            }),
            success: function success(response) {
                $("#show-token-dialog #plain-token").val(response.passwordValue);
                AJS.dialog2("#show-token-dialog").show();
                updateTable();
            }
        });
    }

    function createTokenHandler(unlimitedExpiration) {
        var label = $("#label-input").val();
        var expiration = $("#expiration-date-input").val();

        if (!isEmpty(label)) {
            if (!isEmpty(expiration) && expiration > 0) {
                createToken(label, expiration);
                $("#label-input").val("");
                $("#expiration-date-input").val("");
                AJS.dialog2("#create-token-dialog").hide();
            } else {
                if (unlimitedExpiration) {
                    createToken(label);
                    $("#label-input").val("");
                    AJS.dialog2("#create-token-dialog").hide();
                }
            }
        } else {
            $("#label-input").val("");
            $("#label-input").focus();
        }
    }

    function checkLicense() {
        return $.get("".concat(TOKENS_API, "license"), function (response) {
            if ($.trim(response) === "false") {
                AJS.messages.error("#license-error", {
                    title: "Invalid License.",
                    body:
                        "<p>API Tokens plugin has no valid license. Access to REST API via Tokens will not work</p>"
                });
            }
        });
    }

    function expirationBuilder(expirationDate) {
        var span = $("<span>");
        if (expirationDate != null) {
            if (expirationDate > getServerTime()) {
                $(span)
                    .addClass("aui-lozenge aui-lozenge-subtle aui-lozenge-success")
                    .attr("title", new Date(expirationDate))
                    .text(timeBetween(new Date(expirationDate), getServerTime()));
            } else {
                $(span)
                    .addClass("aui-lozenge aui-lozenge-subtle aui-lozenge-removed")
                    .attr("title", new Date(expirationDate))
                    .text("Expired");
            }
        } else {
            $(span)
                .addClass("aui-lozenge aui-lozenge-subtle")
                .attr("title", "The token will never expire")
                .text("Unlimited");
        }

        return span;
    }

    function getServerTime() {
        var dateTime = new Date();
        $.get(SERVER_TIME, function (response) {
            if (response != null) {
                dateTime = new Date(+response);
                console.log(dateTime + " this date time was created from server side")
            } else {
                console.log(dateTime + " this date time was created from client side")
            }
        });
        return dateTime;
    }

})(jQuery);
