(function($) {

    var ContentManager = {

        init: function () {
            this.container = $('.quiz-questions-editor');
            this.content = this.container.find('.content-wrapper');
            this.categoryReport = $('#quiz-categories-report');
            this.questionsReport = $('.quiz-questions-report');

            this.bindAddCategory();
            this.bindUpdateQuestionRandomize();
            this.bindCategoryRename();
            this.bindRemoveQuestion();
            this.bindRemoveCategory();
            this.bindBulkOperations();
            this.makeSortable();
        },

        bindAddCategory: function () {
            var self = this;

            self.container.find('#add-category').click(function () {
                self.addCategory();
            });
        },

        addCategory: function () {
            var self = this;

            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/category/add",
                data: {
                    quizId: QuizCommons.getContentId(),
                    name: ""
                },
                dataType: 'json',
                success: function(id) {
                    self.showNewCategory(id);
                },
                error: QuizErrorHandler.showError
            });
        },

        showNewCategory: function(id) {
            var self = this;
            var category = $(QuizQuestionsEditor.category({category: {id: id, name: '', stats: []}, canCreate: true}));
            self.content.append(category);
            self.makeSortable();
            category.find('.category-name').focus();
        },

        updateCategoryReport: function () {
            this.updateReport(this.categoryReport, 'categories');
        },

        updateQuestionReport: function() {
            this.updateReport($('#quiz-questions-report'), 'questions');
        },

        updateReport: function(container, type) {
            var self = this;
            var btns = $('a[href="#quiz-content-report"], .content-report-selector');

            function update() {
                $.ajax({
                    type: 'GET',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/" + type + "/report",
                    data: {
                        quizId: QuizCommons.getContentId()
                    },
                    dataType: 'html',
                    success: function (html) {
                        container.html(html);
                    }
                });
            }

            btns.unbind('click.' + type);

            if (container.is(':visible')) {
                update();
            } else {
                btns.one('click.' + type, function () {
                    self.updateReport(container, type)
                });
            }
        },

        bindCategoryRename: function() {
            var self = this;

            self.container.on('keyup keydown', '.category-name', function (e) {
                if (e.which === 13) {
                    self.saveCategoryName($(this));
                    return false;
                }
                $(this).parents('.category-table').find('.save-category-name').css('display', 'table-cell');
            }).on('change', '.category-name', function () {
                self.saveCategoryName($(this));
            });
        },

        saveCategoryName: function(input) {
            var self = this;
            var id = input.parents('.quiz-category').data('id');

            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/category/rename",
                data: {
                    quizId: QuizCommons.getContentId(),
                    name: input.val(),
                    id: id
                },
                dataType: 'json',
                success: function() {
                    self.categoryReport.find("tr[data-id=" + id + "] .category-name").text(input.val());
                },
                error: QuizErrorHandler.showError
            });

            input.blur();
            input.parents('.category-table').find('.save-category-name').hide();
        },
        
        bindUpdateQuestionRandomize: function () {
            var self = this;

            self.container.on('click', '.randomize-answers', function () {
                var btn = $(this);
                var id = btn.data('id');
                var newState = btn.hasClass('disabled');

                $.ajax({
                    type: 'POST',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/change-answers-randomize",
                    data: {
                        pageIds: id,
                        values: newState
                    },
                    dataType: 'json',
                    success: function() {
                        btn.replaceWith(QuizQuestionsEditor.randomizeAnswersBtn({questionId: id, isRandom: newState}));
                        QuizMessageHelper.showSuccessMsg(newState ?
                            AJS.I18n.getText('quiz.question.answers.random.enabled') :
                            AJS.I18n.getText('quiz.question.answers.strict.enabled')
                        );
                    },
                    error: QuizErrorHandler.showError
                });
            })
        },

        bindRemoveQuestion: function () {
            var self = this;
            $('.quiz-question-remove').click(function() {
                var pageId = $(this).data('id');
                self.showRemoveQuestionsDialog([pageId]);
            });

            $('#qqr-remove').click(function() {
                var questionIds = self.container.find('.qqr-select:checked').map(function () { return $(this).data('id')}).toArray();
                var categoryIds = self.container.find('.bulk-category-select:checked').map(function () { return $(this).data('id')}).toArray();
                self.showRemoveQuestionsDialog(questionIds, categoryIds);

                return false;
            });
        },

        showRemoveQuestionsDialog: function (pageIds, categoryIds) {
            var self = this;

            var dialog = new AJS.Dialog({
                width: 550,
                height: 210,
                id: "remove-question-dialog",
                closeOnOutsideClick: false
            });

            var bodyText = AJS.I18n.getText("quiz.question.report.dialog.body");
            bodyText += pageIds.length > 1 ? AJS.I18n.getText("quiz.question.report.dialog.body2", AJS.Meta.get('page-title'))
                : AJS.I18n.getText("quiz.question.report.dialog.body1", AJS.Meta.get('page-title'));
            dialog.addHeader(AJS.I18n.getText("quiz.question.report.dialog.title"));
            dialog.addPanel('panel', '<div>' + bodyText + '</div>', 'remove-question-dialog-panel');
            dialog.addButton(AJS.I18n.getText("quiz.create-wizard.dialog.answer.remove"), function() {
                $.ajax({
                    type: 'POST',
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/remove-question",
                    data: {
                        pageIds: pageIds,
                        quizIds: QuizCommons.getContentId()
                    },
                    success: function() {
                        dialog.remove();
                        self.removeQuestionsFromReport(pageIds);
                        self.updateCategoryReport();
                        self.updateCategoriesMeta();
                        AJS.trigger('quiz-questions-report-updated');
                    },
                    error: QuizErrorHandler.showError
                });
                if (categoryIds && categoryIds.length) {
                    $.ajax({
                        type: 'DELETE',
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/category/delete",
                        data: {
                            ids: categoryIds,
                            quizId: QuizCommons.getContentId()
                        },
                        success: function () {
                            categoryIds.forEach(function (id) {
                                self.getCategory(id).remove();
                                self.categoryReport.find('tr[data-id="' + id + '"]').remove();
                            });
                        },
                        error: QuizErrorHandler.showError
                    });
                }
            }, "remove-question-dialog-ok");
            dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                dialog.remove();
                QuizQuestionsReport.hideReusePanel();
            }, '#');

            dialog.show();
        },

        bindRemoveCategory: function () {
            var self = this;

            $(document).on('click', '.quiz-category-remove', function () {
                var id = $(this).data('id');
                if (self.getCategory(id).find('.quiz-question').length) {
                    self.showRemoveCategoriesDialog(id);
                } else {
                    self.removeCategory(id);
                }
            });
        },

        showRemoveCategoriesDialog: function (id) {
            var self = this;
            var dialog = $(QuizQuestionsEditor.removeCategoryDialog());
            AJS.dialog2(dialog).show();

            dialog.find('#quiz-remove-category-cancel').click(function() {
                AJS.dialog2(dialog).remove();
            });

            dialog.find('#quiz-remove-category-confirm').click(function() {
                self.removeCategory(id, function () {
                    AJS.dialog2(dialog).remove();
                });
            });
        },

        removeCategory: function(id, onSuccess) {
            var self = this;

            $.ajax({
                type: 'DELETE',
                url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/category/delete",
                data: {
                    ids: id,
                    quizId: QuizCommons.getContentId()
                },
                success: function(newCategoryId) {
                    if (onSuccess) onSuccess();
                    var category = self.getCategory(id);
                    var questionIds = category.find('.quiz-question').map(function () { return $(this).data('id')}).toArray();
                    self.removeQuestionsFromReport(questionIds);
                    category.remove();
                    self.categoryReport.find('tr[data-id="' + id + '"]').remove();

                    if (self.getCategory(newCategoryId).length === 0) {
                        self.showNewCategory(newCategoryId);
                    }
                },
                error: QuizErrorHandler.showError
            });
        },

        getCategory: function(id) {
            return this.container.find('.quiz-category[data-id="' + id + '"]');
        },

        removeQuestionsFromReport: function (ids) {
            var self = this;

            $('.quiz-question, .quiz-questions-report tr').filter(function () {
                return ids.indexOf($(this).data('id')) > -1;
            }).remove();

            var total = self.container.find('.quiz-question').length;
            if (total === 0) {
                AJS.trigger('quiz-questions-update-needed');
                return;
            }

            $('.quiz-questions-report-content .quiz-macro-view-description span').text(total);

            self.questionsReport.trigger('update', false);
            QuizQuestionsReport.hideReusePanel();
        },

        bindBulkOperations: function () {
            var self = this;

            self.container.on('change', '.bulk-category-select', function () {
                var checkBox = $(this);
                var childCheckBoxes = checkBox.parents('.quiz-category').find('.bulk-question-select');
                childCheckBoxes.prop('checked', checkBox.is(':checked'));
                self.togglePanel();
            });

            self.container.on('change', '.qqr-select', function () {
                var checkBox = $(this);
                var category = checkBox.parents('.quiz-category');
                var categoryCheckBox = category.find('.bulk-category-select');
                if (!checkBox.is(':checked')) {
                    categoryCheckBox.prop('checked', false);
                }
                self.togglePanel();
            });
        },

        togglePanel: function () {
            var self = this;
            if (self.container.find('.qqr-select:checked, .bulk-category-select:checked').length) {
                QuizQuestionsReport.showReusePanel()
            } else {
                QuizQuestionsReport.hideReusePanel();
            }
        },

        makeSortable: function () {
            var self = this;

            self.content.sortable({
                containment: 'parent',
                items: ".quiz-category",
                handle: '.category-handle',
                axis: 'y',
                update: function () {
                    var ids = self.content.find('.quiz-category').map(function () { return $(this).data('id'); }).toArray();

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/category/reorder",
                        data: {
                            quizId: QuizCommons.getContentId(),
                            ids: ids
                        },
                        dataType: 'json',
                        success: function() {
                            self.updateCategoryReport();
                            self.updateQuestionReport();
                        },
                        error: QuizErrorHandler.showError
                    });
                }
            });

            self.content.find('.questions').sortable({
                containment: self.content,
                items: ".quiz-question",
                handle: '.question-handle',
                connectWith: '.questions',
                axis: 'y',
                start: function (e, ui) {
                    self.content.addClass('sorting');
                },
                stop: function (e, ui) {
                    self.content.removeClass('sorting');
                },
                update: function (e, ui) {
                    var ids = self.content.find('.quiz-question').map(function () { return $(this).data('id'); }).toArray();

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/question/reorder",
                        data: {
                            quizId: QuizCommons.getContentId(),
                            ids: ids,
                            id: ui.item.data('id'),
                            categoryId: ui.item.parents('.quiz-category').data('id')
                        },
                        dataType: 'json',
                        success: function() {
                            self.updateCategoryReport();
                            self.updateQuestionReport();
                            self.updateCategoriesMeta();
                        },
                        error: QuizErrorHandler.showError
                    });
                }
            })
        },

        updateCategoriesMeta: function () {
            var self = this;

            self.container.find('.quiz-category').each(function () {
                var category = $(this);
                category.find('.question-count').text(category.find('.quiz-question').length);
            });
        }
    };

    AJS.toInit(function() {
        ContentManager.init();
        AJS.bind('quiz-questions-updated', function () {
            ContentManager.init();
            AJS.trigger('quiz-questions-report-updated');
        });
    });
})(AJS.$);


