import Papa from 'papaparse';

(function($) {

    const QUESTION_TYPES = {
        MULTIPLE_CHOICE: "0",
        SINGLE_CHOICE: "1",
        TRUE_FALSE: "2",
        FREE_TEXT: "3",
        AUTO_CHECKED_FREE_TEXT: "4"
    };

    const VALIDATION_RESULTS = {
        OK: 0,
        NO_QUESTION_TEXT: 1,
        NO_VARIANTS: 2,
        ONLY_ONE_VARIANT_FOR_SC_QUESTION: 3,
        NO_RIGHT_VARIANT_FRO_SC_QUESTION: 4
    }

    window.QuizCreateQuestion = {

        dialog: null,
        saveBtn: null,
        answerInput: null,
        questionInput: null,
        typeInput: null,
        wasAdded: false,
        saveEditorLink: null,
        prevType: "1",

        init: function(e, categoryId) {
            var self = QuizCreateQuestion;
            self.categoryId = categoryId || 0;
            self.wasAdded = false;
            self.typeSwitched = false;

            self.dialog = new AJS.Dialog({
                width: 900,
                height: Math.min($(window).height() * 0.85, 850),
                id: "create-wizard-dialog",
                closeOnOutsideClick: false
            });

            self.dialog.addHeader(AJS.I18n.getText("quiz.create-wizard.dialog.title"));
            self.dialog.addPanel('panel', QuizWizard.dialog(), 'create-wizard-dialog-panel');
            self.dialog.addButton(AJS.I18n.getText("quiz.create-wizard.dialog.create"), function() {
                $('.wizard-dialog .aui-message-error').remove();

                var result = self.validate();

                if (result === VALIDATION_RESULTS.OK) {
                    self.save(false);
                } else if (result === VALIDATION_RESULTS.NO_QUESTION_TEXT) {
                    AJS.messages.error($('.wizard-dialog .quiz-dialog-param-wrapper.question-container'), {
                        body: AJS.I18n.getText("quiz.create-wizard.dialog.message.question-empty"),
                        closeable: true
                    });
                    self.questionInput.focus();
                } else if (result === VALIDATION_RESULTS.NO_VARIANTS) {
                    AJS.messages.error($('.wizard-dialog .quiz-dialog-param-wrapper.answers .answers-error'), {
                        body: AJS.I18n.getText("quiz.create-wizard.dialog.message.answers-empty"),
                        closeable: true
                    });
                    self.answerInput.focus();
                } else if (result === VALIDATION_RESULTS.ONLY_ONE_VARIANT_FOR_SC_QUESTION) {
                    AJS.messages.error($('.wizard-dialog .quiz-dialog-param-wrapper.answers .answers-error'), {
                        body: AJS.I18n.getText("quiz.create-wizard.dialog.message.one-answer"),
                        closeable: true
                    });
                    self.answerInput.focus();
                } else if (result === VALIDATION_RESULTS.NO_RIGHT_VARIANT_FRO_SC_QUESTION) {
                    AJS.messages.error($('.wizard-dialog .quiz-dialog-param-wrapper.answers .answers-error'), {
                        body: AJS.I18n.getText("quiz.create-wizard.dialog.message.no-correct"),
                        closeable: true
                    });
                }
            }, "create-wizard-dialog-save");
            self.dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                dialog.remove();
                if (self.wasAdded) {
                    updateQuestionsReportMacro();
                }
            }, '#');

            self.saveBtn = $('.create-wizard-dialog-save');
            self.answerInput = $('.wizard-dialog .answer');
            self.questionInput = $('.wizard-dialog .question');
            self.questionBodyInput = $('.wizard-dialog .question-body');
            self.questionLimit = $('.wizard-dialog .question-length');
            self.typeInput = $('.wizard-dialog .type');
            self.reuseWrong = $('.wizard-dialog .reuse-wrong');
            self.reuseCorrect = $('.wizard-dialog .reuse-correct');
            self.reuseFree = $('.wizard-dialog .reuse-free');

            $('.wizard-dialog .free-feedback-wrapper').hide();
            self.reuseCorrect.change(function () {
                if (!self.typeSwitched) {
                    self.reuseFree.prop('checked', self.reuseCorrect.is(':checked'));
                }
            });

            self.reuseFree.change(function () {
                if (!self.typeSwitched) {
                    self.reuseCorrect.prop('checked', self.reuseFree.is(':checked'));
                }
            });

            self.saveBtn.parent()
                .prepend(QuizWizard.createAnother({checked: getCookie("quiz-ca") != 'false'}))
                .prepend(QuizWizard.saveEditor());

            self.saveEditorLink = $('.save-editor');

            self.dialog.show();

            $('#add-answer').click(self.addVariant);
            self.answerInput.keypress(function(e) {
                if (e.which == 13) {
                    self.addVariant();
                }
            });
            self.questionInput.bind('input', self.validate);

            self.typeInput.change(function() {
                var selectedType = $(this).val();

                if (self.prevType === QUESTION_TYPES.TRUE_FALSE) {
                    $('.wizard-dialog .quiz-dialog-param-wrapper.answers').removeClass('true-false-type');
                    $('.wizard-dialog .answer-variant').remove();
                }

                if (self.prevType === QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT) {
                    $('.wizard-dialog .answer-variant.hidden-input').each(function() {
                        $(this).removeClass("hidden-input");
                    });
                }

                if (selectedType === QUESTION_TYPES.FREE_TEXT) {
                    $('.wizard-dialog .answers, .wizard-dialog .random-wrapper').hide();
                    $('.wizard-dialog .free-text-hint').show();
                    $('.wizard-dialog .auto-checked-free-text-hint').hide();
                    $('.wizard-dialog .wrong-feedback-wrapper').hide();
                    $('.wizard-dialog .correct-feedback-wrapper').hide();
                    $('.wizard-dialog .free-feedback-wrapper').show();

                    if ($('.correct-feedback').val() && !self.typeSwitched) {
                        self.typeSwitched = true;
                        $('.free-feedback').val($('.correct-feedback').val());
                    }
                } else if (selectedType === QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT) {
                    $('.wizard-dialog .answers').show();
                    $('.wizard-dialog .random-wrapper').hide();
                    $('.wizard-dialog .auto-checked-free-text-hint').show();
                    $('.wizard-dialog .wrong-feedback-wrapper').show();
                    $('.wizard-dialog .correct-feedback-wrapper').show();
                    $('.wizard-dialog .free-feedback-wrapper').hide();
                    $('.wizard-dialog .free-text-hint').hide();
                    $('.wizard-dialog .description.answer-desc').hide();
                    $('.wizard-dialog .description.free-text-answer-desc').show();

                    $('.wizard-dialog input.text.answer').attr('placeholder', AJS.I18n.getText('quiz.create-wizard.dialog.answer.free-text-ph'));

                    $('.wizard-dialog .answer-variant').each(function() {
                        $(this).addClass('hidden-input');
                    });

                    $('.wizard-dialog .variant-correct').each(function() {
                        $(this).clone().attr("type", "checkbox").insertBefore(this);
                    }).remove();

                    if ($('.free-feedback').val() && !self.typeSwitched) {
                        self.typeSwitched = true;
                        $('.correct-feedback').val($('.free-feedback').val());
                    }
                } else {
                    $('.wizard-dialog .wrong-feedback-wrapper').show();
                    $('.wizard-dialog .correct-feedback-wrapper').show();
                    $('.wizard-dialog .free-feedback-wrapper').hide();
                    $('.wizard-dialog .free-text-hint').hide();
                    $('.wizard-dialog .auto-checked-free-text-hint').hide();
                    $('.wizard-dialog .answers, .wizard-dialog .random-wrapper').show();
                    $('.wizard-dialog .description.free-text-answer-desc').hide();

                    $('.wizard-dialog input.text.answer').attr('placeholder', AJS.I18n.getText('quiz.create-wizard.dialog.answer.ph'));


                    if (selectedType === QUESTION_TYPES.TRUE_FALSE) {
                        $('.wizard-dialog .description.answer-desc').hide();
                        $('.wizard-dialog .quiz-dialog-param-wrapper.answers').addClass('true-false-type');
                        $('.wizard-dialog .answer-variant').remove();
                        $('.wizard-dialog #answer-variants')
                            .append(QuizWizard.variant({type: "radio", text: AJS.I18n.getText("quiz.create-wizard.dialog.answer.true"), hiddenInput: false}))
                            .append(QuizWizard.variant({type: "radio", text: AJS.I18n.getText("quiz.create-wizard.dialog.answer.false"), hiddenInput: false}));
                        $('.wizard-dialog .variant-correct:eq(0)').attr('checked', 'checked');
                    } else {
                        $('.wizard-dialog .description.answer-desc').show();
                        $('.wizard-dialog .variant-correct').each(function() {
                            $(this).clone().attr("type", selectedType === QUESTION_TYPES.SINGLE_CHOICE ? "radio" : "checkbox").insertBefore(this);
                        }).remove();
                    }

                    if ($('.free-feedback').val() && !self.typeSwitched) {
                        self.typeSwitched = true;
                        $('.correct-feedback').val($('.free-feedback').val());
                    }
                }

                self.prevType = selectedType;
                self.validate();
                self.showAnswerDescription();
                $('.wizard-dialog .feedback textarea').trigger('change');
            });

            $('.variant-correct').live('change', self.validate);

            var createAnother = $('.create-another').change(function() {
                setCookie('quiz-ca', $(this).is(':checked'));
            });

            $('.create-another-wrapper').click(function(e) {
                if (!$(e.target).is(createAnother)) {
                    createAnother.click();
                }
            });

            $('.random').prop('checked', getCookie("quiz-random") != 'false').change(function() {
                setCookie('quiz-random', $(this).is(':checked'));
            });

            self.saveEditorLink.click(function() {
                if ($('.wizard-dialog .answer-variant').length === 0 && !self.questionInput.val().trim()) {
                    self.saveEditorLink.unbind('click');
                    $('.quiz-create-question').click();
                    $('#create-dialog').css('visibility', 'hidden');
                } else {
                    self.save(true);
                }
                return false;
            });

            $('.wizard-dialog #answer-variants').sortable({
                cancel: ".variant-remove, input, .variant-text",
                containment: 'parent',
                tolerance: "pointer",
                scroll: false,
                axis: 'y'
            });

            self.spinner = $('<img class="wait-icon" src="' + Confluence.getContextPath() +'/images/icons/wait.gif">');
            self.saveBtn.before(self.spinner);

            $('.wizard-dialog .clear-correct-feedback').click(function () {
                $('.correct-feedback').val('').trigger('change');
                return false;
            });
            $('.wizard-dialog .clear-wrong-feedback').click(function () {
                $('.wrong-feedback').val('').trigger('change');
                return false;
            });
            $('.wizard-dialog .clear-free-feedback').click(function () {
                $('.free-feedback').val('').trigger('change');
                return false;
            });

            var settingsTable = $('#quiz-settings-table');
            if (settingsTable.length && !settingsTable.data('show-feedback')) {
                var settingsDialog = null;
                var input = null;

                var inputs = $('.wizard-dialog .feedback textarea');
                var closeDialog = function () {
                    settingsDialog.remove();
                    inputs.unbind('input.show-feedback change.show-feedback');
                };
                inputs.bind('input.show-feedback change.show-feedback', function () {
                    var thisInput = $(this);
                    if (thisInput.val().trim() && !thisInput.is(input) && thisInput.is(':visible')) {
                        if (settingsDialog) {
                            settingsDialog.remove();
                        }
                        settingsDialog = AJS.InlineDialog(thisInput, "quiz-show-feedback", function(contents, trigger, showPopup) {
                            contents.html(QuizWizard.enableFeedbackDialog());
                            showPopup();
                            contents.find('.aui-iconfont-close-dialog').click(closeDialog);
                            contents.find('#enable-feedback').click(function () {
                                QuizSettings.input = $('#show-res-input').add($('#show-feedback-input').prop('checked', true));
                                QuizSettings.save();
                                QuizMessageHelper.showSuccessMsg(AJS.I18n.getText('quiz.create-wizard.dialog.enable-feedback.success'));
                                closeDialog();
                                return false;
                            });
                        }, {hideDelay: null, closeOnTriggerClick: false, noBind: true, container: thisInput.parent(),
                            width: 310, gravity: 's', persistent: true, calculatePositions: function (popup) {
                            return {
                                displayAbove: true,
                                popupCss: {
                                    left: 400,
                                    top: - popup.height() - 8
                                },
                                arrowCss: {
                                    left: popup.width() / 2 - 8,
                                    top:  popup.height()
                                }
                            };
                        }});
                        input = thisInput;
                        settingsDialog.show();
                    } else if ((!thisInput.val().trim() || !thisInput.is(':visible')) && settingsDialog && thisInput.is(input)) {
                        settingsDialog.remove();
                        settingsDialog = null;
                        input = null;
                        inputs.trigger('change');
                    }
                });
            }

            $('#question-body-link').click(function () {
                $(this).parent().toggleClass('collapsed expanded');
                return false;
            })
        },

        save: function(editor) {
            var self = QuizCreateQuestion;
            var question = $.trim(self.questionInput.val());

            if (question.length > 255) {
                AJS.messages.error($('.wizard-dialog .quiz-dialog-param-wrapper.question-container'), {
                    body: AJS.I18n.getText("page.title.too.long"),
                    closeable: true
                });
                self.questionInput.focus();
            } else if (!Confluence.Blueprint.canCreatePage(AJS.Meta.get('space-key'), question)) {
                AJS.messages.error($('.wizard-dialog .quiz-dialog-param-wrapper.question-container'), {
                    body: AJS.I18n.getText("quiz.blueprint.form.page.error"),
                    closeable: true
                });
                self.questionInput.focus();
                return;
            }

            var answers = [];
            var correct = [];
            $('.wizard-dialog li.answer-variant').each(function(i) {
                var elem = $(this);
                answers.push(elem.find('.variant-text').val());
                if (elem.find('.variant-correct').is(':checked')) {
                    correct.push(i + 1);
                }
            });

            self.saveBtn.prop('disabled', true);
            self.spinner.css('visibility', 'visible');
            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/create",
                data: {
                    pageId: QuizCommons.getContentId(),
                    text: question,
                    extension: self.questionBodyInput.val(),
                    type: self.prevType === QUESTION_TYPES.TRUE_FALSE ? QUESTION_TYPES.SINGLE_CHOICE : self.prevType,
                    quizIds: $('.quiz-create-wizard').data('quizid'),
                    answers: answers,
                    correct: correct.join(','),
                    random: $('.random').is(':checked'),
                    correctFeedback: self.prevType == QUESTION_TYPES.FREE_TEXT ? $('.free-feedback').val() : $('.correct-feedback').val(),
                    wrongFeedback: self.prevType == QUESTION_TYPES.FREE_TEXT ? '' : $('.wrong-feedback').val(),
                    categoryId: self.categoryId
                },
                dataType: 'json',
                success: function(pageId) {
                    if (editor) {
                        window.location.href = Confluence.getContextPath() + "/pages/editpage.action?pageId=" + pageId;
                        return;
                    }

                    QuizMessageHelper.showSuccessMsg(AJS.I18n.getText('quiz.create-wizard.dialog.success.message'));

                    self.wasAdded = true;
                    if (!$('.create-another').is(':checked')) {
                        self.dialog.remove();
                        updateQuestionsReportMacro();
                    } else {
                        self.questionInput.val('');
                        self.questionBodyInput.val('');
                        if (!self.reuseCorrect.is(':checked')) $('.correct-feedback').val('');
                        if (!self.reuseWrong.is(':checked')) $('.wrong-feedback').val('');
                        if (!self.reuseFree.is(':checked')) $('.free-feedback').val('');
                        if (self.prevType != QUESTION_TYPES.TRUE_FALSE) {
                            self.answerInput.val('');
                            $('.answer-variant').remove();
                        }
                        self.questionLimit.text('255').removeClass('red');
                    }

                    $('.create-wizard-dialog-panel').scrollTop(0);
                },
                complete: function () {
                    self.saveBtn.prop('disabled', false);
                    self.spinner.css('visibility', 'hidden');
                },
                error: QuizErrorHandler.showError
            });
        },

        addVariant: function() {
            var self = QuizCreateQuestion;
            var text = self.answerInput.val().trim();
            var type = self.typeInput.val();
            if (text.length) {
                var variant = $(QuizWizard.variant({
                    type: type === QUESTION_TYPES.SINGLE_CHOICE ? "radio" : "checkbox",
                    text: text,
                    hiddenInput: type === QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT
                }));
                $('.wizard-dialog #answer-variants').append(variant);
                variant.find('.quiz-variant-draggable').height(Math.floor((variant.height() - 2) / 4) * 4);
                self.answerInput.val('');
                variant.find('.variant-remove').click(function() {
                    variant.remove();
                    self.validate();
                    self.showAnswerDescription();
                });
                variant.find('.variant-text').blur(function() {
                    if (!$(this).val().trim()) {
                        variant.find('.variant-remove').click();
                    }
                });
                self.validate();
                self.showAnswerDescription();

                $('.wizard-dialog .quiz-dialog-param-wrapper.answers .answers-error .aui-message').remove();
            }
        },

        showAnswerDescription: function() {
            var self = QuizCreateQuestion;
            var type = self.typeInput.val();
            var descriptions = $('.wizard-dialog .quiz-dialog-param-wrapper.answers .description:not(.answer-desc, .free-text-answer-desc)').hide();

            descriptions.hide();
            if ($('.wizard-dialog #answer-variants .answer-variant').length) {
                if (type === QUESTION_TYPES.TRUE_FALSE) {
                    descriptions.filter('.true-false').show();
                } else if (type === QUESTION_TYPES.SINGLE_CHOICE) {
                    descriptions.filter('.single').show();
                } else if (type === QUESTION_TYPES.MULTIPLE_CHOICE) {
                    descriptions.filter('.multiple').show();
                }
            }
        },

        validate: function() {
            var self = QuizCreateQuestion;
            var variants = $('.wizard-dialog li.answer-variant');
            var checked = $('.wizard-dialog .variant-correct:checked');
            var type = self.typeInput.val();
            var text = self.questionInput.val().trim();

            var charLeft = 255 - text.length;
            self.questionLimit.text(charLeft);
            if (charLeft < 0) {
                self.questionLimit.addClass('red');
            } else {
                self.questionLimit.removeClass('red');
            }

            if (variants.length === 0 && !text) {
                self.saveEditorLink.html(AJS.I18n.getText("quiz.create-wizard.dialog.open-editor"));
            } else if (!text) {
                self.saveEditorLink.html('');
            } else {
                self.saveEditorLink.html(AJS.I18n.getText("quiz.create-wizard.dialog.save-editor"));
            }

            if (!text) {
                return VALIDATION_RESULTS.NO_QUESTION_TEXT;
            } else if ((type === QUESTION_TYPES.SINGLE_CHOICE || type === QUESTION_TYPES.MULTIPLE_CHOICE || type === QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT) && variants.length === 0) {
                return VALIDATION_RESULTS.NO_VARIANTS;
            } else if (type === QUESTION_TYPES.SINGLE_CHOICE && variants.length === 1) {
                return VALIDATION_RESULTS.ONLY_ONE_VARIANT_FOR_SC_QUESTION;
            } else if (type === QUESTION_TYPES.SINGLE_CHOICE && checked.length === 0) {
                return VALIDATION_RESULTS.NO_RIGHT_VARIANT_FRO_SC_QUESTION;
            } else {
                return VALIDATION_RESULTS.OK;
            }
        }
    };

    window.QuizImportCSV = {

        files: [],
        questions: [],

        init: function () {
            var self = QuizImportCSV;
            self.done = false;
            self.progress = false;
            self.quizIds = $('.quiz-create-wizard').data('quizid');
            self.cancel = false;
            self.paused = false;
            self.categoryId = getCategoryIdForImport();
            
            self.dialog = new AJS.Dialog({
                width: 550,
                height: self.quizIds ? 400 : 480,
                id: "import-csv-dialog",
                closeOnOutsideClick: false
            });

            self.dialog.addHeader(AJS.I18n.getText("quiz.import-csv.dialog.title"));
            self.dialog.addPanel('panel', QuizWizard.csvDialog(), 'import-csv-dialog-panel');
            self.dialog.addButton(AJS.I18n.getText("quiz.import-csv.dialog.import"), function() {
                if (self.progress) {
                    self.saveBtn.prop('disabled', true);
                    self.cancel = true;
                    if (self.pageIds.length) {
                        self.cancelLink.addClass('disabled').css('pointer-events', 'none');

                        self.removeImports(self.pageIds, function () {
                            self.dialog.remove();
                        });
                    } else {
                        self.dialog.remove();
                    }
                } else if (self.done) {
                    self.dialog.remove();
                    if (self.reload) {
                        updateQuestionsReportMacro();
                    }
                } else {
                    self.parseCSV();
                }
            }, "import-csv-dialog-save");
            self.dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                if (self.progress) {
                    if (self.paused) {
                        self.progressContainer.show();
                        $('#cancel-import').hide();
                        self.paused = false;
                        self.saveBtn.prop('disabled', true).text(AJS.I18n.getText("quiz.import-csv.dialog.import"));
                        self.cancelLink.text(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"));
                        self.imported.notify();
                    } else {
                        self.paused = true;
                        self.progressContainer.hide();
                        $('#cancel-import').show();
                        self.saveBtn.prop('disabled', false).text(AJS.I18n.getText("quiz.import-csv.dialog.confirm-cancel"));
                        self.cancelLink.text(AJS.I18n.getText("quiz.import-csv.dialog.reject-cancel"));
                    }
                } else {
                    dialog.remove();
                }
            }, '#');

            if (self.quizIds) {
                self.quizIds = [self.quizIds];
            } else {
                self.quizIds = [];
                self.initQuizAutocomplete();
            }

            self.saveBtn = $('.import-csv-dialog-save');
            self.cancelLink = self.saveBtn.parent().find('.button-panel-link');
            self.saveBtn.prop('disabled', true);
            self.paramsContainer = $('#csv-import-params');
            self.progressContainer = $('#csv-import-progress');

            var fileInput = $('#quiz-csv-file');
            fileInput.change(function () {
                self.files = fileInput.prop('files');
                if (self.files.length) {
                    self.saveBtn.prop('disabled', false);
                } else {
                    self.saveBtn.prop('disabled', true);
                }
            });

            var form = $('.import-csv-form');
            form.bind('drop', function (e) {
                if (self.paramsContainer.is(':visible') && e.originalEvent.dataTransfer) {
                    if (e.originalEvent.dataTransfer.files.length) {
                        self.files = e.originalEvent.dataTransfer.files;
                        fileInput.prop('files', self.files);
                        self.saveBtn.prop('disabled', false);
                    }
                }
                form.removeClass('drop');
                return false;
            }).bind('dragover', function() {
                form.addClass('drop');
            }).bind('dragleave', function() {
                form.removeClass('drop');
            });

            $(document).click();

            self.dialog.show();

            return false;
        },

        parseCSV: function () {
            var self = this;
            self.progress = true;

            self.saveBtn.prop('disabled', true);

            Papa.parse(self.files[0], {
                complete: function(results) {
                    self.questions = results.data;
                    self.importCSV();
                }
            });
        },

        importCSV: function () {
            var self = this;
            self.pageIds = [];
            self.paramsContainer.hide();
            var progressContainer = self.progressContainer.show();

            var pageId = QuizCommons.getContentId();
            var failed = [];
            failed.push(self.questions[0]);
            var containsErrorCol = self.questions[0][0] == AJS.I18n.getText('quiz.import-csv.error.header');
            if (!containsErrorCol) {
                failed[0].unshift(AJS.I18n.getText('quiz.import-csv.error.header'));
            }
            var total = self.questions.length - 1;    
            if (self.questions[total].length == 1) {
                total--;
            }

            var index = 1;
            self.imported = $.Deferred();
            self.imported.progress(function() {
                AJS.progressBars.update("#csv-import-progress-indicator", index / total);
                if (!self.paused) importQuestion(index++);
            });
            importQuestion(index++);

            function importQuestion(i) {
                //todo улучшить прозрачность в области обращений к question[index]
                var question = self.questions[i];

                if (i <= total) {
                    if (containsErrorCol) {
                        question.shift();
                    }

                    if (question.length === 1) {
                        self.questions.splice(i, 1);
                        total--;
                        index--;
                        self.imported.notify();
                        return;
                    }
                    if (question.length < 3) {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.type.empty'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }
                    if (!question[0].trim()) {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.title.empty'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }
                    if (question[0].trim().length > 255) {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.title.long'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }
                    if (question[0].trim() === "Sample question: What planets have rings?") {
                        self.imported.notify();
                        return;
                    }

                    var type = question[2].toLocaleLowerCase().trim();
                    if (type === QUESTION_TYPE_NAMES.MULTIPLE_CHOICE.toLowerCase()) {
                        type = QUESTION_TYPES.MULTIPLE_CHOICE;
                    } else if (!type || type === QUESTION_TYPE_NAMES.SINGLE_CHOICE.toLowerCase()) {
                        type = QUESTION_TYPES.SINGLE_CHOICE;
                    } else if (type === QUESTION_TYPE_NAMES.FREE_TEXT.toLowerCase() || type === QUESTION_TYPE_NAMES.MANUALLY_CHECKED_FREE_TEXT.toLowerCase()) {
                        type = QUESTION_TYPES.FREE_TEXT;
                    } else if (type === QUESTION_TYPE_NAMES.AUTO_CHECKED_FREE_TEXT.toLowerCase()) {
                        type = QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT;
                    } else {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.type.empty'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }

                    var random = 'false';
                    if (question.length >= 4 && type !== QUESTION_TYPES.FREE_TEXT && type !== QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT) {
                        random = question[3].trim().toLocaleLowerCase();
                        random = random.replace(/yes/i, "true").replace(/no/i, "false");
                        if (random && random != "true" && random != "false") {
                            question.unshift(AJS.I18n.getText('quiz.import-csv.error.random.wrong'));
                            failed.push(question);
                            self.imported.notify();
                            return;
                        }
                    }

                    var answers = [];
                    for (var j = 7; j < question.length; j++) {
                        if (question[j].trim()) answers.push(question[j]);
                    }

                    if (type !== QUESTION_TYPES.FREE_TEXT && answers.length === 0) {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.answers.empty'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }
                    if (type === QUESTION_TYPES.SINGLE_CHOICE && answers.length < 2) {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.answers.one'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }

                    var correct = type !== QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT ? question[6].trim() : '';

                    if (type === QUESTION_TYPES.SINGLE_CHOICE && (correct == "" || correct.split(',').length != 1)) {
                        question.unshift(AJS.I18n.getText('quiz.import-csv.error.correct.one'));
                        failed.push(question);
                        self.imported.notify();
                        return;
                    }

                    if (type !== QUESTION_TYPES.FREE_TEXT && type !== QUESTION_TYPES.AUTO_CHECKED_FREE_TEXT) {
                        var integers = correct.length ? correct.split(',') : [];
                        var nan = false;
                        integers.forEach(function (integer) {
                            nan = nan || isNaN(parseInt(integer))
                        });
                        var maxCorrect = Math.max.apply(Math, integers);
                        if (nan || maxCorrect > answers.length) {
                            question.unshift(AJS.I18n.getText('quiz.import-csv.error.correct.wrong'));
                            failed.push(question);
                            self.imported.notify();
                            return;
                        }
                    }

                    $.ajax({
                        type: 'POST',
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/create",
                        data: {
                            pageId: pageId,
                            text: question[0],
                            extension: question[1],
                            type: parseInt(type),
                            quizIds: self.quizIds,
                            answers: answers,
                            correct: correct,
                            random: random,
                            correctFeedback: question.length >= 5 ? question[4] : '',
                            wrongFeedback: question.length >= 6 ? question[5] : '',
                            categoryId: self.categoryId
                        },
                        dataType: 'json',
                        success: function(pageId) {
                            self.pageIds.push(pageId);
                            if (self.cancel) {
                                self.removeImports(pageId);
                            }
                        },
                        error: function (data) {
                            if (data.responseText.indexOf('com.atlassian.confluence.api.service.exceptions.ReadOnlyException') !== -1) {
                                QuizErrorHandler.showError(data);
                            } else {
                                question.unshift('1.' + JSON.parse(data.responseText).message);
                                failed.push(question);
                            }
                        },
                        complete: function () {
                            self.imported.notify();
                        }
                    });
                } else {
                    self.reload = true;
                    $('#csv-import-progress-indicator-container').hide();
                    if (failed.length === 1) {
                        progressContainer.find('.success').show().find('.csv-number').text(self.pageIds.length);
                    } else {
                        if (failed.length - 1 === total) {
                            progressContainer.find('.failed').show();
                            self.reload = false;
                        } else {
                            var message = progressContainer.find('.part').show();
                            message.find('.csv-number.total').text(self.pageIds.length);
                            message.find('.csv-number.failed').text(failed.length - 1);
                            if (failed.length === 2) {
                                message.find('.many').hide();
                                message.find('.one').show();
                            }
                        }
                        var csvContent = "data:text/csv;charset=utf-8," + Papa.unparse(failed);
                        var encodedUri = encodeURI(csvContent);
                        progressContainer.find('.download-failed').attr("href", encodedUri);
                    }

                    self.saveBtn.text(AJS.I18n.getText('done.name')).prop('disabled', false);
                    self.cancelLink.hide();
                    self.done = true;
                    self.progress = false;
                }
            };
        },

        removeImports: function (ids, success) {
            $.ajax({
                type: 'POST',
                url: Confluence.getContextPath() + "/rest/quiz/1.0/service/remove-imports",
                data: {
                    ids: ids
                },
                dataType: 'json',
                success: success,
                error: QuizErrorHandler.showError
            });
        },

        initQuizAutocomplete: function () {
            var self = this;
            var container = $(QuizzesMacro.pagesContainer());
            var wrapper = $('#quiz-autocomplete').show().append(container);
            var autocomplete = wrapper.find('input');

            var updateIds = function() {
                self.quizIds = [];
                $('.quiz-row').each(function() {
                    self.quizIds.push($(this).attr('id'));
                });
            };

            $('.quiz-remove').live('click', function() {
                var row = $(this).parents('.quiz-row').remove();
                updateIds();
            });

            CQAutocomplete.init(autocomplete, function($suggestion) {
                if ($('#' + $suggestion.data('key')).length == 0) {
                    container.append(QuizzesMacro.page({id: $suggestion.data('key'), title: $suggestion.text()}));
                    updateIds();
                }
            }, function(token, onSuccess, onError) {
                $.ajax({
                    type: 'GET',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/service/search/quizzes",
                    data: {
                        title: token
                    },
                    dataType: 'json',
                    success: function(variants) {
                        var data = [];
                        $.each(variants, function(i, variant) {
                            if ($('#quizzes-page-container #' + variant.key).length == 0) {
                                data.push(variant);
                            }
                        });
                        onSuccess(data);
                    },
                    error: function (errorResp) {
                        onError('Unable to find quizzes');
                    }
                });
            });
        }
    };

    window.QuizImportQuestions = {

        questionIds: [],

        init: function () {
            var self = QuizImportQuestions;
            self.categoryId = getCategoryIdForImport();
            self.quizIds = $('.quiz-create-wizard').data('quizid');
            self.existingQuestionIds = [];
            $('.quiz-questions-report tr[data-id]').each(function () {
                self.existingQuestionIds.push($(this).data('id'));
            });

            self.dialog = new AJS.Dialog({
                width: 810,
                height: 500,
                id: "import-questions-dialog",
                closeOnOutsideClick: false
            });

            self.dialog.addHeader(AJS.I18n.getText("quiz.import-questions.dialog.title"));

            var hasQuestions = true;
            $.ajax({
                type: 'POST',
                cache: false,
                async: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/service/search/questions",
                data: {
                    title: ''
                },
                dataType: 'json',
                success: function(variants) {
                    hasQuestions = variants.length > 0;
                },
                error: QuizErrorHandler.showError
            });

            if (!hasQuestions) {
                self.dialog.addPanel('panel', QuizWizard.noQuestionsMsg(), 'import-questions-dialog-panel');
            } else {
                self.dialog.addPanel('panel', QuizWizard.questionsDialog(), 'import-questions-dialog-panel');
                self.dialog.addButton(AJS.I18n.getText("quiz.import-questions.dialog.add"), function () {
                    self.save(function () {
                        QuizMessageHelper.showSuccessMsg(self.questionIds.length == 1 ? AJS.I18n.getText('quiz.import-questions.dialog.message.1') :
                            AJS.I18n.getText('quiz.import-questions.dialog.message.pl'));
                        self.dialog.remove();
                        updateQuestionsReportMacro();
                    });
                }, "import-questions-dialog-save");


                self.saveBtn = $('.import-questions-dialog-save');
                self.saveBtn.prop('disabled', true);
                self.autocompeteInput = $('.import-questions-form .reuse');
                self.autocompleteContainerQuiz = $('.import-questions-form #quizzes-page-container');
                self.autocompleteContainerQuestion = $('.import-questions-form #questions-page-container');
                self.hr = $('.import-questions-form hr');
            }

            self.dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                dialog.remove();
            }, '#');
            
            $(document).click();
            self.dialog.show();

            self.initAutocomplete();

            return false;
        },

        initAutocomplete: function () {
            var self = this;

            CQAutocomplete.init(self.autocompeteInput, function($suggestion) {
                if ($('#' + $suggestion.data('key')).length == 0) {
                    var type = $suggestion.data('type');
                    var id = $suggestion.data('key');
                    var autocompleteContainer = type === 'quiz' ? self.autocompleteContainerQuiz : self.autocompleteContainerQuestion;
                    var row = $(QuizWizard.page({
                        id: id,
                        title: $suggestion.text(),
                        icon: $suggestion.data('img'),
                        type: $suggestion.data('type'),
                        hint: $suggestion.attr('title')
                    }));

                    autocompleteContainer.append(row);

                    if (type == 'quiz') {
                        self.saveBtn.prop('disabled', true);
                        var waiterRow = $(QuizWizard.questionsWaiter({
                            parentQuiz: id
                        }));
                        self.autocompleteContainerQuiz.append(waiterRow);

                        $.ajax({
                            type: 'GET',
                            cache: false,
                            url: Confluence.getContextPath() + "/rest/quiz/1.0/service/get/questions",
                            data: {
                                id: id
                            },
                            dataType: 'json',
                            success: function(variants) {
                                waiterRow.remove();
                                
                                if (variants.length) {
                                    $.each(variants, function (i, variant) {
                                        self.autocompleteContainerQuiz.append(QuizWizard.page({
                                            id: variant.key,
                                            title: variant.name,
                                            icon: getPathToImage("question-blue"),
                                            type: 'question',
                                            parentQuiz: id,
                                            hint: $suggestion.attr('title'),
                                            alreadyExists: self.existingQuestionIds.indexOf(variant.key) !== -1
                                        }));
                                    });
                                    self.getIds();

                                    if (row.hasClass('expanded')) {
                                        row.find('.quiz-show').click();
                                    }
                                    row.data('questions-count', variants.length);
                                    row.find('.question-in-quiz').text(variants.filter(function (v) {
                                        return self.existingQuestionIds.indexOf(v.key) === -1
                                    }).length + '/' + variants.length);
                                } else {
                                    row.addClass('empty');
                                    self.getIds();
                                }
                            }
                        });
                    } else {
                        self.getIds();
                    }

                    self.updateAutocompleteContainers();
                }
            }, function(token, onSuccess, onError) {
                $.ajax({
                    type: 'GET',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/service/search/quizzes",
                    data: {
                        title: token,
                        limit: 6,
                        restrictedIds: [].concat(self.quizIds)
                    },
                    dataType: 'json',
                    success: function(variants) {
                        var data = [];
                        var hasSection = false;
                        $.each(variants, function(i, variant) {
                            if ($('#quizzes-page-container #' + variant.key).length == 0) {
                                if (!hasSection) {
                                    hasSection = true;
                                    data.push({type: 'suggestions-section', name: AJS.I18n.getText("com.stiltsoft.confluence.quiz.quizzes.label")});
                                }
                                variant.type = 'quiz';
                                variant.icon = getPathToImage("quiz-blue");
                                data.push(variant);
                            }
                        });
                        $.ajax({
                            type: 'POST',
                            cache: false,
                            url: Confluence.getContextPath() + "/rest/quiz/1.0/service/search/questions",
                            data: {
                                title: token,
                                restrictedIds: [].concat(self.questionIds).concat(self.existingQuestionIds)
                            },
                            dataType: 'json',
                            success: function(variants) {
                                var hasSection = false;
                                $.each(variants, function(i, variant) {
                                    if ($('#quizzes-page-container #' + variant.key).length == 0) {
                                        if (!hasSection) {
                                            hasSection = true;
                                            var section = {type: 'suggestions-section', name: AJS.I18n.getText("quiz.import-questions.dialog.questions")};
                                            if (data.length) {
                                                section.class = "top-border";
                                            }
                                            data.push(section);
                                        }
                                        variant.type = 'question';
                                        variant.icon = getPathToImage("question-blue");
                                        data.push(variant);
                                    }
                                });
                                onSuccess(data);
                            },
                            error: function (errorResp) {
                                onError('Unable to find questions');
                            }
                        });
                    },
                    error: function (errorResp) {
                        onError('Unable to find quizzes');
                    }
                });
            });

            $('.quiz-remove').live('click', function() {
                var row = $(this).parents('.quiz-row');
                row.parent().find('tr[data-parent="' + row.attr('id') + '"]').remove();
                row.remove();
                self.getIds();
                self.updateAutocompleteContainers();
            });

            $('.check-quiz').live('change', function() {
                var input = $(this);
                var row = input.parents('.quiz-row');
                var checked = input.is(':checked');
                var checkedQuestions = row.parent().find('tr[data-parent="' + row.attr('id') + '"]').find('input:not(:disabled)').prop('checked', checked);
                var questionsCount = row.data('questions-count');
                row.find('.question-in-quiz').text((checked ? checkedQuestions.length : 0) + '/' + questionsCount);
                self.getIds();
            });

            $('.check-question').live('change', function() {
                self.getIds();
                var row = $(this).parents('.quiz-row');
                var parentID = row.data('parent');
                var quizRow = $('#' + parentID);
                var count = $('.quiz-row.question[data-parent="' + parentID + '"] .check-question:checked').length;
                quizRow.find('.question-in-quiz').text(count + '/' + quizRow.data('questions-count'));
            });

            $('.quiz-show').live('click', function() {
                var row = $(this).parents('.quiz-row');
                row.parent().find('tr[data-parent="' + row.attr('id') + '"]').show();
                row.addClass('expanded');
            });

            $('.quiz-hide').live('click', function() {
                var row = $(this).parents('.quiz-row');
                row.parent().find('tr[data-parent="' + row.attr('id') + '"]').hide();
                row.removeClass('expanded');
            });
        },

        getIds: function () {
            var self = this;

            self.questionIds = [];
            $('.import-questions-form tr.quiz-row.question').each(function () {
                var row = $(this);
                if (row.data('parent')) {
                    if (row.find('input').is(':checked')) {
                        self.questionIds.push(row.attr('id'));
                    }
                } else {
                    self.questionIds.push(row.attr('id'));
                }
            });

            if (self.questionIds.length) {
                self.saveBtn.prop('disabled', false);
            } else {
                self.saveBtn.prop('disabled', true);
            }
        },

        updateAutocompleteContainers: function() {
            var self = this;

            var hasQuizzes = self.autocompleteContainerQuiz.find('tr').length > 0;
            var hasQuestions = self.autocompleteContainerQuestion.find('tr').length > 0;

            if (hasQuizzes) {
                self.autocompleteContainerQuiz.show();
            } else {
                self.autocompleteContainerQuiz.hide();
            }
            if (hasQuestions) {
                self.autocompleteContainerQuestion.show();
            } else {
                self.autocompleteContainerQuestion.hide();
            }

            if (hasQuizzes && hasQuestions) {
                self.hr.show();
            } else {
                self.hr.hide();
            }
        },

        save: function (success) {
            var self = this;
            self.saveBtn.prop('disabled', true);

            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/add-questions",
                data: {
                    quizIds: self.quizIds,
                    questionIds: self.questionIds,
                    categoryId: self.categoryId
                },
                dataType: 'json',
                success: success,
                error: QuizErrorHandler.showError
            });
        }
    };
    
    AJS.toInit(function() {
        $(".quiz-create-wizard").click(QuizCreateQuestion.init);
        $(".quiz-import-csv").click(QuizImportCSV.init);
        $(".quiz-import-questions").click(QuizImportQuestions.init);

        AJS.bind('quiz-questions-update-needed', updateQuestionsReportMacro);
    });

    function getCategoryIdForImport() {
        return $('.active[aria-owns="quiz-import-csv-dd"]').data('id') || 0;
    }
    
    function updateQuestionsReportMacro() {
        var content = $('.conf-macro[data-macro-name="quiz-questions-report"]');
        if (content.length) {
            $.ajax({
                type: 'GET',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/quiz-question/macro",
                data: {
                    quizId: QuizCommons.getContentId()
                },
                dataType: 'html',
                success: function (html) {
                    content.replaceWith(html);
                    AJS.trigger('quiz-questions-updated');
                }
            });
        }
    }

    function setCookie(name, value) {
        document.cookie = name + "=" + encodeURIComponent(value) + "; max-age=" + 60 * 60 * 24 * 3000;
    }

    function getCookie(name) {
        var cookie = document.cookie;
        if (cookie.length != 0) {
            var re = new RegExp('[; ]' + name + '=([^\\s;]*)');
            var matcher = (' ' + document.cookie).match(re);
            if (matcher) {
                return decodeURIComponent(matcher[1]);
            }
        }
        return '';
    }

    function getPathToImage(imageName) {
        return Confluence.getContextPath() + "/download/resources/com.stiltsoft.confluence.quiz/images/" + imageName + ".png";
    }

})(AJS.$);


