import initDueDateSettings from "../utils/due-date-settings";

(function($) {

    const REPORT_LIMIT = 50;
    var REPORT_IS_LOADED = false;

    var Course = {

        init: function () {
            var self = this;
            self.container = $('#course-content');
            self.hideable = $('body:not(.edit)').find('#main-header, .page-metadata');
            self.bulkRemove = self.container.find('#remove-bulk');
            self.enrollBtn = self.container.find('#course-enroll');
            self.sectionsContainer = self.container.find('#course-sections');

            self.table = self.container.find('.course-participant-report');
            self.reportLoadingSpinner = self.container.find('.report-loading-spinner-wrapper');

            self.tbody = self.table.find('tbody');
            self.tableWrapper = self.table.parents('.table-wrap');
            self.panel = self.container.find('#qar-panel');
            self.removeBtn = self.panel.find('#remove-course-records');
            self.endPoint = self.container.find('.end-point');
            self.tableCheckBoxes = [];

            self.emptyCourseMessage = self.container.find('.empty-course-message');
            self.courseIsEmpty = false;
            self.loadingReportMessage = self.container.find('.loading');
            self.loadingProgressBar = self.loadingReportMessage.find('.aui-progress-indicator-value');
            self.loadingProgressText = self.loadingReportMessage.find('.progress-loading-text');

            self.loadReport(function(total) {
                self.loadingReportMessage.css('display','none');
                REPORT_IS_LOADED = true;
                if (total === 0) {
                    self.table.after($(QuizCourse.emptyReportMessage()));
                } else {
                    self.initStickyElements();
                    self.bindReportCollapser();
                    self.fixBugWithDuplicateTHead();
                    self.initCheckBoxes();
                    self.bindCheckBoxes();

                    self.table.removeClass('not-visible-report');
                    self.table.tablesorter();
                    registerResizeEventListener('#course-content', self.updatePanelPosition);
                }
            }, function (errorResp) {
                console.log(JSON.stringify(errorResp));
                self.loadingReportMessage.hide();
                self.table.after(CourseReport.loadError());
            });

            self.bindCollapse();
            self.bindEnroll();
            self.bindSwitchContent();
            self.bindAddSection();
            self.bindBulkRemove();
            self.bindModuleType();
            self.bindSection();
            self.bindModule();
            self.bindAddModule();
            self.bindSettings();
            self.bindRemove();
            self.bindCancel();
            self.bindCreateModule();
            self.checkCourseIsEmpty();
        },

        loadReport: function(onSuccess, onError) {
            var self = this;
            var startFrom = 0;
            var isEditable = self.container.data('editable') === true;
            var isPreview = self.container.data('preview') === true;

            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/course/total-participant-stats",
                data: {
                    courseIds: self.container.data('id'),
                    studentIds: []
                },
                dataType: 'json',
                success: function(total) {
                    if (total === 0) {
                        onSuccess(total);
                    } else {
                        if (total > REPORT_LIMIT) self.loadingReportMessage.css('display','block');
                        getResults(startFrom, total, onSuccess, onError);
                    }
                },
                error: function(xhr) {
                    onError(xhr);
                }
            });

            function getResults(start, total, onSuccess, onError) {
                $.ajax({
                    type: 'GET',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/course/participant-stats",
                    data: {
                        courseIds: self.container.data('id'),
                        studentIds: [],
                        start: start,
                        limit: REPORT_LIMIT
                    },
                    dataType: 'json',
                    success: function(resp) {
                        resp.forEach(function(value) {
                            var reportRow = CourseReport.courseAdminPageReportRow({
                                isEditable: isEditable,
                                isPreview: isPreview,
                                stat: value
                            });
                            self.table.append(reportRow);
                        });

                        var progress = Math.round((start + resp.length) / total * 100) + '%';
                        self.loadingProgressBar.css('width', progress);
                        self.loadingProgressText.text(progress);

                        if (resp.length < REPORT_LIMIT) {
                            if (total > REPORT_LIMIT) {
                                setTimeout(function() {onSuccess(total);}, 400);
                            } else {
                                onSuccess(total);
                            }
                        } else {
                            getResults(start + REPORT_LIMIT, total, onSuccess, onError);
                        }
                    },
                    error: function(xhr) {
                        onError(xhr);
                    }
                });
            }
        },

        // https://jira.atlassian.com/browse/CONFSERVER-56489
        fixBugWithDuplicateTHead: function () {
            var self = this;
            var tables = self.container.find('table');

            tables.bind('tablesorter-initialized', function () {
                TableSorterFixer.fix($(this));
                self.bindReportCollapser();
            });

            TableSorterFixer.fix(tables);

            self.container.find('#course-participants-report-link').click(function () {
                setTimeout(function () {
                    self.updatePanelPosition();
                }, 20);
            });
        },

        initStickyElements: function () {
            var self = this;

            $(document).one('scroll', function () {
                setTimeout(function () {
                    var header = $('#header');
                    if (header.hasClass('fixed-header')) {
                        var headerHeight = header.height();
                        self.container.find('.sticky').each(function () {
                            var sticky = $(this);
                            sticky.css('top', (parseInt(sticky.css('top')) + headerHeight) + 'px');
                        });
                    }
                }, 50);
            });
        },

        bindReportCollapser: function () {
            var self = this;

            self.container.find('.collapser-container').unbind('click').click(function () {
                var btn = $(this);
                btn.toggleClass('expanded collapsed');
                btn.parents('td').find('.module-tree').toggle();
                self.updatePanelPosition();
                
                return false;
            });
        },

        bindEnroll: function () {
            var self = this;

            self.enrollBtn.click(function () {
                self.table.addClass('not-visible-report');

                var mailServer = true;
                var emptyQuiz = false;

                $.ajax({
                    type: 'GET',
                    cache: false,
                    async: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/service/mail-server",
                    success: function(data) {
                        mailServer = data;
                    }
                });

                $.ajax({
                    type: 'GET',
                    cache: false,
                    async: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/course/check-empty-quiz",
                    data: {
                        courseId: self.container.data('id')
                    },
                    success: function(data) {
                        emptyQuiz = data;
                    }
                });

                var dialog = new AJS.Dialog({
                    width: 610,
                    height: 500,
                    id: "share-course-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.share.dialog.title"));
                dialog.addPanel('panel',
                    QuizCourse.shareDialog({
                        desc: self.container.find('#course-description').val() || '',
                        mailServer: mailServer,
                        emptyQuiz: emptyQuiz,
                        isOverDue: isOverDue()
                    }),
                    'share-course-dialog-panel');

                if (mailServer && !emptyQuiz) {
                    dialog.addButton(AJS.I18n.getText("quiz.share.dialog.share"), function() {
                        submitBtn.prop('disabled', true);
                        enrollParticipantsDialogSpinner.css('display','inline-block');

                        var recipients = {
                            user: [],
                            group: [],
                            email: []
                        };

                        $('.course-share-dialog li.recipient').each(function() {
                            var elem = $(this);
                            recipients[elem.data('type')].push(elem.data('key'));
                        });

                        $.ajax({
                            type: 'POST',
                            cache: false,
                            url: Confluence.getContextPath() + "/rest/quiz/1.0/course/enroll",
                            data: {
                                courseId: self.container.data('id'),
                                desc: $('.course-share-dialog .course-desc').val(),
                                users: recipients.user,
                                groups: recipients.group,
                                emails: recipients.email
                            },
                            dataType: 'json',
                            success: function(data) {
                                enrollParticipantsDialogSpinner.css('display','none');
                                self.reportLoadingSpinner.css('display','block');
                                dialog.remove();
                                self.showEnrollResult(data);
                            },
                            error: function (xhr) {
                                submitBtn.enable(xhr);
                                enrollParticipantsDialogSpinner.css('display','none');
                                QuizErrorHandler.showError();
                            }
                        });
                    }, "share-course-dialog-save");
                }

                dialog.addLink(AJS.I18n.getText("share.dialog.form.cancel"), function (dialog) {
                    $('.quiz-suggestions').remove();
                    self.reportLoadingSpinner.css('display','block');
                    dialog.remove();

                    self.showReport();
                }, '#');

                var submitBtn = $('.share-course-dialog-save').prop('disabled', true);
                var enrollParticipantsDialogSpinner = $(QuizCourse.spinner());
                submitBtn.before(enrollParticipantsDialogSpinner);

                CQAutocomplete.init($('.autocomplete-course-share'), function($suggestion) {
                        CQAutocomplete.autocompleteParticipants($suggestion, $('.course-share-dialog'), submitBtn);
                    }, CQAutocomplete.findParticipants);

                dialog.show();
            });

            function isOverDue() {
                var dueDate = $('.cs-due-date-option-container .due-date-wrapper .due-date').data('due-date');
                var today = new Date();
                today.setHours(0, 0, 0, 0);

                return dueDate && today > $.datepicker.parseDate('yy-mm-dd', dueDate);
            }
        },

        showEnrollResult: function (data) {
            var self = this;

            var dialog = new AJS.Dialog({
                width: 600,
                height: 500,
                id: "course-enroll-results-dialog",
                closeOnOutsideClick: false
            });

            dialog.addHeader(AJS.I18n.getText("quiz.share.dialog.title"));
            dialog.addPanel('panel', QuizCourse.enrollResults({data: data}), 'course-enroll-results-dialog-panel');

            dialog.addLink(AJS.I18n.getText("close.name"), function (dialog) {
                dialog.remove();
                if (data.enrolledCount) {
                    self.container.find('.empty-report-message').remove();
                    location.reload();
                } else {
                    self.showReport();
                }
            }, '#');

            dialog.show();
        },

        bindCollapse: function () {
            var self = this;

            self.container.find('#course-collapse').click(function () {
                var expanded = self.container.hasClass('collapsed');
                if (expanded) {
                    self.container.removeClass('collapsed').addClass('expanded');
                    self.hideable.addClass('course-view').css('display', '');
                    $('#main').removeClass('course-expanded');
                } else {
                    self.container.removeClass('expanded').addClass('collapsed');
                    self.hideable.hide();
                    $('#main').addClass('course-expanded');
                }

                $.ajax({
                    type: 'POST',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/course/expand",
                    data: {
                        courseId: self.container.data('id'),
                        expanded: expanded
                    },
                    dataType: 'json',
                    error: QuizErrorHandler.showError
                });
            });

            if (self.container.hasClass('collapsed')) {
                $('#main').addClass('course-expanded');
            }
        },

        bindSwitchContent: function () {
            var self = this;

            var sideBarBtns = self.container.find('div.course-sidebar-btn').click(function () {
                var btn = $(this);
                sideBarBtns.removeClass('active');
                btn.addClass('active');
                self.container.find('.course-body-content, .course-title').removeClass('active');
                self.container.find('.' + btn.data('type')).addClass('active');
                if (btn.data('type') === 'course-reports') {
                    self.showReportTabContent();
                } else {
                    self.hideReportTabContent();
                }

                location.hash = '#' + btn.data('type');
            });

            self.container.find('.course-sidebar-btn[data-type="' + location.hash.replace('#', '') + '"]').click();
        },

        bindAddSection: function () {
            var self = this;

            self.container.find('#add-section').click(function () {
                $.ajax({
                    type: 'POST',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/course/section/add",
                    data: {
                        courseId: self.container.data('id')
                    },
                    dataType: 'json',
                    success: function (id) {
                        var newSection = $(QuizCourse.section({
                            'id': id,
                            'name': '',
                            'spaceKey': self.container.data('spacekey'),
                            'pageId': self.container.data('id'),
                            'bpId': self.container.data('bpid'),
                            'modules': []
                        }));
                        self.sectionsContainer.append(newSection);
                        self.bindAutocomplete(newSection);
                        self.makeCourseTreeSortable();
                        self.checkCourseIsEmpty();

                        $('html, body').animate({
                            scrollTop: newSection.offset().top - 200
                        }, 300);
                    },
                    error: QuizErrorHandler.showError
                });
            });
        },

        bindBulkRemove: function () {
            var self = this;

            self.bulkRemove.click(function () {
                var dialog = new AJS.Dialog({
                    width: 500,
                    height: 200,
                    id: "delete-bulk-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.course.bulk.remove.dialog.title"));
                dialog.addPanel('panel', "<div>" + AJS.I18n.getText("quiz.course.bulk.remove.dialog.body") + "</div>", 'delete-bulk-dialog-panel');
                dialog.addButton(AJS.I18n.getText("quiz.report.dialog.delete.confirm"), function() {
                    var sectionIds = [];
                    var sections = $();
                    self.container.find('.bulk-section-select:checked').each(function () {
                        var input = $(this);
                        var section = input.parents('.course-section');
                        if (section.find('.bulk-module-select:not(:checked)').length === 0) {
                            sectionIds.push(section.data('id'));
                            sections = sections.add(section);
                        } else {
                            input.prop('checked', false);
                        }
                    });

                    var moduleIds = [];
                    var modules = $();
                    self.container.find('.bulk-module-select:checked').each(function () {
                        var module = $(this).parents('.course-module');
                        moduleIds.push(module.data('id'));
                        modules = modules.add(module);
                    });

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/course/bulk/remove",
                        data: {
                            courseId: self.container.data('id'),
                            sectionIds: sectionIds,
                            moduleIds: moduleIds
                        },
                        dataType: 'json',
                        success: function() {
                            $(sections).remove();
                            $(modules).remove();
                            dialog.remove();
                            self.checkSectionEmpty();
                        },
                        error: QuizErrorHandler.showError
                    });
                }, "delete-bulk-dialog-save");
                dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                    dialog.remove();
                }, '#');

                dialog.show();
            });
        },

        bindModuleType: function () {
            var self = this;

            self.container.delegate('.module-type', 'change', function () {
                var select = $(this);
                var parent = select.parents('.add-module');
                if (select.val() === '0') {
                    parent.addClass('page').removeClass('quiz');
                } else {
                    parent.removeClass('page').addClass('quiz');
                }
            });

            $(document).delegate('.select-module-dd a', 'click', function (e) {
                var a = $(this);
                var container = $('.add-module[data-id="' + a.data('id') + '"]').removeClass('none').addClass(a.data('class'));
                container.find('.module-type').val(a.data('val'));
                container.find('.' + a.data('class') + '-autocomplete').focus();
                e.stopPropagation();
            });

            $(document).click(function (e) {
                var target = $(e.target);
                if (!target.is('.course-section,.remove-section,.remove-module,.section-name')) {
                    target = target.parents('.course-section');
                }
                self.hideModuleAdders(target.data('id'));
            });
        },

        hideModuleAdders: function(excludeSectionId) {
            var self = this;

            self.container.find('.add-module').not('[data-id="' + excludeSectionId + '"]').removeClass('quiz page').addClass('none');
        },

        saveSectionTitle: function(input) {
            var self = this;
            var id = input.parents('.course-section').data('id');

            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/course/section/rename",
                data: {
                    courseId: self.container.data('id'),
                    name: input.val(),
                    id: id
                },
                dataType: 'json',
                error: QuizErrorHandler.showError
            });

            input.blur();
            input.parents('.section-table').find('.save-section-name').hide();
            self.checkCourseIsEmpty();
        },

        bindSection: function () {
            var self = this;

            self.container.delegate('.save-section-name', 'click', function () {
                var link = $(this);
                if (link.is(':visible')) {
                    var input = $(this).parents('.section-table').find('.section-name');
                    self.saveSectionTitle(input);
                }
                return false;
            });

            self.container.delegate('.section-name', 'keyup keydown', function (e) {
                var saveLink = $(this).parents('.section-table').find('.save-section-name');

                if (e.which === 13) {
                    saveLink.click();
                    return false;
                } else {
                    saveLink.show();
                }
            }).delegate('.section-name', 'change', function (e) {
                $(this).parents('.section-table').find('.save-section-name').click();
            });

            self.container.delegate('.remove-section', 'click', function () {
                var section = $(this).parents('.course-section');

                if (section.hasClass('empty')) {
                    self.removeSection(section);
                    return;
                }

                var dialog = new AJS.Dialog({
                    width: 500,
                    height: 200,
                    id: "delete-section-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.course.remove.dialog.title"));
                dialog.addPanel('panel', "<div>" + AJS.I18n.getText("quiz.course.remove.dialog.body") + "</div>", 'delete-section-dialog-panel');
                dialog.addButton(AJS.I18n.getText("quiz.report.dialog.delete.confirm"), function() {
                    self.removeSection(section, function () {
                        dialog.remove();
                    });
                }, "delete-section-dialog-save");
                dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                    dialog.remove();
                }, '#');

                dialog.show();
            });

            self.container.delegate('.bulk-section-select', 'change', function () {
                var input = $(this);
                var section = input.parents('.course-section');
                var isChecked = input.is(':checked');
                section.find('.bulk-module-select').prop('checked', isChecked);
                self.enableBulkRemove();
            });

            self.container.delegate('.bulk-module-select', 'change', function () {
                var input = $(this);
                var isChecked = input.is(':checked');
                if (!isChecked && input.parents('.modules').find('.bulk-module-select:checked').length === 0) {
                    input.parents('.course-section').find('.bulk-section-select').prop('checked', false);
                }
            });

            self.makeCourseTreeSortable();
        },

        removeSection: function (section, callback) {
            var self = this;

            $.ajax({
                type: 'POST',
                cache: false,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/course/section/remove",
                data: {
                    courseId: self.container.data('id'),
                    id: section.data('id')
                },
                dataType: 'json',
                success: function() {
                    section.remove();
                    self.enableEnroll();
                    self.checkCourseIsEmpty();
                    if (callback) callback();
                },
                error: QuizErrorHandler.showError
            });
        },

        makeCourseTreeSortable: function () {
            var self = this;

            self.sectionsContainer.sortable({
                containment: 'parent',
                items: ".course-section",
                handle: '.section-handle',
                axis: 'y',
                update: function () {
                    var ids = [];
                    self.container.find('.course-section').each(function () {
                        ids.push($(this).data('id'));
                    });

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/course/section/reorder",
                        data: {
                            courseId: self.container.data('id'),
                            ids: ids
                        },
                        dataType: 'json',
                        error: QuizErrorHandler.showError
                    });
                }
            });
            
            self.container.find('.modules').sortable({
                containment: self.sectionsContainer,
                items: ".course-module",
                handle: '.module-handle',
                connectWith: '.modules',
                axis: 'y',
                start: function (e, ui) {
                    self.sectionsContainer.addClass('sorting');
                },
                stop: function (e, ui) {
                    self.sectionsContainer.removeClass('sorting');
                },
                update: function (e, ui) {
                    var ids = [];
                    self.container.find('.course-module').each(function () {
                        ids.push($(this).data('id'));
                    });

                    self.checkSectionEmpty();

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/course/module/reorder",
                        data: {
                            courseId: self.container.data('id'),
                            ids: ids,
                            id: ui.item.data('id'),
                            sectionId: ui.item.parents('.course-section').data('id')
                        },
                        dataType: 'json',
                        error: QuizErrorHandler.showError
                    });
                }
            })
        },

        checkSectionEmpty: function () {
            var self = this;
            self.container.find('.course-section').each(function () {
                var section = $(this);
                if (section.find('.course-module').length) {
                    section.removeClass('empty');
                } else {
                    section.addClass('empty');
                }
            });

            self.checkCourseIsEmpty();
        },

        checkCourseIsEmpty: function() {
            var self = this;
            var sections = self.container.find('.course-section');
            if (sections.length === 0 || (sections.length === 1 && sections.hasClass('empty'))) {
                self.sectionsContainer.addClass('default');
                if (sections.length === 0) {
                    self.container.find('#add-section').click();
                }

                self.emptyCourseMessage.toggleClass("hidden-by-content", false);
            } else {
                self.sectionsContainer.removeClass('default');
                self.emptyCourseMessage.toggleClass("hidden-by-content", true)
            }
        },

        bindModule: function () {
            var self = this;

            self.container.delegate('.remove-module', 'click', function () {
                var module = $(this).parents('.course-module');

                var dialog = new AJS.Dialog({
                    width: 500,
                    height: 200,
                    id: "delete-module-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.course.module.remove.dialog.title"));
                dialog.addPanel('panel', "<div>" + AJS.I18n.getText("quiz.course.module.remove.dialog.body") + "</div>", 'delete-module-dialog-panel');
                dialog.addButton(AJS.I18n.getText("quiz.report.dialog.delete.confirm"), function() {
                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/course/module/remove",
                        data: {
                            courseId: self.container.data('id'),
                            id: module.data('id')
                        },
                        dataType: 'json',
                        success: function() {
                            module.remove();
                            dialog.remove();
                            self.enableEnroll();
                            self.checkSectionEmpty();
                        },
                        error: QuizErrorHandler.showError
                    });
                }, "delete-module-dialog-save");
                dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                    dialog.remove();
                }, '#');

                dialog.show();
            });

            self.container.delegate('.bulk-module-select', 'change', function () {
                self.enableBulkRemove();
            });
        },

        bindAddModule: function () {
            var self = this;

            self.container.find('.course-section').each(function () {
                self.bindAutocomplete($(this));
            })
        },

        updateReminderDays: function(input, previousValues) {
            var self = this;

            var dataForServer = {
                courseId: self.container.data('id'),
                value: input.val().trim(),
                updateEnrolled: false
            };

            if ($('.course-reports #course-participants-report tbody tr').length > 0) {
                var dialog = new AJS.Dialog({
                    width: 550,
                    height: 250,
                    id: "update-reminder-days-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.course.settings.dialog.reminder-days"));
                dialog.addPanel('panel', QuizSettingsTemplate.updateEnrolledDialog({type: 'reminderdays'}), 'update-reminder-days-dialog');
                dialog.addButton(AJS.I18n.getText("update.word"), function() {
                    dataForServer.updateEnrolled = $('#update-enrolled-input').is(':checked');
                    sendCourseSetting(dataForServer, input.data('type'));

                    previousValues[input.data('type')] = dataForServer.value;

                    dialog.remove();
                }, "update-reminder-days-dialog-ok");
                dialog.addLink(AJS.I18n.getText("cancel.name"), function() {
                    input.val(previousValues[input.data('type')]);

                    dialog.remove();
                }, '#');

                dialog.show();
            } else {
                sendCourseSetting(dataForServer, input.data('type'));
                previousValues[input.data('type')] = dataForServer.value;
            }
        },

        bindSettings: function () {
            var self = this;

            QuizNumberUtils.bindNumberInputs(self.container);

            //todo временное решение для хранения предыдущего значения количества дней для напоминалки
            // будет рефакторинг в рамках задачи QUIZ-608
            var previousValues = {};
            previousValues['reminder-days'] = self.container.find('#reminder-days')[0].value;
            self.container.find('.course-setting').bind('change', function () {
                var input = $(this);
                if (input.data('type') == 'reminder-days') {
                    self.updateReminderDays(input, previousValues);
                } else {
                    var dataForServer = {
                        courseId: self.container.data('id'),
                        value: input.val().trim()
                    };
                    sendCourseSetting(dataForServer, input.data('type'));
                }
            });

            $("#course-categories").bind('change', function (e) {
                $.ajax({
                    type: 'POST',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/course/settings/category/" + (e.added ? 'add' : 'delete'),
                    data: {
                        courseId: self.container.data('id'),
                        category: e.added ? e.added.id : e.removed.id
                    },
                    dataType: 'json',
                    error: QuizErrorHandler.showError
                });
            }).auiSelect2({formatNoMatches: function () {
                    return AJS.I18n.getText('quiz.course.categories.empty');
                }});

            self.container.find('.cs-self-enrollment-option-container').each(function() {
                var SPECIFIC_GROUP_TYPE = 3; //соотносится с CourseSelfEnrollmentType.class

                var container = $(this);
                var viewContainer = container.find('.quiz-settings-view');
                var viewOptions = viewContainer.find('.cs-option-label');
                var editContainer = container.find('.quiz-settings-edit');
                var editOptions = editContainer.find('.cs-option-label');
                var tipMessage = container.find('.cs-self-enrollment-option-tip');
                var viewGroupsContainer = viewContainer.find('.cs-self-enrollment-groups');
                var groupSelector = editContainer.find('.cs-self-enrollment-group-selector');
                var groupAutocompleteDropdown = editContainer.find('.aui-dd-parent.autocomplete');
                var editGroupsContainer = editContainer.find('.cs-self-enrollment-groups');
                var previouslyCheckedType = getCheckedOptionType(viewContainer);
                var previouslySelectedGroups = getGroups(viewGroupsContainer);
                var selectedGroups = [];
                var isEditing = false;

                toggleEmptyCourseMessage();

                viewOptions.on('click', function(e) {
                    if (!isEditing) {
                        startEditing($(this).find('input').data('type'));
                    }

                    e.stopPropagation();
                    return false;
                });

                viewContainer.on('click', function(e) {
                    if (!isEditing) {
                        startEditing(previouslyCheckedType);
                    }

                    e.stopPropagation();
                    return false;
                });

                viewOptions.find('input').on('change', function(e) {
                    var input = $(this);
                    if (input.data('type') === SPECIFIC_GROUP_TYPE) {
                        viewGroupsContainer.show();
                    } else {
                        viewGroupsContainer.hide();
                    }
                });

                editOptions.find('input').on('change', function(e) {
                    var input = $(this);
                    if (input.data('type') === SPECIFIC_GROUP_TYPE) {
                        updateGroups(previouslySelectedGroups, true, editGroupsContainer);
                        selectedGroups = getGroups(editGroupsContainer);
                        groupSelector.show();
                        groupSelector.focus();
                        editGroupsContainer.show();
                    } else {
                        updateGroups([], true, editGroupsContainer);
                        selectedGroups = [];
                        groupAutocompleteDropdown.find('.aui-dropdown:not(".hidden")').each(function() {
                            $(this).toggleClass('hidden');
                        });
                        groupSelector.val(null);
                        groupSelector.hide();
                        editGroupsContainer.hide();
                    }
                });

                editContainer.on('click', function(e) {
                    e.stopPropagation();
                });

                editContainer.on('click', '.cs-self-enrollment-group .cs-remove-element', function() {
                    var groupElement = $(this).parent();
                    var name = groupElement.data('name');
                    groupElement.remove();
                    selectedGroups.splice(selectedGroups.indexOf(name), 1);
                });

                groupSelector.on("selected.autocomplete-user-or-group", function(e, data) {
                    groupSelector.val(null);
                    if (data && selectedGroups.indexOf(data.content.name) === -1) {
                        selectedGroups.push(data.content.name);
                        appendGroup(data.content.name, true, editGroupsContainer);
                    }
                });

                $(document).on('click', function(e) {
                    if (isEditing) {
                        finishEditing();
                    }
                });

                editContainer.find('.save-options button.cancel').click(finishEditing);
                editContainer.find('.save-options button.submit').on('click', function() {
                    var checkedType = getCheckedOptionType(editContainer);
                    var groups = checkedType === SPECIFIC_GROUP_TYPE ? getGroups(editGroupsContainer) : [];

                    if (checkedType === SPECIFIC_GROUP_TYPE && groups.length === 0) {
                        AJS.flag({
                            type: 'error',
                            close: 'auto',
                            body: AJS.I18n.getText("quiz.course.settings.param.self-enrollment.message.groups-missed")
                        })
                    } else {
                        $.ajax({
                            type: 'POST',
                            cache: false,
                            async: false,
                            url: Confluence.getContextPath() + "/rest/quiz/1.0/course/settings/self-enrollment-type",
                            data: {
                                courseId: self.container.data('id'),
                                type: checkedType,
                                groups: groups.join(',')
                            },
                            dataType: 'json',
                            success: function () {
                                previouslyCheckedType = checkedType;
                                previouslySelectedGroups = groups;
                                finishEditing();
                                toggleEmptyCourseMessage();
                            },
                            error: QuizErrorHandler.showError
                        });
                    }

                    return false;
                });

                function startEditing(checkedType) {
                    isEditing = true;

                    viewContainer.hide();
                    editContainer.show();
                    tipMessage.show();

                    forceCheckInput(editOptions, checkedType);
                    updateGroups(previouslySelectedGroups, true, editGroupsContainer);
                    selectedGroups = getGroups(editGroupsContainer);
                }

                function finishEditing() {
                    isEditing = false;
                    tipMessage.hide();
                    editContainer.hide();
                    viewContainer.show();

                    forceCheckInput(viewOptions, previouslyCheckedType);
                    updateGroups(previouslySelectedGroups, false, viewGroupsContainer);
                }

                function forceCheckInput(options, type) {
                    options.each(function(index, element) {
                        var label = $(element);
                        var input = label.find('input');
                        if (input.data('type') === type) {
                            input.prop('checked', true);
                            input.trigger('change');
                        }
                    });
                }

                function getCheckedOptionType(container) {
                    var input = container.find('input:checked');
                    if (input) {
                        input.removeAttr('checked');
                        input.prop('checked', true);

                        return input.data('type');
                    } else {
                        return null;
                    }
                }

                function getGroups(container) {
                    var names = [];

                    container.find('.cs-self-enrollment-group').each(function() {
                        names.push($(this).data('name'));
                    });

                    return names;
                }

                function appendGroup(name, isRemovable, container) {
                    container.append(CourseSettings.group({name: name, isRemovable: isRemovable}));
                }

                function updateGroups(groups, isRemovable, container) {
                    container.find('.cs-self-enrollment-group').remove();
                    groups.forEach(function(name) {
                        appendGroup(name, isRemovable, container);
                    });
                }

                function toggleEmptyCourseMessage() {
                    if (previouslyCheckedType > 0) {
                        self.emptyCourseMessage.toggleClass("hidden-by-se-type", false);
                    } else {
                        self.emptyCourseMessage.toggleClass("hidden-by-se-type", true);
                    }
                }
            });

            initDueDateSettings(AJS, self.container.find('.course-settings .cs-due-date-option-container .quiz-settings-edit'));

            //todo в рамках QUIZ-575 можно будет избавиться от костылей(разной инициализации для editable-option)
            self.container.find('.cs-due-date-option-container').each(function() {
                var settings = $(this);
                var views = settings.find('.quiz-settings-view');
                var edits = settings.find('.quiz-settings-edit');
                var manager = {
                    dueDateText: edits.find('.due-date-text')
                };

                edits.click(function(e) {
                    e.stopPropagation();
                });

                views.click(function(e) {
                    if ($(e.target).is('a')) {
                        return;
                    }

                    startEditing($(this));

                    return false;
                });

                $(document).click(function() {
                    cancelEditing();
                });

                edits.find('input').keydown(function(e) {
                    if (e.which == 13) {
                        save();
                        return false;
                    } else if (e.which == 27) {
                        cancelEditing();
                    }
                });
                edits.find('.save-options button.cancel').click(cancelEditing);
                //todo по необъяснимой причине enter в инпуты(particiapnt reminder и self enrollment group selector) вызывает триггер этого события
                edits.find('.save-options button.submit').click(function() {
                    if (manager.editing) {
                        save();
                    }
                    return false;
                });

                function startEditing(view) {
                    var edit = view.parent().find('.quiz-settings-edit');

                    cancelEditing();
                    view.hide();
                    manager.editing = true;
                    manager.input = edit.show().find('input[data-type]');
                    manager.prevValue = [];
                    manager.input.each(function() {
                        manager.prevValue.push($(this).val());
                    });
                }

                function cancelEditing() {
                    if (manager.isDialog) return;
                    if (!manager.editing) return;
                    manager.editing = false;
                    manager.input.each(function(i) {
                        var thisInput = $(this).val(manager.prevValue[i]);
                        if (thisInput.is('input')) {
                            var parent = thisInput.parent();
                            parent.find('input[type="radio"]').prop('checked', manager.prevValue[i].length > 0);
                            var stuff = parent.find('.param-colon, .due-date-text, input[data-type="due-days"], .clear-due-date');
                            if (!manager.prevValue[i]) {
                                stuff.hide();
                            } else {
                                stuff.show();
                            }
                        }
                    });
                    manager.dueDateText.text(views.find('.due-date').text());
                    views.show();
                    edits.hide();
                    $('.aui-datepicker-dialog').hide();
                }

                function save(updateEnrolled) {
                    var type = manager.input.data('type');

                    if (updateEnrolled === undefined && $('.course-reports #course-participants-report tbody tr').length > 0) {
                        manager.isDialog = true;

                        var dialog = new AJS.Dialog({
                            width: 550,
                            height: 250,
                            id: "update-attempts-dialog",
                            closeOnOutsideClick: false
                        });

                        dialog.addHeader(AJS.I18n.getText("quiz.answers-report.due-date.dialog.header"));
                        dialog.addPanel('panel', QuizSettingsTemplate.updateEnrolledDialog({type: type.replace('-', '')}), 'update-attempts-dialog');
                        dialog.addButton(AJS.I18n.getText("update.word"), function() {
                            updateEnrolled = $('#update-enrolled-input').is(':checked');
                            dialog.remove();
                            manager.isDialog = false;
                            save(updateEnrolled);
                        }, "update-attempts-dialog-ok");
                        dialog.addLink(AJS.I18n.getText("cancel.name"), function () {
                            dialog.remove();
                            manager.isDialog = false;
                            cancelEditing();
                        }, '#');

                        dialog.show();
                        return;
                    }

                    manager.prevValue = [];
                    manager.input.each(function() {
                        manager.prevValue.push($(this).val());
                    });
                    var insertValue = manager.prevValue;
                    var updatedEnrolled = false;

                    manager.input.each(function(i, elem) {
                        var inputType = $(elem).data('type');
                        var dataForServer = {
                            courseId: self.container.data('id'),
                            value: manager.prevValue[i],
                            updateEnrolled: updateEnrolled && !updatedEnrolled && (manager.prevValue[i] != '' || manager.input.length - 1 === i)
                        };
                        sendCourseSetting(dataForServer, inputType);

                        updatedEnrolled = updatedEnrolled || manager.prevValue[i] != '';
                    });

                    manager.input.each(function(i) {
                        var thisInput = $(this);
                        var thisView = thisInput.parents('div.field-group').find('.quiz-settings-view');
                        var thisViewInput = thisView.find('[data-view-type="' + thisInput.data('type') + '"]');

                        if (thisInput.hasClass('due-date')) {
                            thisViewInput.text(manager.dueDateText.text());
                            thisViewInput.data('due-date', insertValue[i]);
                        } else {
                            thisViewInput.text(insertValue[i]);
                        }

                        if (thisInput.is('input')) {
                            thisView.find('input[type="radio"]').eq(i).prop('checked', insertValue[i].length > 0);
                            if (insertValue[i].length) {
                                thisView.find('.param-colon').eq(i).show();
                            } else {
                                thisView.find('.param-colon').eq(i).hide();
                            }
                        }
                    });

                    cancelEditing();
                }
            });
        },

        bindAutocomplete: function (section) {
            var self = this;

            var sectionId = section.data('id');
            var container = section.find('.modules');
            self.initAutocomplete(container, section.find('.page-autocomplete'), sectionId, 0, section);
            self.initAutocomplete(container, section.find('.quiz-autocomplete'), sectionId, 1, section);
        },

        addModule: function(sectionId, pageId, pageTitle, moduleType, container, section, async, callback) {
            var self = this;

            $.ajax({
                type: 'POST',
                cache: false,
                async: async,
                url: Confluence.getContextPath() + "/rest/quiz/1.0/course/module/add",
                data: {
                    courseId: self.container.data('id'),
                    pageId: pageId,
                    type: moduleType,
                    sectionId: sectionId
                },
                dataType: 'json',
                success: function (module) {
                    container.append(QuizCourse.module({module: {
                            id: module.id,
                            pageId: pageId,
                            name: pageTitle,
                            type: moduleType,
                            emptyQuiz: module.emptyQuiz,
                            quizAttempts: module.quizAttempts
                        }}));
                    section.removeClass('empty');
                    self.checkCourseIsEmpty();
                    self.enableEnroll();

                    if (callback) callback();
                },
                error: QuizErrorHandler.showError
            });
        },

        initAutocomplete: function (container, input, sectionId, type, section) {
            var self = this;

            CQAutocomplete.init(input, function($suggestion) {
                if (self.container.find('.course-module[data-pageId="' + $suggestion.data('key') + '"]').length === 0) {
                    self.addModule(sectionId, $suggestion.data('key'), $suggestion.text(), type, container, section, true);
                }
            }, function(token, success, onError) {
                var restrictedIds = [];
                self.container.find('.course-module').each(function () {
                    restrictedIds.push($(this).data('pageid'));
                });

                var entities = type ? "quizzes" : "pages";
                $.ajax({
                    type: 'GET',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/service/search/" + entities,
                    data: {
                        title: token,
                        restrictedIds: [].concat(restrictedIds),
                        limit: 20
                    },
                    dataType: 'json',
                    success: function(variants) {
                        var data = [];
                        $.each(variants, function(i, variant) {
                            if (container.find('.course-module[data-pageId="' + variant.key + '"]').length === 0) {
                                data.push(variant);
                            }
                        });
                        success(data);
                    },
                    error: function (errorResp) {
                        onError('Unable to find ' + entities);
                    }
                });
            });
        },

        enableBulkRemove: function () {
            var self = this;
            if (self.container.find('.section-checkbox-wrap input:checked').length) {
                self.bulkRemove.show();
            } else {
                self.bulkRemove.hide();
            }
        },

        enableEnroll: function () {
            this.enrollBtn.prop('disabled', this.container.find('.course-module').length === 0);
        },

        initCheckBoxes: function () {
            var self = this;

            self.table.each(function (i) {
                self.tableCheckBoxes[i] = $(this).find('.record-checkbox');
            });

            self.checkBoxes = $(self.tableCheckBoxes).map(function () { return this.toArray(); } );

            self.table.bind('tablesorter-initialized', function() {
                self.initCheckBoxes();
            });

            TableSorterFixer.fix(self.table);
        },

        bindCheckBoxes: function () {
            var self = this;

            self.table.each(function (i) {
                $(this).find('#select-all-records').live('change', function() {
                    if ($(this).is(':checked')) {
                        self.tableCheckBoxes[i].attr('checked', 'checked');
                    } else {
                        self.tableCheckBoxes[i].attr('checked', false);
                    }

                    self.onCheckBoxesChange();
                });
            });

            self.checkBoxes.bind('change', function() {
                self.onCheckBoxesChange();
            });
        },

        onCheckBoxesChange: function() {
            var self = this;

            if (self.checkBoxes.filter(':checked').length) {
                self.showPanel();
            } else {
                self.hidePanel();
            }
        },

        bindRemove: function () {
            var self = this;

            self.removeBtn.click(function() {
                self.table.addClass('not-visible-report');
                self.panel.css('display', 'none');

                var dialog = new AJS.Dialog({
                    width: 500,
                    height: 200,
                    id: "delete-answers-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.report.dialog.delete.title"));
                dialog.addPanel('panel', "<div>" + AJS.I18n.getText("quiz.report.dialog.delete.body") + "</div>", 'delete-answers-dialog-panel');
                dialog.addButton(AJS.I18n.getText("quiz.report.dialog.delete.confirm"), function() {
                    submitBtn.prop('disabled', true);
                    removeParticipantsDialogSpinner.css('display','inline-block');

                    var checked = self.checkBoxes.filter(':checked');
                    var ids = [];
                    checked.each(function() {
                        ids.push($(this).attr('id'));
                    });

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        url: Confluence.getContextPath() + "/rest/quiz/1.0/course/answers/remove",
                        data: {
                            courseId: self.container.data('id'),
                            ids: ids
                        },
                        dataType: 'json',
                        success: function() {
                            removeParticipantsDialogSpinner.css('display','none');
                            self.reportLoadingSpinner.css('display', 'block');

                            self.checkBoxes = self.checkBoxes.not(checked);
                            checked.parents('tr').remove();
                            self.hidePanel();
                            if (self.checkBoxes.length) {
                                self.table.trigger('update', false);
                            }
                            dialog.remove();

                            self.showReport();
                        },
                        error: function(xhr) {
                            removeParticipantsDialogSpinner.css('display','none');
                            QuizErrorHandler.showError(xhr);
                        }
                    });
                }, "delete-answers-dialog-save");
                dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                    self.reportLoadingSpinner.css('display','block');
                    self.panel.css('display','none');
                    dialog.remove();

                    self.showReport(function() {
                        self.showPanel();
                    });
                }, '#');

                var submitBtn = $('.delete-answers-dialog-save');
                var removeParticipantsDialogSpinner = $(QuizCourse.spinner());
                submitBtn.before(removeParticipantsDialogSpinner);

                dialog.show();

                return false;
            });
        },

        bindCancel: function () {
            var self = this;
            $('#qcr-cancel').click(function () {
                self.hidePanel();
                return false;
            });
        },

        showPanel: function () {
            var self = this;
            self.endPoint.show();
            self.updatePanelPosition();
            self.panel.show();
        },

        hidePanel: function () {
            var self = this;
            self.panel.hide();
            self.checkBoxes.attr('checked', false);
            $('#select-all-records').attr('checked', false);
            self.endPoint.css('margin-bottom', '').hide();
        },

        updatePanelPosition: function() {
            var self = Course;

            if (!self.endPoint.offset()) return;
            var jqueryWindow = $(window);
            var windowPosition = jqueryWindow.height() + jqueryWindow.scrollTop();
            var windowBottom = windowPosition - self.endPoint.offset().top;
            self.panel.css({'bottom': Math.min(Math.max(5, windowBottom - 47), windowPosition - self.tbody.offset().top - 43),
                'left': self.tableWrapper.offset().left,
                'width': self.tableWrapper.width() - 12});
            if (windowBottom > 5) {
                self.endPoint.css('margin-bottom', Math.min(windowBottom - 5, 50) + 'px');
            } else {
                self.endPoint.css('margin-bottom', '');
            }
        },

        showReport: function(afterCallback) {
            var self = this;

            setTimeout(function() {
                self.table.removeClass('not-visible-report');
                self.reportLoadingSpinner.css('display', 'none');
                if (typeof afterCallback === 'function') afterCallback();
            }, 1000);
        },

        bindCreateModule: function () {
            var self = this;

            self.container.delegate('.create-quiz', 'click', function () {
                var a = $(this);
                var data = a.data();
                var autocompleteInput = a.parents('tr').find('.quiz-autocomplete');
                data.title = {
                    title: autocompleteInput.val(),
                    courseId: data.courseId,
                    sectionId: data.sectionId
                };
                Confluence.Blueprint.loadDialogAndOpenTemplate(data);
                autocompleteInput.val('');

                return false;
            });

            self.container.delegate('.create-page', 'click', function (e) {
                var a = $(this);
                var data = a.data();
                var autocompleteInput = a.parents('tr').find('.page-autocomplete');
                var title = autocompleteInput.val();
                autocompleteInput.val('');

                var dialog = new AJS.Dialog({
                    width: 500,
                    height: 260,
                    id: "create-module-dialog",
                    closeOnOutsideClick: false
                });

                dialog.addHeader(AJS.I18n.getText("quiz.course.create-module.dialog.title"));
                dialog.addPanel('panel', QuizCourse.createModule({title: title}), 'create-module-dialog-panel');
                dialog.addButton(AJS.I18n.getText("quiz.create-wizard.dialog.create"), function() {
                    var input = $('#course-new-module-title');
                    var errorContainer = $('#course-new-module-error');
                    var title = input.val().trim();

                    if (!title) {
                        errorContainer.text(AJS.I18n.getText("quiz.course.create-module.dialog.error.empty")).show();
                        return;
                    } else if (!Confluence.Blueprint.canCreatePage(data.spaceKey, title)) {
                        errorContainer.text(AJS.I18n.getText("quiz.course.create-module.dialog.error")).show();
                        return;
                    } else {
                        errorContainer.hide();
                    }

                    $.ajax({
                        type: 'POST',
                        cache: false,
                        async: false,
                        url: Confluence.getContextPath() + "/rest/api/content",
                        contentType: 'application/json',
                        data: JSON.stringify({
                            type: 'page',
                            title: title,
                            space: {key: data.spaceKey},
                            ancestors: [{"id": data.courseId}],
                            body: {
                                storage: {
                                    value: '<p><br/></p>',
                                    representation: 'storage'
                                }
                            }
                        }),
                        dataType: 'json',
                        success: function(page) {
                            var section = a.parents('.course-section');
                            var container = section.find('.modules');
                            self.addModule(data.sectionId, page.id, title, 0, container, section, false, function () {
                                dialog.remove();
                                window.open(Confluence.getContextPath() + '/pages/editpage.action?pageId=' + page.id,'_blank');
                            });
                        },
                        error: function (xhr) {
                            if (xhr.responseText.indexOf('com.atlassian.confluence.api.service.exceptions.ReadOnlyException') !== -1) {
                                QuizErrorHandler.showError(xhr);
                            } else {
                                errorContainer.text(AJS.I18n.getText("quiz.course.create-module.dialog.error")).show();
                            }
                        }
                    });
                }, "create-module-dialog-save");
                dialog.addLink(AJS.I18n.getText("quiz.create-wizard.dialog.cancel"), function (dialog) {
                    dialog.remove();
                }, '#');

                dialog.show();

                e.preventDefault();
            });
        },

        showReportTabContent() {
            var self = this;

            if (REPORT_IS_LOADED) {
                self.reportLoadingSpinner.css('display','block');
                self.showReport();
            }
        },

        hideReportTabContent() {
            var self = this;
            self.table.addClass('not-visible-report');
        }
    };

    function sendCourseSetting(dataForServer, setting) {
        $.ajax({
            type: 'POST',
            cache: false,
            url: Confluence.getContextPath() + "/rest/quiz/1.0/course/settings/" + setting,
            data: dataForServer,
            dataType: 'json',
            error: QuizErrorHandler.showError
        });
    }

    AJS.toInit(function() {
        Course.init();
    });
})(AJS.$);

