import {IFRAME_DELETED_ENROLL_MESSAGE, IFRAME_OVERDUE_ENROLL_MESSAGE} from "../../utils/constants/iframe-messages";
import {MODULE_ANSWERED} from "../../utils/constants/submit-module-results";
import {ENROLL_NOT_FOUND, ENROLL_OVERDUE} from "../../utils/constants/enroll-states";
import { iframeResizer } from "iframe-resizer";

(function($) {

    var SideBar = {
        sidebarWidthKey: "quiz-course-sbw",
        sidebarCollapsedKey: "quiz-course-sbc",

        init: function () {
            this.sidebar = AJS.sidebar('.aui-sidebar');
            this.sidebar.isViewportNarrow = function () { return false; };
            this.sidebarWrap = $('#course-content .aui-sidebar-wrapper');
            this.main = $('#course-main, body.linchpin-theme-plugin-resources #footer');
            
            $('#main-header').remove();

            if (localStorage.getItem(this.sidebarWidthKey)) {
                this.sidebarWrap.width(localStorage.getItem(this.sidebarWidthKey) + 'px');
            }

            if (localStorage.getItem(this.sidebarCollapsedKey)) {
                this.sidebar.collapse();
            } else {
                this.sidebar.expand();
            }

            this.resizeMain();
            this.bindDrag();
            this.bindCollapse();
        },

        bindDrag: function () {
            var self = this;

            $('.ia-splitter-handle').draggable({axis: 'x', drag: function(e) {
                var x = e.originalEvent.clientX - self.sidebarWrap.offset().left;
                if (x > 60) {
                    self.sidebarWrap.width(x + 'px');
                    localStorage.setItem(self.sidebarWidthKey, x);
                    if (self.sidebar.isCollapsed()) {
                        self.sidebar.expand();
                    }
                    self.resizeMain();
                } else if (x > 0) {
                    self.sidebar.collapse();
                    self.sidebarWrap.width('');
                    localStorage.removeItem(self.sidebarWidthKey);
                    self.resizeMain();
                }
            }});
        },

        bindCollapse: function () {
            var self = this;
            self.sidebar.on('collapse-end', function () {
                self.sidebarWrap.width('');
                localStorage.removeItem(self.sidebarWidthKey);
                localStorage.setItem(self.sidebarCollapsedKey, true);
                self.resizeMain();
            });

            self.sidebar.on('expand-end', function () {
                localStorage.removeItem(self.sidebarCollapsedKey);
                self.resizeMain();
            });
        },

        resizeMain: function () {
            this.main.css('padding-left', (this.sidebarWrap.width() + 20) + 'px');
        },

        getCurrentModuleId: function () {
            var selectedModuleNav = this.sidebarWrap.find(".aui-nav-selected > a");
            if (selectedModuleNav.length > 0) {
                return selectedModuleNav.data("id");
            } else {
                return this.getFirstAvailableModuleId();
            }
        },

        getFirstAvailableModuleId: function() {
            return this.sidebarWrap.find('.aui-nav.module-tree .aui-nav-item.module:eq(0)').data("id");
        }
    };

    var Course = {

        init: function (currentModuleId) {
            this.isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
            if (this.isMobile) {
                $('#main').css('padding', '10px');
                $('#header .aui-header-secondary').hide();
                $('#header .aui-header-primary .aui-buttons').hide();
            }

            this.qsd = $.getUrlVar("qsd");
            this.submitBtn = $('#submit-module');
            this.spinner = $('#course-spinner');
            this.frame = $('#course-module-frame');
            this.sidebar = $('.cq-sidebar');
            this.moduleHeader = $('#module-title');
            this.btnsContainer = $('#course-controls');
            //TODO разобраться, почему мы храним данные об активных модулях прямо в HTML
            var modules = this.sidebar.attr('data-cq-active-modules');
            this.activeModules = modules ? modules.split(',') : [];
            this.moduleId = currentModuleId;

            this.bindFrame();
            this.bindSubmit();
            this.bindNavigation();
            this.bindIframeEvents();

            this.getModule();
            this.reactivateNext();
        },

        reactivateNext: function() {
            var self = this;

            if ($('#course-content .aui-nav-item.current-module').length === 0) {
                var firstDisabled = $('#course-content .aui-nav-item.module.disabled').first();

                if (firstDisabled) {
                    self.getModule(firstDisabled.data("id"));
                }
            }
        },

        getModule: function(moduleId) {
            if (moduleId === this.moduleId) return;

            this.submitBtn.prop('disabled', true).hide();
            this.btnsContainer.hide();
            this.btnsContainer.removeClass('completed');
            $(document).scrollTop(0);

            if (moduleId) {
                this.moduleId = moduleId;
            }

            this.moduleHeader.text($('#course-content .aui-nav-item[data-id="' + this.moduleId + '"]').eq(0).text());
            this.frame.css('height', 0).hide();
            this.spinner.show();
            this.frame.attr('src', Confluence.getContextPath() + '/quiz/course/viewModule.action?qsd=' + this.qsd + '&questionId=' + (moduleId || this.moduleId));
        },

        bindFrame: function () {
            var self = this;

            self.frame.on('load', function() {
                self.setFrameContent();
                self.spinner.hide();
                self.frame.show();

                new QuizFileViewerHelper(self.frame, self.frameContent);

                if (self.frameContent.hasClass('course-module')) {
                    if (self.frameContent.attr('data-passed') !== 'true') {
                        self.submitBtn.prop('disabled', false);
                        self.submitBtn.show();
                    } else {
                        self.btnsContainer.addClass('completed');
                    }
                    self.btnsContainer.show();
                } else {
                    self.frameContent = self.frame.contents().find('#quiz-test-container');

                    if (self.frameContent.find('#quiz-test-progress-container').length === 0) {
                        if (self.frameContent.data('error')) {
                            self.frame.height(self.frameContent.height());
                        } else if (self.frameContent.data('correct') >= self.frameContent.data('target')) {
                            if (self.sidebar.find('.aui-nav-selected .aui-nav-item > .module-in-progress').length) {
                                self.submitBtn.prop('disabled', false).show();
                                self.btnsContainer.show();
                            } else if (self.sidebar.find('.aui-nav-selected .aui-nav-item > .aui-iconfont-approve').length) {
                                self.btnsContainer.addClass('completed');
                                self.btnsContainer.show();
                            }
                        } else if (self.frameContent.data('review') > 0) {
                            self.showReviewed();
                        } else if (self.frameContent.find('#restart-quiz').length === 0) {
                            if (isQuizAction(self.frame)) {
                                CourseCommon.showFailed();

                                $('#course-content a.aui-nav-item.module[data-id="' + self.moduleId + '"]')
                                    .find('.module-in-progress')
                                    .removeClass('module-in-progress')
                                    .addClass('aui-iconfont-devtools-task-disabled');
                            } else {
                                self.frame.contents().find("#header").hide();
                                self.frame.height(self.frame.contents().height());
                            }
                        }
                    }

                    self.frame[0].contentWindow.onbeforeunload = function () {
                        self.spinner.show();
                        $(document).scrollTop(0);
                        self.frame.css('height', 0).hide();
                    };
                }

                // костыль для QUIZ-714
                self.frameContent.find('li[data-inline-task-id]').each(function() {
                    var li = $(this);
                    li.prepend('<input type="checkbox"/>');
                    li.removeAttr('data-inline-task-id');
                });

                self.moduleId = self.frameContent.data('id');
                self.setActiveNav();

                if (self.isMobile) {
                    self.frameContent.find('iframe').css('height', '');
                }
            });
        },
        
        bindSubmit: function () {
            var self = this;

            self.submitBtn.click(function () {
                self.submitBtn.prop('disabled', true);
                $.ajax({
                    type: 'POST',
                    cache: false,
                    url: Confluence.getContextPath() + "/rest/quiz/1.0/course/submit",
                    data: {
                        param: $.getUrlVar("qsd"),
                        pid: self.frameContent.data("id"),
                        attempt: self.submitBtn.data("attempt")
                    },
                    dataType: 'json',
                    success: function(data) {
                        if (data.value === ENROLL_NOT_FOUND) {
                            self.showEnrollNotFoundMessage();
                        } else if (data.value === ENROLL_OVERDUE) {
                            self.showEnrollOverdueMessage();
                        } else {
                            self.submitBtn.prop('disabled', true).hide();
                            self.btnsContainer.addClass('completed');
                            if (self.processAnswered(data.value, self.frameContent.data("id"))) {
                                self.getModule();
                            }
                        }
                    },
                    error: QuizErrorHandler.showError
                });
            });
        },

        bindIframeEvents: function() {
            let self = this;
            window.addEventListener("message", (e) => {
                if (e.data === IFRAME_DELETED_ENROLL_MESSAGE) {
                    self.showEnrollNotFoundMessage();
                }
                if (e.data === IFRAME_OVERDUE_ENROLL_MESSAGE) {
                    self.showEnrollOverdueMessage();
                }
            });

            iframeResizer({log: false, checkOrigin: false}, '#course-module-frame');
        },

        showEnrollNotFoundMessage: function() {
            $('#course-content').empty();
            $('.deleted-enroll-message').show();
        },

        showEnrollOverdueMessage: function() {
            $('#course-content').empty();
            $('.enroll-overdue-message').show();
        },

        bindNavigation: function () {
            var self = this;

            $('#course-content a.aui-nav-item.module').click(function () {
                var a = $(this);
                $('#course-content .aui-nav-selected').removeClass('aui-nav-selected');
                a.parent().addClass('aui-nav-selected');
                self.getModule(a.data('id'));
                return false;
            });

            $('#course-content a.aui-nav-item.section').click(function () {
                $(this).parent().find('a.aui-nav-subtree-toggle').click();
                return false;
            });

            $('body').delegate('#course-content .aui-sidebar-submenu-dialog a.aui-nav-item', 'click', function () {
                self.getModule($(this).data('id'));
            });
        },

        setFrameContent: function () {
            this.frameContent = this.frame.contents().find('#main-content');
        },

        setActiveNav: function () {
            var self = this;

            $('#course-content .aui-nav-selected').removeClass('aui-nav-selected');
            $('#course-content .aui-nav-child-selected').removeClass('aui-nav-child-selected');

            var a = $('#course-content .aui-nav-item[data-id="' + self.moduleId + '"]');
            a.removeClass('disabled');
            self.changeIconToProgress(a);
            a.parent().addClass('aui-nav-selected');
            var sectionNavItem = $('#course-content a.aui-nav-item.section[data-id="' + a.data('section') + '"]');
            self.changeIconToProgress(sectionNavItem);
            sectionNavItem.parents('li').addClass('aui-nav-child-selected');
            self.moduleHeader.text(a.eq(0).text());
        },

        processAnswered: function (result, id) {
            var self = this;

            if (result === MODULE_ANSWERED) {
                var item = $('#course-content a.aui-nav-item.module[data-id="' + id + '"]');
                item.find('.module-in-progress').toggleClass('module-in-progress aui-iconfont-approve');

                if ($('#course-content a.aui-nav-item.module[data-section="' + item.data('section') + '"]').find('.aui-iconfont-locked, .aui-iconfont-unlocked').length === 0) {
                    $('#course-content a.aui-nav-item.section[data-id="' + item.data('section') + '"] .module-in-progress').toggleClass('module-in-progress aui-iconfont-approve');
                }

                var index = self.activeModules.indexOf('' + id);
                if (self.activeModules.length === 1) {
                    self.showFinish();
                    return false;
                }

                var nextIndex = self.activeModules.length > index + 1 ? index + 1 : 0;
                var nextItem = $('#course-content a.aui-nav-item.module[data-id="' + self.activeModules[nextIndex] + '"]');
                self.moduleId = nextItem.data('id');
                self.activeModules.splice(index, 1);

                return true;
            } else {
                return false;
            }
        },

        showFinish: function () {
            var homeLink = $('.aui-page-header-main a');

            // todo заменить на AJS.dialog2
            var dialog = new AJS.Dialog({
                width: 500,
                height: 300,
                id: "course-finished-dialog",
                closeOnOutsideClick: false
            });

            dialog.addHeader(AJS.I18n.getText("quiz.course.finish.header"));
            dialog.addPanel('panel', QuizCourseAction.finishDialog({name: homeLink.text().trim(), url: homeLink.attr('href')}), 'course-finished-dialog-panel');
            dialog.addLink(AJS.I18n.getText("close.name"), function (dialog) {
                dialog.remove();
            }, '#');

            dialog.show();
        },
        
        showReviewed: function () {
            var dialog = new AJS.Dialog({
                width: 500,
                height: 330,
                id: "quiz-reviewed-dialog",
                closeOnOutsideClick: false
            });

            dialog.addHeader(AJS.I18n.getText("quiz.course.quiz.review.header"));
            dialog.addPanel('panel', QuizCourseAction.reviewDialog(), 'quiz-reviewed-dialog-panel');
            dialog.addLink(AJS.I18n.getText("close.name"), function (dialog) {
                dialog.remove();
            }, '#');

            dialog.show();
        },

        changeIconToProgress: function (container) {
            container
                .find('.aui-iconfont-locked, .aui-iconfont-unlocked')
                .removeClass('aui-iconfont-locked aui-iconfont-unlocked')
                .addClass('module-in-progress');
        }
    };

    AJS.toInit(function () {
        SideBar.init();
        $('#ia-splitter').css('margin-top', $('#header').height() + 'px');
        $(window).scroll();
        //todo тут невероятная взаимосвязь присутствует между сайдбаром и курсом, их бы конечно разъединить через третью (высшую) сущность, но пока так
        Course.init(SideBar.getCurrentModuleId());
    });

    function isQuizAction(frame) {
        return frame[0].contentWindow.location.pathname.includes("viewQuiz.action");
    }
})(AJS.$);
