(function ($, AnalyticsClient) {
    if (!AJS || !AJS.toInit) {
        return;
    }

    window.HandyStatus = {

        init: function () {
            var self = this;
            self.bindInlineComments();
            self.initStatuses();
            self.bindDOMChange();
            self.bindTableSorterHeader();
            self.bindTableFilterTransformer();
            self.bindStatusChange();
        },

        initStatuses: function () {
            var self = HandyStatus;
            var statuses = $('.handy-status-view:not(.ready)');
            if (statuses.length === 0) return;
            statuses.addClass('ready');

            self.setMarkup();
            self.bindStatusClick(statuses);
            if (!window.ConfluenceMobile) {
                self.bindHistory(statuses);
            }
            self.bindDDInit();
        },

        setMarkup: function () {
            $('body').append($('[id^="asdd-"]'));
        },

        bindStatusClick: function (statuses) {
            return statuses.filter('.handy-status-editable').click(function (e) {
                e.stopPropagation();
            });
        },
        
        bindHistory: function (statuses) {
            statuses.filter('.history').each(function () {
                const status = $(this);
                if (status.parents('.ic-content').length) {
                    status.parents('p').css('white-space', 'normal');
                }

                HandyStatus.initStatusHistory(status);
                HandyStatus.onChangeStatusMacroId(status, HandyStatus.changeHistoryInlineDialogId);
            });
        },

        bindStatusChange: function () {
            var pageVersionMeta = $('meta[name="page-version"]');
            var xhr, timer;
            $(document).delegate('.handy-status-option', 'click', function () {
                const option = this;
                const macroId = option.dataset.macroid;
                const pageId = option.dataset.pageid;
                const statusSpans = document.querySelectorAll('[id="' + macroId + '"][data-pageid="' + pageId + '"]');
                const dd = document.getElementById('asdd-' + macroId + '-' + pageId);
                const oldStyle = statusSpans[0].style.cssText;
                const oldStatus = statusSpans[0].dataset.status;
                const oldText = statusSpans[0].textContent;
                const changeStatus = function () {
                    const optionStyle = option.children[0].style.cssText;
                    const optionStatus = option.dataset.status;
                    statusSpans.forEach(function (statusSpan) {
                        statusSpan.textContent = option.textContent.toUpperCase();
                        statusSpan.style.cssText = optionStyle;
                        statusSpan.dataset.status = optionStatus;
                        statusSpan.dispatchEvent(new Event('DOMCharacterDataModified'));
                    });
                };
                if (statusSpans[0].dataset.status === option.dataset.status) return;

                if ($('body.content-preview').length) {
                    changeStatus();
                    return;
                }

                var statusesCount = option
                    .closest('aui-section')
                    .querySelectorAll('aui-item-link')
                    .length;

                const url = Confluence.getContextPath() + '/rest/handy-macros/2.0/status-macro/' + macroId + '/value';

                $.ajax({
                    contentType: "application/x-www-form-urlencoded; charset=utf-8",
                    type: 'PUT',
                    cache: false,
                    url: url,
                    data: {
                        contentId: option.dataset.pageid,
                        statusId: option.dataset.status,
                        name: option.textContent.trim()
                    },
                    dataType: 'json',
                    success: function (data) {
                        AnalyticsClient.sendStatusChangedEvent(statusesCount);
                        if ((data.pageVersion - pageVersionMeta.attr('content')) === 1) {
                            pageVersionMeta.attr('content', data.pageVersion);
                        }

                        if (dd) {
                            const auiMessage = dd.querySelector('.aui-message');
                            if (auiMessage) auiMessage.remove();

                            dd.querySelectorAll('.handy-status-option')
                                .forEach(function (element) {element.dataset.macroid = data.id});
                        }

                        statusSpans.forEach(function (statusSpan) {statusSpan.id = data.id});

                        document.dispatchEvent(new CustomEvent('handy-status-changed', {detail: {oldId: macroId, currentId: data.id}}));
                    },
                    error: function (xhr) {
                        statusSpans.forEach(function (statusSpan) {
                            statusSpan.textContent = oldText;
                            statusSpan.style.cssText = oldStyle;
                            statusSpan.dataset.status = oldStatus;
                        });
                        statusSpans[0].click();
                        var isReadOnly = xhr.responseText.indexOf('com.atlassian.confluence.api.service.exceptions.ReadOnlyException') !== -1;
                        AJS.messages.error(dd, {
                            title: AJS.I18n.getText("handy.macros.change.error.title"),
                            body: isReadOnly ? AJS.I18n.getText("handy.macros.read-only-mode") : AJS.I18n.getText("handy.macros.change.error.body")
                        });
                    }
                });

                changeStatus();
            })
                .delegate(".handy-status-search", "input", function () {
                    var input = $(this);
                    var container = input.closest("div");
                    var spinner = container.find(".handy-status-search-spinner").css("display", "inline-block");

                    clearTimeout(timer);
                    xhr && xhr.abort();
                    timer = setTimeout(function () {
                        getOptions(input, container, spinner, input.val().trim());
                        container.find(".handy-status-search-show-all").show();
                    }, 300)
                })
                .delegate(".handy-status-search-show-all", "click", function () {
                    var a = $(this).hide();
                    var container = a.closest("div");
                    var spinner = container.find(".handy-status-show-all-spinner").show();
                    container.find(".handy-status-search").val("");
                    clearTimeout(timer);
                    xhr && xhr.abort();
                    getOptions(a, container, spinner, "", 1000000);
                    return false;
                });

            function getOptions(el, container, spinner, query, limit) {

                const macroId = el.data('id');
                const url = Confluence.getContextPath() + '/rest/handy-macros/2.0/status-macro/' + macroId + '/options';

                xhr = $.ajax({
                    contentType: "application/x-www-form-urlencoded; charset=utf-8",
                    type: 'GET',
                    cache: false,
                    url: url,
                    data: {
                        macroId: el.data('id'),
                        contentId: el.data('pageid'),
                        query: query,
                        limit: limit,
                    },
                    dataType: 'json',
                    success: function (data) {
                        container.children(":gt(0)").remove();
                        container.append(HandyStatusTemplate.options({options: data[0].items}));
                        spinner.hide();
                        container.find(".handy-status-search-shown").text(data[0].items.length);
                    },
                    complete: function() {
                        spinner.hide();
                    }
                });
            }
        },

        bindInlineComments: function () {
            var self = this;

            if (_ && Backbone && Backbone.Events.listenTo) {
                $(document).bind('ajaxComplete', function (e, xhr, data) {
                    if (data.url.indexOf('/rest/inlinecomments/1.0/comments') >= 0) {
                        self.initStatuses();
                    }
                });
                $('#rte-button-publish, #rte-button-cancel').on('click', function () {
                    self.initStatuses();
                });

                var listener = {};
                _.extend(listener, Backbone.Events);
                listener.listenTo(Backbone, "ic:view", function () {
                    setTimeout(self.initStatuses, 50);
                });
            }
        },

        bindDOMChange: function () {
            var self = this;
            var timer = null;
            var onInsert = function (target) {
                if (target.hasClass('handy-status-view') || target.find('.handy-status-view').length) {
                    if (timer) {
                        clearTimeout(timer);
                        timer = null;
                    }
                    timer = setTimeout(self.initStatuses, 100);
                }
            };

            var wikiContent = $('.wiki-content');
            if (wikiContent.length && wikiContent.find('.microBlogWrapper').length === 0) {
                var observer = new MutationObserver(function (events) {
                    events.some(function (event) {
                        if (event.addedNodes.length) {
                            return onInsert($(event.addedNodes));
                        }
                    });
                });
                observer.observe(wikiContent[0], {
                    childList: true,
                    subtree: true
                });
            }
        },

        bindTableSorterHeader: function () {
            var self = this;

            $('table').bind('tablesorter-initialized', function () {
                var statuses = $(this).find('thead .tablesorter-headerRow .handy-status-view');
                self.bindStatusClick(statuses).bind('mousedown mouseup', function (e) {
                    e.preventDefault();
                    return false;
                });
                self.bindHistory(statuses);
            });
        },

        bindTableFilterTransformer: function () {
            const self = this;

            $('.tj-source').bind('tj-executed', function () {
                const tableJoiner = $(this.parentElement);
                const macroRenderWrapper = tableJoiner.find('.wrapper.conf-macro-render');
                if (macroRenderWrapper.length > 0) {
                    const statuses = macroRenderWrapper.find('.handy-status-view.history.ready');
                    self.bindHistory(statuses);
                    self.bindStatusClick(statuses);
                }
            });
        },

        bindDDInit: function () {
            $('aui-dropdown-menu[id^="asdd-"]').on("aui-dropdown2-show", function () {
                var dropdown = this;
                var $dropdown = $(this);
                var sections = $dropdown.find('aui-section');

                if (sections.length) {
                    var section = sections[0];
                    var resolved = section.hasAttribute("resolved");

                    if (resolved) {
                        HandyStatus.initDD($dropdown);
                    } else {
                        var observer = new MutationObserver(function (e) {
                            if (section.hasAttribute("resolved")) {
                                observer.disconnect();
                                HandyStatus.initDD($dropdown);
                            }
                        });
                        observer.observe(section, {attributes: true});
                    }
                } else {
                    var observer = new MutationObserver(function (e) {
                        if (HandyStatus.getNotRenderedDropdownOptions($dropdown).length) {
                            observer.disconnect();
                            HandyStatus.initDD($dropdown);
                        }
                    });
                    observer.observe(dropdown, {childList: true, subtree: true});
                }
            });
        },

        onChangeStatusMacroId: function (status, onChange) {
            const observer = new MutationObserver(function (mutations) {
                for (const mutation of mutations) {
                    if (mutation.attributeName === 'id') {
                        const newMacroId = mutation.target.getAttribute('id');
                        const oldMacroId = mutation.oldValue;
                        if (newMacroId !== oldMacroId) {
                            onChange(status, oldMacroId);
                        }
                    }
                }
            });

            const statusSpan = status[0];
            observer.observe(statusSpan, {
                attributes: true,
                attributeOldValue: true,
                attributeFilter: ['id'],
            });
        },

        clearTimeoutForStatusHistory: function (timer, inlineDialog) {
            inlineDialog.respondsTo = '';
            clearTimeout(timer);
        },

        changeHistoryInlineDialogId: function (status, oldMacroId) {
            const newMacroId = status.attr('id');
            const pageId = status.data('pageid');
            const oldAsdd = `asdd-${oldMacroId}-${pageId}`;
            const newAsdd = `asdd-${newMacroId}-${pageId}`;
            const dialog = document.getElementById(`${oldAsdd}-history`);

            const newDialogId = `${newAsdd}-history`;
            dialog.id = newDialogId;
            status.attr("aria-controls" , newDialogId);
        },

        getStatusHistoryInlineDialog: function (status) {
            const macroId = status.attr('id');
            const pageId = status.data('pageid');
            const asdd = `asdd-${macroId}-${pageId}`;
            return document.getElementById(`${asdd}-history`);
        },

        loadHistoryAndOpenDialog: function (status) {
            const that = this;

            const macroId = status.attr('id');
            const inlineDialog = that.getStatusHistoryInlineDialog(status);
            const contentDiv = inlineDialog.querySelector('.aui-inline-dialog-contents');

            const url = Confluence.getContextPath() +
                '/rest/handy-macros/2.0/status-macro/' + macroId + '/history';

            $.ajax({
                contentType: 'application/x-www-form-urlencoded; charset=utf-8',
                type: 'GET',
                cache: false,
                url: url,
                data: {
                    contentId: status.data('pageid'),
                },
                dataType: 'json',
                success: function (data) {
                    const popupContent = HandyStatusTemplate.history({revisions: data});
                    $(contentDiv).html(popupContent);
                    inlineDialog.respondsTo = 'hover';
                    status.prop('history-ready', true);
                    status.trigger('mouseover');
                },
            });
        },

        getLoadHistoryAndOpenDialogTimer: function (status) {
            return setTimeout(function () {
                HandyStatus.loadHistoryAndOpenDialog(status);
            }, 600);
        },

        initStatusHistory: function (status) {
            const that = this;

            let timer;
            status.one('mouseover', function () {
                const inlineDialog = that.getStatusHistoryInlineDialog(status);

                timer = that.getLoadHistoryAndOpenDialogTimer(status);

                status.on('click', function () {
                    inlineDialog.open = false;
                    that.clearTimeoutForStatusHistory(timer, inlineDialog);
                });

                inlineDialog.addEventListener('aui-hide', function () {
                    status.prop('history-ready', false);
                    that.clearTimeoutForStatusHistory(timer, inlineDialog);
                });

            }).on('mouseleave', function () {
                const inlineDialog = that.getStatusHistoryInlineDialog(status);
                if (!inlineDialog.open) {
                    status.prop('history-ready', false);
                    that.clearTimeoutForStatusHistory(timer, inlineDialog);
                }
            });

            status.on('mouseover', function () {
                if (status.prop('history-ready')) {
                    const inlineDialog = that.getStatusHistoryInlineDialog(status);
                    inlineDialog.open = true;
                    return;
                }
                if (timer) clearTimeout(timer);

                const isDropdownNotExpanded = !status.attr('aria-expanded') ||
                    status.attr('aria-expanded') === 'false' ;

                const isHistoryDialogNotExpanded = status.parent().attr('aria-expanded') === 'false';

                if (isDropdownNotExpanded && isHistoryDialogNotExpanded) {
                    timer = that.getLoadHistoryAndOpenDialogTimer(status);
                }
            });
        },

        getNotRenderedDropdownOptions: function (dropdown) {
            return dropdown.find("a:not(.handy-status-option):not(.binded)");
        },

        initDD: function (dds) {
            return this.getNotRenderedDropdownOptions(dds).each(function () {
                var $dropdownOption = $(this);
                var searchOption = $dropdownOption.find('.handy-status-search');
                var auiMessageOption = $dropdownOption.find('.aui-message');
                var handyStatusOption = $dropdownOption.find('.handy-status-option');

                if (!handyStatusOption.length && !searchOption.length && !auiMessageOption.length) {
                    $dropdownOption.html($dropdownOption.text());
                } else {
                    $dropdownOption.addClass("binded");
                }
            }).length;
        }
    };

    if (window.ConfluenceMobile) {
        ConfluenceMobile.contentEventAggregator.on("displayed", function () {
            if (!Confluence.getContextPath) {
                Confluence.getContextPath = AJS.contextPath;
            }
            HandyStatus.init();
            WRM.require("com.atlassian.auiplugin:aui-dropdown2");
        });
    } else {
        AJS.toInit(function () {
            HandyStatus.init();
        });
    }

})(AJS.$ || $, require('stiltsoft/handy-macros/analytics/client'));
