window.define(
    'stiltsoft/HandyStatusPageEditorPropertyPanel',
    [
        'jquery',
        'ajs',
        'confluence/root',
        'confluence-macro-browser/macro-parameter-serializer',
        'stiltsoft/HandyStatusRestAPI',
        'stiltsoft/handy-macros/text-utils',
    ],
    ($, ajs, confluence, macroParameterSerializer, HandyStatusRestAPI, TextUtils) => {

        const PERSONAL_SETS_LINK = `${confluence.getContextPath()}/users/handy-macros.action`;

        const getMacroParameters = ($macro) =>
            macroParameterSerializer.deserialize($macro.attr('data-macro-parameters'));

        const fixPropertyPanelPosition = ($panel) => {
            const diff = $(window).width() - parseInt($panel.css('left')) - $panel.width() - 25;
            if (diff >= 0) return;
            $panel.css('left', parseInt($panel.css('left')) + diff);
            const arrow = $panel.find('.aui-tip-parent');
            arrow.css('left', parseInt(arrow.css('left')) - diff);
        };

        return class HandyStatusPageEditorPropertyPanel {
            $macro;
            $panel;
            $setSelect;
            $statusSelect;
            $copyStatusMacroIdBtn;
            $copySetBtn;

            storageMacroParams;
            statusMacroId;
            macroData;
            sets;
            statuses;

            constructor(props) {
                this.$macro = $(props.anchor);
                this.$panel = $(props.panel);
            }

            init = () => {
                this.showPanelSpinner();
                this.fetchMacroParameters()
                    .then(this.renderAndInitControls)
                    .catch(error => {
                        console.error('Cannot get macro. ' + error.status);
                        this.hideControls();
                    })
                    .finally(this.hidePanelSpinner);
            };

            fetchMacroParameters = () => {
                const that = this;
                this.storageMacroParams = getMacroParameters(this.$macro);
                this.statusMacroId = this.storageMacroParams.id;
                const contentId = ajs.params.contentId ? ajs.params.contentId : undefined;

                return new Promise((resolve, reject) => {
                    HandyStatusRestAPI
                        .fetchStatusMacro(this.statusMacroId, contentId, false)
                        .then(macroData => {
                            that.macroData = macroData;

                            Promise.all([
                                HandyStatusRestAPI.fetchStatusSets(macroData.set, ajs.params.spaceKey),
                                HandyStatusRestAPI.fetchSetStatuses(macroData.set),
                            ]).then(responses => {
                                that.sets = responses[0];
                                that.statuses = responses[1];
                                resolve();
                            });
                        }).catch(reject);
                });
            };

            renderAndInitControls = () => {
                if (!this.macroData.licensed) return;
                this.renderControls();
                this.bindControls();
            };

            renderControls = () => {
                const renderedControls = Stiltsoft.HandyMacros.HandyStatus.Editor.propertyPanelControls();
                const panelButtons = this.$panel.find('.panel-buttons');
                panelButtons.append(renderedControls);

                const renderedInlineDialogs = Stiltsoft.HandyMacros.HandyStatus.Editor.propertyPanelDialogs({
                    personalSetsLink: PERSONAL_SETS_LINK
                });
                this.$panel.append(renderedInlineDialogs);

                panelButtons.wrap(Stiltsoft.HandyMacros.HandyStatus.Editor.formWrapper());

                this.findControls();
                this.renderSelectSetOptions();
                this.renderSelectStatusOptions();
                fixPropertyPanelPosition(this.$panel);
            };

            renderSelectSetOptions = () => {
                const globalSets = [];
                const spaceSets = [];
                const personalSets = [];

                this.sets.forEach(set => {
                    if (set.owner.startsWith("global")) {
                        globalSets.push(set);
                    } else if (set.owner.startsWith("space")) {
                        spaceSets.push(set);
                    } else if (set.owner.startsWith("user")) {
                        personalSets.push(set);
                    } else {
                        console.error('Unknown owner type', JSON.stringify(set));
                    }
                });

                const renderedOptions = Stiltsoft.HandyMacros.HandyStatus.Editor.selectSetOptions({
                    globalSets: globalSets,
                    spaceSets: spaceSets,
                    personalSets: personalSets,
                    selectedSetId: this.macroData.set
                });

                this.$setSelect.append(renderedOptions);
            };

            renderSelectStatusOptions = () => {
                this.$statusSelect.empty();
                if (!this.statuses.length) return;

                const renderedOptions = Stiltsoft.HandyMacros.HandyStatus.Editor.selectStatusOptions({
                    statuses: this.statuses,
                    selectedStatusId: this.macroData.status,
                });

                this.$statusSelect.append(renderedOptions);

                this.$statusSelect.trigger('change');
            };

            findControls = () => {
                this.$setSelect = this.$panel.find('#handy-status-set-select');
                this.$statusSelect = this.$panel.find('#handy-status-select');
                this.$copyStatusMacroIdBtn = this.$panel.find('#copy-status-btn');
                this.$copySetBtn = this.$panel.find('#copy-set-btn');
            };

            bindControls = () => {
                this.$setSelect.change(this.onChangeSet);
                this.$copyStatusMacroIdBtn.click(this.onClickCopyStatusMacroId);
                this.$copySetBtn.click(this.onClickCopyStatusSet);
                this.$statusSelect.bind('change', this.onChangeStatus);
            };

            onChangeSet = () => {
                const option = this.$setSelect.find('option:selected');
                const statusSetId = this.$setSelect.val();
                const isOwn = option.text().trim().indexOf(ajs.params.currentUserFullname) === 0;
                const isCopied = this.$setSelect.find(`option[data-parent="${statusSetId}"]`).length > 0;
                if (isOwn || isCopied) {
                    this.$copySetBtn.prop('disabled', true);
                    this.$copySetBtn.attr('title', isOwn ? ajs.I18n.getText('handy.status.clone.set.own') :
                        ajs.I18n.getText('handy.status.clone.set.cloned'));
                } else {
                    this.$copySetBtn.prop('disabled', false);
                    this.$copySetBtn.attr('title', ajs.I18n.getText('handy.status.clone.set'));
                }

                if (this.macroData.set !== this.$setSelect.val()) {
                    this.macroData.status = undefined;
                }
                this.macroData.set = this.$setSelect.val();
                this.macroData.width = this.$setSelect.find('option:selected').data('width');

                const that = this;
                HandyStatusRestAPI.fetchSetStatuses(this.macroData.set)
                    .then(statuses => {
                        that.statuses = statuses;
                        that.renderSelectStatusOptions();
                    });
            };

            onChangeStatus = () => {
                this.$statusSelect.disable();

                this.macroData.status = this.$statusSelect.val();
                this.storageMacroParams['Status'] = this.$statusSelect.find('option:selected').text();

                this.storeMacroParameters().finally(() => this.$statusSelect.enable());
            };

            onClickCopyStatusMacroId = (event) => {
                event.preventDefault();
                this.replaceCopyStatusMacroIdIconWithCheckMarkIcon();
                TextUtils.copyTextToClipboard(this.macroData.id);
            };

            onClickCopyStatusSet = () => {
                this.$copySetBtn.prop('disabled', true);
                void HandyStatusRestAPI.cloneStatusSet(this.macroData.set, false);
                return false;
            };

            replaceCopyStatusMacroIdIconWithCheckMarkIcon = () => {
                this.$copyStatusMacroIdBtn.find('#copy-status')
                    .replaceWith(Stiltsoft.HandyMacros.HandyStatus.Editor.copiedIcon());
            };

            storeMacroParameters = () => {
                const requestParams = {
                    contentId: ajs.params.contentId ? ajs.params.contentId : undefined,
                    statusId: this.macroData.status,
                    statusSetId: this.macroData.set,
                    outline: this.macroData.outline,
                    labelled: this.macroData.labelled,
                    prefix: this.macroData.prefix,
                };

                return HandyStatusRestAPI.draftStatusMacro(this.macroData.id, requestParams, false)
                    .then((newStatusMacroId) => {
                        this.updateStorageAttributesForMacro(newStatusMacroId);
                        fixPropertyPanelPosition(this.$panel);
                    });
            };

            updateStorageAttributesForMacro = (newStatusMacroId) => {
                this.storageMacroParams.id = '' + newStatusMacroId;

                const {set, outline, status, width} = this.macroData;
                const src = `/plugins/servlet/status-handy/placeholder?set=${set}&outline=${outline}&status=${status}&width=${width}`;

                this.$macro.attr('src', confluence.getContextPath() + src);
                this.$macro.attr('width', width);
                this.$macro.attr('data-mce-src', ajs.params.baseUrl + src);
                this.$macro.attr('data-macro-parameters', confluence.MacroParameterSerializer.serialize(this.storageMacroParams));
            };

            showPanelSpinner = () => {
                this.$panel.find('.panel-buttons').append(Stiltsoft.HandyMacros.HandyStatus.Editor.panelSpinner());
                ajs.$(this.$panel.find('.status-handy-panel-spinner')).spin();
            };

            hidePanelSpinner = () => {
                ajs.$(this.$panel.find('.status-handy-panel-spinner')).spinStop();
                this.$panel.find('.status-handy-panel-spinner-wrapper').remove();
            };

            hideControls = () => {
                this.$setSelect.hide();
                this.$statusSelect.hide();
                this.$copyStatusMacroIdBtn.hide();
                this.$copySetBtn.hide();
            };
        };
    });