;(function($) {
	var COMMON = ARCHIVING_CONTENT_STATUS_COMMON;
	var gridCssSelector = "#content-status-browser-grid";
	var grid;
	var dataView;
	var dataModel;

	function getRestUrlForPage(pageId) {
		return AJS.contextPath() + COMMON.REST_BASE_URL + pageId + "/children";
	}

	function getRestUrlForSpace(spaceKey) {
		return AJS.contextPath() + COMMON.REST_BASE_URL + "space/" + spaceKey + "/children";
	}

	function convertResponseToModel(response, parentId, indent) {
		return $.map(response, function(value, index) {
			return {
				id: value.id,
				title: value.title,
				url: value.url,
				status: value.status.summary,
				statusCode: value.status.code,
				statusDescription: value.status.description,
				statusSubDescription: value.status.subDescription,
				lastUpdated: value.lastUpdated ? value.lastUpdated.shortInfoAsHtml : "",
				lastViewed: value.lastViewed ? value.lastViewed.shortInfoAsHtml : "",
				hasChildren: !!value.hasChildren,
				isHomePage: !!value.isHomePage,
				parentId: parentId || "root",
				indent: indent || 0,
				_collapsed: true // all nodes are collapsed by default
			};
		});
	}

	var CellFormatters = {
		pageTitle: function(row, cell, value, columnDef, dataContext) {
			if (typeof value == "undefined") {
				return "";
			}
			value = value.toString().replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
			var spacer = "<span style='display:inline-block;height:1px;width:" + (15 * dataContext["indent"]) + "px'></span>";
			var idx = dataView.getIdxById(dataContext.id);
			var pageClass = "aui-icon aui-icon-small icon-page" + (dataContext['isHomePage'] ? " icon-home-page content-type-home" : " content-type-page");
			var link = '<span style="vertical-align: sub;" class="' + pageClass + '"></span><a href="' + AJS.contextPath() + dataContext['url'] + '">' + value + '</a>';
			var expanderClasses = (dataModel[idx] && dataModel[idx].hasChildren ?
				(" expand-control-icon " + (dataContext._collapsed ? "opened" : "expanded"))
				: "");
			return spacer + " <div class='click-zone'><span class='toggle" + expanderClasses + "'></span></div>&nbsp;" + link;
		},
		status: function(row, cell, value, columnDef, dataContext) {
			if (!value) {
				return "";
			}
			var cssClass = "";
			var code = dataContext["statusCode"];
			if (code == "up-to-date") {
				cssClass = "up-to-date";
			} else if (code == "not-viewed") {
				cssClass = "not-viewed";
			} else if (code.indexOf("expired") === 0) {
				cssClass = "expired";
			} else if (code.indexOf("to-be-archived") === 0) {
				cssClass = "to-be-archived";
			} else if (code == "excluded") {
				cssClass = "excluded";
			} else if (code.indexOf("unknown") === 0) {
				cssClass = "unknown";
			}
			var description = dataContext["statusDescription"];
			var subDescription = $.trim(dataContext["statusSubDescription"]);
			var escapedTooltip = AJS.escapeHtml(description) +
				(subDescription != "" ? (" - " + AJS.escapeHtml(subDescription)) : "");
			return '<span class="aui-lozenge ' + cssClass + '" title="' + escapedTooltip + '">' + value.toUpperCase() + '</span>';
		},
		date: function(row, cell, value, columnDef, dataContext) {
			// simply return the value, that will tell slickgrid that the value is already sanitized
			return value;
		}
	};

	function updateGridWithData(newItems) {
		dataView.beginUpdate();
		dataView.setItems(newItems);
		dataView.endUpdate();
		grid.invalidateAllRows();
		grid.updateRowCount();
		grid.render();
	}

	function handleCollapseExpand(item) {
		function findAllParentIds(startItem) {
			var parentIds = ["root"];
			var pid = startItem.parentId;
			while (pid != "root" && !!pid) {
				parentIds.push(pid);
				var parentIndex = dataView.getIdxById(pid);
				var parent = parentIndex ? dataModel[parentIndex] : null;
				pid = parent ? parent.parentId : null;
			}
			return parentIds;
		}

		if (!item._collapsed) {
			item._collapsed = true;
			var startIndex = dataView.getIdxById(item.id);
			var index = startIndex + 1;
			// step over rows until the next row on the same or higher hierarchy level like the row being closed
			var parentIds = findAllParentIds(item);
			while (index < dataModel.length && parentIds.indexOf(dataModel[index].parentId) === -1) {
				index++;
			}
			var itemsToRemove = index - startIndex - 1;
			var newItems = dataView.getItems();
			newItems.splice(startIndex + 1, itemsToRemove);
			updateGridWithData(newItems);
		} else {
			item._collapsed = false;
			$.ajax({
				url: getRestUrlForPage(item.id),
				type: "GET",
				dataType: "json",
				contentType: "application/json"
			}).done(function(response) {
				var data = convertResponseToModel(response, item.id, item.indent + 1);
				var newItems = dataView.getItems();
				var startIndex = dataView.getIdxById(item.id) + 1;
				for (var i = 0; i < data.length; i++) {
					newItems.splice(startIndex + i, 0, data[i]);
				}
				updateGridWithData(newItems);
			}).fail(genericAjaxErrorHandler);
		}
		dataView.updateItem(item.id, item);
	}

	function maintainDimensions() {
		$(window).resize(function() {
			grid.resizeCanvas();
		});
		$(".expand-collapse-trigger").click(function() {
			grid.resizeCanvas();
			return true;
		});
	}

	function applyDocumentationThemeFix() {
		var paddingAdjustment = 40;
		var containingCell = $("#content.space-operations").find("td.in-page-menu-content,#space-tools-body").first();
		function fixContainingCellWidth() {
			var gridElement = $(gridCssSelector);
			gridElement.hide();
			var remainingHorizontalSpace = $(document).innerWidth() - containingCell.offset().left - paddingAdjustment;
			gridElement.width(remainingHorizontalSpace).show();
			grid.resizeCanvas();
		}
		var $body = $("body");
		var isDocumentationTheme = $body.hasClass("theme-documentation");
		var isBitVoodooTheme = $body.hasClass("GetunikCore400");
		var isScrollVersionsTheme = $body.hasClass("sv-theme");
		if (isDocumentationTheme || isBitVoodooTheme || isScrollVersionsTheme) {
			$(window).resize(fixContainingCellWidth);
			fixContainingCellWidth();
		}
	}

	var errorFlag;

	var genericAjaxErrorHandler = function(jqXHR) {
		var text = '<div style="word-wrap: break-word;">' + COMMON.buildAndLogErrorMessage(jqXHR) + '</div>';
		require(['aui/flag'], function(flag) {
			if (errorFlag) {
				errorFlag.close();
			}
			errorFlag = flag({
				type: 'error',
				title: 'Error',
				body: text
			});
			if (typeof grid != "undefined") {
				grid.resizeCanvas();
			}
		});
	};

	$(function() {
		$.ajax({
			url: getRestUrlForSpace(AJS.params.spaceKey),
			type: "GET",
			dataType: "json",
			contentType: "application/json"
		}).done(function(response) {
			var columns = [
				{id: "page", name: "Page", field: "title", width: 300, cssClass: "page-name", formatter: CellFormatters.pageTitle},
				{id: "status", name: "Status", field: "status", width: 150, formatter: CellFormatters.status},
				{id: "last-updated", name: "Last Updated", field: "lastUpdated", width: 200, formatter: CellFormatters.date},
				{id: "last-viewed", name: "Last Viewed", field: "lastViewed", width: 200, formatter: CellFormatters.date}
			];
			var options = {
				enableCellNavigation: true,
				enableColumnReorder: false,
				forceFitColumns: true,
				autoHeight: true
			};

			dataModel = convertResponseToModel(response);

			// initialize the model
			dataView = new Slick.Data.DataView();
			dataView.beginUpdate();
			dataView.setItems(dataModel);
			dataView.endUpdate();

			grid = new Slick.Grid(gridCssSelector, dataView, columns, options);

			grid.onClick.subscribe(function(e, args) {
				var $target = $(e.target);
				if ($target.hasClass("click-zone") || $target.hasClass("toggle")) {
					var item = dataView.getItem(args.row);
					if (item) {
						handleCollapseExpand(item);
					}
					e.stopImmediatePropagation();
				}
			});

			maintainDimensions();
			applyDocumentationThemeFix();

		}).fail(genericAjaxErrorHandler);
	});
})(jQuery);
