/*
 * Decompiled with CFR 0.152.
 */
package com.midori.confluence.plugin.archiving.service.eventindex;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.confluence.pages.Page;
import com.atlassian.sal.api.transaction.TransactionTemplate;
import com.google.common.base.Functions;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.midori.confluence.plugin.archiving.model.base.ContentEvent;
import com.midori.confluence.plugin.archiving.model.util.ModelUtils;
import com.midori.confluence.plugin.archiving.service.support.DateTimeProvider;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import net.java.ao.DBParam;
import net.java.ao.Query;
import net.java.ao.RawEntity;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

public abstract class AbstractContentEventTracker<T extends ContentEvent> {
    public static final int CONTENT_EVENT_JOURNAL_MAXSIZE = 5000;
    private ActiveObjects ao;
    private TransactionTemplate transactionTemplate;
    private Cache<Long, String> clusterWideJournal;

    protected AbstractContentEventTracker(ActiveObjects ao, CacheManager cacheManager, TransactionTemplate transactionTemplate) {
        this.ao = ao;
        this.transactionTemplate = transactionTemplate;
        this.clusterWideJournal = this.createClusterWideJournal(cacheManager);
    }

    private Cache<Long, String> createClusterWideJournal(CacheManager cacheManager) {
        String cacheName = String.format("Archiving Plugin: Content %s Event Journal", StringUtils.capitalize((String)this.getEventName()));
        return cacheManager.getCache(cacheName, null, new CacheSettingsBuilder().expireAfterAccess(30L, TimeUnit.MINUTES).expireAfterWrite(30L, TimeUnit.MINUTES).replicateViaCopy().maxEntries(5000).build());
    }

    private Map<Long, T> captureAndFlushJournal(boolean flush) {
        HashMap<Long, T> map = new HashMap<Long, T>();
        for (Long key : this.clusterWideJournal.getKeys()) {
            String valueSerialized = (String)this.clusterWideJournal.get((Object)key);
            if (valueSerialized == null) continue;
            try {
                T value = this.fromSerializedFormat(valueSerialized);
                map.put(key, value);
                if (!flush) continue;
                this.clusterWideJournal.remove((Object)key);
            }
            catch (Throwable t2) {
                this.getLogger().error(String.format("Failed to deserialize value <%s>", valueSerialized), t2);
            }
        }
        return map;
    }

    private Map<Long, T> captureJournal() {
        return this.captureAndFlushJournal(false);
    }

    private Map<Long, T> captureAndFlushJournal() {
        return this.captureAndFlushJournal(true);
    }

    protected void enjournal(long contentEntityId, T contentEvent) {
        if (this.clusterWideJournal.getKeys().size() < 5000) {
            this.clusterWideJournal.put((Object)contentEntityId, (Object)((ContentEvent)contentEvent).serialize());
            this.getLogger().debug(String.format("%s on page #%d by <%s> enjournalled (%d %ss in journal)", StringUtils.capitalize((String)this.getEventName()), contentEntityId, ((ContentEvent)contentEvent).getOriginatorName(), this.clusterWideJournal.getKeys().size(), this.getEventName()));
        } else {
            this.getLogger().warn(String.format("%s not enjournalled, as journal size reached the limit %d", StringUtils.capitalize((String)this.getEventName()), 5000));
        }
    }

    public synchronized void persistContentEventJournal() {
        Map<Long, T> journal = this.captureAndFlushJournal();
        this.getLogger().debug(String.format("Persisting %d %ss in journal", journal.size(), this.getEventName()));
        if (!journal.isEmpty()) {
            this.ao.executeInTransaction(() -> {
                this.persistContentEvents(journal);
                this.getLogger().debug(String.format("%d %ss remained in journal", this.clusterWideJournal.getKeys().size(), this.getEventName()));
                return null;
            });
        }
        this.getLogger().debug(String.format("%s journal persisted", StringUtils.capitalize((String)this.getEventName())));
    }

    public void persistContentEvent(long pageId, T event) {
        this.persistContentEvents(ImmutableMap.of(pageId, event));
    }

    public void persistContentEventIfNotExisting(long pageId, T event) {
        long start = DateTimeProvider.currentTimeMillis();
        boolean actuallyPersisted = (Boolean)this.transactionTemplate.execute(() -> {
            boolean isExisting;
            boolean bl = isExisting = this.findLastEventsByPageIdsLow(Collections.singletonList(pageId), Optional.empty()).length > 0;
            if (!isExisting) {
                this.createRecordsForPages(ImmutableMap.of(pageId, event));
            }
            return !isExisting;
        });
        this.getLogger().trace(String.format("New %s record %s in %d ms", this.getEventName(), actuallyPersisted ? "inserted" : "not re-inserted", DateTimeProvider.currentTimeMillis() - start));
    }

    private void persistContentEvents(Map<Long, T> journal) {
        this.transactionTemplate.execute(() -> {
            this.deleteRecordsForPages(journal.keySet());
            this.createRecordsForPages(journal);
            return null;
        });
    }

    private void deleteRecordsForPages(Collection<Long> pageIds) {
        long start = DateTimeProvider.currentTimeMillis();
        this.processPagesInChunks(pageIds, (pageIdArray, placeholderCommaList) -> this.ao.deleteWithSQL(this.getActiveObjectsClass(), String.format("PAGE_ID IN (%s)", placeholderCommaList), pageIdArray));
        this.getLogger().trace(String.format("%d old %s records deleted in %d ms", pageIds.size(), this.getEventName(), DateTimeProvider.currentTimeMillis() - start));
    }

    private void createRecordsForPages(Map<Long, T> events) {
        long start = DateTimeProvider.currentTimeMillis();
        for (Long pageId : events.keySet()) {
            ContentEvent event = (ContentEvent)events.get(pageId);
            List<DBParam> params = this.createDbParams(pageId, event);
            this.ao.create(this.getActiveObjectsClass(), params.toArray(new DBParam[0]));
        }
        this.getLogger().trace(String.format("%d new %s records inserted in %d ms", events.keySet().size(), this.getEventName(), DateTimeProvider.currentTimeMillis() - start));
    }

    protected List<DBParam> createDbParams(Long pageId, T event) {
        return new LinkedList<DBParam>(Arrays.asList(new DBParam("PAGE_ID", (Object)pageId), new DBParam(this.getOriginatorNameDbField(), (Object)((ContentEvent)event).getOriginatorName()), new DBParam(this.getDateDbField(), (Object)((ContentEvent)event).getDate())));
    }

    public Map<Page, T> findLastEventsByPages(Set<Page> pages) {
        return this.findLastEventsByPagesMergedWithJournal(pages, Optional.empty());
    }

    public Map<Page, T> findLastEventsByPages(Set<Page> pages, long minimumAge) {
        return this.findLastEventsByPagesMergedWithJournal(pages, Optional.of(minimumAge));
    }

    private Map<Page, T> findLastEventsByPagesMergedWithJournal(Set<Page> pages, Optional<Long> minimumAge) {
        Optional<Date> before = Optional.of(new Date(DateTimeProvider.currentTimeMillis() - minimumAge.orElse(0L)));
        Map<Page, T> persistedEventsByPages = this.findLastEventsByPages(pages, before);
        Map<Page, T> journalledEventsByPages = this.getJournalledEventsByPages(pages, before);
        persistedEventsByPages.putAll(journalledEventsByPages);
        return persistedEventsByPages;
    }

    private Map<Page, T> findLastEventsByPages(Set<Page> pages, Optional<Date> before) {
        if (pages.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<Long, Page> pageIdentityMap = ModelUtils.getPageIdentityMap(pages);
        RawEntity<T>[] events = this.findLastEventsByPageIdsLow(pageIdentityMap.keySet(), before);
        HashMap<Page, T> lastEventsByPages = new HashMap<Page, T>(events.length);
        for (Page page : pages) {
            lastEventsByPages.put(page, null);
        }
        for (RawEntity<T> event : events) {
            T eventModel = this.fromAo(event);
            long pageId = ((ContentEvent)eventModel).getPageId();
            Page page = pageIdentityMap.get(pageId);
            lastEventsByPages.put(page, eventModel);
        }
        return lastEventsByPages;
    }

    private Map<Page, T> getJournalledEventsByPages(Set<Page> pages, Optional<Date> before) {
        if (pages.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<Long, Page> pageIdentityMap = ModelUtils.getPageIdentityMap(pages);
        HashMap<Page, ContentEvent> journalledEventsByPages = new HashMap<Page, ContentEvent>();
        for (Map.Entry<Long, T> journalEntry : this.captureJournal().entrySet()) {
            Long pageId = journalEntry.getKey();
            Page page = pageIdentityMap.get(pageId);
            ContentEvent event = (ContentEvent)journalEntry.getValue();
            if (page == null || before.isPresent() && event.getDate().after(before.get())) continue;
            journalledEventsByPages.put(page, event);
        }
        return journalledEventsByPages;
    }

    private RawEntity<T>[] findLastEventsByPageIdsLow(Collection<Long> pageIds, Optional<Date> before) {
        long start = DateTimeProvider.currentTimeMillis();
        ArrayList events = new ArrayList();
        this.processPagesInChunks(pageIds, (pageIdArray, placeholderCommaList) -> {
            Query query = before.isPresent() ? Query.select((String)"ID").where("PAGE_ID IN (" + placeholderCommaList + ") AND " + this.getDateDbField() + " < ?", ArrayUtils.add((Object[])pageIdArray, before.get())) : Query.select((String)"ID").where("PAGE_ID IN (" + placeholderCommaList + ")", pageIdArray);
            events.addAll(Arrays.asList(this.ao.find(this.getActiveObjectsClass(), query)));
        });
        this.getLogger().debug(String.format("%d %s records %s found for %d pages in %d ms", events.size(), this.getEventName(), before.map(date -> String.format("(before %s)", Instant.ofEpochMilli(date.getTime()).toString())).orElse(""), pageIds.size(), DateTimeProvider.currentTimeMillis() - start));
        return events.toArray(new RawEntity[0]);
    }

    private void processPagesInChunks(Collection<Long> pageIds, ChunkProcessor callback) {
        int PAGE_ID_CHUNK_SIZE = 900;
        for (List<Long> pageIdsChunk : Iterables.partition(pageIds, 900)) {
            Object[] pageIdArray = Iterables.toArray(pageIdsChunk, Object.class);
            String placeholderCommaList = Joiner.on(", ").join(Iterables.transform(pageIdsChunk, Functions.constant("?")));
            callback.apply(pageIdArray, placeholderCommaList);
        }
    }

    protected abstract String getEventName();

    protected abstract Class<? extends RawEntity> getActiveObjectsClass();

    protected abstract T fromAo(RawEntity var1);

    protected abstract T fromSerializedFormat(String var1);

    protected abstract String getOriginatorNameDbField();

    protected abstract String getDateDbField();

    protected abstract Logger getLogger();

    @FunctionalInterface
    private static interface ChunkProcessor {
        public void apply(Object[] var1, String var2);
    }
}

