AJS.toInit(function($) {
	var COMMON = ARCHIVING_CONTENT_STATUS_COMMON;

	/**
	 * UI helper methods.
	 */
	var UiHelper = (function() {
		var iconClassPrefix = "aui-icon aui-icon-small aui-button-subtle status-loaded aui-iconfont-";

		function getIconClassForResource(response) {
			var upToDateIconClasses = iconClassPrefix + "approve status-up-to-date";
			var notViewedIconClasses = iconClassPrefix + "view status-not-viewed";
			var expiredIconClasses = iconClassPrefix + "error status-expired";
			var toBeArchivedIconClasses = iconClassPrefix + "devtools-task-disabled status-to-be-archived";
			var excludedIconClasses = iconClassPrefix + "devtools-task-cancelled status-excluded";
			var unknownIconClasses = iconClassPrefix + "help status-unknown";
			var iconClasses;
			var status = response["status"]["code"];
			if (status === "up-to-date") {
				iconClasses = upToDateIconClasses;
			} else if (status.indexOf("expired") === 0) {
				iconClasses = expiredIconClasses;
			} else if (status.indexOf("to-be-archived") === 0) {
				iconClasses = toBeArchivedIconClasses;
			} else if (status === "excluded") {
				iconClasses = excludedIconClasses;
			} else if (status === "not-viewed") {
				iconClasses = notViewedIconClasses;
			} else if (status.indexOf("unknown") === 0) {
				iconClasses = unknownIconClasses;
			}
			return iconClasses;
		}

		function getErrorIconClasses() {
			return iconClassPrefix + "warning status-error";
		}

		function themeSpecificAdjustments($webItem) {
			(function brikitThemePress() {
				// Brikit Theme Press for Confluence (https://marketplace.atlassian.com/plugins/com.brikit.themepress/versions)
				var brikitTargetContainer = $(".brikit-page-metadata .page-metadata-modification-info");
				if (brikitTargetContainer.length > 0) {
					$webItem.appendTo(brikitTargetContainer);
				}
			})();
		}

		return {
			"getIconClassForResource": getIconClassForResource,
			"getErrorIconClasses": getErrorIconClasses,
			"themeSpecificAdjustments": themeSpecificAdjustments
		};
	})();

	/**
	 * Page context helper functions.
	 */
	var PageContextHelper = (function() {
		function getAutoQuickAction() {
			var validQuickActionCodes = [
				"discuss",
				"update",
				"confirm",
				"own",
				"expire",
				"unexpire",
				"archive",
				"unarchive",
				"exclude",
				"include"
			];
			var autoQuickActionParameter = MIDORI.getUrlParameter("quick-action");
			return (validQuickActionCodes.indexOf(autoQuickActionParameter) !== -1) ? autoQuickActionParameter : null;
		}

		function getPageId() {
			return AJS.params.pageId;
		}

		function forwardWithNoPageViewTrackingParamQueryString() {
			return window.location.search.match(/[\?&]ap_novt=/) ? "?ap_novt=1" : "";
		}

		/**
		 * Reload page labels using an AJAX call.
		 */
		function reloadLabels() {
			var spaceKey = AJS.Meta.get('space-key');
			var entityType = AJS.Meta.get('content-type');
			var entityId = AJS.Meta.get('page-id');
			$.ajax({
				url: AJS.Labels.routes.list(entityType, entityId),
				cache: false
			}).done(function(response) {
				var labels = response.labels;
				var labelSection = $('.labels-section-content').filter(function() {
					return this.getAttribute('entityid') === entityId && this.getAttribute('entitytype') === entityType;
				});
				if (labelSection.length) {
					AJS.Meta.set('nu-labels', labels.length);
					var labelsFragment = Confluence.Templates.Labels.dialogLabels({labels: labels, spaceKey: spaceKey});
					labelSection.find('li.aui-label').remove();
					labelSection.find('li.labels-edit-container').before(labelsFragment);
					var zeroLabelsShown = labelSection.find('.aui-label').length === 0;
					if (zeroLabelsShown) {
						var noLabelsMessageShown = labelSection.find('.no-labels-message').length !== 0;
						if (!noLabelsMessageShown) {
							labelSection.find('.labels-edit-container').before(Confluence.Templates.Labels.nolabels());
						}
					} else {
						labelSection.find('.no-labels-message').remove();
					}
				}
			});
		}

		return {
			"getAutoQuickAction": getAutoQuickAction,
			"getPageId": getPageId,
			"forwardWithNoPageViewTrackingParamQueryString": forwardWithNoPageViewTrackingParamQueryString,
			"reloadLabels": reloadLabels
		};
	})();

	/**
	 * Content Status Indicator component.
	 */
	var ContentStatusIndicator = (function() {
		var dialogId = "content-status-indicator-dialog";
		var dataLoaded = false; // load inline dialog content only once
		var autoQuickAction = PageContextHelper.getAutoQuickAction();
		var autoQuickActionShown = false;

		function init(forceReload) {
			var $webItem = $("#content-status-indicator");

			if ($webItem.length > 0) {
				var pageId = PageContextHelper.getPageId();

				if (forceReload === true) {
					dataLoaded = false;
					// remove old dialogs
					$("#inline-dialog-content-status-indicator-dialog, #archiving-quick-actions-dialog").remove();
					// replace indicator icon with a new one without events
					var clonedWebItem = $webItem.clone(false);
					clonedWebItem.insertAfter($webItem);
					$webItem.remove();
					$webItem = clonedWebItem;
				}

				// set indicator tooltip to "loading"
				$webItem.prop({
					"title": AJS.I18n.getText("com.midori.confluence.plugin.archiving.web.contentStatusIndicator.loading.label")
				}).tooltip({
					"html": true
				});

				// get page status data for indicator
				$.ajax({
					url: AJS.contextPath() + COMMON.REST_BASE_URL + pageId + (PageContextHelper.forwardWithNoPageViewTrackingParamQueryString()),
					type: "GET",
					dataType: "json",
					contentType: "application/json"
				}).done(function(response) {
					// create indicator inline dialog
					reCreate($webItem, pageId, response, forceReload);

					// set indicator CSS classes and final tooltip
					$webItem.attr({
						"class": UiHelper.getIconClassForResource(response)
					}).prop({
						"title": AJS.I18n.getText("com.midori.confluence.plugin.archiving.web.contentStatusIndicator.label")
					});
				}).fail(function(jqXHR) {
					// set indicator to error state
					$webItem.attr({
						"class": UiHelper.getErrorIconClasses()
					}).click(function() {
						return false;
					});
					// let the AJAX call's always() method run first, then set error message as indicator tooltip
					setTimeout(function() {
						$webItem.prop({
							"title": COMMON.buildAndLogErrorMessage(jqXHR)
						});
					}, 100);
				}).always(function() {
					$webItem.closest("li").addClass("has-button");
					UiHelper.themeSpecificAdjustments($webItem);
				});
			}
		}

		function reCreate($webItem, pageId, indicatorResponse) {
			var dialog = AJS.InlineDialog($webItem, dialogId,
				function(content, trigger, showPopup) {
					if (!dataLoaded) {
						// render and show without quick actions
						render(pageId, content, indicatorResponse, null); // quick actions are not available yet
						showPopup();
						// get data for quick actions
						$.ajax({
							url: AJS.contextPath() + "/archiving/quick/init.action?" + $.param({
								"pageId": pageId,
								"statusCode": indicatorResponse.status.code,
								"viaAncestor": indicatorResponse.status.viaAncestor,
								"ancestorId": indicatorResponse.status.ancestorId
							}),
							type: "GET",
							dataType: "json",
							contentType: "application/json"
						}).done(function (initResponse) {
							dataLoaded = true;
							render(pageId, content, indicatorResponse, initResponse);
							QuickActions.initializeQuickActionLinks(pageId, initResponse.actions);
							// fire quick action automatically if requested in URL parameter
							if (autoQuickAction && !autoQuickActionShown) {
								QuickActions.onQuickActionClick(pageId, autoQuickAction, initResponse.actions);
								autoQuickActionShown = true;
							}
						}).fail(function (jqXHR) {
							alert(COMMON.buildAndLogErrorMessage(jqXHR));
						});
					} else {
						showPopup();
						QuickActions.enableQuickActionLinkTooltips();
					}
					return false;
				}, {
					width: $(".content-status-indicator-dialog .indicator-layout").outerWidth()
				}
			);

			// workaround to dismiss the inline dialog when clicking on web-item again until https://ecosystem.atlassian.net/browse/AUI-1175 is done
			$webItem.click(function() {
				if ($("#inline-dialog-" + dialogId).is(":visible")) {
					dialog.hide();
				}
			});

			// open indicator automatically if requested in URL parameter
			if (autoQuickAction && !autoQuickActionShown) {
				$webItem.click();
			}
		}

		function render(pageId, content, indicatorResponse, initResponse) {
			// set up template data for the indicator...
			var templateInput = indicatorResponse;
			templateInput.pageId = pageId;
			templateInput.contextPath = AJS.contextPath();
			templateInput.defaultDate = moment().format('YYYY-MM-DD');
			// ...plus for the quick actions
			templateInput.actions = initResponse ? initResponse.actions : null;
			templateInput.primaryActions = initResponse ? initResponse.primaryActions : null;
			templateInput.secondaryActions = initResponse ? initResponse.secondaryActions : null;
			templateInput.isReadOnlyAccessModeEnabled = initResponse ? initResponse.isReadOnlyAccessModeEnabled : null;

			// render and initialize indicator
			content.attr("id", "content-status-indicator-dialog-container").html(Confluence.Templates.Plugins.Content.Status.Indicator.info(templateInput));
		}

		return {
			"init": init
		};
	})();

	/**
	 * Quick actions.
	 */
	var QuickActions = (function() {
		/**
		 * Enable quick action link tooltips.
		 *
		 * This method and its "disable" counterpart are a workaround for a bug that would otherwise make a tooltip
		 * re-appear in the top-left corner after closing the quick actions dialog).
		 */
		function enableQuickActionLinkTooltips() {
			findQuickActionLinks().tooltip("enable");
		}

		function disableQuickActionLinkTooltips() {
			findQuickActionLinks().tooltip("disable");
		}

		function initializeQuickActionLinks(pageId, actions) {
			findQuickActionLinks().click(function(e) {
				e.preventDefault();
				onQuickActionClick(pageId, $(this).data("action"), actions);
			}).tooltip();
		}

		function onQuickActionClick(pageId, action, actions) {
			switch(action) {
				case "update":
					window.location = AJS.contextPath() + "/pages/editpage.action?pageId=" + pageId;
					break;

				case "jump":
					window.location = AJS.contextPath() + "/pages/viewpage.action?pageId=" + actions[action].context.ancestorId;
					break;

				case "more":
					$("#moreLink").hide();
					$(".secondaryActions").show();
					break;

				default:
					disableQuickActionLinkTooltips();
					// hide content status indicator
					$("#inline-dialog-content-status-indicator-dialog").hide();
					// initialize dialog
					var dialogContainer = $("#archiving-quick-actions-dialog");
					if (dialogContainer.prop("INITIALIZED") !== true) {
						initializeQuickActionControls(pageId);
						dialogContainer.prop("INITIALIZED", true);
					}
					// reset dialog state to selected action
					var actionParameterSelector = ".action-" + action;
					dialogContainer.find(".archiving-action-input-parameters").hide();
					dialogContainer.find(".archiving-action-input-parameters" + actionParameterSelector).show();
					// fix header layout
					var header = dialogContainer.find("h2.archiving-action-input-parameters" + actionParameterSelector);
					header.insertBefore(header.siblings("h2").first());
					dialogContainer.find("input[name=quick-action]").val(action);
					clearValidationErrors();
					// show quick action dialog
					AJS.dialog2("#archiving-quick-actions-dialog").show();
					break;
			}
		}

		function findQuickActionLinks() {
			return $(".archiving-quick-actions-container a");
		}

		function initializeQuickActionControls(pageId) {
			function onSubmit(e) {
				e.preventDefault();
				switchToBusy();

				var dialog = $("#archiving-quick-actions-dialog");
				var action = dialog.find("input[name=quick-action]").val();

				$.ajax({
					url: AJS.contextPath() + "/archiving/quick/" + action + ".action?pageId=" + pageId,
					type: "POST",
					data: collectDialogData(dialog, action)
				}).done(function(response) {
					clearValidationErrors();
					if (response.validationErrors && response.validationErrors.length > 0) {
						showValidationErrors(response.validationErrors);
					} else {
						AJS.dialog2("#archiving-quick-actions-dialog").hide();
						ContentStatusIndicator.init(true);
						MIDORI.showQuickActionStatusFlag(response);
						PageContextHelper.reloadLabels();
					}
				}).fail(function(jqXHR) {
					alert(COMMON.buildAndLogErrorMessage(jqXHR)); // FIXME may return the Confluence error page in jqXHR, therefore the error message will be generic
				}).always(function() {
					switchToIdle();
				});
			}

			$("#archiving-quick-actions-close-button").click(function(e) {
				e.preventDefault();
				AJS.dialog2("#archiving-quick-actions-dialog").hide();
				switchToIdle();
			});

			$(".archiving-quick-actions-submit-button").click(onSubmit);

			$('#archiving-quick-actions-dialog-form input').keypress(function(e) {
				if (e.which === 13) {
					onSubmit(e);
				}
			});

			MIDORI.initMultiUserPicker($("#archiving-discuss-recipients"), true);
			MIDORI.initMultiUserPicker($("#archiving-page-owners-single"), true);
			MIDORI.initMultiUserPicker($("#archiving-page-owners-recursive"), true);
			// initialize dates selectors - DO NOT merge the jQuery selectors into one (and to one .datePicker() call) because it would break positioning!
			$("#archiving-expire-date").datePicker({'overrideBrowserDefault': true});
			$("#archiving-archive-date").datePicker({'overrideBrowserDefault': true});
			// initialize radio buttons
			initArchivingDateRadioButtons();
			$("input[type=radio][name=archiving-archive-date-options]").change(initArchivingDateRadioButtons);
		}

		function collectDialogData(dialog, action) {
			var data = {};

			if (action === "confirm") {
				data.recursive = dialog.find("#archiving-confirm-recursive").is(":checked");
				data.message = dialog.find("#archiving-confirm-message").val();
			} else if (action === "own") {
				data.ownersSingle = dialog.find("#archiving-page-owners-single").val();
				data.ownersRecursive = dialog.find("#archiving-page-owners-recursive").val();
			} else if (action === "expire") {
				data.date = dialog.find("#archiving-expire-date").val().replace(/-/g, "/").slice(2);
				data.recursive = dialog.find("#archiving-expire-recursive").is(":checked");
			} else if (action === "archive") {
				data.archiveByDate = dialog.find("#archiving-archive-date-option-givendate").is(":checked");
				data.date = dialog.find("#archiving-archive-date").val().replace(/-/g, "/").slice(2);
				data.message = dialog.find("#archiving-archive-reason").val();
				data.recursive = dialog.find("#archiving-archive-recursive").is(":checked");
			} else if (action === "exclude") {
				data.recursive = dialog.find("#archiving-exclude-recursive").is(":checked");
			} else if (action === "discuss") {
				data.recipients = dialog.find("#archiving-discuss-recipients").val();
				data.message = dialog.find("#archiving-discuss-message").val();
			}

			return data;
		}

		function switchToBusy() {
			$(".archiving-quick-actions-submit-button").prop("disabled", true);
			$("#archiving-quick-actions-loader").css("display", "inline-block");
		}

		function switchToIdle() {
			$("#archiving-quick-actions-loader").hide();
			$(".archiving-quick-actions-submit-button").prop("disabled", false);
		}

		function clearValidationErrors() {
			$("#archiving-quick-actions-dialog-form-validation-errors").hide().empty();
		}

		function showValidationErrors(validationErrors) {
			var errorContainer = $("#archiving-quick-actions-dialog-form-validation-errors");
			validationErrors.forEach(function(element) {
				errorContainer.append("<p>" + element + "</p>");
			});
			errorContainer.show();
		}

		function initArchivingDateRadioButtons() {
			var archiveDateControl = $("#archiving-archive-date");
			if ($("#archiving-archive-date-option-nextexecution").is(':checked')) {
				archiveDateControl.val('').prop("disabled", true);
			} else if ($("#archiving-archive-date-option-givendate").is(':checked')) {
				archiveDateControl.prop("disabled", false);
			}
		}

		return {
			"enableQuickActionLinkTooltips": enableQuickActionLinkTooltips,
			"initializeQuickActionLinks": initializeQuickActionLinks,
			"onQuickActionClick": onQuickActionClick
		}
	})();

	ContentStatusIndicator.init();
});
