var accesslog = (function() {
  var self = {};

  // Prevent clashing with other addons Handlebars versions
  var accesslogHandlebars = Handlebars;
  Handlebars.noConflict();

  var summaryTemplate;
  var pageInfoTemplate;

  var query = '';
  var filter;
  var sortParams = [];
  var page = 1;

  self.init = function(pageParam, queryParam, sortParamsInit, filterParams) {
    filter = filterParams;
    query = queryParam;
    page = pageParam;
    sortParams = sortParamsInit;
    summaryTemplate = compileTemplate("access-record-summary-template");
    pageInfoTemplate = compileTemplate("page-info-template");
    setUpConfig();
    setUpSearch();
    setUpFilter();
    setUpCleanOption();
    update(page);
  }

  self.sortLog = function(sortKey) {
    var prevOrder;
    for (var i = 0; i < sortParams.length; i++) {
      if (sortParams[i].key == sortKey) {
        prevOrder = sortParams[i].order;
        sortParams.splice(i, 1);
      }
    }
    var order = prevOrder == "ASC" ? "DESC" : "ASC";
    updateOrderIcon(sortKey, order);

    sortParams.push({
      key : sortKey,
      order : order
    });
    update(1);
  }

  function updateOrderIcon(sortKey, order) {
    var jquerySortKeyThId = "#" + sortKey;
    var unsortedKeys = new Set([ "TIME", "USER", "IP_ADDRESS", "CONTENT_TYPE", "NEW_CONTENT_DISPLAY_NAME", "ACTION" ]);
    unsortedKeys.delete(jquerySortKeyThId);
    
    for (var it = unsortedKeys.values(), unsortedKey= null; unsortedKey=it.next().value; ) {
      var jqueryUnsortedKeyThId = "#" + unsortedKey;
      AJS.$(jqueryUnsortedKeyThId).attr("class", "tablesorter-header tablesorter-headerUnSorted");
      AJS.$(jqueryUnsortedKeyThId).attr("aria-sort", "none");
      AJS.$(jqueryUnsortedKeyThId).attr("aria-label", unsortedKey + 
          ": No sort applied, activate to apply a "+ 
          (order == "ASC" ? "descending" : "ascending") +" sort")
    }

    var classValue = "tablesorter-header "
        + (order == "ASC" ? "tablesorter-headerAsc" : "tablesorter-headerDesc");
    var ariaSortValue = order == "ASC" ? "ascending" : "descending";
    var ariaLabelValue = sortKey + ": " + (order == "ASC" ? "Ascending" : "Descending")
        + " sort applied, activate to apply a " + (order == "ASC" ? "descending" : "ascending") + " sort";

    AJS.$(jquerySortKeyThId).attr("class", classValue);
    AJS.$(jquerySortKeyThId).attr("aria-sort", ariaSortValue);
    AJS.$(jquerySortKeyThId).attr("aria-label", ariaLabelValue)
  }

  function setUpCleanOption() {
    AJS.$('#cleanLogButton').click(function() {
      AJS.dialog2('#deleteWarningDialog').show();
    });
    AJS.$('#confirmDelete').click(function() {
      cleanAccessLog();
    });
    AJS.$('.delete-warning-dialog-close-button').click(function() {
      AJS.dialog2('#deleteWarningDialog').hide();
    });
  }

  function cleanAccessLog() {
    var url = AJS.contextPath() + "/rest/accesslog/1.0/log";
    AJS.$.ajax({
      url : url,
      type : "DELETE",
      contentType : "application/json",
      success : function(res) {
        AJS.dialog2("#deleteWarningDialog").hide();
        update(1);
      }
    });
  }

  function compileTemplate(templateId) {
    return accesslogHandlebars.compile(AJS.$("#" + templateId).html())
  }

  function update(page) {
    var url = AJS.contextPath() + '/rest/accesslog/1.0/log.json?page=' + page + '&'
        + getUrlParameters();
    AJS.$.ajax({
      url : url,
      dataType : 'json'
    }).done(function(log) {
      updateUrl(log.pageNumber);
      setPaginationInfo(log);
      clearTable();

      const areRecordsPresent = log.accessRecordsApi.length !== 0;
      if (!areRecordsPresent) {
        // In PII Protector we have a comment
        // ## colspan property doesn't work well with sortable aui tables
        // And we have a separate task to test whether can we use colspan or not
        // https://bitbucket.org/enhancera/access-log/issues/273/investigate-can-we-use-colspan-for-instead
        // and we will check this moment in the scope of this task.
        AJS.$('#access-record-table > tbody').append("<tr><td><i>Nothing found</i></td><td></td><td></td><td></td><td></td><td></td></tr>");
      }
      AJS.$('#auditing-export-data').attr('disabled', !areRecordsPresent);

      for (var i = 0; i < log.accessRecordsApi.length; i++) {
        var accessRecordApi = log.accessRecordsApi[i];
        appendRow(accessRecordApi);
      }
    });
  }

  function updateUrl(page) {
    history.pushState(null, null, AJS.contextPath() + '/plugins/servlet/accesslog/view?page='
        + page + '&' + getUrlParameters());
  }

  function clearTable() {
    AJS.$('#access-record-table > tbody > tr').remove();
  }

  function appendRow(accessRecordApi) {
    AJS.$('#access-record-table > tbody').append(renderSummaryTemplate(accessRecordApi));
  }

  function setPaginationInfo(log) {
    if (log.isLastPage) {
      AJS.$('#last-page-info').show();
    } else {
      AJS.$('#last-page-info').hide();
    }
    AJS.$('.page-info').html(pageInfoTemplate(log));
    AJS.$('.pagination-control a[page-num]').click(function(e) {
      e.preventDefault();
      if (AJS.$(this).attr('aria-disabled')) {
        return;
      }
      var page = AJS.$(this).attr('page-num');
      update(page);
    });
  }

  function setUpSearch() {
    AJS.$('#search-form').submit(function(e) {
      e.preventDefault();
      query = AJS.$('#search-box').val().trim();
      update(1);
    });
  }

  function setUpFilter() {

    function getAjaxSelect2Params(paramName) {
      return {
        multiple : true,
        minimumInputLength : 3,
        ajax : {
          delay : 250,
          url : function(param) {
            return AJS.contextPath() + '/rest/accesslog/1.0/filter/' + paramName + '/' + param;
          },
          dataType : 'json',
          type : 'GET',
          results : function(data) {
            return {
              results : data
            };
          }
        }
      }
    }

    AJS.$('#search-box').val(query);

    AJS.$('#date-picker-from').datePicker({
      'overrideBrowserDefault' : true
    });
    AJS.$('#date-picker-from').attr("placeholder", "yyyy-mm-dd");
    AJS.$('#date-picker-to').datePicker({
      'overrideBrowserDefault' : true
    });
    AJS.$('#date-picker-to').attr("placeholder", "yyyy-mm-dd");

    AJS.$('#filter-user-select').auiSelect2(getAjaxSelect2Params('USER'));
    AJS.$('#filter-ip-address-select').auiSelect2(getAjaxSelect2Params('IP_ADDRESS'));
    AJS.$('#filter-content-type').auiSelect2();
    AJS.$('#filter-action').auiSelect2();

    setupInitialParamsForTheFilter();

    AJS.$('#filter-dialog-show-button').click(function() {
      AJS.dialog2('#filter-dialog').show();
      AJS.$('#filter-error-message').hide();
    });
    AJS.$('.filter-dialog-close-button').click(function(e) {
      e.preventDefault();
      AJS.dialog2('#filter-dialog').hide();
      setupInitialParamsForTheFilter();
    });
    AJS.$('#filter-clean-button').click(function(e) {
      e.preventDefault();
      AJS.$('#date-picker-from').val("");
      AJS.$('#date-picker-to').val("");
      AJS.$('#filter-user-select').select2('data', null).trigger("change");
      AJS.$('#filter-ip-address-select').select2('data', null).trigger("change");
      AJS.$('#filter-content-type').select2('data', null).trigger("change");
      AJS.$('#filter-content-id').val("");
      AJS.$('#filter-action').select2('data', null).trigger("change");
      AJS.$('#filter-error-message').hide();
    });

    updateUiFilterText();
    AJS.$('#filter-dialog-submit-button').click(function(e) {
      e.preventDefault();
      submitFilter(1);
    });
  }

  function setupInitialParamsForTheFilter() {

    // yyyy-mm-dd
    function formatDate(date) {
      return date.getFullYear() + "-" + (date.getMonth() + 1) + "-" + date.getDate();
    }

    AJS.$('#filter-user-select').select2('data', filter.users).trigger("change");
    AJS.$('#filter-ip-address-select').select2('data', getSelect2Data(filter.ipAddresses)).trigger(
        "change");
    AJS.$('#filter-content-type').select2('val', filter.contentTypes).trigger('change');
    AJS.$('#filter-content-id').val(filter.contentIds.toString()).trigger(
        "change");
    AJS.$('#filter-action').select2('val', filter.actions).trigger('change');
    if (filter.fromTime) {
      AJS.$('#date-picker-from').val(formatDate(new Date(filter.fromTime)));
    }
    if (filter.toTime) {
      AJS.$('#date-picker-to').val(formatDate(new Date(filter.toTime)));
    }
  }

  function getSelect2Data(data) {
    var result = [];
    if (!data) {
      return result;
    }
    for (var i = 0; i < data.length; i++) {
      result.push({
        id : data[i],
        text : data[i]
      });
    }
    return result;
  }

  function submitFilter(page) {
    var isValid = true;
    var errors = [];

    var fromDateString = AJS.$('#date-picker-from').val();
    if (isEmpty(fromDateString)) {
      filter.fromTime = null;
      AJS.$('#date-picker-from-error').hide();
    } else {
      var date = parseDate(fromDateString);
      if (date) {
        filter.fromTime = date.valueOf();
        AJS.$('#date-picker-from-error').hide();
      } else {
        filter.fromTime = null;
        AJS.$('#date-picker-from-error').show();
        isValid = false;
      }
    }

    var toDateString = AJS.$('#date-picker-to').val();
    if (isEmpty(toDateString)) {
      filter.toTime = null;
      AJS.$('#date-picker-to-error').hide();
    } else {
      var date = parseDate(toDateString);
      if (date) {
        filter.toTime = date.valueOf();
        AJS.$('#date-picker-to-error').hide();
      } else {
        filter.toTime = null;
        AJS.$('#date-picker-to-error').show();
        isValid = false;
      }
    }
    if(!isEmpty(fromDateString) && !isEmpty(toDateString)) {
      var fromDate = parseDate(fromDateString);
      var toDate = parseDate(toDateString);
      if (fromDate >= toDate) {
        isValid = false;
        errors.push("\"Before\" field should have date > date in the \"After\" field.")
      }
    }

    filter.users = AJS.$('#filter-user-select').select2('data');
    filter.ipAddresses = AJS.$('#filter-ip-address-select').select2('val');
    filter.contentTypes = AJS.$('#filter-content-type').select2('val');
    // .split(',').map(item=>item.trim()).filter(item => item); - used to split content ids by ',',
    // trim it and filter out empty strings.
    filter.contentIds = AJS.$('#filter-content-id').val().split(',')
        .map(item => item.trim()).filter(item => item);
    filter.actions = AJS.$('#filter-action').select2('val');

    if (isValid) {
      update(page);
      updateUiFilterText()
      AJS.dialog2('#filter-dialog').hide();
    } else {
      showFilterErrors(errors);
    }
  }

  function showFilterErrors(errors) {
    if (!AJS.$.isEmptyObject(errors)) {
      AJS.$('#filter-error-message-details').empty();
      var errorDetails = AJS.$('#filter-error-message-details');
      for (var i = 0; i < errors.length; i++) {
        errorDetails.append("<li>" + errors[i] + "</li>");
      }
      AJS.$('#filter-error-message').show();
    }
  }

  function updateUiFilterText() {
    var filterConditionCount = 0;
    if (filter.toTime || filter.fromTime) {
      filterConditionCount++;
    }
    if (filter.users && filter.users.length > 0) {
      filterConditionCount++;
    }
    if (filter.ipAddresses && filter.ipAddresses.length > 0) {
      filterConditionCount++;
    }
    if (filter.contentTypes && filter.contentTypes.length > 0) {
      filterConditionCount++;
    }
    if (filter.contentIds && filter.contentIds.length > 0) {
      filterConditionCount++;
    }
    if (filter.actions && filter.actions.length > 0) {
      filterConditionCount++;
    }
    var filterSummary = filterConditionCount ? " (" + filterConditionCount + ")" : "";
    AJS.$('#filter-summary').text(filterSummary);
  }

  function isEmpty(string) {
    return !string || !string.trim();
  }

  /*
   * returns null if dateString is not valid yyyy-mm-dd
   */
  function parseDate(dateString) {
    if (isNaN(new Date(dateString).valueOf())) {
      return null;
    }
    var parts = dateString.split("-");
    var year = parts[0];
    var month = parts[1] - 1; // months are zero indexed
    var day = parts[2];
    var date = new Date(year, month, day);
    return date;
  }

  function getUrlParameters() {
    var res = 'query=' + encodeURIComponent(query);
    if (sortParams) {
      for (var i = 0; i < sortParams.length; i++) {
        res += "&sortParams=" + sortParams[i].key + "+" + sortParams[i].order;
      }
    }
    if (filter.fromTime) {
      res += '&fromTime=' + encodeURIComponent(filter.fromTime);
    }
    if (filter.toTime) {
      res += '&toTime=' + encodeURIComponent(filter.toTime);
    }
    if (filter.users) {
      for (var i = 0; i < filter.users.length; i++) {
        res += '&users=' + encodeURIComponent(filter.users[i].id);
      }
    }
    if (filter.ipAddresses) {
      for (var i = 0; i < filter.ipAddresses.length; i++) {
        res += '&ipAddresses=' + encodeURIComponent(filter.ipAddresses[i]);
      }
    }
    if (filter.contentTypes) {
      for (var i = 0; i < filter.contentTypes.length; i++) {
        res += '&contentTypes=' + encodeURIComponent(filter.contentTypes[i]);
      }
    }
    if (filter.contentIds) {
      for (var i = 0; i < filter.contentIds.length; i++) {
        res += '&contentIds=' + encodeURIComponent(filter.contentIds[i]);
      }
    }
    if (filter.actions) {
      for (var i = 0; i < filter.actions.length; i++) {
        res += '&actions=' + encodeURIComponent(filter.actions[i]);
      }
    }
    return res;
  }

  function renderSummaryTemplate(context) {
    context.baseUrl = AJS.contextPath();
    context.contentIdViewUri = AJS.contextPath() + context.contentIdViewUri;
    return summaryTemplate(context);
  }

  self.csvExport = function() {
    window.location.assign(AJS.contextPath() + "/plugins/servlet/accesslog/access-log.csv?"
        + getUrlParameters());
  }

  function loadConfig(callback) {
    var url = AJS.contextPath() + '/rest/accesslog/1.0/config.json';
    AJS.$.ajax({
      url : url,
      method : 'GET',
      dataType : "json"
    }).done(callback);
  }

  // Creates HTML element list for events config.
  function fillEnabledContentTypeActionCheckboxes(config) {
    for (const contentType in config.enabledContentTypeActions) {
      for (const action of config.enabledContentTypeActions[contentType]) {
        document.getElementById("checkbox-content-type-" + contentType).checked = true;
        document.getElementById("action-checkbox-" + contentType + "-" + action).checked = true;
      }
    }
  }

  // Mapping events config from HTML element to JSON object, to save it in DB.
  function getEnabledContentTypeActions() {
    let enabledContentTypeActions = {};
    const contentTypeCheckboxes = document.getElementById("content-type-actions-list").querySelectorAll(".content-type-checkbox");
    for (const contentTypeCheckbox of contentTypeCheckboxes) {
      const contentTypeName = contentTypeCheckbox.id.substring(contentTypeCheckbox.id.lastIndexOf('-') + 1);
      const actionCheckboxes = document.getElementById("actions-sublist-" + contentTypeName).querySelectorAll(".action-checkbox");
      const selectedActions = [];
      for (const actionCheckbox of actionCheckboxes) {
        if (actionCheckbox.checked) {
          const actionName = actionCheckbox.id.substring(actionCheckbox.id.lastIndexOf('-') + 1)
          selectedActions.push(actionName);
        }
      }
      enabledContentTypeActions[contentTypeName] = selectedActions;
    }

    const enabledContentTypeActionsObject = {};
    enabledContentTypeActionsObject["enabledContentTypeActions"] = enabledContentTypeActions;
    return JSON.stringify(enabledContentTypeActionsObject);
  }

  function saveConfig(config, callback) {
    var url = AJS.contextPath() + '/rest/accesslog/1.0/config.json';
    AJS.$.ajax({
      url : url,
      headers : {
        'X-Atlassian-Token' : 'no-check'
      },
      type : 'POST',
      data : config
    }).done(function(errors) {
        AJS.$('#config-error-message-details').empty();
        if (!AJS.$.isEmptyObject(errors)) {
          var errorDetails = AJS.$('#config-error-message-details');
          for (var i = 0; i < errors.length; i++) {
              errorDetails.append("<li>" + errors[i] + "</li>");
          }
          AJS.$('#config-error-message').show();
        } else {
          AJS.$('#config-error-message').hide();
          callback();
        }
    });
  }

  function setUpConfig() {
    // Hides the dialog without saving.
    AJS.$(".config-dialog-close-button").click(function(e) {
      e.preventDefault();
      AJS.dialog2("#config-dialog").hide();
    });
    // Saves config and hides the dialog.
    AJS.$("#config-dialog-submit-button").click(function(e) {
      e.preventDefault();
      saveConfig({
        retentionPeriod : AJS.$('#retention-period').val(),
        numberOfRecordsPerPage : AJS.$('#number-of-records-per-page').val(),
        writeToSyslog: AJS.$('#writeToSyslog').is(":checked"),
        syslogHost: AJS.$('#syslogHost').val(),
        syslogFacility: AJS.$('#syslogFacility').val(),
        syslogLevel: AJS.$('#syslogLevel').val(),
        writeToFile: AJS.$('#writeToFile').is(":checked"),
        filePath: AJS.$('#filePath').val(),
        eventsConfigMap: getEnabledContentTypeActions()
      }, function() {
        AJS.dialog2("#config-dialog").hide();
        update(1);
      });
    });

    AJS.$('#writeToSyslog').click(function(){
        AJS.$('.syslogHostDetails').toggle(this.checked);
      });

    AJS.$('#writeToFile').click(function(){
      AJS.$('.filePathDetails').toggle(this.checked);
    });
    // Initializes the dialog.
    AJS.$('#accesslog-config').click(function() {
      // Initialize from current config.
      loadConfig(function(config) {
        AJS.$('#retention-period').val(config.retentionPeriod);
        AJS.$('#number-of-records-per-page').val(config.numberOfRecordsPerPage);
        AJS.$('#writeToSyslog').prop('checked', config.writeToSyslog);
        AJS.$('#syslogHost').val(config.syslogHost);
        AJS.$('#syslogFacility').val(config.syslogFacility);
        AJS.$('#syslogLevel').val(config.syslogLevel);
        if (config.writeToSyslog) {
          AJS.$('.syslogHostDetails').show();
        } else {
          AJS.$('.syslogHostDetails').hide();
        }
        AJS.$('#writeToFile').prop('checked', config.writeToFile);
        AJS.$('#filePath').val(config.filePath);
        if (config.writeToFile) {
          AJS.$('.filePathDetails').show();
        } else {
          AJS.$('.filePathDetails').hide();
        }
        AJS.dialog2('#config-dialog').show();

        AJS.$('#config-error-message-details').empty();
        AJS.$('#config-error-message').hide();

        fillEnabledContentTypeActionCheckboxes(config);

        AJS.$(".action-checkbox").change((e) => {
          const targetActionCheckbox = e.target;
          const actionCheckboxes = targetActionCheckbox.closest(".events-config-actions-list").querySelectorAll("input");
          const parentListId = targetActionCheckbox.closest(".events-config-actions-list").id;
          // Extracts content type name e.g. 'ISSUE' from 'checkbox-content-type-ISSUE'.
          const contentTypeKey = parentListId.substring(parentListId.lastIndexOf('-') + 1)
          const contentTypeCheckbox = document.getElementById("checkbox-content-type-" + contentTypeKey);

          // If parent content type checkbox is unchecked, and user checked some action if this content type, then we need to check parent content type checkbox.
          if (targetActionCheckbox.checked && !contentTypeCheckbox.checked) {
            contentTypeCheckbox.checked = true;
            return;
          }

          // Checking if all actions checkboxes are unchecked.
          if (!targetActionCheckbox.checked) {
            let areAllActionsUnchecked = true;
            for (let actionCheckbox of actionCheckboxes) {
              if (actionCheckbox.checked) {
                areAllActionsUnchecked = false;
                break;
              }
            }

            // Unchecking parent content type checkbox.
            if (areAllActionsUnchecked) {
              contentTypeCheckbox.checked = false;
            }
          }
        });

        // If parent content type checkbox getting checked/unchecked then checking/unchecking all children action checkboxes.
        AJS.$(".content-type-checkbox").change((e) => {
          const checkboxId = e.target.id;
          // Extracts content type name e.g. 'ISSUE' from 'checkbox-content-type-ISSUE'.
          const contentTypeKey = checkboxId.substring(checkboxId.lastIndexOf('-') + 1);
          const contentTypeActionsCheckboxList = document.getElementById("actions-sublist-" + contentTypeKey).querySelectorAll("input");
          const isContentTypeEnabled = e.target.checked;

          for (const contentTypeActionCheckbox of contentTypeActionsCheckboxList) {
            contentTypeActionCheckbox.checked = isContentTypeEnabled;
          }
        });
      });
    });
  }
  return self;
}());
